/*
 * RankListModel.java:  a TextListModel handles Rank
 * for TaxoNote based on Nomencurator
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: RankListModel.java,v 1.1 2002/03/26 03:56:38 nozomi Exp $
 * $Log: RankListModel.java,v $
 * Revision 1.1  2002/03/26 03:56:38  nozomi
 * use RankChoice to handle additional ranks
 *
 */	


package org.nomencurator.editor.model;

import java.util.Observable;
import java.util.Observer;

import jp.kyasu.awt.DefaultTextListModel;

import org.nomencurator.Rank;

/**
 * The TextListModel handles <code>Rank</code>
 *
 * @see 	org.nomencurator.Rank
 *
 * @version 	26 Mar 2002
 * @author 	Nozomi `James' Ytow
 */
public class RankListModel
    extends DefaultTextListModel
    implements Observer
{
    /** Initial contents of <code>RankListModel</code> */
    protected static String[][] ranks = {
	{"Domain"},
	{"Kingdom"},
	{"Phylum"},
	{"Class"},
	{"Order"},
	{"Family"},
	{"Tribe"},
	{"Genus"},
	{"Section"},
	{"Series"},
	{"Species"}, 
	{"Variety"},
	{"Form"},
	{"other..."},
    };

    protected static String[][] others = {{"other..."}};

    public RankListModel()
    {
	super();
	Rank.getHighest().addObserver(this);
	setRankItems();
    }

    protected void setRankItems()
    {
	replaceItems(0, 0, others);
	replaceItems(0, 0, Rank.getRankNameArray());
    }

    public void insert(String newName, String higher, String lower)
    {
	if(newName == null)
	    return;

	if(Rank.get(newName) != null)
	    return;

	Rank newRank = new Rank(newName);
	newRank.setHigher(Rank.get(higher));
	newRank.setLower(Rank.get(lower));

	insert(newRank);
    }

    public void insert(Rank rank)
    {
	if(Rank.get(rank.getName()) != null)
	    return;

	Rank.put(rank);

	int items = getItemCount();

	Rank next = rank.getHigher();
	if(next != null) {
	    int index = getIndex(next);
	    if(index != -1) {
		index++;
		replaceItems(index, index, new String[][]{{rank.getName()}});
		return;
	    }
	}

	next = rank.getLower();
	if(next != null) {
	    int index = getIndex(next);
	    if(index != -1) {
		index--;
		replaceItems(index, index, new String[][]{{rank.getName()}});
		return;
	    }
	}

	items -= 2;
	replaceItems(items, items, new String[][]{{rank.getName()}});
    }

    /**
     * Returns index of <code>rank</code> in this <code>TextListModel</code>
     * or -1 if the <code>rank</code> is not contained.
     *
     * @param rank <code>Rank</code> to be looked for in this list model
     * @return index of given <code>rank</code> or -1 if not found in the list model
     */
    public int getIndex(Rank rank)
    {
	int items = getItemCount();
	for(int i = 0; i  < items; i++) {
	    if(getItem(i, 0).equals(rank.getName()))
		return i;
	}

	return -1;
    }

    public void update(Observable obs, Object arg)
    {
	replaceItems(0, getItemCount() - 1, Rank.getRankNameArray());	
    }
}
