/*
 * HTMLWriter.java
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */
/*
 * $Id: HTMLWriter.java,v 1.6 2002/09/06 17:13:17 nozomi Exp $
 * $Log: HTMLWriter.java,v $
 * Revision 1.6  2002/09/06 17:13:17  nozomi
 * skip Text.LINE_SEPARATOR_CHAR
 *
 * Revision 1.5  2002/09/06 13:56:31  nozomi
 * reintroduce paragraph tag
 *
 * Revision 1.4  2002/09/06 11:46:53  nozomi
 * remove paragraph tags
 *
 * Revision 1.3  2002/09/06 09:40:11  nozomi
 * use <BR></BR> pair to cheat XML parser
 *
 * Revision 1.2  2002/09/05 05:31:48  nozomi
 * support only single paragraph tentatively
 *
 * Revision 1.1  2002/09/04 17:55:43  nozomi
 * introduce pseudo HTML support
 *
 */

package org.nomencurator.graphics.html;

import java.io.IOException;
import java.io.StringWriter;

import java.util.Stack;

import jp.kyasu.graphics.ParagraphStyle;
import jp.kyasu.graphics.Text;
import jp.kyasu.graphics.TextStyle;

import jp.kyasu.graphics.html.HTMLText;

/**
 * A <CODE>jp.kyasu.graphics.html.HTMLWriter<CODE> specialized to
 * convert <CODE>HTMLText</CODE> into a pseudo HTML document,
 * i.e. contents of BODY part
 *
 * @see jp.kyasu.graphics.html.HTMLWriter
 *
 * @version 	06 Sep 2002
 * @author 	Nozomi `James' Ytow
 */
public class HTMLWriter
    extends jp.kyasu.graphics.html.HTMLWriter
{
    /**
     * Constructs a html writer with the specified html text.
     *
     * @param htmlText the html text to be saved.
     */
    public HTMLWriter(HTMLText htmlText)
    {
	super(htmlText);
    }
    
    /** Returns <CODE>String</CODE> containing the HTML body */
    public String bodyToHTMLString()
    {
	int htmlLength = htmlText.length();
	writer = new StringWriter(htmlLength + htmlText.length()/4);

	listStack = new Stack();
	bqLevel = 0;
	currentParagraphStyle = null;
	inPreFormatted = false;

	try {
	    //	    writeHeader();
	    writeHTML();
	    //	    writeFooter();
	    writer.flush();
	}
	catch (IOException e) { }
	String result = writer.toString();
	writer = null;
	return result;
    }

    /**
     * Writes the paragraph in the specified range of the specified text
     * as a HTML with the specified paragraph style.
     *
     * @param begin  the beginning index of the paragraph, inclusive.
     * @param end    the ending index of the paragraph, exclusive.
     * @param text   the text.
     * @param pStyle the paragraph style of the paragraph.
     */
    protected void writeParagraph(int begin, int end, Text text,
				  ParagraphStyle pStyle)
	throws IOException
    {
	TextStyle baseStyle = pStyle.getBaseStyle();
	if (baseStyle == null) {
	    baseStyle = htmlStyle.getDefaultTextStyle();
	}
	write("<P>");
	writeText(begin, end, text, baseStyle);
	write("</P>");
    }

    /**
     * Writes the specified range of the specified text as a HTML data.
     *
     * @param begin     the beginning index of the text, inclusive.
     * @param end       the ending index of the text, exclusive.
     * @param text      the text.
     * @param firstData true if the data written is a first data in the
     *                  paragraph.
     */
    protected void writeData(int begin, int end, Text text, boolean firstData)
	throws IOException
    {
	char lastChar = (firstData ? ' ' : 'a');
	for (int i = begin; i < end; i++) {
	    char c = text.getChar(i);
	    switch (c) {
	    case '<':
		write("&lt;");
		break;
	    case '>':
		write("&gt;");
		break;
	    case '&':
		write("&amp;");
		break;
	    case ' ':
		if (inPreFormatted || lastChar != ' ') {
		    write(' ');
		}
		else {
		    write("&nbsp;");
		    c = 'a'; // lastChar = 'a';
		}
		break;
	    case Text.LINE_SEPARATOR_CHAR:
		break;
	    case Text.LINE_BREAK_CHAR:
		if (inPreFormatted) {
		    writeln();
		}
		else {
		    writeln("<BR></BR>");
		}
		break;
	    case Text.ATTACHMENT_CHAR:
		writeTextAttachment(text.getAttachmentAt(i));
		break;
	    default:
		write(c);
		break;
	    }
	    lastChar = c;
	}
    }

}
