/*
 * IndexManager.java: index data manager for local database system.
 *
 * Copyright (c) 2001, 2002 Ryo FUJIMOTO
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: IndexManager.java,v 1.11 2002/08/14 12:36:30 ryo Exp $
 * $Log: IndexManager.java,v $
 * Revision 1.11  2002/08/14 12:36:30  ryo
 * remove unused line
 *
 * Revision 1.10  2002/07/02 10:42:46  t.okada
 * It changes so that unicode can be used.
 *
 * Revision 1.9  2002/06/07 14:11:15  ryo
 * change the analysis method of an index
 *
 * Revision 1.8  2002/06/04 13:48:39  ryo
 * add path setting function
 *
 * Revision 1.7  2002/05/29 07:14:00  ryo
 * add the file extension to the index file for null data
 *
 * Revision 1.6  2002/03/28 07:13:49  ryo
 * add copyright.
 *
 * Revision 1.5  2002/03/12 17:44:14  nozomi
 * CRLF conversion
 *
 * Revision 1.4  2002/03/12 12:38:08  ryo
 * disable debug print
 *
 * Revision 1.3  2002/03/11 12:02:21  ryo
 * remove convertHexToString()
 *
 * Revision 1.2  2002/03/07 13:23:55  ryo
 * add method: IndexFilenameFilter#convertHexToString()
 *
 * Revision 1.1  2002/03/05 07:47:56  ryo
 * initial import into CVS.
 *
 */

package org.nomencurator.localdb;

import java.io.File;
import java.io.FilenameFilter;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.util.Vector;
import org.nomencurator.util.Util;

class IndexManager
{
	private String _rootDir = null;
	public static  String _fileExtention = ".idx";
	private static String _indexRecordSeparator = "\n";

	/**
	 * Constractor.
	 */
	public IndexManager(String rootDir) {
		_rootDir = rootDir;
		File dir = new File(_rootDir);
		if (!dir.exists())
			dir.mkdirs();
	}
	
	/**
	 * read object id from index file.
	 * @param filename the file name.
	 * @return The object id list read from the file.
	 */
	private Vector readStringFromIndexFile(String filename) {
		Vector strings = new Vector();
		try {
			File file = new File(_rootDir, filename);
			BufferedReader in = new BufferedReader(new InputStreamReader(new FileInputStream(file), "UTF8"));
			String line;
			while ((line = in.readLine()) != null) {
				strings.add(line);
			}
			in.close();
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
		return strings;
	}
	
	/**
	 * write object id to index file.
	 * @param filename the file name.
	 * @param string the contents.
	 * @return true if it succeeds in the writing of a file; false otherwise.
	 */
	public synchronized boolean writeStringToIndexFile(String filename, String string) {
		try {
			File file = new File(_rootDir, filename);
			BufferedWriter out = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file), "UTF8"));
			out.write(string + _indexRecordSeparator);
			out.flush();
			out.close();
			return true;
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}
	}
	
	/**
	 * search object id from index file.
	 * @param searchWord the search word.
	 * @return the object id list.
	 */
	public Vector searchIndexData(String searchWord) {
		Vector indexes = new Vector();
		File dir = new File(_rootDir);
		String[] fileList = dir.list(new IndexFilenameFilter(searchWord));
		for (int i=0; i<fileList.length; i++) {
			Vector v = readStringFromIndexFile(fileList[i]);
			String filename = fileList[i].substring(0, fileList[i].length() - _fileExtention.length());
			for (int j=0; j<v.size(); j++) {
				String line = (String)v.elementAt(j);
				int n = line.indexOf("\t");
				if (n >= 0) {
					String[] columns = new String[4];
					columns[0] = new String(Util.convertHexToString(filename));
					int begin = 0;
					int end = line.indexOf("\t");
					int k = 1;
					while (line.length() > 0) {
						String buffer = line.substring(0, end);
						columns[k] = new String(buffer);
						if (line.length() < end + 1)
							break;
						line = line.substring(end + 1);
						end = line.indexOf("\t");
						if (end == -1)
							end = line.length();
						k++;
					}
					indexes.add(columns);
				}
			}
		}
		return indexes;
	}
	
	/**
	 * add index data.
	 * @param filename the file name.
	 * @param contents the contents.
	 * @return true if it succeeds in the writing of a file; false otherwise.
	 */
	public synchronized boolean addIndexData(String filename, String contents) {
		Vector v = new Vector();
		v.addElement(contents);
		return addIndexData(filename, v);
	}
	
	/**
	 * add index data.
	 * @param filename the file name.
	 * @param contents the contents.
	 * @return true if it succeeds in the writing of a file; false otherwise.
	 */
	public synchronized boolean addIndexData(String filename, Vector contents) {
//		if (filename == null || filename.length() == 0)
//			filename = "_"; // dummy index file name
		try {
			File file = new File(_rootDir, filename + _fileExtention);
//			BufferedWriter out = new BufferedWriter(new FileWriter(file.getPath(), true));
			BufferedWriter out = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file.getPath(), true), "UTF8"));
			for (int i=0; i<contents.size(); i++) {
				out.write(contents.elementAt(i) + _indexRecordSeparator);
			}
			out.flush();
			out.close();
			return true;
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}
	}
	
	public static void main(String[] args) {
		IndexManager im = new IndexManager(".");
		im.searchIndexData(".");
	}
}


class IndexFilenameFilter implements FilenameFilter {
	
	private String _searchWord = null;
	
	/**
	 * Constractor.
	 */
	public IndexFilenameFilter(String searchWord) {
		_searchWord = searchWord;
	}

	/**
	 * Tests if a specified file should be included in a file list.
	 * @param dir the directory in which the file was found.
	 * @param name the name of the file.
	 * @return true if the name should be included in the file list; false otherwise.
	 */
	public boolean accept(File dir, String name) {
		String origName = "";
		name = name.substring(0, name.length() - IndexManager._fileExtention.length());
		if (name.length() != 0)
			origName = org.nomencurator.util.Util.convertHexToString(name);
		if (origName.indexOf(_searchWord) != -1) {
			return true;
		} else {
			return false;
		}
	}

}
