/*
 * LocalDatabase.java: local database system for TaxoNote.
 *
 * Copyright (c) 2001, 2002 Ryo FUJIMOTO
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 *
 * $Id: LocalDatabase.java,v 1.21 2002/09/07 04:05:07 nozomi Exp $
 * $Log: LocalDatabase.java,v $
 * Revision 1.21  2002/09/07 04:05:07  nozomi
 * support partial match
 *
 * Revision 1.20  2002/09/02 09:18:27  ryo
 * change method name from getName to getUsedName
 *
 * Revision 1.19  2002/08/15 05:16:55  ryo
 * change type of year into String
 *
 * Revision 1.18  2002/08/14 12:40:10  ryo
 * change the join algorithm to hash join from nested loop
 *
 * Revision 1.17  2002/07/16 07:10:45  ryo
 * add constructor with no argument
 *
 * Revision 1.16  2002/06/27 13:15:01  ryo
 * make XML_RESPONSE_* public
 *
 * Revision 1.15  2002/06/21 13:03:39  ryo
 * add _top and its accessor
 *
 * Revision 1.14  2002/06/07 14:11:31  ryo
 * add authorListSummary to data and index
 *
 * Revision 1.13  2002/06/04 13:48:39  ryo
 * add path setting function
 *
 * Revision 1.12  2002/05/24 13:12:27  ryo
 * ignore if obj is null in addObjectData()
 *
 * Revision 1.11  2002/05/14 13:37:00  ryo
 * modify specification of addObjectData()
 *
 * Revision 1.10  2002/05/08 11:10:40  ryo
 * remove unused import line
 *
 * Revision 1.9  2002/04/19 22:55:31  nozomi
 * method name change from getAppearanceList to getAppearances
 *
 * Revision 1.8  2002/04/19 21:46:27  nozomi
 * use Enumeration istead of for
 *
 * Revision 1.7  2002/04/16 03:53:38  nozomi
 * migration to NameUsage from NameRecord
 *
 * Revision 1.6  2002/04/08 01:52:23  nozomi
 * minor edtition in source code style
 *
 * Revision 1.5  2002/03/28 07:13:49  ryo
 * add copyright.
 *
 * Revision 1.4  2002/03/12 17:44:14  nozomi
 * CRLF conversion
 *
 * Revision 1.3  2002/03/11 12:03:04  ryo
 * remove convertStringToHex()
 *
 * Revision 1.2  2002/03/07 13:24:34  ryo
 * add method: convertStringToHex()
 *
 * Revision 1.1  2002/03/05 07:47:56  ryo
 * initial import into CVS.
 *
 */

package org.nomencurator.localdb;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import org.nomencurator.Appearance;
import org.nomencurator.Author;
import org.nomencurator.NamedObject;
import org.nomencurator.NameUsage;
import org.nomencurator.Publication;

import org.nomencurator.util.Util;

public class LocalDatabase
{

	private String _top;
	private DataManager _dataManager;
	private IndexManager _nameIndexManager;
	private IndexManager _authorIndexManager;
	private IndexManager _yearIndexManager;
	public final static String XML_RESPONSE_HEADER = "<?xml version=\"1.0\"?>\n"
		+ "<!DOCTYPE NOM-XML>\n"
		+ "<response status=\"0\" text=\"OK\">\n";
	public final static String XML_RESPONSE_FOOTER = "</response>";

	/** hash table size which used in hash join */
	private static int HASHSIZE = 1009;

	/**
	 * Constructor
	 */
	public LocalDatabase() {
	}

	/**
	 * Constructor
	 */
	public LocalDatabase(String topDirectory) {
		setTopDirectory(topDirectory);
	}

	/**
	 * Set top directory for local database.
	 */
	public void setTopDirectory(String topDirectory) {
		_top = topDirectory;
		String sep = System.getProperty("file.separator");
		_dataManager = new DataManager(topDirectory + sep + "data");
		_nameIndexManager = new IndexManager(topDirectory + sep + "index" + sep + "name");
		_authorIndexManager = new IndexManager(topDirectory + sep + "index" + sep + "author");
		_yearIndexManager = new IndexManager(topDirectory + sep + "index" + sep + "year");
	}
	
	/**
	 * Get top directory for local database.
	 */
	public String getTopDirectory() {
		return _top;
	}

    /**
     * Add data and index to local database.
     * @param v NamedObject list.
     * @return inserted row count.
     */
    public int addObjectData(Vector v) {
	int insertedRows = 0;
	String xml = "";
	for (int i=0; i<v.size(); i++) {
	    NamedObject obj = (NamedObject)v.elementAt(i);
	    if (obj == null) continue;
	    xml = obj.toXMLString();
	    if (obj instanceof NameUsage) {
				// save link between NameUsage and Appearance
		Appearance appearance = ((NameUsage)obj).getAppearance();
		if(appearance == null)
		    appearance = new Appearance();
		_nameIndexManager.addIndexData(
					       Util.convertStringToHex(((NameUsage)obj).getUsedName()),
					       obj.getPersistentID() + "\t" + 
					       appearance.getPersistentID());
	    } else if (obj instanceof Publication) {
				// save link between Author and Publication
		Vector authors = ((Publication)obj).getAuthors();
		if (authors != null) {
		    for (int j=0; j<authors.size(); j++) {
			_authorIndexManager.addIndexData(
							 Util.convertStringToHex(((Author)authors.elementAt(j)).getSurname()),
							 ((Author)authors.elementAt(j)).getPersistentID() + "\t" +
							 obj.getPersistentID());
		    }
		}
	    } else if (obj instanceof Appearance) {
				// save link between Publication and Appearance
		Publication publication = ((Appearance)obj).getPublication();
		_yearIndexManager.addIndexData(
					       Util.convertStringToHex(publication.getYear()),
					       publication.getPersistentID() + "\t" +
					       obj.getPersistentID() + "\t" +
					       publication.getAuthorListSummary());
	    }
	    _dataManager.addObjectData(Util.convertStringToHex(obj.getPersistentID()), xml);
	    insertedRows++;
	}
	return insertedRows;
    }
    
    
    
    /**
     * search index data.
     * @param name the search word of name in NameUsage object.
     * @param author the search word of author in Author object.
     * @param year the search word of year in Publication object.
     * @return the NameUsage list for displaying on a search pane. (XML format)
     */
    public String search(String name, String author, String year) {
	//long n = System.currentTimeMillis();
	// get index data
	Vector nameVector /* name, oid, appearance */ = _nameIndexManager.searchIndexData(name);
	Vector authorVector /* author, oid, publication(s) */ = _authorIndexManager.searchIndexData(author);
	Vector yearVector /* year, oid, appearance(s) */ = _yearIndexManager.searchIndexData(year);
	//n = System.currentTimeMillis() - n;
	//System.out.println(n);
	//n = System.currentTimeMillis();
	// create hash
	Vector[] nameAppHash = new Vector[HASHSIZE];
	for (Enumeration e = nameVector.elements(); e.hasMoreElements(); ) {
	    String[] elem = (String[])e.nextElement();
	    int index = hash(elem[2]); // appearance oid
	    if (nameAppHash[index] == null)
		nameAppHash[index] = new Vector();
	    nameAppHash[index].addElement(elem);
	}
	Vector[] authorPubHash = new Vector[HASHSIZE];
	for (Enumeration e = authorVector.elements(); e.hasMoreElements(); ) {
	    String[] elem = (String[])e.nextElement();
	    int index = hash(elem[2]); // publication oid
	    if (authorPubHash[index] == null)
		authorPubHash[index] = new Vector();
	    authorPubHash[index].addElement(elem);
	}
	//n = System.currentTimeMillis() - n;
	//System.out.println(n);
	//n = System.currentTimeMillis();
	// hash join
	Hashtable hash = new Hashtable();
	//System.out.println("name size : " + nameVector.size());
	//System.out.println("author size : " + authorVector.size());
	//System.out.println("year size : " + yearVector.size());
	for (int i=0; i<yearVector.size(); i++) {
	    //if ((i % 100) == 0)
	    //	System.out.println("year : " + i);
	    // get pointer to appearance from Publication
	    String pub_str = ((String[])yearVector.elementAt(i))[0];		// year
	    String pub_oid = ((String[])yearVector.elementAt(i))[1];		// oid
	    String pub_appoid = ((String[])yearVector.elementAt(i))[2];		// appearance
	    String pub_authors = ((String[])yearVector.elementAt(i))[3];	// authors string
	    authorVector = authorPubHash[hash(pub_oid)];
	    if (authorVector == null)
		continue;
	    for (int j=0; j<authorVector.size(); j++) {
		String author_str = ((String[])authorVector.elementAt(j))[0];	// author
		String author_oid = ((String[])authorVector.elementAt(j))[1];	// oid
		String author_puboid = ((String[])authorVector.elementAt(j))[2];	// publication
		if (pub_oid.equals(author_puboid)) {
		    nameVector = nameAppHash[hash(pub_appoid)];
		    if (nameVector == null)
			continue;
		    for (int k=0; k<nameVector.size(); k++) {
			String name_str = ((String[])nameVector.elementAt(k))[0];	// name
			String name_oid = ((String[])nameVector.elementAt(k))[1];	// oid
			String name_appoid = ((String[])nameVector.elementAt(k))[2];	// appearance
			if (pub_appoid.equals(name_appoid)) {
			    hash.put(name_oid, new String[]{
				name_str,
				(pub_authors != null) ? pub_authors : author_str,
				pub_str});
			}
		    }
		}
	    }
	}
	//n = System.currentTimeMillis() - n;
	//System.out.println(n);
	//n = System.currentTimeMillis();
	StringBuffer xml = new StringBuffer();
	for (Enumeration e = hash.keys(); e.hasMoreElements();) {
	    String name_oid = (String)e.nextElement();
	    String[] strings = (String[])hash.get(name_oid);
	    xml.append("<item>\n<oid>");
	    xml.append(name_oid);
	    xml.append("</oid>\n<name>");
	    xml.append(strings[0]); // name string
	    xml.append("</name>\n<author>");
	    xml.append(strings[1]); // author string
	    xml.append("</author>\n<year>");
	    xml.append(strings[2]); // publication string
	    xml.append("</year>\n</item>\n");
	}
	//n = System.currentTimeMillis() - n;
	//System.out.println(n);
	return addResponseTag(xml.toString());
    }
    
    /**
     * Gets a hash value
     * @param key a hash key
     * @return a hash value
     */
    private int hash(String key) {
	int n = key.length();
	int val = 0;
	for (int i=0; i<n; i++) {
	    val = ((val << 8) + key.charAt(i)) % HASHSIZE;
	}
	return val;
    }
    
//	public String search(Vector nameVector, Vector authorVector, Vector yearVector) {
//		int i, j, k;
//		Hashtable hash = new Hashtable();
//		for (i=0; i<yearVector.size(); i++) {
//			// get pointer to appearance from Publication
//			String pub_str = ((String[])yearVector.elementAt(i))[0];		// year
//			String pub_oid = ((String[])yearVector.elementAt(i))[1];		// oid
//			String pub_appoid = ((String[])yearVector.elementAt(i))[2];		// appearance
//			String pub_authors = ((String[])yearVector.elementAt(i))[3];	// authors string
//			for (j=0; j<authorVector.size(); j++) {
//				String author_str = ((String[])authorVector.elementAt(j))[0];	// author
//				String author_oid = ((String[])authorVector.elementAt(j))[1];	// oid
//				String author_puboid = ((String[])authorVector.elementAt(j))[2];	// publication
//				if (pub_oid.equals(author_puboid)) {
//					for (k=0; k<nameVector.size(); k++) {
//						String name_str = ((String[])nameVector.elementAt(k))[0];	// name
//						String name_oid = ((String[])nameVector.elementAt(k))[1];	// oid
//						String name_appoid = ((String[])nameVector.elementAt(k))[2];	// appearance
//						if (pub_appoid.equals(name_appoid)) {
////							hash.put(name_oid, new String[]{name_str, pub_oid, pub_str});
//							hash.put(name_oid, new String[]{
//								name_str,
//								(pub_authors != null) ? pub_authors : author_str,
//								pub_str});
//						}
//					}
//				}
//			}
//		}
//		StringBuffer xml = new StringBuffer();
//		for (Enumeration e = hash.keys(); e.hasMoreElements();) {
//			String name_oid = (String)e.nextElement();
//			String[] strings = (String[])hash.get(name_oid);
//
////			Publication publication = (Publication)
////			String authors
//
//			xml.append("<item>\n<oid>");
//			xml.append(name_oid);
//			xml.append("</oid>\n<name>");
//			xml.append(strings[0]); // name string
//			xml.append("</name>\n<author>");
////			xml.append(authors); // authors string
//			xml.append(strings[1]); // author string
//			xml.append("</author>\n<year>");
//			xml.append(strings[2]); // publication string
//			xml.append("</year>\n</item>\n");
//		}
//		return addResponseTag(xml.toString());
//	}

	/**
	 * get object data.
	 * @param objectID the object id.
	 * @return the object data. (XML format)
	 */
	public String getObjectData(String objectID) {
		if (objectID == null || objectID.length() == 0) return null;
		Hashtable hash = new Hashtable();
		hash.put(Util.convertStringToHex(objectID), new Integer(0));
		return getObjectData(hash);
	}

	/**
	 * get object data.
	 * @param objectIDs the object id list.
	 * @return the object data. (XML format)
	 */
	public String getObjectData(Hashtable objectIDs) {
		String s = getObjectDataWithoutResponseTag(objectIDs);
		return addResponseTag(s);
	}

	/**
	 * get object data without response tag.
	 * @param objectID the object id.
	 * @return the object data. (XML format)
	 */
	public String getObjectDataWithoutResponseTag(String objectID) {
		if (objectID == null || objectID.length() == 0) return null;
		Hashtable hash = new Hashtable();
		hash.put(Util.convertStringToHex(objectID), new Integer(0));
		return getObjectDataWithoutResponseTag(hash);
	}

	/**
	 * get object data without response tag.
	 * @param objectIDs the object id list.
	 * @return the object data. (XML format)
	 */
	public String getObjectDataWithoutResponseTag(Hashtable objectIDs) {
		Enumeration keys = objectIDs.keys();
		StringBuffer s = new StringBuffer();
		while (keys.hasMoreElements()) {
			s.append(_dataManager.getObjectData((String)keys.nextElement()));
		}
		return s.toString();
	}

	/**
	 * add response tag.
	 * @param xml XML data.
	 * @return XML data with response tag.
	 */
	public String addResponseTag(String xml) {
		return XML_RESPONSE_HEADER + xml + XML_RESPONSE_FOOTER;
	}

	/**
	 * add response tag.
	 * @param xml XML data.
	 * @return XML data with response tag.
	 */
	public StringBuffer addResponseTag(StringBuffer xml) {
		StringBuffer s = new StringBuffer(XML_RESPONSE_HEADER);
		s.append(xml);
		s.append(XML_RESPONSE_FOOTER);
		return s;
	}

}
