/*
 * ObjectPool.java: 
 *
 * Copyright (c) 2002 t.okada, Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: ObjectPool.java,v 1.1 2002/11/20 05:15:37 nozomi Exp $
 * $Log: ObjectPool.java,v $
 * Revision 1.1  2002/11/20 05:15:37  nozomi
 * ported from org.nomencurator.broker with enhancements
 *
 */

package org.nomencurator.util;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.NoSuchElementException;
import java.util.Vector;

import org.nomencurator.Nameable;

/**
 * <code>ObjectPool</code> provides a pool of <code>Object</code>s
 *
 * @version 	20 Nov 2002
 * @author 	t.okada
 * @author 	Nozomi `James' Ytow
 */
public class ObjectPool
    extends Hashtable
{

    public static final int MAX_SIZE = 65535;

    /** pool size */
    protected int capacity;
	
    private ObjectList  list;
	

    /** Constructs default size pool */
    public ObjectPool()
    {
	this(MAX_SIZE);
    }

    /**
     * Constructs an <code>ObjectPool</code>
     * with specified <code>size</code>
     *
     * @param size int specifing size of this <code>ObjectPool</code>
     */
    public ObjectPool(int initialCapacity)
    {
	super(initialCapacity);
	capacity = initialCapacity;
	list = new ObjectList();
    }
	
    /**
     * Sets <code>size</code> as size of this <code>ObjectPool</code>
     *
     * @param size int specifing size to be set
     */
    public void setCapacity(int capacity)
    {
	this.capacity = capacity;
    }
    
    /**
     * Returns size of the <code>ObjectPool</code>
     *
     * @return int representing size of this <code>ObjectPool</code>
     */
    public int getCapacity()
    {
	return capacity;
    }

    /**
     * Maps <CODE>key</CODE> to <CODE>value</CODE> in this <CODE>ObjectPool</CODE>.
     * Neither <CODE>key</CODE> nor <CODE>value</CODE> may be null.
     * The <CODE>value</CODE> is available by calling <CODE>get()</CODE> method
     * with specifying <CODE>key</CODE> same to original <CODE>key</CODE>.
     * 
     * @param key <CODE>Hashtable</CODE> key
     * @param value <CODE>Object</CODE> to be pooled
     *
     * @return previous value mapped to specified <CODE>key</CODE> or null
     * if unavailable.
     *
     * @exception NullPointerException if <CODE>key</CODE> or <CODE>value</CODE> is null
     *
     * @see Object.equals(Object)
     * @see get(Object)
     */
    public Object put(Nameable name)
    {
	System.err.println("put(N:" + name.getName() + ")");
	return put(name.getName(), name);
    }
	

    /**
     * Maps <CODE>key</CODE> to <CODE>value</CODE> in this <CODE>ObjectPool</CODE>.
     * Neither <CODE>key</CODE> nor <CODE>value</CODE> may be null.
     * The <CODE>value</CODE> is available by calling <CODE>get()</CODE> method
     * with specifying <CODE>key</CODE> same to original <CODE>key</CODE>.
     * 
     * @param key <CODE>Hashtable</CODE> key
     * @param value <CODE>Object</CODE> to be pooled
     *
     * @return previous value mapped to specified <CODE>key</CODE> or null
     * if unavailable.
     *
     * @exception NullPointerException if <CODE>key</CODE> or <CODE>value</CODE> is null
     *
     * @see Object.equals(Object)
     * @see get(Object)
     */
    public Object put(Object key, Object value)
    {
	if(capacity == 0)
	    return null;

	if(key == null ||
	   value == null)
	    throw new NullPointerException();

	System.err.println("put(O, O:" + key + ")");
	new Error(value.getClass().toString()).printStackTrace();
	Object previous = super.get(key);
	if(previous != null) {
	    previous = ((ObjectNode)previous).getObject();
	    if(value == previous)
		return previous;
	}

	ObjectNode objectNode = list.addFirst(key, value);
	if(list.size() >= capacity) {
	    ObjectNode toBeRemoved = list.removeLastNode();
	    if(toBeRemoved != null) {
		super.remove(toBeRemoved.getKey());
	    }
	}

	super.put(key, objectNode);

	return previous;
    }

    /**
     * Returns value mapped to specified <CODE>key</CODE> in this <CODE>ObjectPool</CODE>.
     *
     * @param key key in the <CODE>ObjectPool</CODE>.
     *
     * @return Object value mapped to the <CODE>key</CODE> in this <CODE>ObjectPool</CODE>
     * or  null if no value is mapped to the <CODE>key</CODE>
     *
     * @see put(Object, Object)
     */
    public Object get(Object key)
    {
	if(capacity == 0)
	    return null;

	if(key == null)
	    throw new NullPointerException();

	ObjectNode objectNode = (ObjectNode)super.get(key);
	if(objectNode == null)
	    return null;

	Object value = objectNode.getObject();
		
	if(value != null) {
	    list.remove(objectNode);
	    super.put(key, list.addFirst(key, value));
	    System.err.println("get:" + value.getClass());
	}
	return value;
    }

    /**
     * Removes <CODE>key</CODE> (and corresponding value) from this <CODE>ObjectPool</CODE>.
     * It does nothing if the <CODE>key</CODE> isn't in this <CODE>ObjectPool</CODE>.
     *
     * @param key key to remove
     *
     * @return Object mapped to the <CODE>key</CODE> in this <CODE>ObjectPool</CODE>,
     * or null if no value is mapped to the <CODE>key</CODE>
     */
    public Object remove(Object key)
    {
	Object node = super.remove(key);
	if(node != null) {
	    list.remove((ObjectNode)node);
	    node = ((ObjectNode)node).getObject();
	}
	return node;
    }

    /**
     * Returns a list of keys in this <CODE>ObjectPool</CODE>
     * as an <CODE>Enumeration</CODE>
     *
     * @return Enumeration of keys in this <CODE>ObjectPool</CODE>
     *
     * @see java.util.Enumeration
     * @see elements()
     */
    public Enumeration keys()
    {
	return new KeyEnumeration(list.getFirstNode());
    }

    /**
     * Returns list of values in this <CODE>ObjectPool</CODE>
     * Use methods of <CODE>Enumeration</CODE> to get values sequencially.
     *
     * @return Enumeration of values in this <CODE>ObjectPool</CODE>
     *
     * @see java.util.Enumeration
     * @see keys()
     */
    public Enumeration elements()
    {
	return new ObjectEnumeration(list.getFirstNode());
    }
    


/*	
	public static void main(String [] args) {

		ObjectPool op = new ObjectPool();
		
		org.nomencurator.NameUsage nu = new org.nomencurator.NameUsage();
		Long t1 = new Long(System.currentTimeMillis());
		System.out.println(t1.toString() + "  START!!");
		
		for(int i = 0; i < 100000; i++) {
			op.addObject(Integer.toString(i), nu);
		}

		Long t15 = new Long(System.currentTimeMillis());
		System.out.println(t15.toString() + "  FLAG!!");

		for(int i = 0; i < 10000; i++) {
			org.nomencurator.NameUsage nu2 = 
			(org.nomencurator.NameUsage)op.getObject(Integer.toString(i));
		}
		
		Long t2 = new Long(System.currentTimeMillis());
		System.out.println(t2.toString() + "  END!!");
		
	}
*/	
}

/**
 * A linked list of <CODE>ObjectNode</CODE> used in <CODE>ObjectPool</CODE>
 *
 * @version 	20 Nov 2002
 * @author 	t.okada
 * @author 	Nozomi `James' Ytow
 */
class ObjectList
{
    /** The first node in the list */
    private ObjectNode first;

    /** The last node of the list */
    private ObjectNode last;

    /** Number of nodes in the list */
    private int size;
    
    /**
     * Constructs an empty <CODE>ObjectList</CODE>
     */
    public ObjectList()
    {
	first = null;
	last = null;
	size = 0;
    }
	
    /**
     * Add <CODE>object</CODE> to the list
     *
     * @param object <CODE>Nameable</CODE> to be added to the list
     */
    public ObjectNode add(Nameable object)
    {
	return add(object.getName(), object);
    }
	
    /**
     * Add <CODE>object</CODE> to the list
     *
     * @param object <CODE>Object</CODE> to be added to the list
     */
    public ObjectNode add(Object object)
    {
	return add(object, object);
    }
	
    /**
     * Add <CODE>object</CODE> to the list with <COCE>key</CODE>
     *
     * @param object <CODE>Object</CODE> to be added to the list
     * @param key <CODE>Object</CODE> to be used to access to the <CODE>object</CODE>
     */
    public ObjectNode add(Object key, Object object)
    {
	ObjectNode objectNode = new ObjectNode(object, key);
	addNode(objectNode);
	return objectNode;
    }
	
    /**
     * Insert <CODE>object</CODE> to the top of the list
     *
     * @param object <CODE>Nameable</CODE> to be added to the list
     */
    public ObjectNode addFirst(Nameable object)
    {
	return addFirst(object.getName(), object);
    }

    /**
     * Insert <CODE>object</CODE> to the top of the list
     *
     * @param object <CODE>Object</CODE> to be added to the list
     */
    public ObjectNode addFirst(Object object)
    {
	return addFirst(object, object);
    }

    /**
     * Insert <CODE>object</CODE> to the top of the list
     *
     * @param object <CODE>Object</CODE> to be added to the list
     * @param key <CODE>Object</CODE> to be used to access to the <CODE>object</CODE>
     */
    public ObjectNode addFirst(Object key, Object object)
    {
	ObjectNode objectNode = new ObjectNode(key, object);
	addNodeFirst(objectNode);
	return objectNode;
    }


    /**
     * Append <CODE>objectNode</CODE> at end of the list
     *
     * @param objectNode <CODE>ObjectNode</CODE> to be appended
     */
    public void addNode(ObjectNode objectNode)
    {
	if(last == null) {
	    objectNode.setPreviousNode(null);
	    objectNode.setNextNode(null);
	    first = objectNode;
	    last = objectNode;
	} else {
	    last.setNextNode(objectNode);
	    objectNode.setPreviousNode(last);
	    last = objectNode;
	}
	size++;
    }
    
    /**
     * Insert <CODE>objectNode</CODE> to the top of the list
     *
     * @param objectNode <CODE>ObjectNode</CODE> to be inserted
     */
    public void addNodeFirst(ObjectNode objectNode)
    {
	objectNode.setPreviousNode(null);
	objectNode.setNextNode(first);
	if(first != null)
	    first.setPreviousNode(objectNode);
	
	first = objectNode;
	
	size++;
    }
    
    /**
     * Returns <CODE>ObjectNode</CODE> at top of the list,
     * or null if the list is empty.
     *
     * @return ObjectNode at top of the list,
     * or null if the list is empty.
     */
    public ObjectNode getFirstNode()
    {
	return first;
    }
    
    /**
     * Returns <CODE>ObjectNode</CODE> at end of the list,
     * or null if the list is empty.
     *
     * @return ObjectNode at end of the list,
     * or null if the list is empty.
     */
    public ObjectNode getLastNode()
    {
	return last;
    }
    
    /**
     * Removes <CODE>objectNode</CODE> from the list.
     *
     * @param objectNode <CODE>ObjectNode</CODE> to be removed from the list.
     */
    public void remove(ObjectNode objectNode)
    {
	if(objectNode == null)
	    return;
	
	ObjectNode nextObjectNode = objectNode.getNextNode();
	ObjectNode prevObjectNode = objectNode.getPreviousNode();
	if(prevObjectNode != null)
	    prevObjectNode.setNextNode(nextObjectNode);
	if(nextObjectNode != null)
	    nextObjectNode.setPreviousNode(prevObjectNode);
	size--;
    }
    
    /**
     * Removes last <CODE>ObjectNode</CODE> from the list
     *
     * @return ObjectNode removed for the list, or null
     * if no <CODE>ObjectNode</CODE> was removed
     */
    public ObjectNode removeLastNode()
    {
	if(last == null || last.getPreviousNode() == null)
	    return null;

	ObjectNode toBeRemoved = last;
	
	last.getPreviousNode().setNextNode(null);
	last = last.getPreviousNode();
	
	size--;
	
	return toBeRemoved;
    }
    
    /**
     * Returns number of <CODE>ObjectNode</CODE>s in the list
     */
    public int size()
    {
	return size;
    }

    /**
     * Returns <CODE>Enumeration</CODE> to iterate <CODE>Object</CODE>s
     * in the list.
     * <P>
     * Note: <CODE>nextElement()</CODE> method of the returned <CODE>Enumeration</CODE>
     * returns <CODE>Object</CODE> but not <CODE>ObjectNode</CODE> wrapping it.
     *
     * @return <CODE>Enumeration</CODE> to iterate <CODE>Object</CODE>s
     */
    public Enumeration elements()
    {
	return new ObjectEnumeration(first);
    }

    /**
     * Returns <CODE>Enumeration</CODE> to iterate <CODE>Object</CODE>s
     * in the list.
     * <P>
     * Note: <CODE>nextElement()</CODE> method of the returned <CODE>Enumeration</CODE>
     * returns <CODE>Object</CODE> but not <CODE>ObjectNode</CODE> wrapping it.
     *
     * @return <CODE>Enumeration</CODE> to iterate <CODE>Object</CODE>s
     */
    public Enumeration keys()
    {
	return new KeyEnumeration(first);
    }

    /**
     * Returns <CODE>Vector</CODE> holding <CODE>Object</CODE>s
     * in this <CODE>ObjectPool</CODE>.
     *
     * @return Vector holding <CODE>Object</CODE>s in this <CODE>ObjectPool</CODE>
     */
    public Vector toVector()
    {
	Vector list = new Vector();
	ObjectNode node = first;

	if(node != null) {
	    list.add(node.getObject());
	    node = node.getNextNode();
	}

	return list;
    }	
}

/**
 * An implemantaiton of <CODE>Enumearation</CODE>
 * to iterate <CODE>Object</CODE>s in the <CODE>ObjectPool</CODE>
 *
 * @version 	20 Nov 2002
 * @author 	Nozomi `James' Ytow
 */
class ObjectNodeEnumeration
    implements Enumeration
{
    /** The next node to be returned by <CODE>nextElement</CODE> method */
    ObjectNode node;

    /**
     * Constructs an <CODE>Enumeration</CODE> handles a list
     * starts with <CODE>node</CODE>
     *
     * @param node the first <CODE>ObjectNode</CODE> in tht list
     */
    public ObjectNodeEnumeration(ObjectNode node)
    {
	this.node = node;
    }

    /**
     * Evaluate whether there is another element in this <CODE>Enumeration</CODE>
     *
     * @return true if there is one or more elements in this <CODE>Enumeartion</CODE>
     * or false
     */
    public boolean hasMoreElements()
    {
	return (node != null && node.getNextNode() != null);
    }

    /**
     * Returns next element if one or more elements remain in this <CODE>Enumeration</CODE>
     *
     * @return next element in this <CDOE>Enumerationr</CODE>
     *
     * @exception java.util.NoSuchElementException if no more elements in the <CODE>Enumeration</CODE>
     */
    public Object nextElement()
    {
	if(!hasMoreElements())
	    throw new NoSuchElementException();
	ObjectNode current = node;
	node = node.getNextNode();
	return current;
    }

}

/**
 * An implemantaiton of <CODE>Enumearation</CODE>
 * to iterate <CODE>Object</CODE>s in the <CODE>ObjectPool</CODE>
 *
 * @version 	20 Nov 2002
 * @author 	Nozomi `James' Ytow
 */
class ObjectEnumeration
    extends ObjectNodeEnumeration
{

    /**
     * Constructs an <CODE>Enumeration</CODE> handles a list
     * starts with <CODE>node</CODE>
     *
     * @param node the first <CODE>ObjectNode</CODE> in tht list
     */
    public ObjectEnumeration(ObjectNode node)
    {
	super(node);
    }

    /**
     * Returns next element if one or more elements remain in this <CODE>Enumeration</CODE>
     *
     * @return next element in this <CDOE>Enumerationr</CODE>
     *
     * @exception java.util.NoSuchElementException if no more elements in the <CODE>Enumeration</CODE>
     */
    public Object nextElement()
    {
	return ((ObjectNode)nextElement()).getObject();
    }

}

/**
 * An implemantaiton of <CODE>Enumearation</CODE>
 * to iterate keys in the <CODE>ObjectPool</CODE>
 *
 * @version 	20 Nov 2002
 * @author 	Nozomi `James' Ytow
 */
class KeyEnumeration
    extends ObjectNodeEnumeration
{

    /**
     * Constructs an <CODE>Enumeration</CODE> handles a list
     * starts with <CODE>node</CODE>
     *
     * @param node the first <CODE>ObjectNode</CODE> in tht list
     */
    public KeyEnumeration(ObjectNode node)
    {
	super(node);
    }

    /**
     * Returns next element if one or more elements remain in this <CODE>Enumeration</CODE>
     *
     * @return next element in this <CDOE>Enumerationr</CODE>
     *
     * @exception java.util.NoSuchElementException if no more elements in the <CODE>Enumeration</CODE>
     */
    public Object nextElement()
    {
	return ((ObjectNode)nextElement()).getKey();
    }

}



/**
 * An implemantaiton of <CODE>Enumearation</CODE>
 * to iterate <CODE>Object</CODE>s in the <CODE>ObjectPool</CODE>
 * @version 	20 Nov 2002
 * @author 	t.okada
 * @author 	Nozomi `James' Ytow
 */
class ObjectNode
{
    /** The previous node in the linked list */
    private ObjectNode previousNode;

    /** The next node in the linked list */
    private ObjectNode nextNode;

    /** The <CODE>Object</CODE> entity wrapped by this node */
    private Object object;

    /** The key to the entity <CODE>Object</CODE> */
    private Object key;
	

    /**
     * Construct a <CODE>ObjectNode</CODE>
     * wrapping <CODE>object</CODE>
     *
     * @param object <CODE>Nameable</CODE> to be proxied by this
     * <CODE>ObjectNode</CODE>
     */
    public ObjectNode(Nameable object)
    {
	this(object.getName(), object);
    }
	
    /**
     * Construct a <CODE>ObjectNode</CODE>
     * wrapping <CODE>object</CODE>
     *
     * @param object <CODE>Object</CODE> to be proxied by this
     * <CODE>ObjectNode</CODE>
     */
    public ObjectNode(Object object)
    {
	this(object, object);
    }
	
    /**
     * Construct a <CODE>ObjectNode</CODE>
     * wrapping <CODE>object</CODE>
     *
     * @param key <CODE>Object</CODE> to be used as key to 
     * the proxied <CODE>Object</CODE>
     * @param object <CODE>Object</CODE> to be proxied by this
     * <CODE>ObjectNode</CODE>
     */
    public ObjectNode(Object key, Object object)
    {
	this(null, null, key, object);
    }
	
    /**
     * Construct a <CODE>ObjectNode</CODE>
     * wrapping <CODE>object</CODE>
     *
     * @param previous previous <CODE>ObjectNode</CODE> on the linked list
     * @param next next <CODE>ObjectNode</CODE> on the linked list
     * @param key <CODE>Object</CODE> to be used as key to 
     * the proxied <CODE>Object</CODE>
     * @param object <CODE>Object</CODE> to be proxied by this
     * <CODE>ObjectNode</CODE>
     */
    public ObjectNode(ObjectNode previous,
		      ObjectNode next,
		      Object key,
		      Object object)
    {
	this.previousNode = previous;
	this.nextNode     = next;
	this.key      = key;
	this.object   = object;
    }

    /**
     * Sets <CODE>previous</CODE> as previous node to this
     * <CODE>ObjectNode</CODE>
     *
     * @param previous an <CODE>ObjectNode</CODE> to be set as previous node
     * of this <CODE>ObjectNode</CODE>, or null to make this <CODE>ObjectNode</CODE>
     * as the first node of the linked list
     *
     * @return ObjectNode prior to this <CODE>ObjectNode</CODE> before
     * setting of <CODE>previous</CODE> by this method, 
     * or null if this <CODE>ObjectNode</CODE>
     * was the first node.
     */
    public ObjectNode setPreviousNode(ObjectNode previous)
    {
	ObjectNode oldNode = previousNode;
	previousNode = previous;
	return oldNode;
    }

    /**
     * Sets <CODE>next</CODE> as next node to this
     * <CODE>ObjectNode</CODE>
     *
     * @param next an <CODE>ObjectNode</CODE> to be set as the next node
     * of this <CODE>ObjectNode</CODE>, or null to make this <CODE>ObjectNode</CODE>
     * as the last node of the linked list
     *
     * @return ObjectNode next to this <CODE>ObjectNode</CODE> before
     * setting of <CODE>next</CODE> by this method, 
     * or null if this <CODE>ObjectNode</CODE>
     * was the last node in the linked list.
     */
    public ObjectNode setNextNode(ObjectNode next)
    {
	ObjectNode oldNode = nextNode;
	nextNode = next;
	return oldNode;
    }

    /**
     * Sets <CODE>object</CODE> as the <CODE>Object</CODE> proxied by 
     * this <CODE>ObjectNode</CODE>.
     *
     * @param object <CODE>Object</CODE> to be proxied by this 
     * <CODE>ObjectNode</CODE>.
     *
     * @return Object proxied by this <CODE>ObjectNode</CODE> before
     * this method was called.
     */
    public Object setObject(Object object)
    {
	Object previousObject = this.object;
	this.object = object;
	return previousObject;
    }

    /**
     * Sets <CODE>key</CODE> as the key to <CODE>Object</CODE> proxied by 
     * this <CODE>ObjectNode</CODE>.
     *
     * @param key key to <CODE>Object</CODE> proxied by this 
     * <CODE>ObjectNode</CODE>.
     *
     * @return Object representing a key to the <CODE>Object</CODE> 
     * proxied by this <CODE>ObjectNode</CODE> before
     * this method was called.
     */
    public Object setKey(Object key)
    {
	Object previousKey = this.key;
	this.key = key;
	return previousKey;
    }

    /**
     * Returns <CODE>ObjectNode</CODE> prior to this 
     * <CODE>ObjectNode</CODE> on the linked list,
     * or null if this <CODE>ObjectNode</CODE> is the
     * first node of the linked list,
     * 
     * @return ObjectNode prior to this <CODE>ObjectNode</CODE>
     * or null if this is the first node on the linked list
     */
    public ObjectNode getPreviousNode()
    {
	return previousNode;
    }


    /**
     * Returns <CODE>ObjectNode</CODE> next to this 
     * <CODE>ObjectNode</CODE> on the linked list,
     * or null if this <CODE>ObjectNode</CODE> is the
     * last node of the linked list,
     * 
     * @return ObjectNode next to this <CODE>ObjectNode</CODE>
     * or null if this is the last node on the linked list
     */
    public ObjectNode getNextNode()
    {
	return nextNode;
    }

    /**
     * Returns <CODE>Object</CODE> proxied by this
     * <CODE>ObjectNode</CODE>
     *
     * @return Object proxied by this
     * <CODE>ObjectNode</CODE>
     */
    public Object getObject()
    {
	return object;
    }
	
    /**
     * Returns key to the <CODE>Object</CODE> proxied by this
     * <CODE>ObjectNode</CODE>
     *
     * @return Object representing the key
     */
    public Object getKey()
    {
	return key;
    }

    /*
    public void getVectorList(Vector list)
    {
	list.add(this);
	if(getNextNode() == null)
	    return;
	getNextNode().getVectorList(list);	
    }
    */
}
