/*
 * DefaultMutableTreeNode.java:  DefaultMutableTreeNode 
 * to replace javax.swing.tree.DefaultMutableTreeNode
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: DefaultMutableTreeNode.java,v 1.15 2002/09/28 02:40:35 nozomi Exp $
 * $Log: DefaultMutableTreeNode.java,v $
 * Revision 1.15  2002/09/28 02:40:35  nozomi
 * modify event handling
 *
 * Revision 1.14  2002/09/20 13:14:14  nozomi
 * restrict index range of child nodes
 *
 * Revision 1.13  2002/09/17 08:34:12  nozomi
 * change Event notification
 *
 * Revision 1.12  2002/08/19 09:20:28  nozomi
 * fix children size bug in getSiblingOf()
 *
 * Revision 1.11  2002/08/09 05:21:55  nozomi
 * translate comments
 *
 * Revision 1.10  2002/08/05 23:54:29  nozomi
 * modify exception handling in getChildAt(int)
 *
 * Revision 1.9  2002/06/09 12:47:49  nozomi
 * change TreeModelEvent handling
 *
 * Revision 1.8  2002/06/06 21:22:45  nozomi
 * create nodeListeners only when necessary
 *
 * Revision 1.7  2002/06/03 04:37:45  nozomi
 * insertion handling in add()
 *
 * Revision 1.6  2002/06/03 01:29:09  nozomi
 * prevents re-insertion of child node
 *
 * Revision 1.5  2002/03/28 03:12:16  nozomi
 * extends Observable
 *
 * Revision 1.4  2002/03/10 08:55:17  nozomi
 * fix multiple insertion bug
 *
 * Revision 1.3  2002/03/10 06:56:06  nozomi
 * support parent node insertion by treeStructureChanged()
 *
 * Revision 1.2  2002/03/08 23:04:37  nozomi
 * add event handlingTreeModelListener related
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */

package org.nomencurator.util.tree;

import java.io.Serializable;
import java.lang.Cloneable;

import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.util.Observable;
import java.util.Stack;
import java.util.Vector;

import org.nomencurator.util.tree.event.TreeModelEvent;
import org.nomencurator.util.tree.event.TreeModelListener;

/**
 * An implementaiont of <code>MutalbeTreeNode<code>
 *
 * @version	27 Sep 2002
 * @author 	Nozomi `James' Ytow
 */
public class DefaultMutableTreeNode
    extends Observable
    implements Cloneable, MutableTreeNode, Serializable
{

    //true if the node may have a child
    protected boolean allowsChildren;

    //array of children
    //It is null if the node does not have any child.
    protected  Vector children;

    //Enumeration always empty which is used when an Enumeration of
    //leaf nodes is required
    static public final Enumeration EMPTY_ENUMERATION
	= new Enumeration() {
		public boolean hasMoreElements() { return false; };
		public Object nextElement() {
		    throw new NoSuchElementException();
		}
	    }; 

    //The parent node of this node in the tree.
    //It is null if the node does not have a parent.
    protected MutableTreeNode parent; 

    //Optional user object
    protected  Object userObject; 

    protected transient Vector nodeListeners;

    //Constructors
    /**
     * Create a tree node without parent nor child
     */
    public DefaultMutableTreeNode() 
    {
	this(null);
    }

    /**
     * Create a tree node without parent nor child
     * but initialise it with the given object
     */
    public DefaultMutableTreeNode(Object userObject)
    {
	this(userObject, true);
    }

    /**
     * Create a tree node without parent nor child
     * but initialise it with the given object and
     * given allowance
     * 
     */
    public DefaultMutableTreeNode(Object userObject, boolean allowsChildren)
    {
	this.allowsChildren = allowsChildren;
	this.userObject     = userObject;
	parent              = null;
	children            = null;
    }

    /**
     * Remove <code>child</code> from its parent if it has,
     * set this node to parent of the <code>child</code> node,
     * and insert the <code>child</code> node as child node of this 
     * node at given <code>index</code>.
     *
     * @param child the MutableTreeNode to be inserted under this node
     * @param index position where the child node should be inserted
     *
     * @exception	ArrayIndexOutOfBoundsException	if
     *				<code>index</code> is out of bounds
     * @exception	IllegalArgumentException	if
     *				<code>child</code> is null or is an
     *				ancestor of this node
     * @exception	IllegalStateException	if this node does not allow
     *						children
     * @see	#isNodeDescendant
     */
    public void insert(MutableTreeNode child, int index)
    {
	if(!allowsChildren) {
	    throw new IllegalStateException();
	}
	else if (child == null) {
	    throw new IllegalArgumentException();
	}
	else if (isNodeAncestor(child)) {
	    throw new IllegalArgumentException();
	}

	if(children != null &&
	   children.contains(child))
	    return;

	DefaultMutableTreeNode previousParent = (DefaultMutableTreeNode)(child.getParent());

	if(previousParent != null && 
	   previousParent.isNodeChild(child) &&
	   previousParent != child) 
	    previousParent.remove(child);

	DefaultMutableTreeNode root = (DefaultMutableTreeNode)(((DefaultMutableTreeNode)child).getRoot());
	child.setParent(this);

	if(children == null) {
	    children = new Vector();
	}

	children.insertElementAt(child, index);

	Object[] path = getObjectPath();
	int[] indices = new int[1];
	indices[0] = index;
	//	((TreeNode)(path[0])).treeNodesInserted(new TreeModelEvent(this, path, indices, new Object[]{child}));

	fireTreeNodesInserted(path, indices, new Object[]{child});
	
	if(getRoot() != this)
	    ((MutableTreeNode)getRoot()).fireTreeNodesInserted(path, indices, new Object[]{child});

	if(root == child)
	    root.fireTreeStructureChanged(new Object[]{this}, null, null);
    }

    /**
     * Remove a child node specified by <code>index</code> from children list of the node
     * and set parent node of the removed child to null.  The child node to be removed must
     * be a MutableTreeNode.
     *
     * @param index index of the node to be removed in child list of this node
     * @exception ArrayIndexOutOfBoundsException if <code>index</code> is out of bounds
     */
    public void remove(int index)
    {
	if(index < 0 || index >= children.size())
	    return;
	MutableTreeNode child = (MutableTreeNode)getChildAt(index);
	child.setParent(null);
	children.removeElementAt(index);
	Object[] path = getObjectPath();
	int[] indices = new int[1];
	indices[0] = index;
	//((TreeNode)(path[0])).treeNodesRemoved(new TreeModelEvent(this, path, indices, new Object[]{child}));

	fireTreeNodesRemoved(path, indices, new Object[]{child});
	if(getRoot() != this)
	    ((DefaultMutableTreeNode)getRoot()).fireTreeNodesRemoved(path, indices, new Object[]{child});

	if(children.isEmpty())
	    children = null;
    }


    /**
     * Sets parent node of the node to <code>parent</code> without changing 
     * children list of parent node.
     * Methods <code>insert()</code> and <code>remove()</code> call this
     * method to re-allocate a child node of a node.  Other methods do not
     * call this method.
     *
     * @param parent new parent to be assigned
     *
     */
    public void setParent(MutableTreeNode parent)
    {
	if(this.parent == parent)
	    return;

	//new Exception().printStackTrace();
	MutableTreeNode root = (MutableTreeNode)getRoot();

	this.parent = parent;
	
	if(root == this) {
	    //	    fireTreeStructureChanged(new Object[]{parent}, null, null);
	}

	//	parent.insert(this, parent.getChildCount());
    }

    /**
     * Returns parent node of this node
     *
     * @return parent TreeNode of this node, or null if it is an orphan
     *
     */
    public TreeNode getParent()
    {
	return parent;
    }

    /**
     * Returns pointer to a child node at given <code>index</code> of
     * children list
     *
     * @param index Index of child node in children list of this node
     *
     * @exception ArrayIndexOutOfBoundsException if <code>index</code> out of boundary
     */
    public TreeNode getChildAt(int index)
    {
	if(children == null ||
	   index >= children.size())
	    throw new ArrayIndexOutOfBoundsException();
	return (TreeNode)(children.elementAt(index));
    }

    /**
     * Returns whther it has a child or more
     *
     * @return true if the node has a child or more
     *
     */
    public boolean hasAChild()
    {
	return getChildCount() != 0;
    }

    /**
     * Returns whther it has a child or more
     *
     * @return true if the node has a child or more
     *
     */
    public boolean hasChildren()
    {
	return getChildCount() > 1;
    }

    /**
     * Returns number of children of this node
     *
     * @return number of children in int
     *
     */
    public int getChildCount()
    {
	if(children == null)
	    return 0;
	return children.size();
    }

    /**
     * Returns index of given <code>child</code> in children list of this node.
     * If <code>child</code> is not child node of this node, returns -1.
     *
     * @param child a TreeNode to find
     *
     * @return index of <code>child</code> node in int.  -1 if it is not child of this node
     *
     * @exception IllegalArgumentException if <code>child</code> is null
     */
    public int getIndex(TreeNode child)
    {
	if(child == null)
	    throw new IllegalArgumentException();

	if(children == null)
	    return -1;

	return children.indexOf(child);
    }


    /**
     * Creates and returns foward Enumeration of children nodes of this node.
     * Modification of children list of this node invalidates any Enumeration
     * created prior to the modification.
     *
     * @return Enumeration of this node's children 
     *
     */
    public Enumeration children()
    {
	if(children == null)
	    return EMPTY_ENUMERATION;
	return children.elements();
    }


    /**
     * Sets whether this node is allowed to have a child node.
     * If <code>allows</code> if false, all child nodes become orphan.
     * <p>
     * Note: a node is allowed to have a child by default.
     *
     * @param allows ture if this node is allowed to have a child node
     *
     */
    public void setAllowsChildren(boolean allows)
    {
	if(allows != allowsChildren) {
	    allowsChildren = allows;
	    if(!allowsChildren) {
		removeAllChildren();
	    }
	}
    }

    /**
     * Returns true if this node is allowed to have a child
     *
     * @return true if this node is allowed to have a child, false for else
     */
    public boolean getAllowsChildren()
    {
	return allowsChildren;
    }

    /**
     * Sets usersObject of this node to <code>object</code>
     *
     * @param object a Object specified by user for this node
     *
     * @see #getUserObject()
     * @see #toString()
     *
     */
    public void setUserObject(Object object)
    {
	this.userObject = userObject;
    }

    /**
     * Returns user specified object to this node
     *
     * @return user specified object to this node
     *
     * @see #setUserObject(java.lang.Object)
     * @see #toString()
     *
     */
    public Object getUserObject()
    {
	return userObject;
    }

    /**
     * Removes a tree of which root is this node from its parent node, 
     * set null as its parent. If this node is the root of the tree,
     * this method does nothing.
     *
     */
    public void removeFromParent()
    {
	if(parent != null) {
	    parent.remove(this);
	}
    }

    /**
     * Removes <code>child</code> node from children list of this node,
     * sets null as parent of the <code>child</code> node.
     *
     * @param child node to be removed
     *
     * @exception IllegalArgumentException if <code>child</code> is null, or not 
     * child node of this node
     *
     */
    public void remove(MutableTreeNode child)
    {
	/*
	if(child == null || 
	   children == null ||
	   !children.contains(child))
	    throw new IllegalArgumentException();
	*/
	if(child == null)
	    throw new IllegalArgumentException();
	if(children == null)
	    throw new IllegalArgumentException();
	if(!children.contains(child))
	    throw new IllegalArgumentException();

	children.removeElement(child);
	child.setParent(null);
    }

    /**
     * Removes all child nodes of this node, sets their parent null.
     * If this node does not have a child, this method does nothing.
     */
    public void removeAllChildren()
    {
	if(children == null)
	    return;
	if(children.isEmpty()) {
	    children = null;
	    return;
	}
	for (Enumeration e = children.elements() ; e.hasMoreElements() ;) {
	    MutableTreeNode child = (MutableTreeNode)e.nextElement();
	    child.setParent(null);
	}
	children.removeAllElements();
	children = null;	
    }

    /**
     * Removes <code>child</code> node from its parent (if it has),
     * make <code>child</code> is a child of this node at end of the children list.
     *
     * @param child node to be add to this node as a child node
     *
     * @exception IllegalArgumentException if <code>child</code> is null
     * @exception IllegalStateException if this node does not allow child
     *
     * @see #insert(javax.swing.tree.MutableTreeNode, int)
     */
    public void add(MutableTreeNode child)
    {
	if(child != null && child.getParent() == this &&
	   children != null) {
	    remove(child);
	    //	    insert(child, getChildCount() - 1);
	}
	//	else
	insert(child, getChildCount());
    }


    /**
     * Returns true if <code>anotherNode</code> is an ancestor node of this, 
     * i.e., if it is this node itself, parent of this node, or ancestor
     * of its parent node.  The node is regarded as anocestor node of itself.
     * Returns null if <code>anotherNode</code> is null.
     * This operation is order of O(h) at best where h is distance from root
     * to this node.
     *
     * @param anotherNode <code>TreeNode</code> to be examined whether
     * it is ancestor or this node or not.
     * @return true if this is a descendant of <code>anotherNode</code>
     * @see isNodeDescendant(DefaultMutableTreeNode),
     * @see getSharedAncestor(DefaultMutableTreeNode)
     */

    public boolean isNodeAncestor(TreeNode node)
    {
	if(node == null)
	    return false;

	TreeNode ancestor = this;

	do {
	    if(ancestor == node)
		return true;
	} while((ancestor = ancestor.getParent()) != null);

	return false;
    }


    /**
     * Returns true if <code>node</code> is a descendant node of this <code>TreeNode</code>,
     * i.e. <code>node</code> is this node itself, one of child nodes of this node, or,
     * a descendant node of one of child nodes.  A node itself is regarded as its
     * descendant node.  This method returns false if <code>node</code> is null.
     * This operation is O(h) at worst where h is distance between root node and
     * <code>node</code>
     *
     * @param node <code>TreeNode</code> to be evaluated whether it's descendant of this.
     *
     * @return true if this node is ancestor of <code>node</code>
     *
     * @see isNodeAncestor(jp.kyasu.util.tree.TreeNode),
     * @getSharedAncestor(jp.kyasu.util.tree.DefaultMutableTreeNode)
     */
    public boolean isNodeDescendant(DefaultMutableTreeNode node)
    {
	if(node == null)
	    return false;

	return !node.isNodeAncestor(this);
    }

    /**
     * Returns the nearest common ancestor node shared by this node and <code>node</code>, or
     * returns null if there is no common ancestor node, i.e. this node and <code>node</code>
     * are in different trees or <code>node</code> is null.  A node itself is regarded
     * as an ancestor node of the node.
     *
     * @param node <code>TreeNode</code> of which common ancestor to be looked for
     *
     * @return TreeNode the nearest common ancestor node shared by this node and <code>node</code>, or
     * null if there is no common ancestor node
     *
     * @see isNodeAncestor(jp.kyasu.util.tree.TreeNode),
     * @see isNodeDescendant(jp.kyasu.util.tree.DefaultMutableTreeNode)
     */
    public TreeNode getSharedAncestor(DefaultMutableTreeNode node)
    {
	if(node == null)
	    return null;
	if(node == this)
	    return this;

	
	int diff = getLevel() - node.getLevel();
	TreeNode node0 = node;
	TreeNode node1 = this;

	if(diff < 0) {
	    diff = -diff;
	    node0 = this;
	    node1 = node;
	}

	while(diff > 0) {
	    node0 = node0.getParent();
	    diff--;
	}

	do {
	    if(node0 == node1)
		return node0;
	    node0 = node0.getParent();
	    node1 = node1.getParent();
	} while(node0 != null);

	return null;
    }


    /**
     * Returns true if <code>node</code> shares the same tree with this node,
     * or false if <code>node</code> is null
     *
     * @return true if <code>node</code> is in the same tree with this node, 
     * or false if <code>node</code> is null
     *
     * @see getSharedAncestor(jp.kyasu.util.tree.DefaultMutableTreeNode)
     * @see getRoot()
     */
    public boolean isNodeRelated(DefaultMutableTreeNode node)
    {
	return (node != null) && (getRoot() == node.getRoot());
    }


    /**
     * Returns depth of tree of which root is this node, i.e. the longest distance
     * between this node and leaves.  It returns zero if this node does not have
     * any child node.  This operation requires much heavier load than <code>getLevel()</code>
     * because this method requires whole tree rooted by this node efficiently.
     *
     * @return depth of tree rooted by this node
     *
     * @see getLevel()
     */
    public int getDepth()
    {
	int depth = 0;
	if(children == null)
	    return depth;

	if(children.isEmpty()) {
	    children = null;
	    return depth;
	}

	for (Enumeration e = children.elements() ; e.hasMoreElements() ;) {
	    DefaultMutableTreeNode child = (DefaultMutableTreeNode)e.nextElement();
	    int d = child.getDepth();
	    if(d > depth)
		depth = d;
	}

	return ++depth;
    }



    /**
     * Returns number of levels above this node, i.e. distance from the root node to
     * this node.  Returns zero if this node is a root node.
     *
     * @return number of levels above this node
     *
     * @see getDepth()
     */

    public int getLevel()
    {
	int level = 0;
	TreeNode node = this.getParent();
	while(node != null) {
	    level++;
	    node = node.getParent();
	}

	return level;
    }


    /**
     * Returns pats from the root to this node.
     * The last element of the path is this node.
     *
     * @return array of <code>TreeNode</code> objects specifying the path.
     * The first element of the array is the root node, and the last element
     * is this node.
     */
    public TreeNode[] getPath()
    {
	int level = getLevel();
	TreeNode[] path = new TreeNode[level + 1];

	TreeNode node = this;

	do {
	    path[level--] = node;
	    node = node.getParent();
	}while(level >= 0);

	return path;
    }

    public Object[] getObjectPath()
    {
	int level = getLevel();
	Object[] path = new Object[level + 1];

	TreeNode node = this;

	do {
	    path[level--] = node;
	    node = node.getParent();
	}while(level >= 0);

	return path;
    }


    /**
     * Returns parent path contains upto the root node.
     * The <code>node</code> is the last element of returned array.
     * Length of the returned array indicates depth of the node
     * in the tree.
     *
     * @param node <code>TreeNode</code> to which path will be obtained
     * @param depth <code>int</code> used to specify size of returned array 
     * which is already obtained by (recursive) call
     *
     * @return array of <code>TreeNode</code> representing the path from the noed
     * to <code>node</code>
     */
    protected TreeNode[] getPathToRoot(TreeNode node, int depth)
    {
	return ((DefaultMutableTreeNode)node).getPath();
    }

    /**
     * Returns user object path from the root to this node.
     * The returned path contains null if some of <code>TreeNode</code>
     * has null user object.
     */
    public Object[] getUserObjectPath()
    {
	TreeNode[] path = getPath();
	if(path == null)
	    return null;

	Object[] objectPath = new Object[path.length];

	for (int i = 0; i < path.length; i++) {
	    objectPath[i] 
		= ((DefaultMutableTreeNode)path[i]).getUserObject();
	}
	path = null;

	return objectPath;
    }


    /**
     * Returns the root of the tree contains this node.
     * A root node is a higher node having null parent.
     *
     * @return TreeNode represents the root of the tree contains this node.
     *
     * @see isNodeAncestor(javax.swing.tree.TreeNode)
     */

    public TreeNode getRoot()
    {
	TreeNode node = this;
	TreeNode parent;
	while((parent = node.getParent()) != null) {
	    node = parent;
	}
	return node;
    }

    /**
     * Returns true if this node is a root node.
     * A root is only the node in a tree havig null parent.
     * All tree have only one root.
     *
     * @return true if this node is a root of the tree
     */
    public boolean isRoot()
    {
	return (parent == null);
    }
	

    /**
     * Return the node next to this node by forward scan of tree of this node.
     * It returns null if this node is the last of the scan.
     * Because it is inefficient method to scan whole tree, enumeration could be
     * used instead.
     *
     * @return DefaultMutableTreeNode next to this node by forward scan, or null
     * if this is the last node
     *
     * @see preorderEnumeration()
     */
    public DefaultMutableTreeNode getNextNode()
    {
	return getNode(1);
	/*
	if(getChildCount() != 0)
	    return (DefaultMutableTreeNode)(children.elementAt(0));

	DefaultMutableTreeNode node = this;
	DefaultMutableTreeNode siblingNode = null;

	do {

	    if(node == null)
		return node;

	    siblingNode = node.getNextSibling();
	    if(siblingNode != null) 
		return siblingNode;

	    node = (DefaultMutableTreeNode)node.getParent();
	} while(true);
	*/
    }

    /**
     * Return the node prior to this node by forward scan of tree of this node.
     * It returns null if this node is the first node (root of the tree).
     * Because it is inefficient method to scan whole tree, enumeration could be
     * used instead.
     *
     * @return DefaultMutableTreeNode prior to this node by forward scan, or null
     * if this is the first node
     *
     * @see preorderEnumeration()
     */
    public DefaultMutableTreeNode getPreviousNode()
    {
	return getNode(-1);
    }


    public DefaultMutableTreeNode getNode(int offset)
    {
	if(getChildCount() != 0)
	    return (DefaultMutableTreeNode)(children.elementAt(0));

	DefaultMutableTreeNode node = this;
	DefaultMutableTreeNode siblingNode = null;

	do {

	    if(node == null)
		return node;

	    siblingNode = node.getSibling(offset);
	    if(siblingNode != null) 
		return siblingNode;

	    node = (DefaultMutableTreeNode)node.getParent();
	} while(true);
	
    }

    /**
     * Creates and returns <CODE>Enumeration</CODE> to forward scan a subtree of
     * which root is this node.
     * The node returned by first <CODE>nextElement()</CODE> call
     * is this node.
     * <P>
     * A modificaion of the tree by insertion, deletion or re-allocation of a node
     * invalidate any <CODE>Enumeration</CODE> created before the modificaiton.
     *
     * @return Enumearation to traverse by forward scan
     *
     * @see postorderEnumeration()
     */
    public Enumeration preorderEnumeration()
    {
	return new PreorderEnumeration(this);
    }

    /**
     * Creates and returns <CODE>Enumeration</CODE> to backward scan a subtree of
     * which root is this node.
     * The node returned by first <CODE>nextElement()</CODE> call
     * is the leftmost node.
     * It is same with depth first scan.
     * <P>
     * A modificaion of the tree by insertion, deletion or re-allocation of a node
     * invalidate any <CODE>Enumeration</CODE> created before the modificaiton.
     *
     * @return Enumearation to traverse by backward scan
     *
     * @see depthFirstEnumeration()
     * @see preorderEnumeration()
     */
    public Enumeration postorderEnumeration()
    {
	return new PostorderEnumeration(this);
    }


    /**
     * Creates and returns a <CODE>Enumeration</CODE> to traverse a subtree under this node
     * in breadth first scan.  The node returned by first <CODE>nextElement()</CODE> call
     * is this node.
     * <P>
     * A modificaion of the tree by insertion, deletion or re-allocation of a node
     * invalidate any <CODE>Enumeration</CODE> created before the modificaiton.
     *
     * @return Enumearation to traverse by breadth first scan
     *
     * @see depthFirstEnumeration()
     */
    public Enumeration breadthFirstEnumeration()
    {
	return new BreadthFirstEnumeration(this);
    }

    /**
     * Creates and returns a <CODE>Enumeration</CODE> to traverse a subtree under this node
     * in depth first scan.  The node returned by first <CODE>nextElement()</CODE> call
     * is the leftmost leaf node.  It is same to postorder scan.
     * <P>
     * A modificaion of the tree by insertion, deletion or re-allocation of a node
     * invalidate any <CODE>Enumeration</CODE> created before the modificaiton.
     *
     * @return Enumearation to traverse by depth first scan
     *
     * @see breadthFirstEnumeration()
     * @see postorderEnumeration()
     */
    public Enumeration depthFirstEnumeration()
    {
	return postorderEnumeration();
    }


    /**
     * Creates and returns an <CODE>Enumeration</CODE> to traverse the path from
     * <CODE>ancestor</CODE> to this node.  The <CODE>nextElement()</CODE> of
     * the <CODE>Enumeration</CODE> returns <CODE>ancestor</CODE> at first,
     * then a child of <CODE>ancestor</CODE> which is ancestor of this node recursively,
     * and finally this node.
     * Creation of the <CODE>Enumearation</CODE> is O(m) where m is number of nodes
     * between this node and <CODE>ancestor</CODE> inclusively.
     * Each <CODE>nextElement()</CODE> message is  O(1).
     * <P>
     * A modificaion of the tree by insertion, deletion or re-allocation of a node
     * invalidate any <CODE>Enumeration</CODE> created before the modificaiton.
     *
     * @return Enumearation to traverse from <CODE>ancestor</CODE> to this node
     *
     * @exception IllegalArgumentException if <CODE>ancestor</CODE> is not
     * ancestor of this node
     *
     * @see isNodeAncestor(TreeNode)
     * @see isNodeDescendant(DefaultMutableTreeNode)
     */
    public Enumeration pathFromAncestorEnumeration(TreeNode ancestor)
    {
	return new PathEnumeration(ancestor, this);
    }


    /**
     * Returns true if <CODE>aNode</CODE> is a child node of this node,
     * or false if <CODE>aNode</CODE> is null.
     *
     * @return true if <CODE>aNode</CODE> is a child of this node,
     * or false if <CODE>aNode</CODE> is null
     */
    public boolean isNodeChild(TreeNode node)
    {
	if(node == null || children == null)
	    return false;
	return children.contains(node);
    }

    /*
     * Returns first child node of this node, or thorws <CODE>NoSuchElementException</CODE>
     * if this node has no child node.
     *
     * @return first child node of this node
     *
     * @exception NoSuchElementException if this node has no child node
     */
    public TreeNode getFirstChild()
    {
	if(children == null || children.isEmpty())
	    throw new NoSuchElementException();

	return (TreeNode)(children.firstElement());
    }

    /**
     * Returns last child node of this node, or thorws <CODE>NoSuchElementException</CODE>
     * if this node has no child node.
     *
     * @return last child node of this node
     *
     * @exception NoSuchElementException if this node has no child node
     */
    public TreeNode getLastChild()
    {
	if(children == null || children.isEmpty())
	    throw new NoSuchElementException();

	return (TreeNode)(children.lastElement());
    }


    /**
     * Returns the node in children array just after <CODE>aChild</CODE>.
     * The <CODE>aChild</CODE> must be a child node of this node.
     * It returns null if <CODE>aChild</CODE> is the last child node of this node.
     * This method executes a linear search of O(n) on children nodes of this node to find
     * <CODE>aChild</CODE> where n is number of chilren nodes.
     *
     * @return this node's child node just after <CODE>aChild</CODE>
     *
     * @exception IllegalArgumentException if <CODE>aChild</CODE> is null,
     * or not child node of this node.
     *
     * @see children
     */
    public TreeNode getChildAfter(TreeNode child)
    {
	return getSiblingOf(child, 1);
    }


    /**
     * Returns the node in children array just befor <CODE>aChild</CODE>.
     * The <CODE>aChild</CODE> must be a child node of this node.
     * It returns null if <CODE>aChild</CODE> is the first child node of this node.
     * This method executes a linear search of O(n) on children nodes of this node to find
     * <CODE>aChild</CODE> where n is number of chilren nodes.
     *
     * @return this node's child node just before <CODE>aChild</CODE>
     *
     * @exception IllegalArgumentException if <CODE>aChild</CODE> is null,
     * or not child node of this node.
     */
    public TreeNode getChildBefore(TreeNode child)
    {
	return getSiblingOf(child, -1);
    }

    
    public TreeNode getSiblingOf(TreeNode child, int offset)
    {
	if(child == null || children == null)
	    throw new IllegalArgumentException();

	int index = children.indexOf(child);
	
	if(index < 0)
	    throw new IllegalArgumentException();

	if(offset == 0)
	    return child;

	index += offset;

	if(index < 0 || index >= children.size())
	    return null;

	return (TreeNode)(children.elementAt(index));
    }



    /**
     * Returns true if <CODE>anotherNode</CODE> shears the same parent node with this node.
     * A node is a sibling node of itself.  
     * It returns false if <CODE>anotherNode</CODE> is null.
     *
     * @param anotherNode node to be examined whether it is sibling node of this node
     *
     * @return true if <CODE>anotherNode</CODE> is sibling node of this node
     */
    public boolean isNodeSibling(TreeNode node)
    {
	if(node == null)
	    return false;

	TreeNode p = node.getParent();

	if(p != null && p == parent)
	    return true;
	
	return false;
    }


    /**
     * Returns number of sibling nodes of this node.  A node is a sibling node of itself.
     * It returns 1 if this node has neither pairent nor sibling node
     *
     * @return int representing number of sibling nodes
     */
    public int getSiblingCount()
    {
	if(parent == null)
	    return 1;

	return ((DefaultMutableTreeNode)parent).children.size();
    }


    /**
     * Returns a sibling node just next to this node in children array of the pairent node,
     * or null if this node does not have pairent or this node is the last child node of
     * the pairent node.  This methods executes a linear search of O(n) where n is number of child nodes.
     * Use enumeration instead to traverse whole array.
     *
     * @return the sibling node just next to this node
     *
     * @see children
     */
    public DefaultMutableTreeNode getNextSibling()
    {
	return getSibling(1);
    }

    /**
     * Returns a sibling node just previous to this node in children array of the pairent node,
     * or null if this node does not have pairent or this node is the first child node of
     * the pairent node.  This methods executes a linear search of O(n) where n is number of child nodes.
     *
     * @return the sibling node just previous to this node
     */
    public DefaultMutableTreeNode getPreviousSibling()
    {
	return getSibling(-1);
    }

    /**
     */
    public DefaultMutableTreeNode getSibling(int offset)
    {
	if(parent == null)
	    return null;

	return (DefaultMutableTreeNode)(((DefaultMutableTreeNode)parent).getSiblingOf(this, offset));
    }

    /**
     * Returns true if this node has no child node.
     * To distinguish node without child and node which can not hold node
     * (e.g. distinguish a file from empty directory), this method and
     * <CODE>getAllowsChildren</CODE> can be used in combination.
     *
     * @definition isLeaf of <CODE>TreeNode</CODE> interface
     *
     * @return true if this node has no child node
     *
     * @see getAllowsChildren()
     */
    public boolean isLeaf()
    {
	return (children == null || children.size() == 0);
    }

    /**
     * Finds and returns the first leaf node of this node's descendent nodes, 
     * i.e. this node itself or the first leaf node of the first child node of this node.
     * Returns this node if it is a leaf node itself.
     *
     * @return the first leaf node in the subtree under this node
     *
     * @see isLeaf()
     * @see isNodeDescendant(DefaultMutableTreeNode)
     */
    public DefaultMutableTreeNode getFirstLeaf() {

	DefaultMutableTreeNode node = this;

	while(!node.isLeaf())
	    node = (DefaultMutableTreeNode)(node.getFirstChild());

	return node;
    }


    /**
     * Finds and returns the last leaf node of this node's descendent nodes, 
     * i.e. this node itself or the last leaf node of the last child node of this node.
     * Returns this node if it is a leaf node itself.
     *
     * @return the last leaf node in the subtree under this node
     *
     * @see isLeaf()
     * @see isNodeDescendant(DefaultMutableTreeNode)
     */
    public DefaultMutableTreeNode getLastLeaf()
    {

	DefaultMutableTreeNode node = this;

	while(!node.isLeaf())
	    node = (DefaultMutableTreeNode)(node.getLastChild());

	return node;
    }

    /**
     * Returns a leaf node next to this node, or null if this node is the
     * last leaf node in the tree.
     * <P>
     * This operation is not efficient in this impelementation of <CODE>MutableNode</CODE> interface.
     * This mothod executes linear search of children list of its parent to find itself and to determine
     * the next node.
     * <P>
     * This implementaiont makes an operation appropriate to do short traversal from known position.
     * To traverse all leaves in a tree, enumerate all nodes in a tree by depthFirstEnumeration and then
     * apply isLeaf to each node to determine which nodes are leaves.
     *
     * @return a leaf node next to this node
     *
     * @see depthFirstEnumeration()
     * @see isLeaf()
     */
    public DefaultMutableTreeNode getNextLeaf()
    {
	if(parent == null)
	    return null;

	DefaultMutableTreeNode node = ((DefaultMutableTreeNode)parent).getNextSibling();

	if(node != null)
	    return node.getFirstLeaf();

	return ((DefaultMutableTreeNode)parent).getNextLeaf();

    }

    /**
     * Returns a leaf node previous to this node, or null if this node is the
     * first leaf node in the tree.
     * <P>
     * This operation is not efficient in this impelementation of <CODE>MutableNode</CODE> interface.
     * This mothod executes linear search of children list of its parent to find itself and to determine
     * the previous node.
     * <P>
     * This implementaiont makes an operation appropriate to do short traversal from known position.
     * To traverse all leaves in a tree, enumerate all nodes in a tree by depthFirstEnumeration and then
     * apply isLeaf to each node to determine which nodes are leaves.
     *
     * @return a leaf node previous to this node
     *
     * @see depthFirstEnumeration()
     * @see isLeaf()
     */
    public DefaultMutableTreeNode getPreviousLeaf()
    {
	if(parent == null)
	    return null;

	DefaultMutableTreeNode node = ((DefaultMutableTreeNode)parent).getPreviousSibling();

	if(node != null)
	    return node.getLastLeaf();

	return ((DefaultMutableTreeNode)parent).getPreviousLeaf();
    }


    /**
     * Returns total number of leaf nodes under this node, or 1 if
     * this is a leaf node.
     * This method is O(n) order where n is number of loewr nodes.
     *
     * @return number of leaf under this node
     * @see isNodeAncestor(TreeNode)
     */
    /*
    public int getLeafCount()
    {
    }

    */

    /**
     * Returns result of toString() send to user object of this node,
     * or null if this node does not have a user objcet.
     *
     * @overrid Object#toString
     * @see getUserObject()
     */
    public String toString()
    {
	if(userObject == null)
	    return null;

	return userObject.toString();
    }

    /*
     * Overrides clone() to be public and returns shallow copy of this node,
     * i.e. new node has neither parent nor child but referes to the same object
     * if given
     *
     * @override Object#clode()
     *
     * @return copy of this node
     */
    public Object clone()
    {
	return new DefaultMutableTreeNode(userObject, allowsChildren);
    }



    public Enumeration getChildren()
    {
	return children();
    }

    class TreeNodeEnumeration implements Enumeration
    {
	TreeNode root;
	int i = 0;

	TreeNodeEnumeration(TreeNode rootNode)
	{
	    root = rootNode;
	}
		    
	public boolean hasMoreElements() {
	    return i < 1;
	}

	public Object nextElement() {
	    i++;
	    return root;
	}
    }

    class PreorderEnumeration implements Enumeration
    {

	Stack stack;

	public PreorderEnumeration(TreeNode root)
	{
	    stack = new Stack();
	    stack.push(new TreeNodeEnumeration(root));
	}

	public boolean hasMoreElements()
	{
	    return (!stack.empty() &&
		    ((Enumeration)stack.peek()).hasMoreElements());
	}

	public Object nextElement()
	{
	    Enumeration	enum = (Enumeration)stack.peek();
	    TreeNode	node = (TreeNode)enum.nextElement();
	    Enumeration	children = node.children();

	    if (!enum.hasMoreElements())
		stack.pop();

	    if (children.hasMoreElements())
		stack.push(children);

	    return node;
	}

    };

    class PostorderEnumeration implements Enumeration 
    {
	TreeNode root;
	Enumeration children;
	Enumeration branch;

	public PostorderEnumeration(TreeNode node)
	{
	    root     = node;
	    children = root.children();
	    branch  = EMPTY_ENUMERATION;
	}

	public boolean hasMoreElements()
	{
	    return root != null;
	}

	public Object nextElement()
	{
	    Object next;

	    if (branch.hasMoreElements())
		next = branch.nextElement();
	    else if (children.hasMoreElements()) {
		branch = new PostorderEnumeration((TreeNode)(children.nextElement()));
		next = branch.nextElement();
	    }
	    else {
		next = root;
		root = null;
	    }
		
	    return next;
	}
    };

    class BreadthFirstEnumeration implements Enumeration
    {
	Stack queue;  //Stack extends Vector
	public BreadthFirstEnumeration(TreeNode node)
	{
	    queue = new Stack();
	    queue.push(new TreeNodeEnumeration(node));
	}

	public boolean hasMoreElements()
	{
	    return (!queue.isEmpty() &&
		    ((Enumeration)queue.firstElement()).hasMoreElements());
	}

	public Object nextElement()
	{
	    Enumeration	enum     = (Enumeration)queue.firstElement();
	    TreeNode	node     = (TreeNode)enum.nextElement();
	    Enumeration	children = node.children();

	    if (!enum.hasMoreElements())
		queue.removeElementAt(0);

	    if (children.hasMoreElements())
		queue.push(children);

	    return node;
	}

    };

    class PathEnumeration implements Enumeration 
    {
	Stack path;

	public PathEnumeration(TreeNode ancestor, TreeNode descendant)
	{
	    if (ancestor == null || descendant == null) {
		throw new IllegalArgumentException();
	    }

	    path = new Stack();
	    path.push(descendant);
	    
	    while(descendant != ancestor) {
		descendant = descendant.getParent();

		if(descendant == null) {
		    path.clear();
		    path = null;
		    throw new IllegalArgumentException();
		}
		path.push(descendant);
	    }
	}

	public boolean hasMoreElements()
	{
	    return path.size() > 0;
	}

	public Object nextElement()
	{
	    return path.pop();
	}

    };

    /**
     * Adds listener as <code>TreeModelListener</code> listening this object
     */
    public synchronized void addTreeModelListener(TreeModelListener listener)
    {
	if(nodeListeners == null)
	    nodeListeners = new Vector();

	nodeListeners.addElement(listener);
    }

    /**
     * Removes listener from listeners list of this object
     */
    public synchronized void removeTreeModelListener(TreeModelListener listener)
    {
	nodeListeners.removeElement(listener);
    }
    
    /*
    protected synchronized Enumeration getListenersEnumeration(Vector listeners)
    {
	return ((Vector)listeners.clone()).elements();
    }
    */

    /**
     * Invokes treeNodeChanged() of <code>nodeListeners</code>
     * when a node or a set of siblings was modified.
     */
    public void fireTreeNodesChanged(Object[] path,
				     int[] indices,
				     Object[] nodes)
    {
	if(nodeListeners == null)
	    return;

	Object[] listeners = null;
	synchronized(nodeListeners) {
	    listeners = nodeListeners.toArray();
	}

	TreeModelEvent event = null;
	for(int i = listeners.length; i > 0; ) {
	    if(event == null)
		event = new TreeModelEvent(this,
					   path,
					   indices,
					   nodes);
	    ((TreeModelListener)listeners[--i]).treeNodesChanged(event);
	}
    }

    /**
     * Invokes treeNodesInserted() of after insertion of a node.
    */
    public void fireTreeNodesInserted(Object[] path,
				      int[] indices,
				      Object[] nodes)
    {
	if(nodeListeners == null)
	    return;

	Object[] listeners = null;
	synchronized(nodeListeners) {
	    listeners = nodeListeners.toArray();
	}

	TreeModelEvent event = null;
	for(int i = listeners.length; i > 0; ) {
	    if(event == null)
		event = new TreeModelEvent(this,
					   path,
					   indices,
					   nodes);
	    ((TreeModelListener)listeners[--i]).treeNodesInserted(event);

	}
    }


    /**
     * Ivoked after removal of a node from a tree.
     * When a sbutree removed a tree, this method is ivoked only once for
     * the root node of the removed subtree but never each sibling nodes removed.
     * <p>
     * Use <code>TreeModelEvent#getPath()</code> to obtain parent node of
     * the modified node.  <code>TreeModelEvent#getChildIndices()</code> retruns
     * indices of modifies child nodes in ascending order.
    */
    public void fireTreeNodesRemoved(Object[] path,
				     int[] indices,
				     Object[] nodes)
    {
	if(nodeListeners == null)
	    return;

	Object[] listeners = null;
	synchronized(nodeListeners) {
	    listeners = nodeListeners.toArray();
	}

	TreeModelEvent event = null;
	for(int i = listeners.length; i > 0; ) {
	    if(event == null)
		event = new TreeModelEvent(this,
					   path,
					   indices,
					   nodes);
	    ((TreeModelListener)listeners[--i]).treeNodesRemoved(event);
	}
    }

    /**
     * Ivoked when tree structure under a node modified drastically.
     * When the path length given by <code>TreeModelEvent#getPath()</code> is one
     * and the first element is not the current root node, the first element is 
     * the root of the new tree.
     * <p>
     * Use <code>TreeModelEvent#getPath()</code> to obtain parent node of
     * the modified node.  <code>TreeModelEvent#getChildIndices()</code> retruns
     * null.
     */
    public void fireTreeStructureChanged(Object[] path,
					 int[] indices,
					 Object[] nodes)
    {
	if(nodeListeners == null ||
	   nodeListeners.isEmpty())
	    return;

	Object[] listeners = null;
	synchronized(nodeListeners) {
	    listeners = nodeListeners.toArray();
	}

	TreeModelEvent event = null;
	for(int i = listeners.length; i > 0; ) {
	    if(event == null)
		event = new TreeModelEvent(this,
					   path,
					   indices,
					   nodes);
	    ((TreeModelListener)listeners[--i]).treeStructureChanged(event);
	}
    }
}
