/*
 * DefaultTreeModel.java:  DefaultTreeModel to replace it in javax.swing.tree.
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: DefaultTreeModel.java,v 1.8 2002/08/09 05:28:24 nozomi Exp $
 * $Log: DefaultTreeModel.java,v $
 * Revision 1.8  2002/08/09 05:28:24  nozomi
 * fix insertion bug
 *
 * Revision 1.7  2002/06/09 12:48:36  nozomi
 * change TreeModelEvent handling
 *
 * Revision 1.6  2002/06/06 21:22:45  nozomi
 * create nodeListeners only when necessary
 *
 * Revision 1.5  2002/06/05 11:18:09  nozomi
 * change Exception message
 *
 * Revision 1.4  2002/03/10 08:58:03  nozomi
 * remove unnecessary reload() call
 *
 * Revision 1.3  2002/03/10 06:55:53  nozomi
 * modify log format
 *
 * Revision 1.2  2002/03/08 23:04:02  nozomi
 * TreeModelListener related modification
 *	
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */

package org.nomencurator.util.tree;

import java.util.Enumeration;
import java.util.Vector;
import java.util.Stack;

import org.nomencurator.util.tree.TreeModel;
import org.nomencurator.util.tree.TreeNode;
import org.nomencurator.util.tree.TreePath;

import org.nomencurator.util.tree.event.TreeModelEvent;
import org.nomencurator.util.tree.event.TreeModelListener;

/**
 * An implementaiont of <code>TreeModel<code>
 *
 * @version	09 Aug 2002
 * @author 	Nozomi `James' Ytow
 */
public class DefaultTreeModel
    implements TreeModel, TreeModelListener
	       //Serializable
{
    /** Root node of the tree */
    protected TreeNode root;

    /** EventListeners. */
    protected Vector eventListeners;

    /**
      * @see TreeNode#getAllowsChildren
      * @see TreeModel#isLeaf
      * @see #setAsksAllowsChildren
      */
    protected boolean asksAllowsChildren;

    //Constructors
    /**
     * Constructs a tree of which any node may have child node.
     *
     * @param root TreeNode object to be root of the tree.
     *
     * @see DefaultTreeModel(TreeNode, boolean)
     */
    
    public DefaultTreeModel(TreeNode node)
    {
	this(node, false);
    }

    /**
     * Constructs a single column text list model with the specified rich text
     * style.
     *
     * @param root TreeNode object to be root of the tree.
     * @param allowance false if any child node may have 
     * child node, or true if only specified node may have children.
     */
    public DefaultTreeModel(TreeNode node, 
		     boolean allowance)
    {
	if (node == null) {
	    throw new IllegalArgumentException(getClass().getName() + ": tried to constract with null root node");
	}
	
	root           = node;
	((MutableTreeNode)root).addTreeModelListener(this);
	asksAllowsChildren = allowance;

    }
    
    //Methods
    /*
     * Sets whether or not leafness of TreeNodes by passing getAllowsChildren()
     * or isLeaf().
     * Sets true to send getAllowsChildren(), or false to send isLeaf().
    */

    public void setAsksAllowsChildren(boolean allowance)
    {
	asksAllowsChildren = allowance;
    }


    /**
     * Tells leafness determination method
     *
     * @return true if a leaf node is only node prohibited to have child note,
     * or false if any node without child node (even if it allowed to have children) 
     * are leaf nodes.
     *
     * @see #asksAllowsChildren
     */
    public boolean asksAllowsChildren()
    {
	return asksAllowsChildren;
    }


    /**
     * Sets <code>node</code> as root node  of the tree.
     *
     * @exception IllegalArgumentException if <code>node</code> is null
     */
    public void setRoot(TreeNode node)
    {
	if(node == null)
	    throw new IllegalArgumentException(getClass().getName() + ".setRoot(TreeNode): can't set null node as root");

	if(root == node)
	    return;

	if(root != null) {
	    ((MutableTreeNode)root).removeTreeModelListener(this);
	}

	nodeStructureChanged(node);
	root = node;
	((MutableTreeNode)root).addTreeModelListener(this);

	//	nodeStructureChanged(node);

    }

    /**
     * Gets root node of the tree.
     * It returns null only if there is no node in the tree (how it can happen?)
     *
     * @return root of the tree
     *
     */
    public Object getRoot()
    {
	return root;
    }

    /**
     * Returns index of <code>child</code> node of <code>parent</code> node
     *
     */

    public int getIndexOfChild(Object parent,
			       Object child)
    {
	if(parent == null)
	    return -1;
	return ((TreeNode)parent).getIndex((TreeNode)child);
    }


    /**
     * Returns <code>index</code>th child node of <code>parent</code>.
     * The <code>parent</code> must be node obtained from this data source.
     * If <code>index</code> is out of valid range (i.e. index < 0 || 
     * index >= getChildCount(parent)) it does not return null.
     *
     * @param parent a node in this tree obtained from this data source
     *
     * @return indexth child of parent
     */

    public Object getChild(Object parent,
			   int index)
    {
	return ((TreeNode)parent).getChildAt(index);
    }


    /**
     * Returns number of children of <code>parent</code> which must be obtaind
     * from this data source previously.  If <code>parent</code> is a leaf or
     * doesn't have a children node, returns 0.
     *
     * @param parent a node in this tree previously obtained from this data source
     * 
     * @return number of <code>parent</code>'s child nodes
     */

    public int getChildCount(Object parent)
    {
	return ((TreeNode)parent).getChildCount();
    }

    /**
     * Returns wheter specifed <code>node</code> is a leaf node
     * by using method depending on value of asksAllowsChildren. 
     *
     * @param node to be determined
     *
     * @return true if the node is a leaf node
     *
     * @see #asksAllowsChildren
     * @see TreeModel#isLeaf(java.lang.Object)
     */
    public boolean isLeaf(Object node)
    {
	if(node == null)
	    return false;

	if(asksAllowsChildren)
	    return !((TreeNode)node).getAllowsChildren();

	return ((TreeNode)node).isLeaf();
    }

    /**
     * Call this method when you modified TreeNode on which this model
     * depends.  The model will notify modification to its all listeners.
     */

    public void reload()
    {
	reload(root);
    }


    /**
     * Sets <code>object</code> to the TreeNode identified by <code>path</code>
     * and post node chnaged event.
     *
     * @param path path to the node
     * @param object to be set as user object
     */
    public void valueForPathChanged(TreePath path,
				    Object object)
    {
	if(path == null)
	    return;

	MutableTreeNode node = (MutableTreeNode)(path.getLastPathComponent());
	node.setUserObject(object);
	nodeChanged(node);
    }

    /**
     * Inserts <code>child</code> node as <code>index</code>th child of
     * <code>parent</code> node, and ivoke nodesWereInserted.
     *
     * @param child node to be inserted
     * @param parent node to have inderted child node
     * @parem index place where the node to be inserted
     *
     */
    public void insertNodeInto(MutableTreeNode child,
			       MutableTreeNode parent,
			       int index)
    {
	if(parent == null || child == null)
	    return;

	parent.insert(child, index);
	
	int[] indices = new int[1];
	indices[0] = index;
	nodesWereInserted(parent, indices);
    }

    /**
     * Remove <code>node</code> from its parent and send
     * <code>nodesWereRemoved</code> message to create appropriate event.
     *
     * @param node a node to be removed from the tree
     *
     * @exception IllegalArgumentException if the node without parent node
     */
    public void removeNodeFromParent(MutableTreeNode node)
    {
	if(node == null)
	    return;

	MutableTreeNode parent = (MutableTreeNode)(node.getParent());

	if(parent == null)
	    throw new IllegalArgumentException("the node is a root node");

	int[] indices = new int[1];
	indices[0] = parent.getIndex(node);

	parent.remove(indices[0]);

	Object[] removed = new Object[1];
	removed[0] = node;

        nodesWereRemoved(parent, indices, removed);
    }

    /**
     * Ivoke after changing tree representaion
     * 
     * @param  node
     */
    public void nodeChanged(TreeNode node)
    {
	if(eventListeners.isEmpty() || node == null)
	    return;

	TreeNode parent = node.getParent();

	if(parent == null) {
	    if(node == root)
		nodesChanged(node, null);
	    return;
	}

	int index = parent.getIndex(node);
	if(index < 0)
	    return;

	int[] indices = new int[1];
	indices[0] = index;

	nodesChanged(parent, indices);
	
    }

    /**
     * Call this method when you modified a TreeNode on which this TreeModel
     * depends.  The model tells all listeners that a tree under <code>node</code>
     * has been modified.
     *
     */
    public void reload(TreeNode node)
    {
	if(node != null)
	    fireTreeStructureChanged(this,
				     getPathToRoot(node),
				     null,
				     null);
    }


    /**
     * Call this method when you inserted TreeNodes as children of <code>node</node>
     * at positions specified by <code>indices</code> which must be sorted in 
     * ascending order.
     *
     * @param node parent node of inserted nodes
     * @param indices where nodes were inserted
     */
     public void nodesWereInserted(TreeNode node,
				   int[] indices)
    {
	if(eventListeners.isEmpty() || //no listenr, or....
	   node == null            || //no node, or...
	   indices == null         || //no indices, or
	   indices.length == 0        //empty indices
	   )
	    return;
	
	int elements   = indices.length;
	Object[] nodes = new Object[elements];

	for(int i = 0; i < elements; i++)
	    nodes[i] = node.getChildAt(indices[i]);

	fireTreeNodesInserted(this, 
			      getPathToRoot(node), 
			      indices,
			      nodes);


    }


    /**
     * Ivoke this method after removal of nodes from <code>node</code>s child list
     * where <code>indices</code> is an array of removed nodes' indices in ascending order,
     * and <code>removed</code> is an array of removed nodes.
     * 
     *
     */
    public void nodesWereRemoved(TreeNode node,
				 int[] indices,
				 Object[] removed)
    {
	if(eventListeners.isEmpty() || //no listenr, or
	   node == null             || //no node, or
	   indices == null          || //no indices, or
	   indices.length == 0      || //empty indices, or
	   removed == null          || //no removed node, or
	   removed.length == 0         //empty removed node array
	   )
	    return;

	fireTreeNodesRemoved(this, 
			     getPathToRoot(node), 
			     indices,
			     removed);
    }

    /**
     * Ivoke this method after changing tree's reprenstaion under nodes
     * specified by <code>indices</code> of a parent <code>node</code>
     */
     public void nodesChanged(TreeNode node,
			      int[] indices)
    {
	if(eventListeners.isEmpty() || //no listenr, or....
	   node == null            //no node, or...
	   )
	    return;

	if(indices == null         || //no indices, or
	   indices.length == 0        //empty indices
	   ) {
	    fireTreeNodesChanged(this, 
				 getPathToRoot(node), 
				 null,
				 null);
	    return;
	}

	int nodes = indices.length;
	Object[] modified =  new Object[nodes];
	for(int i = 0; i < nodes; i++)
	    modified[i] = node.getChildAt(indices[i]);

	fireTreeNodesChanged(this,
			     getPathToRoot(node),
			     indices,
			     modified);
	
    }


    /**
     *
     * Ivoke when tree structure changed recursively
     *
     */

    public void nodeStructureChanged(TreeNode node)
    {
	if(node != null)
           fireTreeStructureChanged(this,
				    getPathToRoot(node),
				    null,
				    null);
    }


    /**
     * Builds a path from the parent to the root node where the original <code>node</code>
     * is the last element of returned array.  Length of the array indicates depth of the
     * <code>node</code> in the tree.
     *
     * @param node TreeNode to get its path from the root
     * @return an array of TreeNodes inidicating path from root to the given node
     */
    public TreeNode[] getPathToRoot(TreeNode node)
    {
	if(node == null)
	    return new TreeNode[0];

	Stack pathStack = new Stack();
	while(node != null) {
	    pathStack.push(node);
	    node = node.getParent();
	}

	TreeNode[] path = new TreeNode[pathStack.size()];
	int level = 0;
	while(!pathStack.isEmpty())
	    path[level++] = (TreeNode)(pathStack.pop());
	
	pathStack = null;

	return path;
    }

    /**
     * Builds and returns an array of TreeNode indicating path from root to 
     * the <code>node</code> inclusively, i.e. an array of which first 
     * element is the root and the last element is <code>node</code>.
     * Length of returned array indicates depth of the <code>node</code>
     * in the tree.
     *
     * @param node TreeNode to obtain its path
     * @param depth an int number of steps already taken towerds the root
     *        (on recursive call) which is used to size the returned array
     * @return an array of TreeNodes indicating the path
     */
    //Do we need it really?

    protected TreeNode[] getPathToRoot(TreeNode node,
				       int depth)
    {
	return getPathToRoot(node);
    }

    public TreePath getTreePath(TreeNode node)
    {
	TreeNode[] nodes = getPathToRoot(node);

	if(nodes == null || nodes.length == 0)
	    return null;
	
	return new TreePath(nodes);
    }

    /**
     * Adds a TreeModelListener
     *
     * @param listener a listner of TreeModelEvent to be add
     */
    public synchronized void addTreeModelListener(TreeModelListener listener)
    {
	if(eventListeners == null)
	    eventListeners = new Vector();
	eventListeners.addElement(listener);
    }


    /**
     * Revmoves a TreeModelLister from listening list
     *
     * @param listener a listner of TreeModelEvent to be removed
     */
    public synchronized void removeTreeModelListener(TreeModelListener listener)
    {
	if(listener != null)
	    eventListeners.removeElement(listener);	    
    }

    /**
     * Notifies tree nodes changed event to all registered TextModelListerns
     *
     */
    protected void fireTreeNodesChanged(Object source,
					Object[] path,
					int[] indices,
					Object[] nodes)
    {
	TreeModelEvent event = null;
	for(int i = eventListeners.size(); i > 0; ) {
	    if(event == null)
		event = new TreeModelEvent(source,
					   path,
					   indices,
					   nodes);
	    ((TreeModelListener)(eventListeners.elementAt(--i))).treeNodesChanged(event);
	}

    }


    /**
     * Notifies tree nodes inserted event to all registered TextModelListerns
     *
     */
    protected void fireTreeNodesInserted(Object source,
					 Object[] path,
					 int[] indices,
					 Object[] nodes)
    {
	TreeModelEvent event = null;
	for(int i = eventListeners.size(); i > 0; ) {
	    if(event == null)
		event = new TreeModelEvent(source,
					   path,
					   indices,
					   nodes);
	    ((TreeModelListener)(eventListeners.elementAt(--i))).treeNodesInserted(event);
	}

    }

    /**
     * Notifies tree nodes removed event to all registered TextModelListerns
     *
     */
    protected void fireTreeNodesRemoved(Object source,
					Object[] path,
					int[] indices,
					Object[] nodes)
    {
	TreeModelEvent event = null;
	for(int i = eventListeners.size(); i > 0; ) {
	    if(event == null)
		event = new TreeModelEvent(source,
					   path,
					   indices,
					   nodes);
	    ((TreeModelListener)(eventListeners.elementAt(--i))).treeNodesRemoved(event);
	}

    }

    /**
     * Notifies tree strucuture changed event to all registered TextModelListerns
     *
     */
    protected void fireTreeStructureChanged(Object source,
					    Object[] path,
					    int[] indices,
					    Object[] nodes)
    {
	TreeModelEvent event = null;
	for(int i = eventListeners.size(); i > 0; ) {
	    if(event == null)
		event = new TreeModelEvent(source,
					   path,
					   indices,
					   nodes);
	    ((TreeModelListener)(eventListeners.elementAt(--i))).treeStructureChanged(event);
	}
    }

    /**
     * Returns EventListers of specified <code>listenerType</code>
     *
     * @returns all objects recieving specified <code>listenerType</code> event
     *
     * @since 1.3
     */

    /* not yet implemented....is EventListener good?
    public EventListener[] getListeners(Class listenerType)
    {
    }
    */

    // Serialization support.  TBI
    /*
    private void writeObject(ObjectOutputStream s) throws IOException {
    }

    private void readObject(ObjectInputStream s) 
        throws IOException, ClassNotFoundException {
    }
    */


    protected synchronized Enumeration getListenersEnumeration()
    {
	return ((Vector)eventListeners.clone()).elements();
    }

    /**
     * Invokes treeNodesChanged() of <code>nodeListeners</code>
     * when a node or a set of siblings was modified.
     */
    public void treeNodesChanged(TreeModelEvent event)
    {
	fireTreeNodesChanged(this,
			     event.getPath(),
			     event.getChildIndices(),
			     event.getChildren());
    }

    /**
     * Invokes treeNodesInserted() of after insertion of a node.
    */
    public void treeNodesInserted(TreeModelEvent event)
    {
	fireTreeNodesInserted(this,
			     event.getPath(),
			     event.getChildIndices(),
			     event.getChildren());
    }


    /**
     * Ivoked after removal of a node from a tree.
     * When a sbutree removed a tree, this method is ivoked only once for
     * the root node of the removed subtree but never each sibling nodes removed.
     * <p>
     * Use <code>TreeModelEvent#getPath()</code> to obtain parent node of
     * the modified node.  <code>TreeModelEvent#getChildIndices()</code> retruns
     * indices of modifies child nodes in ascending order.
    */
    public void treeNodesRemoved(TreeModelEvent event)
    {
	fireTreeNodesRemoved(this,
			     event.getPath(),
			     event.getChildIndices(),
			     event.getChildren());
    }

    /**
     * Ivoked when tree structure under a node modified drastically.
     * When the path length given by <code>TreeModelEvent#getPath()</code> is one
     * and the first element is not the current root node, the first element is 
     * the root of the new tree.
     * <p>
     * Use <code>TreeModelEvent#getPath()</code> to obtain parent node of
     * the modified node.  <code>TreeModelEvent#getChildIndices()</code> retruns
     * null.
     */
    public void treeStructureChanged(TreeModelEvent event)
    {
	Object[] path   = event.getPath();
	TreeNode root = (TreeNode)getRoot();
	if(path != null &&
	   path.length == 1 && 
	   root != path[0]) {
	    TreeNode newRoot = (TreeNode)path[0];
	    if(root.getParent() != root) {
		//		((DefaultMutableTreeNode)newRoot).add((MutableTreeNode)root);
	    }
	    setRoot(newRoot);
	    return;
	}

	fireTreeStructureChanged(this,
				 path,
				 event.getChildIndices(),
				 event.getChildren());
    }
}
