/*
 * MutableTreeNode.java:  MutableTreeNode to replace javax.swing.tree.MutableTreeNode
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: MutableTreeNode.java,v 1.3 2002/08/05 23:55:09 nozomi Exp $
 * $Log: MutableTreeNode.java,v $
 * Revision 1.3  2002/08/05 23:55:09  nozomi
 * add getUserObject()
 *
 * Revision 1.2  2002/06/09 12:49:42  nozomi
 * change TreeModelEvent handling
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */

package org.nomencurator.util.tree;

import org.nomencurator.util.tree.event.TreeModelListener;

/**
 * Interface of a mutable <code>TreeModel<code>
 *
 * @version	05 Aug 2002
 * @author 	Nozomi `James' Ytow
 */
public interface MutableTreeNode
    extends TreeNode
{
    
     /**
     * Remove <code>child</code> from its parent if it has,
     * set this node to parent of the <code>child</code> node,
     * and insert the <code>child</code> node as child node of this 
     * node at given <code>index</code>.
     *
     * @param child the MutableTreeNode to be inserted under this node
     * @param index position where the child node should be inserted
     *
     */
    public void insert(MutableTreeNode child, int index);

    /**
     * Remove a child node specified by <code>index</code> from children list of the node
     * and set parent node of the removed child to null.  The child node to be removed must
     * be a MutableTreeNode.
     *
     * @param index index of the node to be removed in child list of this node
     */
    public void remove(int index);

    /**
     * Removes <code>child</code> node from children list of this node,
     * sets null as parent of the <code>child</code> node.
     *p
     * @param child node to be removed
     *
     * @exception IllegalArgumentException if <code>child</code> is null, or not 
     * child node of this node
     *
     */
    public void remove(MutableTreeNode node);

    /**
     * Sets usersObject of this node to <code>object</code>
     *
     * @param object a Object specified by user for this node
     *
     * @see #getUserObject()
     * @see #toString()
     *
     */
    public void setUserObject(Object object);

    /**
     * Returns user specified object to this node
     *
     * @return user specified object to this node
     *
     * @see #setUserObject(java.lang.Object)
     * @see #toString()
     *
     */
    public Object getUserObject();

    /**
     * Removes a tree of which root is this node from its parent node, 
     * set null as its parent. If this node is the root of the tree,
     * this method does nothing.
     *
     */
    public void removeFromParent();

    /**
     * Sets parent node of the node to <code>parent</code> without changing 
     * children list of parent node.
     * Methods <code>insert()</code> and <code>remove()</code> call this
     * method to re-allocate a child node of a node.  Other methods do not
     * call this method.
     *
     * @param parent new parent to be assigned
     *
     */
    public void setParent(MutableTreeNode newParent);

    /**
     * Adds listener as <CODE>TreeModelListener</CODE> listening this object
     *
     * @param listener <CODE>TreeModelListener</CODE> to be added to
     * the listener list
     */
    public void addTreeModelListener(TreeModelListener listener);

    /**
     * Removes listener from listeners list of this object
     *
     * @param listener <CODE>TreeModelListener</CODE> to be remvoed from
     * the listener list
     */
    public void removeTreeModelListener(TreeModelListener listener);

    /**
     * Invokes treeNodeChanged() of <code>TreeModelListeners</code>
     * when a node or a set of siblings was modified.
     */
    public void fireTreeNodesChanged(Object[] path,
				     int[] indices,
				     Object[] nodes);

    /**
     * Invokes treeNodesInserted() of after insertion of a node.
    */
    public void fireTreeNodesInserted(Object[] path,
				      int[] indices,
				      Object[] nodes);

    /**
     * Ivoked after removal of a node from a tree.
     * When a sbutree removed a tree, this method is ivoked only once for
     * the root node of the removed subtree but never each sibling nodes removed.
     * <p>
     * Use <code>TreeModelEvent#getPath()</code> to obtain parent node of
     * the modified node.  <code>TreeModelEvent#getChildIndices()</code> retruns
     * indices of modifies child nodes in ascending order.
    */
    public void fireTreeNodesRemoved(Object[] path,
				     int[] indices,
				     Object[] nodes);

    /**
     * Ivoked when tree structure under a node modified drastically.
     * When the path length given by <code>TreeModelEvent#getPath()</code> is one
     * and the first element is not the current root node, the first element is 
     * the root of the new tree.
     * <p>
     * Use <code>TreeModelEvent#getPath()</code> to obtain parent node of
     * the modified node.  <code>TreeModelEvent#getChildIndices()</code> retruns
     * null.
     */
    public void fireTreeStructureChanged(Object[] path,
					 int[] indices,
					 Object[] nodes);
}
