/*
 *
 * AbstractRowColumnLayout.java:  a LayoutManager imelementation supporting multiple rows
 *
 * Copyright (c) 2001 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */
/*
 * $Id: AbstractRowColumnLayout.java,v 1.3 2002/09/05 05:10:29 nozomi Exp $
 * $Log: AbstractRowColumnLayout.java,v $
 * Revision 1.3  2002/09/05 05:10:29  nozomi
 * change copyright to KFC style
 *
 * Revision 1.2  2002/01/25 05:43:38  nozomi
 * Components can fill width of parent Container
 */

package jp.kyasu.awt;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.LayoutManager;

/**
 * The <code>AbstractRowColumnLayout</code> class is a layout manager that lays out a
 * container's components.
 */
public abstract class AbstractRowColumnLayout 
    implements RowColumnLayout 
{
    int orientation;
    int gap;
    int horizontalPlacement;
    int verticalPlacement;
    int lines;
    
    int lineHead;
    int lineAdvance;
    int lineOrigin;
    int lineDirection;
    int lineWidth;
    int lineLength;
	
    boolean equalSize;
    
    int maxAdvance;
    int maxHeight;
    
    int initialLines;
    int initialComponentsInLine;
    
  
    /**
     * Constructs a <code>AbstractRowColumnLayout</code> object.
     */
    public AbstractRowColumnLayout(int horizontalPlacement, 
		      int verticalPlacement, 
		      int orientation, int gap, 
		      boolean equalSize,
		      int initialLines, int initialComponentsInLine)
    {
	setPlacements(horizontalPlacement, verticalPlacement);
	setOrientation(orientation);
	setGap(gap);
	setEqualSize(equalSize);
	setInitialLines(initialLines);
	setInitialComponentsInLine(initialComponentsInLine);
    }
    
    public void setHorizontalPlacement(int horizontalPlacement)
    {
	this.horizontalPlacement = horizontalPlacement;
    }
    
    public void setVerticalPlacement(int verticalPlacement)
    {
	this.verticalPlacement = verticalPlacement;
    }
    
    public void setPlacements(int horizontalPlacement, int verticalPlacement)
    {
	setHorizontalPlacement(horizontalPlacement);
	setVerticalPlacement(verticalPlacement);
    }
    
    public void setOrientation(int orientation)
    {
	this.orientation = orientation;
    }
    
    public void setGap(int gap)
    {
	this.gap = gap;
    }
    
    public int getGap()
    {
	return gap;
    }
    
    public void setEqualSize(boolean equalSize)
    {
	this.equalSize = equalSize;
    }
    
    public boolean getEqualSize()
    {
	return equalSize;
    }
    
    public void setInitialLines(int initialLines)
    {
	this.initialLines = initialLines;
    }
    
    public void setInitialComponentsInLine(int initialComponentsInLine)
    {
	this.initialComponentsInLine = initialComponentsInLine;
    }
    
    
    /**
     * Adds the specified component with the specified name to
     * the layout.
     * @param name the component name
     * @param comp the component to be added
     */
    public void addLayoutComponent(String name, Component comp)
    {
	// do nothing
    }
    
    /**
     * Removes the specified component from the layout.
     * @param comp the component ot be removed
     */
    public void removeLayoutComponent(Component comp)
    {
	// do nothing
    }
    
    /**
     * Calculates the preferred size dimensions for the specified
     * panel given the components in the specified parent container.
     * @param parent the component to be laid out
     *
     * @see #minimumLayoutSize
     */
    public Dimension preferredLayoutSize(Container parent)
    {
	return getLayoutSize(parent, true);
    }
    
    /**
     * Calculates the minimum size dimensions for the specified
     * panel given the components in the specified parent container.
     * @param parent the component to be laid out
     * @see #preferredLayoutSize
     */
    public Dimension minimumLayoutSize(Container parent)
    {
	return getLayoutSize(parent, false);
    }

    /**
     * Calculates the preferred size dimensions for the specified
     * panel given the components in the specified parent container.
     * @param parent the component to be laid out
     * @param sizePreference true to get preferred size
     *
     * @see #minimumLayoutSize
     */
    public Dimension getLayoutSize(Container parent, boolean sizePreference)
    {
	maxAdvance   = 0;
	maxHeight  = 0;
	lines = 0;
	Dimension parentAdvances = getAdvances(parent);
	return determineSize(parent, 
			     parentAdvances, 
			     determineLines(parent, parentAdvances, sizePreference));
    }
    
    protected Dimension getSize(Component component, boolean preferSize)
    {
	if(preferSize)
	    return component.getPreferredSize();
	
	return component.getMinimumSize();
    }

    protected int determineLines(Container parent, 
				 Dimension parentAdvances,
				 boolean sizePreference)
    {
	int visibleComponents = 0;
	int totalAdvance = 0;
	int currentLineLength = 0;
	int maxHeightInLine = 0;
	int maxLineLength = 0;

	int compCount = parent.getComponentCount();

	//trick...
	/*
	if(parentAdvances.width == 0 && compCount > 4)
	    compCount /= 2;
	*/
	lineLength = 0;
	lines = 0;

	for (int i = 0; i < compCount; i++) {
	    Component c = parent.getComponent(i);
	    if (!c.isVisible())
		continue;

	    visibleComponents++;

	    Dimension d = getAdvances(c, sizePreference);

	    if(d.height > maxHeightInLine)
		maxHeightInLine = d.height;
	    
	    lineLength += d.width;
	    currentLineLength += d.width;
	    totalAdvance += d.width; 
	    if(parentAdvances.width > 0 && currentLineLength > parentAdvances.width) {

		lines++;
		currentLineLength -= d.width;
		currentLineLength -= gap;
		if(currentLineLength < 0)
		    currentLineLength = 0;
		if(currentLineLength > maxLineLength)
		    maxLineLength = currentLineLength;
		currentLineLength = d.width;
	    }
	    
	    currentLineLength += gap;
	    
	    if(d.width > maxAdvance)
		maxAdvance = d.width;
	    if(d.height > maxHeight)
		maxHeight = d.height;
	}

	if(parentAdvances.width > maxAdvance)
	    maxAdvance = parentAdvances.width / (parentAdvances.width/maxAdvance);
	
	if(lineLength != 0)
	    lines++;
	lineLength = maxLineLength;

	int height = lines * maxHeight;
	int width  = maxAdvance * visibleComponents/lines;
	/*
	if(getComponentAdvance(parent) == 0) {
	    while(width > height * 2) {
		lines *= 2;
		height = lines * maxHeight;
		width  = maxAdvance * visibleComponents/lines;
	    }
	}
	*/
	return visibleComponents;
    }

    protected Dimension determineSize(Container parent, 
				      Dimension parentAdvances,
				      int visibleComponents)
    {

	int x = 0;
	int y = 0;

	if (visibleComponents > 1) {
	    int parentWidth = parentAdvances.width;
	    if(parentWidth == 0) {
		if((initialLines == 0 &&
		    initialComponentsInLine == 0) ||
		   initialLines == 1)
		    parentWidth = lineLength + gap * (visibleComponents - 1);
		else if(initialLines == 0) {
		    parentWidth = (gap + maxAdvance) * initialComponentsInLine - gap;
		}
	    }

	    if(parentWidth != 0) {
		int componentsInLine = (parentWidth + gap)/(maxAdvance + gap);
		if(componentsInLine != 0) {
		    lines = visibleComponents / componentsInLine;
		    if(visibleComponents % componentsInLine != 0)
			lines++;
		    x = componentsInLine * (maxAdvance + gap) - gap;
		    y = maxHeight * lines;
		}
	    }
	    else {
		x = parentAdvances.width;
		y = maxHeight * initialLines;
	    }
	}

	Insets insets = getInsets(parent);
	return new Dimension(x + (insets.left + insets.right),
			     y + (insets.top  + insets.bottom));
    }

    
    /**
     * Lays out the container in the specified panel.
     * @param parent the component which needs to be laid out
     */
    public void layoutContainer(Container parent)
    {
	setLineParameters(parent);

	drawComponents(parent);
    }

    protected void drawComponents(Container parent)
    {
	drawComponents(parent, 0, parent.getComponentCount());
    }

    protected void drawComponents(Container parent,
				 int from,
				 int to)
    {
	int componentAt      = lineOrigin;
	int remaining        = lineWidth;
	int componentAdvance = maxAdvance;
	
	for (int i = from; i < to; ++i) {
	    Component c = parent.getComponent(i);
	    if (!c.isVisible())
		continue;
	    if(!equalSize)
		componentAdvance = getComponentAdvance(c);
	    if(remaining < componentAdvance) {

		lineHead += lineAdvance;
		remaining = lineWidth;
		componentAt = lineOrigin;
	    }

	    if(componentAt == lineOrigin && lineDirection == -1)
		componentAt -= componentAdvance;

	    setBounds(c, componentAt, lineHead, componentAdvance, maxHeight);
	    remaining -= componentAdvance;
	    componentAt += lineDirection * componentAdvance;
	}

    }

    protected Dimension swap(Dimension d)
    {
	return new Dimension(d.height, d.width);
    }

    public abstract Dimension getAdvances(Component component, boolean preferSize);

    public abstract Dimension getAdvances(Component component);

    public abstract int getComponentAdvance(Component component);

    public abstract int getComponentAdvance(Component component,
				      boolean preferSize);

    public abstract int getLineAdvance(Component component,
				 boolean preferSize);

    protected abstract Insets getInsets(Container container);


    protected abstract void setBounds(Component component,
			     int componentAt,
			     int lineHead,
			     int componentAdvance,
			     int componentHeight);

    protected abstract void setLineParameters(Container parent);

}

