/*
 * ColumnarLayout.java:  a LayoutManager imelementation supporting multiple rows
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */
/*
 * $Id: ColumnarLayout.java,v 1.2 2002/09/22 19:47:19 nozomi Exp $
 * $Log: ColumnarLayout.java,v $
 * Revision 1.2  2002/09/22 19:47:19  nozomi
 * KFC style copyright
 *
 * Revision 1.1  2002/01/29 08:25:41  nozomi
 * initial import into CVS
 *
 */

package jp.kyasu.awt;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.LayoutManager;

/**
 * The <code>ColumnarLayout</code> class is a layout manager that lays out a
 * container's components.
 *
 * @version 19 Jan 2002
 * @author Nozomi `James' Ytow
 */
public class ColumnarLayout
    implements LayoutManager
{
    /**
     *
     */
    protected Insets insets;

    protected static Insets defaultInsets = new Insets(6, 6, 5, 6);

    protected int horizontalPlacement;

    protected int verticalGap;

    protected static int defaultGap = 5;

    public static final int LEFT   = 0;
    public static final int CENTER = 1;
    public static final int RIGHT  = 2;

    protected boolean filling;

    /**
     * Constructs a <code>ColumnarLayout</code> object.
     */
    public ColumnarLayout()
    {
	this(defaultInsets);
    }

    /**
     * Constructs a <code>ColumnarLayout</code> object.
     */
    public ColumnarLayout(Insets insets)
    {
	this(insets, CENTER);
    }


    /**
     * Constructs a <code>ColumnarLayout</code> object.
     */
    public ColumnarLayout(Insets insets,
			  int placement)
    {
	this(insets, placement, defaultGap);
    }

    /**
     * Constructs a <code>ColumnarLayout</code> object.
     */
    public ColumnarLayout(Insets insets,
			  int placement, 
			  int verticalGap)
    {
	this(insets, placement, verticalGap, true);
    }

    /**
     * Constructs a <code>ColumnarLayout</code> object.
     */
    public ColumnarLayout(Insets insets,
			  int placement, 
			  int verticalGap,
			  boolean filling)
    {
	this.insets = insets;
	horizontalPlacement = placement;
	this.verticalGap = verticalGap;
	this.filling = filling;
    }


    /**
     * Lays out the container in the specified panel.
     *
     * @param parent the component which needs to be laid out
     */
    public void layoutContainer(Container parent)
    {
	Insets parentInsets = parent.getInsets();
	Dimension parentSize = parent.getSize();

	int width  = parentSize.width;
	int origin = parentInsets.left + insets.left;
	if(horizontalPlacement == RIGHT)
	    origin = width - parentInsets.right - insets.right;

	int componentCount = parent.getComponentCount();
	int x = origin;
	    
	int y = parentInsets.top + insets.top;

	int maxComponentWidth = 
	    preferredLayoutSize(parent).width - parentInsets.right - parentInsets.left - insets.left - insets.right;

	for (int i = 0; i < componentCount; ++i) {
	    Component c = parent.getComponent(i);
	    if (!c.isVisible())
		continue;

	    Dimension prefSize = c.getPreferredSize();
	    if(!filling)
		maxComponentWidth = prefSize.width;

	    switch(horizontalPlacement) {
	    case LEFT:
		break;
	    case CENTER:
		x = origin + (width - maxComponentWidth)/2;
		break;
	    case RIGHT:
		x = origin - maxComponentWidth;
		break;
	    }
	    c.setBounds(x, y, maxComponentWidth, prefSize.height);

	    y += prefSize.height + verticalGap;
	}
    }

    /**
     * Calculates the preferred size dimensions for the specified
     * panel given the components in the specified parent container.
     * @param parent the component to be laid out
     *
     * @see #minimumLayoutSize
     */
    public Dimension preferredLayoutSize(Container parent)
    {
	return getLayoutSize(parent, true);
    }
    
    /**
     * Calculates the minimum size dimensions for the specified
     * panel given the components in the specified parent container.
     * @param parent the component to be laid out
     * @see #preferredLayoutSize
     */
    public Dimension minimumLayoutSize(Container parent)
    {
	return getLayoutSize(parent, false);
    }


    /**
     * Calculates the preferred size dimensions for the specified
     * panel given the components in the specified parent container.
     * @param parent the component to be laid out
     * @param sizePreference true to get preferred size
     *
     * @see #minimumLayoutSize
     */

    public Dimension getLayoutSize(Container parent, boolean sizePreference)
    {
	int compCount = parent.getComponentCount();
	int width  = 0;
	int height = 0;
	int visibleComponents = 0;

	Dimension d = null;

	for (int i = 0; i < compCount; i++) {
	    Component c = parent.getComponent(i);
	    if (!c.isVisible())
		continue;

	    visibleComponents++;

	    if(sizePreference)
		d = c.getPreferredSize();
	    else
		d = c.getMinimumSize();

	    if(width < d.width)
		width = d.width;
	    height += d.height;
	}

	Insets parentInsets = parent.getInsets();
	width += parentInsets.right + parentInsets.left + insets.right + insets.left;
	height += parentInsets.top + parentInsets.bottom + insets.top + insets.bottom;

	if (visibleComponents > 1)
	    height += (visibleComponents - 1) * verticalGap;

	return new Dimension(width, height);
    }

    /**
     * Adds the specified component with the specified name to
     * the layout.
     * @param name the component name
     * @param comp the component to be added
     */
    public void addLayoutComponent(String name, Component comp)
    {
	// do nothing
    }
    
    /**
     * Removes the specified component from the layout.
     * @param comp the component ot be removed
     */
    public void removeLayoutComponent(Component comp)
    {
	// do nothing
    }
    
}
