/*
 * Tab.java
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */
/*
 * $Id: Tab.java,v 1.6 2002/08/27 01:48:25 nozomi Exp $
 * $Log: Tab.java,v $
 * Revision 1.6  2002/08/27 01:48:25  nozomi
 * add null component handling
 *
 * Revision 1.5  2002/08/14 11:29:35  nozomi
 * fix setName call
 *
 * Revision 1.4  2002/08/14 05:58:52  nozomi
 * change get/setTabbedComponent() to get/setComponent()
 *
 * Revision 1.3  2002/05/14 07:03:05  nozomi
 * modify setName method to (re-)add Tab to the model
 *
 * Revision 1.2  2002/05/14 05:19:58  nozomi
 * add pointer to tabbed Component
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */

package jp.kyasu.awt;

import jp.kyasu.graphics.Text;
import jp.kyasu.graphics.Visualizable;
import jp.kyasu.graphics.VAbstractButton;
import jp.kyasu.graphics.VButton;
import jp.kyasu.graphics.VTab;
import jp.kyasu.graphics.VText;

import java.awt.AWTEventMulticaster;
import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.ItemSelectable;
import java.awt.Panel;

import java.awt.event.ActionEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

/**
 * The <code>Tab</code> class implements a labeled button that can
 * be in either an "on" (<code>true</code>) or "off" (<code>false</code>)
 * state. Clicking on a toggle button changes its state from "on" to "off,"
 * or from "off" to "on."
 *
 * @see 	jp.kyasu.awt.Checkbox
 * @see 	jp.kyasu.awt.ButtonController
 *
 * @version 	27 Aug 2002
 * @author 	Nozomi `James' Ytow
 */
public class Tab
    extends ToggleButton
{
    //tab placement constants
    public static final int TOP    = VTab.TOP;
    public static final int LEFT   = VTab.LEFT;
    public static final int BOTTOM = VTab.BOTTOM;
    public static final int RIGHT  = VTab.RIGHT;

    /** The default background color when the toggle button is selected */
    static protected final Color TAB_BACKGROUND =
	AWTResources.getResourceColor("kfc.button.tabBackground",
				       new Color(144, 144, 144));

    /** <code>Component</code> tabbed by this <code>Tab</code> object</code> */
    Component tabbedComponent;

    /**
     * Creates a <code>Tab</code> with empty label. The state of this toggle
     * button is set to "off," and it is not part of any group.
     */
    public Tab()
    {
	this(TOP);
    }

    /**
     * Creates a <code>Tab</code> with empty label. The state of this toggle
     * button is set to "off," and it is not part of any group.
     */
    public Tab(int placement)
    {
	this("", placement);
    }

    /**
     * Creates a <code>Tab</code> with the specified label. The state of this
     * <code>Tab</code> is set to "off," and it is not part of any group.
     * @param label a string label for this <code>Tab</code>.
     */
    public Tab(Component component)
    {
	this(component.getName(), component);
    }

    /**
     * Creates a <code>Tab</code> with the specified label. The state of this
     * <code>Tab</code> is set to "off," and it is not part of any group.
     * @param label a string label for this <code>Tab</code>.
     */
    public Tab(String label, Component component)
    {
	this(label, TOP, component);
    }

    /**
     * Creates a <code>Tab</code> with the specified label. The state of this
     * <code>Tab</code> is set to "off," and it is not part of any group.
     * @param label a string label for this <code>Tab</code>.
     */
    public Tab(String label, int placement, Component component)
    {
	this(label, false, placement, component);
    }

    /**
     * Creates a <code>Tab</code> with the specified label. The state of this
     * <code>Tab</code> is set to "off," and it is not part of any group.
     * @param label a string label for this <code>Tab</code>.
     */
    public Tab(String label, boolean state, int placement, Component component)
    {
	this(label, state, new TabSelectionModel(), placement, component);
    }

    /**
     * Creates a <code>Tab</code> with the specified label. The state of this
     * <code>Tab</code> is set to "off," and it is not part of any group.
     * @param label a string label for this <code>Tab</code>.
     */
    public Tab(String label)
    {
	this(label, TOP);
    }


    /**
     * Creates a <code>Tab</code> with the specified label. The state of this
     * <code>Tab</code> is set to "off," and it is not part of any group.
     * @param label a string label for this <code>Tab</code>.
     */
    public Tab(String label, int placement)
    {
	this(label, false, null, placement);
    }

    /**
     * Creates a <code>Tab</code> with the specified label. The state of this
     * <code>Tab</code> is as specified by the <code>state</code> argument, and
     * it is not part of any group.
     * @param label a string label for this <code>Tab</code>.
     * @param state the initial state of this <code>Tab</code>.
     */
    public Tab(String label, boolean state)
    {
	this(label, state, TOP);
    }

    /**
     * Creates a <code>Tab</code> with the specified label. The state of this
     * <code>Tab</code> is as specified by the <code>state</code> argument, and
     * it is not part of any group.
     * @param label a string label for this <code>Tab</code>.
     * @param state the initial state of this <code>Tab</code>.
     */
    public Tab(String label, boolean state, int placement)
    {
	this(label, state, new TabSelectionModel(), placement);
    }

    /**
     * Creates a <code>Tab</code> with the specified label, in the specified
     * group, and set to the specified state.
     * @param label a string label for this <code>Tab</code>.
     * @param state the initial state of this <code>Tab</code>.
     * @param group a group for this <code>Tab</code>.
     */
    public Tab(String label, boolean state, TabSelectionModel model)
    {
	this(label, state, model, TOP);
    }

    /**
     * Creates a <code>Tab</code> with the specified label, in the specified
     * group, and set to the specified state.
     * @param label a string label for this <code>Tab</code>.
     * @param state the initial state of this <code>Tab</code>.
     * @param group a group for this <code>Tab</code>.
     */
    public Tab(String label, boolean state, TabSelectionModel model, int placement)
    {
	this(label, state, model, placement, null);
    }

    /**
     * Creates a <code>Tab</code> with the specified label, in the specified
     * group, and set to the specified state.
     * @param label a string label for this <code>Tab</code>.
     * @param state the initial state of this <code>Tab</code>.
     * @param group a group for this <code>Tab</code>.
     */
    public Tab(String label, 
	       boolean state,
	       TabSelectionModel model,
	       int placement,
	       Component component)

    {
	this(new Text(label == null ? "" : label), state, model, placement, component);
	setName(label);
    }

    /**
     * Creates a <code>Tab</code> with the specified text. The state of this
     * <code>Tab</code> is set to "off," and it is not part of any group.
     * @param text a text for this <code>Tab</code>.
     */
    public Tab(Text text)
    {
	this(text, TOP);
    }

    /**
     * Creates a <code>Tab</code> with the specified text. The state of this
     * <code>Tab</code> is set to "off," and it is not part of any group.
     * @param text a text for this <code>Tab</code>.
     */
    public Tab(Text text, int placement)
    {
	this(text, false, null, placement);
    }

    /**
     * Creates a <code>Tab</code> with the specified text. The state of this
     * <code>Tab</code> is as specified by the <code>state</code> argument, and
     * it is not part of any group.
     * @param text  a text for this <code>Tab</code>.
     * @param state the initial state of this <code>Tab</code>.
     */
    public Tab(Text text, boolean state)
    {
	this(text, state, TOP);
    }

    /**
     * Creates a <code>Tab</code> with the specified text. The state of this
     * <code>Tab</code> is as specified by the <code>state</code> argument, and
     * it is not part of any group.
     * @param text  a text for this <code>Tab</code>.
     * @param state the initial state of this <code>Tab</code>.
     */
    public Tab(Text text, boolean state, int placement) 
    {
	this(text, state, null, placement);
    }

    /**
     * Creates a <code>Tab</code> with the specified text, in the specified
     * group, and set to the specified state.
     * @param text  a text label for this <code>Tab</code>.
     * @param state the initial state of this <code>Tab</code>.
     * @param group a group for this <code>Tab</code>.
     */
    public Tab(Text text, boolean state, TabSelectionModel model)
    {
	this(text, state, model, TOP);
    }

    /**
     * Creates a <code>Tab</code> with the specified text, in the specified
     * group, and set to the specified state.
     * @param text  a text label for this <code>Tab</code>.
     * @param state the initial state of this <code>Tab</code>.
     * @param group a group for this <code>Tab</code>.
     */
    public Tab(Text text, boolean state, TabSelectionModel model, int placement)
    {
	this(text, state, model, placement, new Panel());
    }

    /**
     * Creates a <code>Tab</code> with the specified text, in the specified
     * group, and set to the specified state.
     * @param text  a text label for this <code>Tab</code>.
     * @param state the initial state of this <code>Tab</code>.
     * @param group a group for this <code>Tab</code>.
     */
    public Tab(Text text, boolean state, TabSelectionModel model, int placement, Component component)
    {
	this(new VText(text), state, model, placement, component, text.toString());
    }

    /**
     * Creates a <code>Tab</code> with the specified text, in the specified
     * group, and set to the specified state.
     * @param text  a text label for this <code>Tab</code>.
     * @param state the initial state of this <code>Tab</code>.
     * @param group a group for this <code>Tab</code>.
     */
    public Tab(Text text, boolean state, TabSelectionModel model, 
	       int placement, Component component,
	       String name)
    {
	this(new VText(text), state, model, placement, component, name);
    }

    /**
     * Creates a <code>Tab</code> with the specified visual object. The state of
     * this <code>Tab</code> is set to "off," and it is not part of any group.
     * @param visualizable a visual object for this <code>Tab</code>.
     */
    public Tab(Visualizable visualizable)
    {
	this(visualizable, TOP);
    }

    /**
     * Creates a <code>Tab</code> with the specified visual object. The state of
     * this <code>Tab</code> is set to "off," and it is not part of any group.
     * @param visualizable a visual object for this <code>Tab</code>.
     */
    public Tab(Visualizable visualizable, int placement)
    {
	this(visualizable, false, null, placement);
    }

    /**
     * Creates a <code>Tab</code> with the specified visual object. The state of
     * this <code>Tab</code> is as specified by the <code>state</code> argument,
     * and it is not part of any group.
     * @param visualizable a visual object for this <code>Tab</code>.
     * @param state        the initial state of this <code>Tab</code>.
     */
    public Tab(Visualizable visualizable, boolean state)
    {
	this(visualizable, state, TOP);
    }

    /**
     * Creates a <code>Tab</code> with the specified visual object. The state of
     * this <code>Tab</code> is as specified by the <code>state</code> argument,
     * and it is not part of any group.
     * @param visualizable a visual object for this <code>Tab</code>.
     * @param state        the initial state of this <code>Tab</code>.
     */
    public Tab(Visualizable visualizable, boolean state, int placement)
    {
	this(visualizable, state, null, placement);
    }

    /**
     * Creates a <code>Tab</code> with the specified visual object, in the
     * specified group, and set to the specified state.
     * @param visualizable a visual object label for this <code>Tab</code>.
     * @param state        the initial state of this <code>Tab</code>.
     * @param group        a group for this <code>Tab</code>.
     */
    public Tab(Visualizable visualizable, boolean state,
			TabSelectionModel model)
    {
	this(visualizable, state, model, TOP);
    }

    /**
     * Creates a <code>Tab</code> with the specified visual object, in the
     * specified group, and set to the specified state.
     * @param visualizable a visual object label for this <code>Tab</code>.
     * @param state        the initial state of this <code>Tab</code>.
     * @param group        a group for this <code>Tab</code>.
     */
    public Tab(Visualizable visualizable, boolean state,
	       TabSelectionModel model, int placement)
    {
	this(visualizable, state, model, placement, null);
    }

    /**
     * Creates a <code>Tab</code> with the specified visual object, in the
     * specified group, and set to the specified state.
     * @param visualizable a visual object label for this <code>Tab</code>.
     * @param state        the initial state of this <code>Tab</code>.
     * @param group        a group for this <code>Tab</code>.
     */
    public Tab(Visualizable visualizable, boolean state,
	       TabSelectionModel model, int placement,
	       Component component)
    {
	this(visualizable, state, 
	     model, placement, component, "");
    }

    /**
     * Creates a <code>Tab</code> with the specified visual object, in the
     * specified group, and set to the specified state.
     * @param visualizable a visual object label for this <code>Tab</code>.
     * @param state        the initial state of this <code>Tab</code>.
     * @param group        a group for this <code>Tab</code>.
     */
    public Tab(Visualizable visualizable, boolean state,
	       TabSelectionModel model, int placement,
	       Component component, String name)
    {
	this(new VTab(visualizable, placement),
	     state, model, component, name);
    }

    /**
     * Creates a <code>Tab</code> with the specified visual button. The state of
     * this <code>Tab</code> is set to "off," and it is not part of any group.
     * @param button a visual button label for this <code>Tab</code>.
     */
    public Tab(VAbstractButton button)
    {
	this(button, false);
    }

    /**
     * Creates a <code>Tab</code> with the specified visual button. The state of
     * this <code>Tab</code> is as specified by the <code>state</code> argument,
     * and it is not part of any group.
     * @param button a visual button label for this <code>Tab</code>.
     * @param state  the initial state of this <code>Tab</code>.
     */
    public Tab(VAbstractButton button, boolean state)
    {
	this(button, state, new TabSelectionModel());
    }

    /**
     * Creates a <code>Tab</code> with the specified visual button, in the
     * specified group, and set to the specified state.
     * @param button a visual button label for this <code>Tab</code>.
     * @param state  the initial state of this <code>Tab</code>.
     * @param group  a group for this <code>Tab</code>.
     */
    public Tab(VAbstractButton button, boolean state,
			TabSelectionModel model)
    {
	this(button, state, model, null);
    }

    /**
     * Creates a <code>Tab</code> with the specified visual button, in the
     * specified group, and set to the specified state.
     * @param button a visual button label for this <code>Tab</code>.
     * @param state  the initial state of this <code>Tab</code>.
     * @param group  a group for this <code>Tab</code>.
     */
    public Tab(VAbstractButton button, boolean state,
	       TabSelectionModel model,
	       Component component)
    {
	this(button, state, model, component, "");
    }

    /**
     * Creates a <code>Tab</code> with the specified visual button, in the
     * specified group, and set to the specified state.
     * @param button a visual button label for this <code>Tab</code>.
     * @param state  the initial state of this <code>Tab</code>.
     * @param group  a group for this <code>Tab</code>.
     */
    public Tab(VAbstractButton button, boolean state,
	       TabSelectionModel model,
	       Component component, String name)
    {
	super(button, state, model, ButtonController.TOGGLE_ON_DOWN);
	setComponent(component);
	setName(name);
    }

    /**
     * Returns <code>Componante</code> inked to this <code>Tab</code>
     *
     * @return <code>Component</code> linked to this <code>Tab</code>
     */
    public Component getComponent()
    {
	return tabbedComponent;
    }

    /**
     * Sets <code>componante</code> as the <code>Component</code>
     * inked to this <code>Tab</code>
     *
     * @param component <code>Component</code> to be linked to this <code>Tab</code>
     */
    public void setComponent(Component component)
    {
	if(tabbedComponent == component)
	    return;

	tabbedComponent = component;
	if(tabbedComponent != null)
	    tabbedComponent.setName(getName());
    }


    /**
     *
     * Returns placement of tab 
     *
     * @see setTabPlacement(int)
     */
    public int getTabPlacement()
    {
	return ((VTab)getVButton()).getTabPlacement();
    }
        
    /**
     *
     * Returns tab placement which is one of:
     * <UL>
     * <LI>VTab.TOP 
     * <LI>VTab.BOTTOM 
     * <LI>VTab.LEFT 
     * <LI>VTab.RIGHT 
     * </UL>
     * Default placement is <code>VTab.TOP</code>
     *
     * @param placement placement of tab to the contents
     *
     * @exception IllegalArgumentException if tabPlacement is not any of above 
     * valid values.
     *
     * @see getTabPlacement()
     */
    public void setTabPlacement(int placement)
    {
	if(placement != getTabPlacement())
	    ((VTab)getVButton()).setTabPlacement(placement);
    }

    /**
     * Returns the background color for the button.
     */
    public Color getButtonBackground()
    {
	return (getVButton().getState() ? getBackground() : TAB_BACKGROUND);
    }

    /**
     * Sets component's name to <code>String</code> specified by <code>name</code>
     * @param name <code>String</code> to be name of this <code>Tab</code> and
     * its <code>tabbedComponent</code>
     */
    public void setName(String name)
    {
	super.setName(name);
	if(tabbedComponent != null)
	    tabbedComponent.setName(name);
	TabSelectionModel model = getModel();
	if(model != null) {
	    model.remove(this);
	    model.add(this);
	}
    }

    public TabSelectionModel getModel()
    {
	return (TabSelectionModel)getBooleanStateGroup();
    }

    public void setModel(TabSelectionModel model)
    {
	if(model == getModel())
	    return;
	setBooleanStateGroup(model);
    }

    /** Internal constant for serialization */
    /*
    static protected final String itemListenerK = "itemL".intern();

    private void writeObject(java.io.ObjectOutputStream s)
	throws java.io.IOException
    {
	s.defaultWriteObject();
	jp.kyasu.awt.ListenerSerializer.write(s,
					      itemListenerK,
					      itemListener);
	s.writeObject(null);
    }

    private void readObject(java.io.ObjectInputStream s)
	throws ClassNotFoundException, java.io.IOException
    {
	s.defaultReadObject();
	Object keyOrNull;
	while ((keyOrNull = s.readObject()) != null) {
	    String key = ((String)keyOrNull).intern();
	    if (key == itemListenerK)
		addItemListener((ItemListener)s.readObject());
	    else // skip value for unrecognized key
		s.readObject();
	}
    }
    */

}
