/*
 * TabSelectionModel.java
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */
/*
 * $Id: TabSelectionModel.java,v 1.6 2002/08/14 05:59:46 nozomi Exp $
 * $Log: TabSelectionModel.java,v $
 * Revision 1.6  2002/08/14 05:59:46  nozomi
 * add getTab(int); change get/setTabbedComponent() to get/setComponent()
 *
 * Revision 1.5  2002/08/11 23:38:59  nozomi
 * modify Tab insertion handling
 *
 * Revision 1.4  2002/05/14 09:48:22  nozomi
 * add getTabCount()
 *
 * Revision 1.3  2002/05/14 06:28:46  nozomi
 * add getTab()
 *
 * Revision 1.2  2002/05/14 01:46:11  nozomi
 * handles Containers of tabs and tabbed panes
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */

package jp.kyasu.awt;

import java.awt.CardLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;

import java.util.Enumeration;
import java.util.Vector;

import jp.kyasu.awt.event.ChangeEvent;
import jp.kyasu.awt.event.ChangeListener;

/**
 * The <code>TabSelectionModel</code> class is used to create a
 * multiple-exclusion scope for a set of <code>Tab</code> objects.
 * <p>
 * Exactly one <code>Tab</code> in a <code>TabSelectionModel</code> can be
 * in the <code>true</code> state at any given time.
 *
 * @see 	jp.kyasu.awt.BooleanStateGroup
 * @see 	jp.kyasu.awt.SingleSelectionModel
 *
 * @version 	14 Aug 2002
 * @author 	Nozomi `James' Ytow
 */
public class TabSelectionModel
    extends BooleanStateGroup
    implements SingleSelectionModel
{

    /** The current selected item index */
    transient protected int selectedIndex;

    protected int tabPlacement;

    /**
     * <code>Vector</code> of <code>Tab</code>s.
     * Note that <code>Tab</code> implements <code>BooleanState</code>
     * via <code>ToggleButton</code>
     */
    protected Vector tabs;

    protected Panel tabContainer;
    protected BorderedTabbedPanel tabbedPaneContainer;
    protected CardLayout paneLayout;

    transient protected Vector changeListeners;
    protected ChangeEvent changeEvent;


    /**
     * Constructs a <code>TabSelectionModel</code>
     * having default tab placement
     */
    public TabSelectionModel()
    {
	this(TabbedPane.TOP);
    }

    /**
     * Constructs a <code>TabSelectionModel</code>
     * with given <code>placement</code> of tabs
     *
     * @param placement <code>int</code> specifies tab placement
     */
    public TabSelectionModel(int placement) 
    {
	super();
	selectedIndex = UNSELECTED;
	tabs = new Vector();
	setTabPlacement(placement);
	createTabView();
	createTabbedPaneView();
    }

    /**
     * Creates <code>Container</code> of tabs.
     * The subclasses may override this method.
     */
    protected void createTabView()
    {
	tabContainer = new Panel(new DefaultTabLayout(this));
    }

    /**
     * Creates <code>Container</code> of tabbed panes.
     * The subclasses may override this method.
     */
    protected void createTabbedPaneView()
    {
	tabbedPaneContainer = new BorderedTabbedPanel(this);
	paneLayout = (CardLayout)tabbedPaneContainer.getLayout();
    }

    /**
     * Returns <code>Panel</code> representing a view of tabs
     *
     * @return Panel representing a view of tabs
     */ 
    public Panel getTabView()
    {
	return tabContainer;
    }

    /**
     * Returns <code>BorderedTabbedPanel</code> representing 
     * container of tabbed panes
     *
     * @return BorderedTabbedPanel representing 
     * container of tabbed panes
     */
    public BorderedTabbedPanel getTabbedPaneView()
    {
	return tabbedPaneContainer;
    }

    /**
     * Sets <code>placement</code> of tabs
     *
     * @param placement <code>int</code> representing placement of tabs
     */
    public void setTabPlacement(int placement)
    {
	if(tabPlacement == placement)
	    return;
	tabPlacement = placement;
	if(tabs == null)
	    return;
	Enumeration e = tabs.elements();
	while(e.hasMoreElements()){
	    ((Tab)e.nextElement()).setTabPlacement(placement);
	}
	if(tabContainer != null)
	    tabContainer.invalidate();
	if(tabbedPaneContainer != null)
	    tabbedPaneContainer.invalidate();
    }

    /**
     * Returns tab placement
     *
     * @return int representing tab placement
     */
    public int getTabPlacement()
    {
	return tabPlacement;
    }

    /**
     * Adds <code>tab</code> at <code>index</code>.
     * The <code>tab</code> will be append tab at the last of tab list
     * if <code>index</code> is -1.
     * It returns <code>int</code> representing index where
     * the <code>tab</code> was added to.
     *
     * @param tab <code>Tab</code> to be added
     * @param index <code>int</code> representing position of tab, or -1 to
     * append <code>tab</code> at last of the tab list
     *
     * @return int represents index where <code>tab</code> added
     */
    public int add(Tab tab, int index)
    {
	if(tabs == null)
	    tabs = new Vector();

	if(tabs.contains(tab)) {
	    int oldIndex = tabs.indexOf(tab);
	    if(oldIndex == index)
		return index;

	    remove(tab);
	    if(index >= oldIndex)
		index--;
	}

	tabContainer.add(tab, index);
	Component component = tab.getComponent();
	tabbedPaneContainer.add(tab.getName(), component);
	((TabLayout)tabContainer.getLayout()).setSize(component, tab);
	if(index == -1)
	    index = tabs.size();
	tabs.insertElementAt(tab, index);

	if(tabs.size() == 1)
	    setSelectedIndex(0);

	return index;
    }


    /**
     * Appends <code>tab</code> at the end of tab list
     *
     * @param tab <code>Tab</code> to be added
     */
    public synchronized void add(Tab tab)
    {
	add(tab, -1);
    }

    /**
     * Removes <code>tab</code> and its component
     *
     * @param tab <code>Tab</code> to be removed
     */
    public void remove(Tab tab)
    {
	if(!tabs.contains(tab))
	    return;

	
	tabContainer.remove(tab);
	tabbedPaneContainer.remove(tab);
	tabs.remove(tab);

	if(tabs.indexOf(tab) == getSelectedIndex())
	    setSelectedIndex(-1);

	invalidate();
	//validate();
    }

    /**
     * Returns index of <code>tab</code>, or -1
     * if this model does not contain <code>tab</code>
     *
     * @param tab <code>Tab</code> of which index to be returned
     * @return int representing index of <code>tab</code>, or -1
     * if this model does not contain <code>tab</code>
     */
    public int indexOf(Tab tab)
    {
	int members = tabs.size();
	int i = 0;
	for( ; i < members; i++) {	
	    if(tabs.elementAt(i) == tab) 
	       break;
	}

	return (i == members)?  UNSELECTED:i;
    }

    /**
     * Returns index of <code>component</code>, or -1
     * if this model does not contain <code>component</code>
     *
     * @param component <code>Component</code> of which index to be returned
     * @return int representing index of <code>component</code>, or -1
     * if this model does not contain <code>component</code>
     */
    public int indexOf(Component component)
    {
	if(component instanceof Tab)
	    return indexOf((Tab)component);

	int members = tabs.size();
	int i = 0;
	for( ; i < members; i++) {	
	    if(component == ((Tab)tabs.elementAt(i)).getComponent()) 
	       break;
	}

	return (i == members)?  UNSELECTED:i;
    }

    /**
     * Selects <code>component</code> if it is a tabbed pane in this model,
     * or do nothing if not.
     *
     * @param component <code>Component</code> to be selected
     */
    public void select(Component component)
    {
	if(component instanceof Tab)
	    setSelectedBooleanState((BooleanState)component);
	else 
	    setSelectedIndex(indexOf(component));
    }

    /**
     * Selects <code>tab</code> if it is in this model,
     * or do nothing if not.
     *
     * @param tab <code>Tab</code> to be selected
     */
    public void select(Tab tab)
    {
	setSelectedBooleanState(tab);
    }

    /**
     * Sets the currently selected boolean state in this group to be the
     * specified boolean state. This method sets the state of that boolean
     * state to <code>true</code> and sets all other boolean state in this
     * group to be <code>false</code>.
     * <p>
     * If the boolean state argument is <code>null</code> or belongs to a
     * different group, then this method does nothing.
     *
     * @param state the <code>BooleanState</code> to set as the current state.
     */
    public synchronized void setSelectedBooleanState(BooleanState booleanState)
    {
	if(!(booleanState instanceof Tab))
	    return;

	super.setSelectedBooleanState(booleanState);

	Tab tab = (Tab)booleanState;

	int i = indexOf(tab);

	if(i == UNSELECTED) {
	    add(tab);
	    i = tabs.size() - 1;
	}

	int oldSelection = selectedIndex;
	selectedIndex = i;
	if(oldSelection != i) {
	    Graphics g = tabbedPaneContainer.getGraphics();
	    if(g != null) {
		Dimension d = tabbedPaneContainer.getSize();
		g.clearRect(0, 0, d.width, d.height);
		g.dispose();
	    }
	    paneLayout.show(tabbedPaneContainer, ((Component)(tabs.elementAt(selectedIndex))).getName());
	    fireStateChanged();
	}
    }

    /**
     * Returns selected index
     *
     * @return selected index or -1 if nothing is selected
     *
     * @see setSelectedIndex(int)
     */
    public int getSelectedIndex()
    {
	return selectedIndex;
    }


    /**
     * Sets selected index to <code>index</code> and notifies to all
     * listeners if model changed (i.e. <code>index</code> != 
     * current selected index).
     *
     * @param index an int specifying selection index
     *
     * @see getSelectedIndex()
     * @see addChangeListener(ChangeListener)
     */
    public void setSelectedIndex(int index)
    {
	if(index == UNSELECTED)
	    clearSelection();
	else
	    setSelectedBooleanState((BooleanState)(tabs.elementAt(index)));
    }

         
    /**
     * Clears selection (sets selected index to -1)
     *
     */
    public void clearSelection()
    {
	selectedBooleanState.setState(false);
	selectedBooleanState = null;
	selectedIndex = UNSELECTED;
    }

    /**
     * Returns selected <code>Tab</code> or null
     * if no tab is selected
     *
     * @return Tab selected, or null if no <code>Tab</code> is selected.
     */
    public Tab getTab()
    {
	if(tabs == null || 
	   selectedIndex == UNSELECTED)
	    return null;
	
	return (Tab)tabs.elementAt(selectedIndex);
    }

    /**
     * Returns <code>Tab</code> at specified <CODE>index</CODE>
     *
     * @param index int specifying tab position
     *
     * @return Tab selected, or null if no <code>Tab</code> is selected.
     */
    public Tab getTab(int index)
    {
	return (Tab)tabs.elementAt(index);	
    }


    /**
     * Returns true if an item is selected in the model
     *
     * @return true if an item is selected
     */
    public boolean isSelected()
    {
	return selectedIndex != UNSELECTED;
    }



    /**
     * Adds <code>listener</code> to the list of <code>ChangeListener</code>s
     *
     * @param listener a <code>ChangeLister</code> to be added
     */
    public synchronized void addChangeListener(ChangeListener listener)
    {
	if(changeListeners == null)
	    changeListeners = new Vector();
	changeListeners.addElement(listener);
    }


    /**
     * Removes <code>listener</code> from the list of <code>ChangeListener</code>s
     *
     * @param listener a <code>ChangeLister</code> to be removed
     */
    public synchronized  void removeChangeListener(ChangeListener listener)
    {
	if(changeListeners == null) 
	    return;

	changeListeners.removeElement(listener);
	if(changeListeners.isEmpty())
	    changeListeners = null;
    }

    /**
     * Send a <code>ChangeEvent</code> to all <code>ChangeListener</code>s
     * resisted.
     */
    protected void fireStateChanged()
    {
	if(changeListeners == null)
	    return;
	Enumeration e = null;
	synchronized(changeListeners) {
	    e = ((Vector)changeListeners.clone()).elements();
	}
	while(e.hasMoreElements()) {
	    if (changeEvent == null)
		changeEvent = new ChangeEvent(this);
	    ((ChangeListener)(e.nextElement())).stateChanged(changeEvent);
	}
    }

    /**
     * Returns number of tabs in this tabbed pane
     *
     * @return int specifying number of tab pages
     */
    public int getTabCount()
    {
	return tabs.size();
    }

    public void validate()
    {
	if(tabContainer != null)
	    tabContainer.validate();
	if(tabbedPaneContainer != null)
	    tabbedPaneContainer.validate();
    }

    public void invalidate()
    {
	if(tabContainer != null)
	    tabContainer.invalidate();
	if(tabbedPaneContainer != null)
	    tabbedPaneContainer.invalidate();
    }
}
