/*
 * TabbedPane.java:  class compatible with javax.swing.JTabbedPane
 *
 * Copyright (c) 2001, 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */
/*
 * $Id: TabbedPane.java,v 1.9 2002/08/14 06:00:11 nozomi Exp $
 * $Log: TabbedPane.java,v $
 * Revision 1.9  2002/08/14 06:00:11  nozomi
 * some methods were implement
 *
 * Revision 1.8  2002/08/13 16:16:18  nozomi
 * fix default Insets
 *
 * Revision 1.7  2002/08/13 15:51:37  nozomi
 * use default Insets, instead of embedded immidiate value
 *
 * Revision 1.6  2002/05/14 09:49:01  nozomi
 * put methods to TabSelectionModel
 *
 * Revision 1.5  2002/02/21 21:14:06  nozomi
 * Add more comments
 *
 * Revision 1.4  2002/02/21 19:34:37  nozomi
 * add minor comment
 *
 * Revision 1.3  2002/01/30 23:58:46  nozomi
 * Uses TabbbedSplitBar
 *
 * Revision 1.2  2002/01/29 08:00:01  nozomi
 * Use TabbedPaneSize via LayoutManager
 *
 * Revision 1.1.1.1  2002/01/16 12:33:33  ryo
 * initial import into CVS
 */

package jp.kyasu.awt;

import jp.kyasu.awt.event.ChangeEvent;
import jp.kyasu.awt.event.ChangeListener;

import jp.kyasu.graphics.Icon;
import jp.kyasu.graphics.Text;
import jp.kyasu.graphics.TextBuffer;

import java.awt.AWTEvent;
import java.awt.AWTEventMulticaster;
import java.awt.CardLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.ItemSelectable;
import java.awt.FlowLayout;
import java.awt.LayoutManager;
import java.awt.LayoutManager2;
import java.awt.Point;
import java.awt.Rectangle;

import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;

import java.util.Enumeration;
import java.util.Hashtable;

/**
 * The <code>TabbledPanel</code> is the lightweight container class that can
 * switch <code>Component</code>s by selection of a <code>Tab</code>
 *
 * @version 	14 Aug 2002
 * @author 	Nozomi `James' Ytow
 */
public class TabbedPane
    extends SplitPanel
    implements ChangeListener, ItemSelectable
{
    //tab placement constants
    /** Place tabs on top */
    public static final int TOP    = Tab.TOP;

    /** Place tabs on left */
    public static final int LEFT   = Tab.LEFT;

    /** Place tabs on bottom */
    public static final int BOTTOM = Tab.BOTTOM;

    /** Place tabs on right */
    public static final int RIGHT  = Tab.RIGHT;

    /** default <CODE>Insets</CODE> */
    static protected final Insets defaultInsets = new Insets(0, 0, 0, 0);

    //tab placement holder
    protected int tabPlacement;

    protected Panel tabPanel;
    protected BorderedTabbedPanel pane;
    protected TabSelectionModel tabSelectionModel;


    /**
     * Listeners of this <code>Component</code>
     */
    transient protected ItemListener itemListeners;

    /** <code>ItemEvent</code> to be passed to listeners */
    protected ItemEvent itemEvent;

    /**
     * Creates a new split panel with default tab placement
     *
     */
    public TabbedPane()
    {
	this(TOP);
    }

    /**
     * Creates a new split panel with the specified orientation.
     *
     * @param orientation the split orientation.
     */
    public TabbedPane(int placement)
    {
	this(placement, 0);
    }

    /**
     * Creates a new split panel with the specified orientation and gaps.
     *
     * @param orientation the split orientation.
     * @param gap         the gap between components.
     */
    public TabbedPane(int placement, int gap)
    {
	this(placement, gap, defaultInsets);
    }

    /**
     * Creates a new split panel with the specified orientation and gaps.
     *
     * @param orientation the split orientation.
     * @param gap         the gap between components.
     * @param insets      the insets of the panel.
     */
    public TabbedPane(int placement, int gap, Insets insets)
    {
	super(getOrientation(placement), gap, insets);
	setTabPlacement(placement);
	tabSelectionModel = new TabSelectionModel(placement);
	tabSelectionModel.addChangeListener(this);
	tabPanel = tabSelectionModel.getTabView();
	/*
	switch(placement){
	case TOP:
	    tabPanel.setLayout(new RowsLayout(RowsLayout.BOTTOM, RowsLayout.LEFT, RowsLayout.BOTTOM_LEFT, 0, true));
	    break;
	case BOTTOM:
	    tabPanel.setLayout(new RowsLayout(RowsLayout.TOP, RowsLayout.LEFT, RowsLayout.TOP_LEFT, 0, true));
	    break;
	default:
	    tabPanel.setLayout(new FlowLayout(FlowLayout.LEFT, 0, 0));
	}
	*/


	pane = tabSelectionModel.getTabbedPaneView();

	switch(placement){
	case TOP:
	case LEFT:
	    super.add(tabPanel);
	    super.add(pane);
	    break;
	case BOTTOM:
	case RIGHT:
	    super.add(pane);
	    super.add(tabPanel);
	    break;
	}
    }

    public static int getOrientation(int placement)
    {
	switch(placement) {
	case TOP:
	case BOTTOM:
	    return VERTICAL;
	case RIGHT:
	case LEFT:
	    return HORIZONTAL;
	default:
	    return -1;
	}
    }

    //==== JTabbedPane IF=========
    /**
     *
     * Returned UI object implementing look and feel of the component
     *
     * @see setUI(TabbedPaneUI)
     */
    //public TabbedPaneUI getUI();

    /**
     *
     * Sets a UI object implementing look and feel of the component
     *
     * @param ui an UI object to be set
     *
     * @see UIDefaults.getUI(Component)
     */
    //public void setUI(TabbedPaneUI ui);


    /**
     *
     * Notification from <code>UIMangaer</code> indicating look and feel
     * was changed.  It is used to change the <code>UI</code> from 
     * default <code>UIFactroy</code> to new one.
     *
     * @override updateUI of JComponent
     *
     * @see Jcomponent#updateUI()
     */
    //public void updateUI();

    /**
     *
     * Retruns name of the <code>UI</code> class implementing
     * look and feel of the component
     *
     * @override getUIClassID of JComponent
     *
     * @return String TabbedPaneUI
     *
     * @see JComponent#getUIClassID()
     * @see UIDefaults.getUI(JComponent)
     */
    //public String getUIClassID();

    
    /**
     * 
     * Override this method in subclasses to return a subclass of
     * <code>ModelListener</code> or other implementations of
     * <code>ChangeListenr</code> when you'd like to process ChangeEvent 
     * in different way
     *
     * @see fireStateChanged()
     */
    //    protected ChangeListener createChangeListener();


    /**
     * Adds a <code>ChangeListener</code> to TabbedPane
     *
     * @param listener <code>ChangeListener</code> to be added
     *
     * @see fireStateChanged()
     * @see removeChangeListener(ChangeListener)
     *
     */
    //    public void addChangeListener(ChangeListener listener);


    /**
     *
     * Removes specified <code>ChangeListener</code> from this
     * TabbedPane
     *
     * @param listener <code>ChangeListener</code> to be removed
     *
     * @see fireStateChanged()
     * @see addChangeListener(ChangeListener)
     */
    //    public void removeChangeListener(ChangeListener listener);

    
    /**
     *
     * Sends each listener <code>ChangeEvent</code> of which source is 
     * this TabbedPane.  This method is invoked whenever <code>ChangeEvent</code>
     * is received from the model
     *
     * @see addChangeListener(ChangeListener)
     * @see EventListenerList
     */
    //    protected void fireStateChanged();

    /**
     * Adds the specified item listener to receive item events from this list.
     *
     * @param listener the item listener.
     */
    public synchronized void addItemListener(ItemListener listener)
    {
	itemListeners = AWTEventMulticaster.add(itemListeners, listener);
	enableEvents(0); // mark newEventsOnly
    }

    /**
     * Returns the selected items on the list in an array of Objects.
     * @see java.awt.ItemSelectable
     */
    public Object[] getSelectedObjects()
    {
	Object o = getSelectedComponent();
	if(o != null)
	  return new Object[]{o};
	return null;
    }

    
    /**
     * Removes the specified item listener so that it no longer receives
     * item events from this list.
     * @param l the item listener.
     */
    public synchronized void removeItemListener(ItemListener listener)
    {
	itemListeners = AWTEventMulticaster.remove(itemListeners, listener);
    }

    /**
     * Returns model relating to this <code>TabbedPane</code>
     *
     * @see setModel(SingleSelectionModel)
     */
    public SingleSelectionModel getModel()
    {
	return tabSelectionModel;
    }


    /**
     * Sets model to be used with this <code>TabbedPane</code>
     *
     * @param model to be used
     *
     * @see getModel()
     */
    public void setModel(SingleSelectionModel model)
    {
	tabSelectionModel = (TabSelectionModel)model;
    }


    /**
     *
     * Returns placement of tab 
     *
     * @see setTabPlacement(int)
     */
    public int getTabPlacement()
    {
	return tabSelectionModel.getTabPlacement();
    }
        
    /**
     *
     * Returns tab placement which is one of:
     * <UL>
     * <LI>TOP 
     * <LI>BOTTOM 
     * <LI>LEFT 
     * <LI>RIGHT 
     * </UL>
     * Default placement is <code>TOP</code>
     *
     * @param tabPlacement placement of tab to the contents
     *
     * @exception IllegalArgumentException if tabPlacement is not any of above 
     * valid values.
     *
     * @see getTabPlacement()
     */
    public void setTabPlacement(int placement)
    {
	if(tabSelectionModel == null)
	    return;

	if(placement == getTabPlacement())
	    return;

	tabSelectionModel.setTabPlacement(placement);
	super.remove(tabPanel);
	super.remove(pane);
	switch(placement) {
	case TOP:
	case LEFT:
	    super.add(tabPanel);
	    super.add(pane);
	    break;
	case BOTTOM:
	case RIGHT:
	    super.add(pane);
	    super.add(tabPanel);
	    break;
	}
    }

    
    /**
     * 
     * Returns index of selected tab, or -1 if no tab is selected
     *
     * @return index of selected tab
     *
     * @see setSelectedIndex(int)
     */
    public int getSelectedIndex()
    {
	return getModel().getSelectedIndex();
    }

    /**
     * Selects tab at <code>index</code>
     *
     * @param index of tab to be selected
     *
     * @exception IllegalArgumentException if sepcified index is out of range
     *
     * @see getSelectedIndex()
     * @see SingleSelectionModel#setSelectedIndex(int)
     */
    public void select(int index)
    {
	setSelectedIndex(index);
    }

    /**
     *
     * Sets index of tab to be selected
     *
     * @param index of tab to be selected
     *
     * @exception IllegalArgumentException if sepcified index is out of range
     *
     * @see getSelectedIndex()
     * @see SingleSelectionModel#setSelectedIndex(int)
     */
    public void setSelectedIndex(int index)
    {
	getModel().setSelectedIndex(index);
    }


    /**
     *
     * Returns <code>Component</code> correstponding to the selected tab
     * in the TabbedPane, or null if no tab is selected
     *
     * @return <code>Compopnent</code> corresponding to the selected tab
     *
     * @see setSelectedComponent(Component)
     */
    public Component getSelectedComponent()
    {
	Tab tab = tabSelectionModel.getTab();
	if(tab == null)
	    return null;
	return tab.getComponent();
    }


    /**
     * Sets selected componet.
     * The selectedIndex is set to the index corresponding to
     * given <code>Component</code>
     *
     * @exception IllegalArgumentException if no <code>Component</code> in the tabbed pane
     *
     * @see getSelectedComponent()
     */
    public void setSelectedComponent(Component component)
    {
	tabSelectionModel.select(component);
    }

    /**
     * Inserts <code>component</code> which is corresponding to either of <code>title</code>
     * or <code>icon</code> (one of them may be null), or both,  at <code>index</code>.
     * It uses <code>java.util.Vector</code> internally.  See <code>insertElementAt</code>
     * for detail of insertion.
     *
     * @param title title to be shown on the tab
     * @param icon icon to be shown on the tab
     * @param component <code>Component</code> to be shown when the tab is clicked
     * @param tip tool hint shown for the tab
     * @param index position where the tab to be inserted
     *
     * @see addTab(String, Icon, Component, String)
     * @see removeTabAt(int)
     */
    public void insertTab(String title,
			  Icon icon,
			  Component component,
			  String tip,
			  int index)
    {
	Tab tab = createTab(title, icon, component, tip);
	tabSelectionModel.add(tab, index);
    }

    /**
     *
     * Adds <code>component</code> which is corresponding to either of <code>title</code>
     * or <code>icon</code> (one of them may be null), or both.  It is a cover method
     * of <code>insertTab</code>.
     *
     * @param title title to be shown on the tab
     * @param icon icon to be shown on the tab
     * @param component <code>Component</code> to be shown when the tab is clicked
     * @param tip tool hint shown for the tab
     *
     * @see insertTab(String, Icon, Component, String, int)
     * @see removeTabAt(int)
     */
    public void addTab(String title,
		       Icon icon,
		       Component component,
		       String tip)
    {
	insertTab(title, icon, component, tip, getTabCount());
    }

    /**
     * Returns a <code>Tab</code> which is corresponding to either of <code>title</code>
     * or <code>icon</code> (one of them may be null), or both,  at <code>index</code>.
     *
     * @param title title to be shown on the tab
     * @param icon icon to be shown on the tab
     * @param component <code>Component</code> to be shown when the tab is clicked
     * @param tip tool hint shown for the tab
     *
     * @return Tab created using those parameters
     */
    protected Tab createTab(String title,
			    Icon icon,
			    Component component,
			    String tip)
    {
	TextBuffer buffer = new TextBuffer();
	if(icon != null)
	    buffer.append(icon);
	if(title != null && title.length() != 0) {
	//	buffer.setTextStyle(bold);
	    buffer.append(title);
	}
	Tab tab = new Tab(buffer.toText(), false, tabSelectionModel, tabSelectionModel.getTabPlacement(), component);
	if(tip != null)
	    tab.setToolTipText(tip);
	return tab;
    }


    /**
     *
     * Adds <code>component</code> which is corresponding to either of <code>title</code>
     * or <code>icon</code> (one of them may be null), or both.  It is a cover method
     * of <code>insertTab</code>.
     *
     * @param title title to be shown on the tab
     * @param icon icon to be shown on the tab
     * @param component <code>Component</code> to be shown when the tab is clicked
     *
     * @see insertTab(String, Icon, Component, String, int)
     * @see removeTabAt(int)
     */
    public void addTab(String title,
		       Icon icon,
		       Component component)
    {
	addTab(title, icon, component, null);
    }

    /**
     *
     * Adds <code>component</code> which is corresponding to <code>title</code>.
     * It is a cover method of <code>insertTab</code>.
     *
     * @param title title to be shown on the tab
     * @param component <code>Component</code> to be shown when the tab is clicked
     *
     * @see insertTab(String, Icon, Component, String, int)
     * @see removeTabAt(int)
     */
    public void addTab(String title,
		       Component component)
    {
	addTab(title, null, component);
    }

    /**
     *
     * Adds <code>component</code> with tab title of component name as default.
     * It is a cover method of <code>insertTab</code>.
     *
     * @override add of Container
     *
     * @param component <code>Component</code> to be shown when the tab is clicked
     *
     * @return component
     *
     * @see insertTab(String, Icon, Component, String, int)
     * @see removeTabAt(int)
     */
    public Component add(Component component)
    {
	return add(component.getName(), component);
    }


    /**
     *
     * Adds <code>component</code> with <code>title</code>.
     * It is a cover method of <code>insertTab</code>.
     *
     * @override add of Container
     *
     * @param title title to be shown on the tab
     * @param component <code>Component</code> to be shown when the tab is clicked
     *
     * @return component
     *
     * @see insertTab(String, Icon, Component, String, int)
     * @see removeTabAt(int)
     */
    public Component add(String title,
			 Component component)
    {
	addTab(title, null, component);
	return component;
    }


    /**
     *
     * Adds <code>component</code> at <code>index</code> with tab title of component name.
     * It is a cover method of <code>insertTab</code>.
     *
     * @override add of Container
     *
     * @param component <code>Component</code> to be shown when the tab is clicked
     * @param index where tab to be inserted
     *
     * @return component
     *
     * @see insertTab(String, Icon, Component, String, int)
     * @see removeTabAt(int)
     */
    public Component add(Component component,
			 int index)
    {
	insertTab(component.getName(), null, component, null, index);
	return component;
    }


    /**
     *
     * Adds <code>component</code> to tabbed pane.
     * The <code>constraints</code> is used as tab title when it is a <code>String</code>  
     * or an <code>Icon</code>, or component name will be used in other cases.
     * It is a cover method of <code>insertTab</code>.
     *
     * @override add of Container
     *
     * @param component <code>Component</code> to be shown when the tab is clicked
     * @param constraints title object displayed on the tab
     *
     * @see insertTab(String, Icon, Component, String, int)
     * @see removeTabAt(int)
     */
    public void add(Component component,
		    Object constraints)
    {
	add(component, constraints, getTabCount());
    }

    /**
     *
     * Adds <code>component</code> to tabbed pane at specified <code>index</code>.
     * The <code>constraints</code> is used as tab title when it is a <code>String</code>  
     * or an <code>Icon</code>, or component name will be used in other cases.
     * It is a cover method of <code>insertTab</code>.
     *
     * @override add of Container
     *
     * @param component <code>Component</code> to be shown when the tab is clicked
     * @param constraints title object displayed on the tab
     * @param index where tab to be inserted
     *
     * @see insertTab(String, Icon, Component, String, int)
     * @see removeTabAt(int)
     */
    public void add(Component component,
		    Object constraints,
		    int index)
    {
	String title = component.getName();
	Icon icon = null;
	if(constraints instanceof String)
	    title = (String)constraints;
	else if(constraints instanceof Icon) {
	    title = null;
	    icon = (Icon)constraints;
	}
	    
	insertTab(title, icon, component, null, index);
    }


    /**
     *
     * Removes tab at <code>index</code>.
     * It sets visibility of the removed component true to make sure
     * that the component will be visible when it will be added to
     * other container.
     *
     * @param index of tab to be removed
     *
     * @see addTab(String, Icon, Component, String)
     * @see insertTab(String, Icon, Component, String, int)
     */
    public void removeTabAt(int index)
    {
	Tab tab = tabSelectionModel.getTab(index);
	tabSelectionModel.remove(tab);
	tab.getComponent().setVisible(true);
	tab.setComponent(null);
    }


    /**
     *
     * Removes tab corresponding to specified <code>component</code>.
     *
     * @override remove of Container
     *
     * @param component <code>Component</code> to be removed from tabbed pane
     *
     * @see addTab(String, Icon, Component, String)
     * @see insertTab(String, Icon, Component, String, int)
     */
    public void remove(Component component)
    {
	int index = 0; 
	int tabs = getTabCount();
	while(index < tabs && component != getComponentAt(index)) {
	    index++;
	}
	if(index < tabs)
	    removeTabAt(index);
    }


    /**
     *
     * Removes tab and <code>Component</code> at given <code>index</code>.
     *
     * @override remove of Container
     *
     * @param index of tab to be removed
     *
     * @see addTab(String, Icon, Component, String)
     * @see removeTabAt(int)
     */
    public void remove(int index)
    {
	removeTabAt(index);
    }

    /**
     *
     * Removes all tabs and corresponding <code>Component</code>s from tabbped pane
     *
     * @override removeAll of Container
     *
     * @see addTab(String, Icon, Component, String)
     * @see removeTabAt(int)
     */
    public synchronized void removeAll()
    {
	for(int i = getTabCount(); i > 0; i--) {
	    removeTabAt(i - 1);
	}
    }


    /**
     * Returns number of tabs in this tabbed pane
     *
     * @return int specifying number of tab pages
     */
    public int getTabCount()
    {
	return tabSelectionModel.getTabCount();
    }


    /**
     *
     * Returns number of runs used to deisplay tabs currently.
     *
     * @return int rows when tabPlacement is TOP or BOTTOM,
     * columns when tabPlacement is LEFT or RIGHT, or zero
     * if no UI is set to this tabbed pane
     */
    //    public int getTabRunCount();


    /**
     * 
     * Returns title of tab at <code>index</code>
     *
     * @param index referred to
     *
     * @return title at index
     *
     * @exception IllegalArgumentException if <code>index</code> is out or range
     *
     * @see setTitleAt(int, String)
     */
    public String getTitleAt(int index)
    {
	return tabSelectionModel.getTab(index).getLabel();
    }



    /**
     * 
     * Returns icon of tab at <code>index</code>
     *
     * @param index referred to
     *
     * @return icon at index
     *
     * @exception IllegalArgumentException if <code>index</code> is out or range
     *
     * @see setIconAt(int, Icon)
     */
    //    public Icon getIconAt(int index);


    /**
     *
     * Returns desabled icon at <code>index</code>
     *
     * @param index referred to
     *
     * @return icon at index
     *
     * @exception IllegalArgumentException if <code>index</code> is out or range
     *
     * @see setDisabledIconAt(int, Icon)
     */
    //    public Icon getDisabledIconAt(int index);


    /**
     *
     * Returns  tool hint's text of tab at <code>index</code>
     *
     * @param index referred to
     *
     * @return String containing text of tool hint at <code>index</code>
     *
     * @exception IllegalArgumentException if <code>index</code> is out or range
     *
     * @see setToolTipTextAt(int, String)
     */
    public String getToolTipTextAt(int index)
    {
	return tabSelectionModel.getTab(index).getToolTipText();
    }


    /**
     *
     * Returns background color of tab at <code>index</code>
     *
     * @param index referred to
     *
     * @return Color for background of tab at <code>index</code>
     *
     * @exception IllegalArgumentException if <code>index</code> is out or range
     *
     * @see setBackgroundAt(int, Color)
     */
    public Color getBackgroundAt(int index)
    {
	return tabSelectionModel.getTab(index).getBackground();
    }



    /**
     *
     * Returns foreground color of tab at <code>index</code>
     *
     * @param index referred to
     *
     * @return Color for foreground of tab at <code>index</code>
     *
     * @exception IllegalArgumentException if <code>index</code> is out or range
     *
     * @see setForegroundAt(int, Color)
     */
    public Color getForegroundAt(int index)
    {
	return tabSelectionModel.getTab(index).getForeground();
    }


    /**
     *
     * Returns wheter tab at <code>index</code> is enabled
     *
     * @param index referred to
     *
     * @return true if tab at <code>index</code> is enabled, or false for others
     *
     * @exception IllegalArgumentException if <code>index</code> is out or range
     *
     * @see setEnabledAt(int, boolean)
     */
    public boolean isEnabledAt(int index)
    {
	return tabSelectionModel.getTab(index).isEnabled();
    }



    /**
     *
     * Returns <code>Component</code> at <code>index</code>
     *
     * @param index referred to
     *
     * @return Component at <code>index</code>
     *
     * @exception IllegalArgumentException if <code>index</code> is out or range
     *
     * @see setComponentAt(int, Component)
     */
    public Component getComponentAt(int index)
    {
	return tabSelectionModel.getTab(index).getComponent();
    }


    /**
     *
     * Returns bounds of tab at <code>index</code>, or null if the tab at <code>index</code>
     * in not visible with current UI or if no UI is set to this tabbed pane.
     *
     * @param index referred to
     *
     * @return Rectangle containing bounds of tab at <code>index</code>, or null if
     * the tab at <code>index</code> is invisible with current UI or if no UI is
     * set to this tabbed pane
     */
    public Rectangle getBoundsAt(int index)
    {
	return tabSelectionModel.getTab(index).getBounds();
    }


    /**
     *
     * Sets <code>title</code> as title of tab at <code>index</code> which may be null.
     * It throws an exception if there is no tab for given <code>index</code>.
     *
     * @param index referred to
     * @param title to be shown on the tab
     *
     * @exception IllegalArgumentException if <code>index</code> is out or range
     *
     * @see getTitleAt(int)
     */
    public void setTitleAt(int index,
			   String title)
    {
	tabSelectionModel.getTab(index).setLabel(title);
    }


    /**
     *
     * Sets <code>icon</code> as icon of tab at <code>index</code> which may be null.
     * It throws an exception if there is no tab for given <code>index</code>.
     *
     * @param index referred to
     * @param icon to be shown on the tab
     *
     * @exception IllegalArgumentException if <code>index</code> is out or range
     *
     * @see getIconAt(int)
     */
    /*
    public void setIconAt(int index,
			  Icon icon);
    */

    /**
     *
     * Sets <code>icon</code> as disabled icon of tab at <code>index</code> which may be null.
     * It throws an exception if there is no tab for given <code>index</code>.
     *
     * @param index referred to
     * @param icon to be shown on the tab when disbaled
     *
     * @exception IllegalArgumentException if <code>index</code> is out or range
     *
     * @see getDisabledIconAt(int)
     */
    /*
    public void setDisabledIconAt(int index,
				  Icon disabledIcon);
    */


    /**
     *
     * Sets <code>toolTipText</code> as tool hint's text of tab at <code>index</code> which may be null.
     * It throws an exception if there is no tab for given <code>index</code>.
     *
     * @param index referred to
     * @param toolTipText text to be shown on the tool tip of tab
     *
     * @exception IllegalArgumentException if <code>index</code> is out or range
     *
     * @see getToolTipTextAt(int)
     */
    public void setToolTipTextAt(int index,
				 String toolTipText)
    {
	tabSelectionModel.getTab(index).setToolTipText(toolTipText);
    }



    /**
     *
     * Sets <code>background</code> as backgroud color  at <code>index</code> which may be null
     * to set default background color.
     * It throws an exception if there is no tab for given <code>index</code>.
     *
     * @param index referred to
     * @param backgournd <code>Color</code> to be shown as background of the tab
     *
     * @exception IllegalArgumentException if <code>index</code> is out or range
     *
     * @see getBackgroundAt(int)
     */
    public void setBackgroundAt(int index,
				Color background)
    {
	tabSelectionModel.getTab(index).setBackground(background);
    }


    /**
     *
     * Sets <code>foreground</code> as backgroud color  at <code>index</code> which may be null
     * to set default foreground color.
     * It throws an exception if there is no tab for given <code>index</code>.
     *
     * @param index referred to
     * @param backgournd <code>Color</code> to be shown as foreground of the tab
     *
     * @exception IllegalArgumentException if <code>index</code> is out or range
     *
     * @see getForegroundAt(int)
     */
    public void setForegroundAt(int index,
				Color foreground)
    {
	tabSelectionModel.getTab(index).setForeground(foreground);
    }


    /**
     *
     * Sets whether tab at <code>index</code> is enabled.
     * It throws an exception if there is no tab for given <code>index</code>.
     *
     * @param index referred to
     * @param enabled whther the tab is enabled
     *
     * @exception IllegalArgumentException if <code>index</code> is out or range
     *
     * @see isEnabledAt(int)
     */
    public void setEnabledAt(int index,
			     boolean enabled)
    {
	tabSelectionModel.getTab(index).setEnabled(enabled);
    }


    /**
     *
     * Sets <code>component</code> as <code>Component</code> at <code>index</code>.
     * It throws an exception if there is no tab for given <code>index</code>.
     *
     * @param index referred to
     * @param component <code>Component</code> of the tab
     *
     * @exception IllegalArgumentException if <code>index</code> is out or range
     *
     * @see getComponentAt(int)
     */
    public void setComponentAt(int index,
			       Component component)
    {
	tabSelectionModel.getTab(index).setComponent(component);
    }

    /**
     *
     * Returns index of first tab having specified <code>title</code>,
     * or -1 if not tab has the <code>title</code>
     *
     * @param title title of tab 
     *
     * @return indx of first tab matched to <code>titel</code> or -1
     * when no tab has the <code>title</code>
     */
    /*
    public int indexOfTab(String title);
    */

    /**
     *
     * Returns index of first tab having specified <code>icon</code>,
     * or -1 if not tab has the <code>icon</code>
     *
     * @param icon icon of tab 
     *
     * @return indx of first tab matched to <code>icon</code> or -1
     * when no tab has the <code>icon</code>
     */
    /*
    public int indexOfTab(Icon icon);
    */


    /**
     *
     * Returns index of first tab having specified <code>component</code>,
     * or -1 if not tab has the <code>component</code>
     *
     * @param component component of tab 
     *
     * @return indx of first tab matched to <code>component</code> or -1
     * when no tab has the <code>component</code>
     */
    /*
    public int indexOfComponent(Component component);
    */


    /**
     *
     * Returns tool hint text at point specified by MouseEvent.
     *
     * @override getToolTipText of JComponent
     *
     * @param event <code>MouseEvent</code> specifying cursor position
     *
     * @return String containing text of tool hint
     *
     * @exception IllegalArgumentException if <code>index</code> is out or range
     */
    /*
    public String getToolTipText(MouseEvent event);
    */


    /**
     *
     * Returns <code>String</code> expression of this JTablePane, which has
     * debug purpose and hence contens and format of contens may vary with
     * implementation by implementation.  The returned <code>String</code>
     * may be empty but never null.
     *
     * @override paramString in JComponent
     *
     * @return String expression of this JTabbedPane
     */
    /*
    protected String paramString();
    */

    /**
     *
     * Returns <code>AccessibleContext</code> correspoing to this <code>JTabbedPane</code>.
     * The <code>AccessibleContext</code> has type of <code>AccessibleJTabbedPane</code>
     * for JTabbedPane.  An instance of <code>AccessibleJTabbedPane</code> will be 
     * constructed when necessary.
     * 
     * @definition getAccessibleContext in interface Accessible
     *
     * @overrides getAccessibleContext in JComponent
     *
     * @return AccessibleJTabbedPane works as AccessibleContext of this JTabbedPane
     */
    //public AccessibleContext getAccessibleContext();

    /**
     * Invoked when a tab selected
     *
     * @param changeEvent <code>ChangeEvent</code> to be handled
     */
    public void stateChanged(ChangeEvent changeEvent)
    {
	/*
	SingleSelectionModel model = (SingleSelectionModel)(changeEvent.getSource());
	int selected = model.getSelectedIndex();
	Graphics g = pane.getGraphics();
	if(g != null) {
	    Dimension d = pane.getSize();
	    g.clearRect(0, 0, d.width, d.height);
	    g.dispose();
	}
	paneLayout.show(pane, (String)(tabNames.elementAt(selected)));
	*/
    }

    /**
     * Creates and returns a <code>SplitBar</code> with given
     * <code>orientation</code>.
     * <P>
     * The subclasses may override this method.
     *
     * @param orientation orientation of <code>SplitBar</code>
     *
     * @return SplitBar created with given <code>orientation</code>
     */
    protected SplitBar createSplitBar(int orientation)
    {
	return new TabbedSplitBar(orientation, tabSelectionModel);
    }

    static public void main(String args[]) {
	jp.kyasu.awt.Frame f = new jp.kyasu.awt.Frame("TabbedPane");
	TabbedPane pane = new TabbedPane(Tab.BOTTOM);
	Panel p = new Panel();
	p.add(new TextField(20));
	pane.addTab("ONE", p);
	pane.addTab("ONE.ONE", new TextField(25));
	pane.addTab("ONE.TWO", new TextField(30));
	pane.addTab("TWO", new Panel());
	pane.addTab("THREE", new Panel());
	f.add(pane);
	f.pack();
	f.setVisible(true);
    }

}

/**
 * <code>Panel</code> containing <code>Tab</code>s.
 * Is it useful for classes other than <code>TabbedPane</code>?
 */
//On going....
class TabPanel
    extends Panel
{
    protected Panel pane;

    public TabPanel(Panel pane)
    {
	this.pane = pane;
    }
}

/**
 * The <code>SplitBar</code> with a break for a <code>Tab</code> on
 * <code>TabbedPane</code>.
 *
 * @version 	31 Jan 2002
 * @author 	Nozomi `James' Ytow
 */
class TabbedSplitBar
    extends SplitBar
{
    protected TabSelectionModel tabStateModel;
    
    TabbedSplitBar(int orientation, TabSelectionModel model)
    {
	super(orientation);
	setTabStateModel(model);
	//	hidden = false;
    }

    /**
     * Returns <code>TabSelectionModel</code> in use.
     *
     * @return TabSelectionModel in use
     * @see    setTabSelectionModel()
     */
    public TabSelectionModel getTabStateModel()
    {
	return tabStateModel;
    }

    /**
     * Sets <code>model</code> as <code>TabSelectionModel</code> to be used.
     *
     * @param model <code>TabSelectionModel</code> to be used
     * @see   getTabSelectionModel()
     */
    public void setTabStateModel(TabSelectionModel model)
    {
	if(tabStateModel == model)
	    return;

	tabStateModel = model;
	setOrientation(TabbedPane.getOrientation(tabStateModel.getTabPlacement()));
    }

    public void paint(Graphics g)
    {
	Tab currentTab = (Tab)(tabStateModel.getSelectedBooleanState());
	
	Rectangle bounds = currentTab.getBounds();
	Color tabColor  = currentTab.getButtonBackground();
	Color save = g.getColor();
	Dimension d = getSize();
	switch(currentTab.getTabPlacement()) {
	    case Tab.BOTTOM:
	    case Tab.RIGHT:
		g.setColor(Color.black); //?
		break;
	    case Tab.TOP:
	    case Tab.LEFT:
		g.setColor(Color.white);
		break;
	}

	if (hidden) {
	    /*
	    int barPosition = SplitPanel.SPLIT_HIDDEN_THICKNESS / 2;
	    if(orientation == SplitPanel.VERTICAL) {
		g.fillRect(0, barPosition - 1, bounds.x, barPosition);
		g.fillRect(bounds.x + bounds.width, barPosition - 1,
			   d.width - (bounds.x + bounds.width), barPosition);
		g.setColor(tabColor);
		g.fillRect(bounds.x, barPosition - 1, bounds.width, barPosition);
	    }
	    else {
		g.fillRect(0, 0, d.width, bounds.y);
		g.fillRect(0, bounds.y + bounds.height, d.width, d.width);
		g.setColor(tabColor);
		g.fillRect(0, bounds.y, d.width, bounds.height);
	    }
	    */
	    thickness = SplitPanel.SPLIT_HIDDEN_THICKNESS / 2 + 
		(orientation == SplitPanel.VERTICAL ? d.height : d.width);
	}
	else {
	    int barPosition = SplitPanel.SPLIT_HIDDEN_THICKNESS / 2;
	    if(orientation == SplitPanel.VERTICAL) {
		g.fillRect(0, barPosition, bounds.x, barPosition + 1);
		g.fillRect(bounds.x + bounds.width, barPosition,
			   d.width - (bounds.x + bounds.width), barPosition + 1);
		g.setColor(tabColor);
		g.fillRect(bounds.x, barPosition, bounds.width, barPosition + 1);
	    }
	    else {
		g.fillRect(0, 0, d.width, bounds.y);
		g.fillRect(0, bounds.y + bounds.height, d.width, d.height - (bounds.y + bounds.height));
		g.setColor(tabColor);
		g.fillRect(0, bounds.y, d.width, bounds.height);
	    }

	    thickness = SplitPanel.SPLIT_LINE_THICKNESS / 2 + 
		(orientation == SplitPanel.VERTICAL ? d.height : d.width);
	}

	g.setColor(save);
    }

}
