/*
 * TabbedPaneSize.java:  calculator of Panel size containing Tabs in a TabbedPane
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */
/*
 * $Id: TabbedPaneSize.java,v 1.2 2002/09/22 19:47:19 nozomi Exp $
 * $Log: TabbedPaneSize.java,v $
 * Revision 1.2  2002/09/22 19:47:19  nozomi
 * KFC style copyright
 *
 * Revision 1.1  2002/01/29 07:45:13  nozomi
 * initial import into CVS
 *
 */

package jp.kyasu.awt;

import java.awt.Component;
import java.awt.Dimension;

/**
 * Calculates size of <code>Panel</code> containing <code>Tab</code>s in a
 * <code>TabbedPane</code>
 *
 * @version 	25 Jan 2002
 * @author 	Nozomi `James' Ytow
 */
public class TabbedPaneSize
{
    /**
     * Size of pane to be used
     */
    Dimension paneSize;

    /**
     * Size of pane containing tabs to be used
     */
    Dimension tabPaneSize;

    /**
     * Size of <code>Tab</code> to be used
     */
    Dimension tabSize;

    int maxTabAdvance;

    /**
     * Layout manager to be used to determine size of components
     */
    AbstractRowColumnLayout layout;

    /**
     * Number of <code>Tab</code>s
     */
    int tabCounts;

    /**
     * Number of lines of <code>Tab</code>s
     */
    int tabLines;

    /**
     * Number of <code>Tab</code>s in a "Tab line".
     */
    int tabsInLine;


    /**
     * Sets <code>model</code> as <code>TabSelectionModel</code> to be used
     *
     * @param model <code>TabSelectionModel</code> to be used
     */
    public TabbedPaneSize(AbstractRowColumnLayout layout)
    {
	this.layout   = layout;
	paneSize      = new Dimension();
	tabPaneSize   = new Dimension();
	tabSize       = new Dimension();
	maxTabAdvance = 0;
	tabCounts          = 0;
	tabLines      = 0;
	tabsInLine    = 0;
    }

    /**
     * Determines size of <code>Tab</code> container using <code>component</code> and <code>tab</code> sizes
     *
     * @param component <code>Component</code> to be <code>Tab</code>bed in a <code>TabbedPane</code>
     * @param tab <code>Tab</code> to associates with <code>component</code>
     */
    public void setSize(Component component, Tab tab)
    {
	subsume(paneSize, layout.getAdvances(component, true));
	tabPaneSize.width = paneSize.width;

	tabSize.width = maxTabAdvance;
	subsume(tabSize, layout.getAdvances(tab, true));
	maxTabAdvance = tabSize.width;
	tabCounts++;

	int totalTabAdvance = tabCounts * tabSize.width;

	if(totalTabAdvance < paneSize.width) {
	    tabPaneSize.height = tabSize.height;
	    tabLines = 1;
	    tabsInLine = tabCounts;
	    if((paneSize.width - totalTabAdvance) <= tabSize.width)
		tabSize.width = paneSize.width / tabsInLine;
	    return;
	}

	if(paneSize.width < tabSize.width)
	    paneSize.width = tabSize.width;
	tabsInLine = paneSize.width / tabSize.width;
	tabSize.width = paneSize.width / tabsInLine;
	totalTabAdvance = tabCounts * tabSize.width;
	tabLines = 0;
	int tabLineWidth = tabSize.width * tabsInLine;
	while(totalTabAdvance > 0) {
	    tabLines++;
	    totalTabAdvance -= tabLineWidth;
	}

	tabPaneSize.height = tabLines * tabSize.height;
    }

    /**
     * Sets <code>subsumptor</cod> to be subsumption of
     * <code>subsumptor</cod> and <code>subsumptant</cod>
     *
     * @param subsumptor <code>Dimension</code> to be subsumption
     * @param subsumptant <code>Dimension</code> to be subsumed
     */
    public void subsume(Dimension subsumptor, Dimension subsumptant)
    {
	if(subsumptor.width < subsumptant.width)
	    subsumptor.width = subsumptant.width;

	if(subsumptor.height < subsumptant.height)
	    subsumptor.height = subsumptant.height;
    }

    /**
     * Returns size of pane to be used
     */
    public Dimension getPaneSize()
    {
	return paneSize;
    }

    /**
     * Returns size of pane to be used
     */
    public void setPaneSize(Dimension size)
    {
	paneSize = size;
    }

    /**
     * Returns size of pane to contain <code>Tab</code>s
     */
    public Dimension getTabPaneSize()
    {
	return tabPaneSize;
    }

    /**
     * Size of <code>Tab</code> to be used
     */
    public Dimension getTabSize()
    {
	return tabSize;
    }

    /**
     * Advance of <code>Tab</code> to be used
     */
    public int getTabAdvance()
    {
	return tabSize.width;
    }


    /**
     * Returns <code>LayoutManager</code> in use
     */
    public AbstractRowColumnLayout getLayout()
    {
	return layout;
    }

    /**
     * Returns number of <code>Tab</code>s
     */
    public int getTabConts()
    {
	return tabCounts;
    }

    /**
     * Returns number of lines of <code>Tab</code>s
     */
    public int getTabLines()
    {
	return tabLines;
    }

    /**
     * Returns number of <code>Tab</code>s in a "Tab line".
     */
    public int getTabsInLine()
    {
	return tabsInLine;
    }
}
