/*
 * CharactorChooser.java:  a class to pick up a character from a table.
 *
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */
/*
 * $Id: CharacterChooser.java,v 1.2 2002/06/23 17:34:06 nozomi Exp $
 * $Log: CharacterChooser.java,v $
 * Revision 1.2  2002/06/23 17:34:06  nozomi
 * modify event handilng
 *
 * Revision 1.1  2002/06/23 13:09:22  nozomi
 * initial import into CVS repository
 *
 */

package jp.kyasu.editor;

import java.awt.AWTEventMulticaster;
import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.ItemSelectable;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import java.util.Hashtable;

import jp.kyasu.awt.BorderedPanel;
import jp.kyasu.awt.Button;
import jp.kyasu.awt.Choice;
import jp.kyasu.awt.DefaultTextListModel;
import jp.kyasu.awt.Panel;
import jp.kyasu.awt.PopupPanel;
import jp.kyasu.awt.TextListModel;

import jp.kyasu.graphics.V3DBorder;
import jp.kyasu.graphics.VActiveButton;


/**
 * The <code>CharactorChooser</code> allows a user to pick up a character
 *
 * @see 	jp.kyasu.awt.PopupPanel#showPopup(java.awt.Component,int,int)
 *
 * @version 	23 Jun 2002
 * @author 	Nozomi `James' Ytow
 */
public class CharacterChooser
    extends PopupPanel 
    implements ActionListener,
	       ItemListener,
	       ItemSelectable
{
    protected Choice charsetChoice;

    protected CardLayout cardLayout;

    protected Panel charPanel;

    protected Panel[] charPanels;

    protected static Insets insets = new Insets(6, 6, 5, 5);

    protected Character selectedChar;

    protected Hashtable charMap;

    protected static TextListModel charsetNameList = 
	new DefaultTextListModel();

    protected static String[][] charsetNames = {
	{EditorResources.getResourceString("westernEuropeCharset")},
	{EditorResources.getResourceString("easternEuropeCharset")},
	{EditorResources.getResourceString("cyrillicCharset")},
	{EditorResources.getResourceString("greekCharset")}
    };

    protected static char[][][] charRange = {
	{
	    {'\u00A1', '\u00FF'}
	},
	{
	    {'\u0100', '\u01F5'}, {'\u01FA', '\u0217'}
	},
	{
	    {'\u0410', '\u044F'},
	    {'\u0401', '\u040C'}, {'\u040E', '\u040F'},
	    {'\u0451', '\u045C'}, {'\u045E', '\u045F'},
	    {'\u0460', '\u0486'}, {'\u0490', '\u04C4'},
	    {'\u04C7', '\u04C8'}, {'\u04CB', '\u04CC'},
	    {'\u04D0', '\u04EB'}, {'\u04EE', '\u04F5'},
	    {'\u04F8', '\u04F9'},
	},
	{
	    {'\u0374', '\u0375'}, {'\u037A', '\u037A'},
	    {'\u037E', '\u037E'}, {'\u0384', '\u038A'},
	    {'\u038C', '\u038C'}, {'\u038E', '\u03A1'},
	    {'\u03A3', '\u03CE'}, {'\u03D0', '\u03D6'},
	    {'\u03DA', '\u03DA'}, {'\u03DC', '\u03DC'},
	    {'\u03DE', '\u03DE'}, {'\u03E0', '\u03E0'},
	    {'\u03E2', '\u03F3'}
	}
    };

    protected static char[][] charTable = new char[charRange.length][];

    static {
	charsetNameList.replaceItems(0, 0, charsetNames);

	for(int i = 0; i < charsetNames.length; i++) {
	    int length = 0;
	    for (int j = 0; j < charRange[i].length; j++)
		length += charRange[i][j][1] - charRange[i][j][0] + 1;
	    charTable[i] = new char[length];
	    int index = 0;
	    for (int j = 0; j < charRange[i].length; j++) {
		for(char c = charRange[i][j][0]; c <= charRange[i][j][1]; c++) {
		    charTable[i][index++] = c;
		}
	    } 
	}
    }

    /** The item listener. */
    transient protected ItemListener itemListener;

    CharacterChooser()
    {
	super();
	selectedChar = null;
	itemListener = null;
	charMap = new Hashtable();
	initializePanel();
    }

    protected void initializePanel()
    {
	cardLayout = new CardLayout();
	charPanel = new Panel(cardLayout);
	charPanel.setLayout(cardLayout);
	charPanels = new Panel[charTable.length];

	charsetChoice = new Choice(charsetNameList);
	charsetChoice.addItemListener(this);

	int maxRows = 0;
	for(int i = 0; i < charTable.length; i++) {
	    int rows = 0;
	    if(charTable[i].length % 16 != 0)
		rows++;
	    rows += charTable[i].length / 16;
	    if(maxRows < rows)
		maxRows = rows;
	}


	for(int i = 0; i < charTable.length; i++) {

	    charPanels[i] = new Panel();
	    charPanels[i].setLayout(new GridLayout(maxRows, 16, 0, 0));
	    charPanel.add(charsetNames[i][0], charPanels[i]);

	    int charIndex = 0;
	    for(int y = 0; y < maxRows /*&& charIndex < charTable[i].length */; y++) {
		for(int x = 0; x < 16 /*&& charIndex < charTable[i].length*/; x++) {
		    Button b = null;
		    if(charIndex < charTable[i].length) {
			char ch = charTable[i][charIndex++];
			b = new Button(new VActiveButton(new String(new char[]{ch})));
			charMap.put(b, new Character(ch));
		    }
		    else {
			b = new Button(new VActiveButton(""));
			b.setEnabled(false);
		    }
		    b.setFocusTraversable(false);
		    b.addActionListener(this);
		    charPanels[i].add(b);
		}
	    }

	}

	BorderedPanel bp = new BorderedPanel(new V3DBorder(true));
	bp.add(charsetChoice, BorderLayout.NORTH);
	bp.add(charPanel, BorderLayout.CENTER);
	add(bp, BorderLayout.CENTER);
	setSize(getPreferredSize());
    }
    
    /**
     * Returns the selected character.
     */
    public char getChar()
    {
	return selectedChar.charValue();
    }

    /**
     * Returns the selected character.
     */
    public Character getCharacter()
    {
	return selectedChar;
    }

    /**
     * Returns the selected items or null if no items are selected.
     * @see java.awt.ItemSelectable
     */
    public Object[] getSelectedObjects()
    {
	return new Character[]{selectedChar};
    }

    /**
     * Add a listener to recieve item events when the state of an item changes.
     * @see java.awt.ItemSelectable
     */
    public void addItemListener(ItemListener listener)
    {
	itemListener = 
	    AWTEventMulticaster.add(itemListener, listener);
    }

    /**
     * Removes an item listener.
     * @see java.awt.ItemSelectable
     */
    public void removeItemListener(ItemListener listener)
    {
	itemListener = 
	    AWTEventMulticaster.remove(itemListener, listener);
    }

    /** Notifies the item event to the item listeners. */
    protected void notifyItemListeners()
    {
	if (itemListener != null) {
	    ItemEvent e = new ItemEvent(this, ItemEvent.ITEM_STATE_CHANGED,
					selectedChar, ItemEvent.SELECTED);
	    itemListener.itemStateChanged(e);
	}
    }

    public void actionPerformed(ActionEvent e)
    {
	Character ch = (Character)charMap.get(e.getSource());
	if(ch != null){
	    selectedChar = ch;
	    hidePopup();
	    notifyItemListeners();
	}
    }

    public void itemStateChanged(ItemEvent event)
    {
	if(event.getSource() == charsetChoice)
	    cardLayout.show(charPanel, charsetChoice.getSelectedItem());
    }

}
