/*
 * Editadaptor.java
 *
 * Copyright (c) 1997, 1998 Kazuki YASUMATSU.  All Rights Reserved.
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */
/*
 * $Id: EditAdaptor.java,v 1.13 2002/10/08 00:41:48 nozomi Exp $
 * $Log: EditAdaptor.java,v $
 * Revision 1.13  2002/10/08 00:41:48  nozomi
 * HTML editor may have URL Field
 *
 * Revision 1.12  2002/10/07 14:33:00  nozomi
 * add EditAdaptor(TextComponent) constructor
 *
 * Revision 1.11  2002/10/07 01:38:48  nozomi
 * improve HTMLEditor support
 *
 * Revision 1.10  2002/10/03 01:35:26  nozomi
 * fix bugs in super/nomal/subscript setting
 *
 * Revision 1.9  2002/10/03 01:26:31  nozomi
 * handle ActionEvent from super/nomal/subscript Buttons
 *
 * Revision 1.8  2002/08/22 10:52:33  nozomi
 * add lastFocused
 *
 * Revision 1.7  2002/08/03 18:59:31  nozomi
 * accept ToolBar as constructor's paramter
 *
 * Revision 1.6  2002/08/02 05:51:55  nozomi
 * fix subcomponent handling in TextComponent switching
 *
 * Revision 1.5  2002/08/01 04:13:14  nozomi
 * make getTool* methods to be static
 *
 * Revision 1.4  2002/06/23 17:34:43  nozomi
 * modify charSet event handilng
 *
 * Revision 1.3  2002/06/23 13:33:58  nozomi
 * use CharacterSelectButton
 *
 * Revision 1.2  2002/06/21 22:20:09  nozomi
 * listen/unlisten return if component is null
 *
 * Revision 1.1  2002/06/21 08:18:56  nozomi
 * intial import of multitext handler
 *
 */

package jp.kyasu.editor;

import java.awt.Component;
import java.awt.CheckboxMenuItem;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.FileDialog;
import java.awt.Font;
import java.awt.Image;
import java.awt.Menu;
import java.awt.MenuBar;
import java.awt.MenuItem;
import java.awt.Point;
import java.awt.PrintJob;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.TextEvent;
import java.awt.event.TextListener;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;

import java.net.URL;
import java.net.URLConnection;
import java.net.MalformedURLException;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Stack;
import java.util.StringTokenizer;
import java.util.Vector;

import jp.kyasu.awt.AbstractButton;
import jp.kyasu.awt.Button;
import jp.kyasu.awt.Choice;
import jp.kyasu.awt.DefaultTextEditModel;
import jp.kyasu.awt.DefaultTextListModel;
import jp.kyasu.awt.Dialog;
import jp.kyasu.awt.Label;
import jp.kyasu.awt.List;
import jp.kyasu.awt.MultipleUndo;
import jp.kyasu.awt.TextComponent;
import jp.kyasu.awt.TextEditModel;
import jp.kyasu.awt.TextField;
import jp.kyasu.awt.TextListModel;
import jp.kyasu.awt.ToggleButton;
import jp.kyasu.awt.ToolBar;
import jp.kyasu.awt.Undo;

import jp.kyasu.awt.event.TextModelEvent;
import jp.kyasu.awt.event.TextPositionEvent;
import jp.kyasu.awt.event.TextPositionListener;

import jp.kyasu.awt.text.TextEditController;
import jp.kyasu.awt.text.TextEditView;

import jp.kyasu.awt.util.HTMLTextEditModel;
import jp.kyasu.awt.util.JavaSyntaxColoringModel;

import jp.kyasu.graphics.BasicPSModifier;
import jp.kyasu.graphics.BasicTSModifier;
import jp.kyasu.graphics.ClickableTextAction;
import jp.kyasu.graphics.ExtendedFont;
import jp.kyasu.graphics.FontModifier;
import jp.kyasu.graphics.ModTextStyle;
import jp.kyasu.graphics.RichText;
import jp.kyasu.graphics.RichTextStyle;
import jp.kyasu.graphics.Text;
import jp.kyasu.graphics.TextAttachment;
import jp.kyasu.graphics.TextBuffer;
import jp.kyasu.graphics.TextStyle;
import jp.kyasu.graphics.TextStyleModifier;
import jp.kyasu.graphics.VActiveButton;
import jp.kyasu.graphics.VColoredWrapper;
import jp.kyasu.graphics.VHRBorder;
import jp.kyasu.graphics.VImage;
import jp.kyasu.graphics.Visualizable;
import jp.kyasu.graphics.VRectangle;
import jp.kyasu.graphics.ParagraphStyle;
import jp.kyasu.graphics.ParagraphStyleModifier;

import jp.kyasu.graphics.html.DefaultHTMLReaderTarget;
import jp.kyasu.graphics.html.HTMLStyle;
import jp.kyasu.graphics.html.HTMLText;
import jp.kyasu.graphics.html.HTMLReader;
import jp.kyasu.graphics.html.HTMLReaderTarget;
import jp.kyasu.graphics.html.HTMLWriter;
import jp.kyasu.graphics.html.VAnchor;

import jp.kyasu.graphics.text.TextChange;

import jp.kyasu.util.RunArray;


/**
 * An implementaiton of <CODE>Editor</CODE> taken from
 * <CODE>TextEditor</CODE> and its subclasses as an
 * abstraction of editorial operations to support
 * multiple <CODE>TextComponent</CODE> operation.
 *
 * @version 	08 Oct 2002
 * @author 	Kazuki YASUMATSU
 * @author 	Nozomi `James' Ytow
 */

public class EditAdaptor
    implements Editor,
	       ActionListener,
	       ItemListener,
	       TextListener,
    	       TextPositionListener
{
    //<TextEditor>
    static public final String L_KFC_URL          = "kfcURL";
    static public final String L_KFC_AUTHOR       = "kfcAuthor";
    static public final String L_KFC_ADDRESS      = "kfcAddress";
    static public final String L_KFC_ADDRESS2     = "kfcAddress2";

    static public final String P_INCREMENTAL_LOAD = "incrementalLoad";

    static public final String L_FILE             = "file";
    static public final String L_READ_ENCODING    = "readEncoding";
    static public final String L_WRITE_ENCODING   = "writeEncoding";

    static public final String P_FILE             = L_FILE;
    static public final String P_READ_ENCODING    = L_READ_ENCODING;
    static public final String P_WRITE_ENCODING   = L_WRITE_ENCODING;
    static public final String P_SUB_COMPS        = "subComps";

    static public final String A_COPY             = "copy";
    static public final String A_CUT              = "cut";
    static public final String A_PASTE            = "paste";
    static public final String A_UNDO             = "undo";
    static public final String A_FIND             = "find";
    static public final String A_GOTO             = "goto";

    static public final String A_OPEN             = "open";
    static public final String A_SAVE             = "save";
    static public final String A_SAVE_AS          = "saveAs";
    static public final String A_PRINT            = "print";

    static public final String L_OPEN_CONFIRM     = "openConfirm";
    static public final String L_CLOSE_CONFIRM    = "closeConfirm";

    static protected final int INC_LINE_COUNT = 10;

    static public final String I_AUTO_INDENT      = TextComponent.P_AUTO_INDENT;
    static public final String I_INCREMENTAL_LOAD = P_INCREMENTAL_LOAD;
    static public final String I_SHOW_MATCH       = TextComponent.P_SHOW_MATCH;
    static public final String I_SOFT_TAB         = TextComponent.P_SOFT_TAB;
    static public final String I_WORD_WRAP        = TextComponent.P_WORD_WRAP;
    //</TextEditor>

    //<CodeEditor>
    static public final String I_SYNTAX_COLOR = "scolor";
    static public final String L_LANG_MODE    = "langMode";

    static public final String P_LANG_MODE    = L_LANG_MODE;
    static public final String P_SYNTAX_COLOR = I_SYNTAX_COLOR;
    //</CodeEditor>

    //<RichTextEditor>
    static public final String L_FORMAT      = "format";
    static public final String L_INSERT      = "insert";
    static public final String L_ALIGN       = "align";
    static public final String L_LIST        = "list";

    static public final String I_BOLD        = "bold";
    static public final String I_ITALIC      = "italic";
    static public final String I_UNDERLINE   = "underline";
    static public final String I_SUPERSCRIPT  = "superscript";
    static public final String I_NORMALSCRIPT = "normalscript";
    static public final String I_SUBSCRIPT    = "subscript";

    static public final String A_BOLD        = I_BOLD;
    static public final String A_ITALIC      = I_ITALIC;
    static public final String A_UNDERLINE   = I_UNDERLINE;
    static public final String A_SUPERSCRIPT  = I_SUPERSCRIPT;
    static public final String A_NORMALSCRIPT = I_NORMALSCRIPT;
    static public final String A_SUBSCRIPT    = I_SUBSCRIPT;

    static public final String A_CLEAR_STYLE = "clearStyle";
    static public final String A_LARGE       = "large";
    static public final String A_SMALL       = "small";
    static public final String A_LEFT        = "left";
    static public final String A_CENTER      = "center";
    static public final String A_RIGHT       = "right";
    static public final String A_INC_INDENT  = "incindent";
    static public final String A_DEC_INDENT  = "decindent";

    static public final String A_IMAGE       = "image";
    static public final String A_HR          = "hr";

    static public final int    NORMALSCRIPT   = ExtendedFont.NORMALSCRIPT;
    static public final int    SUPERSCRIPT    = ExtendedFont.SUPERSCRIPT;
    static public final int    SUBSCRIPT      = ExtendedFont.SUBSCRIPT;
    //</RichTextEditor>

    //<DocumentEditor>

    static public final String L_EDIT             = "edit";
    static public final String L_VIEW             = "view";

    static public final String L_FONT             = "font";
    static public final String L_FONT_NAME        = "fontName";
    static public final String L_FONT_STYLE       = "fontStyle";
    static public final String L_FONT_SIZE        = "fontSize";
    static public final String L_FONT_COLOR       = "fontColor";
    static public final String A_SAVE_AS_TEXT   = "saveAsText";
    static public final String A_SAVE_AS_OBJECT = "saveAsObject";

    static public final String A_LIST           = "list";
    static public final String A_UNLIST         = "unlist";


    /** The indentation of the list. */
    static protected final int LIST_INDENT         = 50;

    /** The heading space for the list. */
    static protected final int LIST_HEADING_SPACE  = 8;

    /** The heading visual object for the list. */
    static protected final Visualizable LIST_HEADING =
		new VColoredWrapper(
			new VRectangle(4, 4, VRectangle.PLAIN), Color.black);

    //</DocumentEditor>

    //<HTMLEditor>
    static public final int MAX_HISTORY = 10;

    static public final String L_PARA_STYLE = "pstyle";
    static public final String L_VARIABLE   = "variableFont";
    static public final String L_FIXED      = "fixedFont";
    static public final String L_S_STYLE    = "smallStyle";
    static public final String L_M_STYLE    = "mediumStyle";
    static public final String L_L_STYLE    = "largeStyle";
    static public final String L_VL_STYLE   = "veryLargeStyle";

    static public final String I_LINK       = "link";
    static public final String I_ULIST      = "list";
    static public final String I_OLIST      = "olist";

    static public final String A_LINK       = I_LINK;
    static public final String A_ULIST      = I_ULIST;
    static public final String A_OLIST      = I_OLIST;

    static public final String A_ANCHOR     = "anchor";
    static public final String A_FORWARD    = "forward";
    static public final String A_BACKWARD   = "backward";
    static public final String A_STOP       = "stop";
    static public final String A_RELOAD     = "reload";
    static public final String A_DOC_TITLE  = "preview";

    static public final String P_URL        = "url";
    static public final String P_TITLE      = "title";
    static public final String P_BACKGROUND = "background";
    static public final String P_FOREGROUND = "foreground";

    static protected final Hashtable PStyles = new Hashtable();

    static {
	PStyles.put(EditorResources.getResourceString("normal"),  "P");
	PStyles.put(EditorResources.getResourceString("h1"),      "H1");
	PStyles.put(EditorResources.getResourceString("h2"),      "H2");
	PStyles.put(EditorResources.getResourceString("h3"),      "H3");
	PStyles.put(EditorResources.getResourceString("h4"),      "H4");
	PStyles.put(EditorResources.getResourceString("h5"),      "H5");
	PStyles.put(EditorResources.getResourceString("h6"),      "H6");
	PStyles.put(EditorResources.getResourceString("address"), "ADDRESS");
	PStyles.put(EditorResources.getResourceString("pre"),     "PRE");
	PStyles.put(EditorResources.getResourceString("li"),      "LI");
	PStyles.put(EditorResources.getResourceString("dd"),      "DD");
	PStyles.put(EditorResources.getResourceString("dt"),      "DT");
    }

    static protected final Hashtable HTMLStyles = new Hashtable();

    static {
	HTMLStyles.put(L_S_STYLE, new HTMLStyle(10));
	HTMLStyles.put(L_M_STYLE, new HTMLStyle(12));
	HTMLStyles.put(L_L_STYLE, new HTMLStyle(14));
	HTMLStyles.put(L_VL_STYLE,
	    new HTMLStyle(
		new TextStyle(HTMLStyle.DEFAULT_BASE_FONT_NAME, Font.PLAIN, 18),
		new ParagraphStyle(ParagraphStyle.LEFT, 8, 8, 0, 0)));
    }

    //</HTMLEditor>

    // focus manage parameters
    /** <CODE>Component</CODE> having focus currently */
    protected TextComponent focused;

    /** <CODE>Component</CODE> had focus previously */
    protected TextComponent lastFocused;

    /** <CODE>Vector</CODE> to hold list of <CODE>TextComponents</CODE>s */
    protected Vector focusListenants;

    // <TextEditor>
    /** <CODE>Hashtable</CODE> to hold <CODE>Cursor</CODE>s */
    protected Hashtable savedTextCursors;

    /** <CODE>Hashtable</CODE> to hold read encoding names */
    protected Hashtable readEncodings;

    /** <CODE>Hashtable</CODE> to hold write encoding names */
    protected Hashtable writeEncodings;

    /** <CODE>Hashtable</CODE> to hold inclreamantal load satate */
    protected Hashtable incrementalLoad;

    /** <CODE>Hashtable</CODE> to hold target <CODE>File</CODE> */
    protected Hashtable writeTargets;

    /** <CODE>Hashtable</CODE> to hold target <CODE>File</CODE> */
    protected Hashtable saveAsObject;

    /** <CODE>boolean</CODE> to hold text changed state */
    protected boolean textChanged;

    /**
     * <CODE>Hashtable</CODE> to hold <CODE>Boolean</CODE>s indicating
     * wheteher text was changed
     */
    protected Hashtable textChangedStatus;

    // </TextEditor>

    //<CodeEditor>
    static public final Vector JavaSuffixes = new Vector();
    static public final Vector CSuffixes    = new Vector();
    static public final Vector CPPSuffixes  = new Vector();

    static {
	String suffixes = EditorResources.getResourceString("javaSuffixes");
	StringTokenizer st = new StringTokenizer(suffixes);
	while (st.hasMoreTokens()) JavaSuffixes.addElement(st.nextToken());

	suffixes = EditorResources.getResourceString("cSuffixes");
	st = new StringTokenizer(suffixes);
	while (st.hasMoreTokens()) CSuffixes.addElement(st.nextToken());

	suffixes = EditorResources.getResourceString("cppSuffixes");
	st = new StringTokenizer(suffixes);
	while (st.hasMoreTokens()) CPPSuffixes.addElement(st.nextToken());
    }
    //</CodeEditor>

    //<HTMLEditor>
    /** <CODE>Hashtable</CODE> to hold HTML history */
    protected Hashtable htmlHistories;

    /** <CODE>Hashtable</CODE> to hold HTML history indices */
    protected Hashtable htmlHistoryIndices;

    transient protected Stack history;
    transient protected int historyIndex;

    /** <CODE>Hashtable</CODE> to hold HTMLEditor's <CODE>loadInputStream</CODE>s */
    protected Hashtable htmlLoadInputStreams;

    /** <CODE>Hashtable</CODE> to hold HTMLEditor's <CODE>urlField</CODE>s */
    protected Hashtable htmlURLFields;

    protected TextField urlField;

    /** <CODE>Hashtable</CODE> to hold HTML <CODE>Thread</CODE>s */
    protected Hashtable backgroundThreads;

    transient protected Thread backgroundThread;

    transient protected ActionListener linkActionListener;
    //</HTMLEditor>

    /** <CODE>ToolBar</CODE> contains components to modify TextComponent contens */
    protected ToolBar toolBar;

    /** <CODE>Vector</CODE> holding sub components of <CODE>toolBar</CODE> */
    protected Vector toolBarComponents;

    /** <CODE>Hashtable</CODE> holding status of sub components in <CODE>toolBar</CODE> */
    protected Hashtable toolBarStates;

    /** <CODE>Menu</CODE> providing a menu GUI */
    protected MenuBar menubar;

    protected Hashtable checkboxMenuMap = new Hashtable();

    protected Vector caretDisableComps;

    /** <CODE>Vector</CODE> to hold <CODE>FocusListener</CODE>s */
    protected Vector focusListeners;

    /** <CODE>Vector</CODE> to hold <CODE>TextListener</CODE>s */
    protected Vector textListeners;

    /** <CODE>Vector</CODE> to hold <CODE>TextPositionListener</CODE>s */
    protected Vector textPositionListeners;

    public final static TextListModel documentFontNameModel = new DefaultTextListModel();

    public final static String[][] documentFontNames = {
	{"SansSerif"}, {"Serif"}, {"Monospaced"}, {"Dialog"}
    };

    static {
	documentFontNameModel.replaceItems(0, 0, documentFontNames);
    }

    public static final int INVALID_EDITOR = -1;
    public static final int CODE_EDITOR = 0;
    public static final int TEXT_EDITOR = CODE_EDITOR + 1;
    public static final int DOCUMENT_EDITOR = TEXT_EDITOR + 1;
    public static final int HTML_EDITOR = DOCUMENT_EDITOR + 1;
    public static final int LAST_EDITOR_TYPE = HTML_EDITOR;

    public EditAdaptor(TextComponent toBeFocused)
    {
	editAdaptor(createToolBar(getEditorType(toBeFocused)));
	setTextComponent(toBeFocused);
    }

    public EditAdaptor(int editorType)
    {
	editAdaptor(createToolBar(editorType));
    }

    public EditAdaptor(ToolBar toolBar)
    {
	editAdaptor(toolBar);
    }

    public void editAdaptor(ToolBar toolBar)
    {
	textChanged = false;
	focusListenants = new Vector();
	toolBarStates = new Hashtable(); 
	setToolBar(toolBar);
	textChangedStatus = new Hashtable();
    }

    public int getEditorType()
    {
	return getEditorType(focused);
    }

    public static int getEditorType(TextComponent component)
    {
	if(component == null)
	    return INVALID_EDITOR;
	if(component.getModel() instanceof JavaSyntaxColoringModel)
	    return CODE_EDITOR;
	if(component.getRichText() instanceof HTMLText)
	    return HTML_EDITOR;
	if(component.getRichText().getRichTextStyle().isVariableLineHeight())
	    return DOCUMENT_EDITOR;
	return TEXT_EDITOR;
    }

    public boolean isHTML()
    {
	return getEditorType() == HTML_EDITOR;
    }


    /**
     * Adds the specified text event listener to recieve text events from
     * this text component.
     * @param l the text event listener.
     */
    public synchronized void addTextListener(TextListener listener)
    {
	if(textListeners == null)
	    textListeners = new Vector();
	textListeners.addElement(listener);
    }

    /**
     * Removes the specified text event listener so that it no longer
     * receives text events from this textcomponent
     * @param l the text event listener.
     */
    public synchronized void removeTextListener(TextListener listener)
    {
	textListeners.remove(listener);

	if(textListeners.isEmpty())
	    textListeners = null;
    }

    /**
     * Invoked when the value of the text has changed.
     * @see java.awt.event.TextListener
     */
    public void textValueChanged(TextEvent event)
    {
	if(textListeners == null)
	    return;
	textChangedStatus.put(focused, Boolean.TRUE);
	Object[] listeners = null;
	synchronized(textListeners) {
	    listeners = textListeners.toArray();
	}
	for(int i = 0; i < listeners.length; i++) {
	    ((TextListener)listeners[i]).textValueChanged(event);
	}
    }

    /**
     * Adds the specified text position event listener to recieve text
     * position events from this text component.
     * @param l the text position event listener.
     */
    public synchronized void addTextPositionListener(TextPositionListener listener)
    {
	if(textPositionListeners == null)
	    textPositionListeners = new Vector();
	textPositionListeners.addElement(listener);
    }

    /**
     * Removes the specified text position event listener so that it no
     * longer receives text position events from this text component
     * @param l the text position event listener.
     */
    public synchronized void removeTextPositionListener(TextPositionListener listener)
    {
	textPositionListeners.remove(listener);

	if(textPositionListeners.isEmpty())
	    textPositionListeners = null;
    }

    /**
     * Invoked when the position of the text has changed.
     * @see java.awt.event.TextPositionListener
     */
    public void textPositionChanged(TextPositionEvent event) 
    {
	if (caretDisableComps != null) {
	    boolean b = !event.selectionIsCaret();
	    for (Enumeration e = caretDisableComps.elements();
		 e.hasMoreElements();
		 )
	    {
		Object obj = e.nextElement();
		if (obj instanceof Component) {
		    ((Component)obj).setEnabled(b);
		}
		else if (obj instanceof MenuItem) {
		    ((MenuItem)obj).setEnabled(b);
		}
	    }
	}

	if(textPositionListeners == null)
	    return;
	
	Object[] listeners = null;
	synchronized(textPositionListeners) {
	    listeners = textPositionListeners.toArray();
	}
	for(int i = 0; i < listeners.length; i++) {
	    ((TextPositionListener)listeners[i]).textPositionChanged(event);
	}
    }

    //<TextComponentAPI>
    /**
     * Returns the model of this text component.
     */
    public TextEditModel getModel() {
	if(focused != null)
	    return focused.getModel();
	return null;
    }

    /**
     * Returns the view of this text component.
     */
    public TextEditView getView() {
	if(focused != null)
	    return focused.getView();
	return null;
    }

    /**
     * Returns the controller of this text component.
     */
    public TextEditController getController() {
	if(focused != null)
	    return focused.getController();
	return null;
    }

    /**
     * Returns the rich text of this text component.
     */
    public RichText getRichText() {
	if(focused != null)
	    return focused.getRichText();
	return null;

    }

    /**
     * Sets the string that is presented by this text component to be the
     * specified string.
     */
    public void setText(String str) {
	if(isHTML())
	    throw new RuntimeException("This operation not allowed");
	focused.setText(str);
	textChangedStatus.put(focused, Boolean.FALSE);
    }

    /**
     * Sets the text of this text component.
     */
    public void setTEXT(Text text) {
	if(isHTML())
	    throw new RuntimeException("This operation not allowed");
	focused.setTEXT(text);
	textChangedStatus.put(focused, Boolean.FALSE);
    }

    /**
     * Sets the rich text of this text component.
     */
    public void setRichText(RichText rtext) {
	if(isHTML())
	    throw new RuntimeException("This operation not allowed");
	focused.setRichText(rtext);
	textChangedStatus.put(focused, Boolean.FALSE);
    }


    //</TextComponentAPI>

    //<TextEditorAPI>
    /**
     * Checks if the text is changed.
     */
    public boolean isTextChanged() {
	return ((Boolean)textChangedStatus.get(focused)).booleanValue();
    }

    /**
     * Sets the text is changed or not.
     */
    public void setTextChanged(boolean b) {
	textChangedStatus.put(focused, Boolean.valueOf(b));
    }
    
    /**
     * Checks if this editor wraps the line at word boundary.
     */
    public boolean isWordWrap() {
	if(focused != null)
	    return focused.isWordWrap();
	return false;
    }

    /**
     * Enables the word wrap.
     */
    public void setWordWrap(boolean wordWrap) {
	if(focused != null)
	    focused.setWordWrap(wordWrap);
    }

    /**
     * Checks if this text component allows soft tab.
     * @see #clearSoftTab()
     * @see #getSoftTab()
     * @see #setSoftTab(int)
     */
    public boolean isSoftTab() {
	if(focused != null)
	    return focused.isSoftTab();
	return false;
    }

    /**
     * Enables the soft tab.
     */
    public void setSoftTab(boolean b) {
	boolean oldValue = focused.isSoftTab();
	if (oldValue == b)
	    return;
	focused.setSoftTab((int)(b ? 4 : 0));
	PropertyChangeSupport change = focused.getPropertyChangeSupport();
	if (change != null) {
	    change.firePropertyChange(TextComponent.P_SOFT_TAB,
				      Boolean.valueOf(oldValue),
				      Boolean.valueOf(b));
	}
    }

    /**
     * Checks if the auto indent is enabled.
     */
    public boolean isAutoIndentEnabled() {
	if(focused != null)
	    return focused.isAutoIndentEnabled();
	else
	    return false;
    }

    /**
     * Enables the auto indent.
     */
    public void setAutoIndentEnabled(boolean autoIndent) {
	focused.setAutoIndentEnabled(autoIndent);
    }

    /**
     * Checks if the show match is enabled.
     */
    public boolean isShowMatchEnabled() {
	if(focused != null)
	    return focused.isShowMatchEnabled();
	else
	    return false;
    }

    /**
     * Enables the show match.
     */
    public void setShowMatchEnabled(boolean showMatch) {
	focused.setShowMatchEnabled(showMatch);
    }

    /**
     * Returns the encoding name for writing.
     */
    public String getWriteEncoding() {
	return (String)writeEncodings.get(focused);
    }

    /**
     * Sets the encoding name for writing to be the specified name.
     */
    public void setWriteEncoding(String enc) {
	if (enc == null)
	    throw new NullPointerException();
	String writeEncoding = (String)writeEncodings.get(focused);
	if (enc == writeEncoding)
	    return;
	String oldValue = writeEncoding;
	writeEncoding = enc;
	PropertyChangeSupport change = focused.getPropertyChangeSupport();
	if (change != null) {
	    change.firePropertyChange(P_WRITE_ENCODING, oldValue, enc);
	}
    }

    /**
     * Returns the encoding name for reading.
     */
    public String getReadEncoding() {
	return (String)readEncodings.get(focused);
    }

    /**
     * Sets the encoding name for reading to be the specified name.
     */
    public void setReadEncoding(String enc) {
	if (enc == null)
	    throw new NullPointerException();
	if(readEncodings == null)
	    return;
	String readEncoding = (String)readEncodings.get(focused);
	if (enc == readEncoding)
	    return;
	String oldValue = readEncoding;
	readEncodings.put(focused, enc);
	PropertyChangeSupport change = focused.getPropertyChangeSupport();
	if (change != null) {
	    change.firePropertyChange(P_READ_ENCODING, oldValue, enc);
	}
    }

    protected File getWirteTarget() {
	return (File)writeTargets.get(focused);
    }

    //File access methods taken from TextEditor
    protected void setWriteTarget(File file) {
	//	File oldFile = writeTarget;
	File oldFile = null;
	if(writeTargets != null)
	    oldFile = (File)writeTargets.get(focused);
	if ((oldFile == null ? file == null : oldFile.equals(file)))
	    return;
	if(writeTargets == null)
	    writeTargets = new Hashtable();
	writeTargets.put(focused, file);
	//	writeTarget = file;
	PropertyChangeSupport change = focused.getPropertyChangeSupport();
	if (change != null) {
	    change.firePropertyChange(P_FILE, oldFile, file);
	}
    }

    protected void warn(Exception e) {
	Dialog.warn(focused.getFrame(),
		    e.getClass().getName() + ": " + e.getMessage());
    }

    protected File getFileFromLoadDialog(String label,
					 String initDir, String initFile)
    {
	FileDialog dialog = new FileDialog(focused.getFrame(), label, FileDialog.LOAD);
	if (initDir != null)
	    dialog.setDirectory(initDir);
	if (initFile != null)
	    dialog.setFile(initFile);
	dialog.setVisible(true);
	String ddir = dialog.getDirectory();
	String dfile = dialog.getFile();
	dialog.dispose();
	if (ddir == null || dfile == null) {
	    return null;
	}
	File f = new File(ddir + dfile);
	if (!f.exists()) {
	    Dialog.warn(focused.getFrame(), f.getPath() + getToolLabel("fileNotExist"));
	    return null;
	}
	else if (!f.isFile()) {
	    Dialog.warn(focused.getFrame(), f.getPath() + getToolLabel("fileNotFile"));
	    return null;
	}
	else if (!f.canRead()) {
	    Dialog.warn(focused.getFrame(), f.getPath() + getToolLabel("fileNotRead"));
	    return null;
	}
	return f;
    }

    protected File getFileFromSaveDialog(String label,
					 String initDir, String initFile)
    {
	FileDialog dialog = new FileDialog(focused.getFrame(), label, FileDialog.SAVE);
	if (initDir != null)
	    dialog.setDirectory(initDir);
	if (initFile != null)
	    dialog.setFile(initFile);
	dialog.setVisible(true);
	String ddir = dialog.getDirectory();
	String dfile = dialog.getFile();
	dialog.dispose();
	if (ddir == null || dfile == null) {
	    return null;
	}
	File f = new File(ddir + dfile);
	if (!f.exists()) {
	    // do not care
	}
	else if (!f.isFile()) {
	    Dialog.warn(focused.getFrame(), f.getPath() + getToolLabel("fileNotFile"));
	    return null;
	}
	else if (!f.canWrite()) {
	    Dialog.warn(focused.getFrame(), f.getPath() + getToolLabel("fileNotWrite"));
	    return null;
	}
	else if (!Dialog.confirm(focused.getFrame(),
				 f.getPath() + getToolLabel("fileOverwrite")))
	{
	    return null;
	}
	return f;
    }

    protected void appendAnyway(Text text, boolean scroll) {
	if (focused.isEditable()) {
	    focused.append(text, scroll);
	}
	else {
	    try {
		focused.setEditable(true);
		focused.append(text, scroll);
	    }
	    finally {
		focused.setEditable(false);
	    }
	}
    }



    /**
     * Checks if the loading is done incrementally.
     */
    public boolean isIncrementalLoad() {
	return ((Boolean)incrementalLoad.get(focused)).booleanValue();
    }

    /**
     * Enables or disables the incremental loading.
     */
    public void setIncrementalLoad(boolean b) {
	boolean oldValue = ((Boolean)incrementalLoad.get(focused)).booleanValue();
	if (b == oldValue)
	    return;
	incrementalLoad.put(focused, Boolean.valueOf(b));
	PropertyChangeSupport change = focused.getPropertyChangeSupport();
	if (change != null) {
	    change.firePropertyChange(P_INCREMENTAL_LOAD,
				      new Boolean(oldValue), new Boolean(b));
	}
    }

    transient protected Cursor savedTextCursor = null;

    /**
     * Disables sub components.
     * @see #enableSubComps()
     */
    public synchronized void disableTextSubComps() {
	Cursor savedTextCursor = 
	    (Cursor)savedTextCursors.get(focused);
	if (savedTextCursor != null)
	    return;

	PropertyChangeSupport change = focused.getPropertyChangeSupport();
	if (change != null) {
	    change.firePropertyChange(P_SUB_COMPS,
				      new Boolean(true), new Boolean(false));
	}

	TextEditView editView = focused.getView();
	savedTextCursor = editView.getCursor();
	savedTextCursors.put(focused, savedTextCursor);
	editView.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
	//setEditable(false);

	focused.getModel().removeTextListener(this);
	editView.removeTextPositionListener(this);

	if (toolBarComponents == null)
	    return;
	Hashtable toolBarStatus = (Hashtable)toolBarStates.get(focused);
	if(toolBarStatus == null) {
	    toolBarStatus = new Hashtable();
	    toolBarStates.put(focused, toolBarStatus);
	}
	for (Enumeration e = toolBarComponents.elements(); e.hasMoreElements(); ) {
	    Component comp = (Component)e.nextElement();
	    toolBarStatus.put(comp, new Boolean(comp.isEnabled()));
	    comp.setEnabled(false);
	}
    }

    /**
     * Enables sub components.
     * @see #disableSubComps()
     */
    public synchronized void enableTextSubComps() {
	if (savedTextCursor == null)
	    return;

	PropertyChangeSupport change = focused.getPropertyChangeSupport();
	if (change != null) {
	    change.firePropertyChange(P_SUB_COMPS,
				      new Boolean(false), new Boolean(true));
	}

	getView().setCursor(savedTextCursor);
	savedTextCursor = null;
	//setEditable(true);

	getModel().addTextListener(this);
	getView().addTextPositionListener(this);

	if (toolBarComponents == null)
	    return;
	/*
	Enumeration s = subCompStates.elements();
	for (Enumeration e = subComps.elements(); e.hasMoreElements(); ) {
	    Component comp = (Component)e.nextElement();
	    boolean enabled = ((Boolean)s.nextElement()).booleanValue();
	    comp.setEnabled(enabled);
	}
	subCompStates = null;
	*/
	loadToolBarStatus();
    }

    /**
     * Add a PropertyChangeListener to the listener list.
     * @param listener The PropertyChangeListener to be added.
     */
    public void addPropertyChangeListener(PropertyChangeListener listener) {
	focused.addPropertyChangeListener(listener);
    }

    /**
     * Remove a PropertyChangeListener from the listener list.
     * @param listener The PropertyChangeListener to be removed
     */
    public void removePropertyChangeListener(PropertyChangeListener listener) {
	focused.removePropertyChangeListener(listener);
    }

    /**
     * Opens the new file.
     */
    public void open_file() {
	if (textChanged &&
	    !Dialog.confirm(focused.getFrame(), getToolLabel(L_OPEN_CONFIRM)))
	{
	    return;
	}
	File writeTarget = (File)writeTargets.get(focused);
	File file = getFileFromLoadDialog(
			getToolTip(A_OPEN),
			(writeTarget != null ? writeTarget.getParent() : null),
			(writeTarget != null ? writeTarget.getName() : null));
	if (file == null)
	    return;
	if(getEditorType() == DOCUMENT_EDITOR)
	    openDocumentFile(file);
	else
	    open_file(file);

	// TODO for HTML.....
    }

    /**
     * Opens the specified file.
     */
    public void open_file(File file) {
	if(isHTML()) {
	    goto_page(fileToURLString(file));
	    return;
	}
	boolean ok = false;
	try {
	    String readEncoding = (String)readEncodings.get(focused);
	    BufferedReader reader = new BufferedReader(
					new InputStreamReader(
						new FileInputStream(file),
						readEncoding));
	    disableSubComps();
	    ok = load(reader);
	    reader.close();
	}
	catch (IOException e) {
	    warn(e);
	}
	finally {
	    enableSubComps();
	}

	if (ok) {
	    setWriteTarget(file);
	}
    }

    /**
     * Saves the text into the current file.
     */
    public void save_file() {
	if(isHTML()) {
	    saveHTMLFile();
	    return;
	}
	File writeTarget = (File)writeTargets.get(focused);
	if (writeTarget == null)
	    save_file_as();
	else if(getEditorType() == DOCUMENT_EDITOR)
	    saveDocumentFileAs(writeTarget);
	else
	    save_file_as(writeTarget);
    }

    /**
     * Saves the text into the selected file.
     */
    public void save_file_as() {
	File writeTarget = (File)writeTargets.get(focused);
	File file = getFileFromSaveDialog(
			getToolTip(A_SAVE),
			(writeTarget != null ? writeTarget.getParent() : null),
			(writeTarget != null ? writeTarget.getName() : null));
	if (file == null)
	    return;
	if(getEditorType() == DOCUMENT_EDITOR)
	    saveDocumentFileAs(file);
	else
	    save_file_as(file);
    }

    /**
     * Saves the text into the specified file.
     */
    public void save_file_as(File file) {
	boolean ok = false;
	try {
	    BufferedWriter writer = new BufferedWriter(
					new OutputStreamWriter(
						new FileOutputStream(file),
						getWriteEncoding()));
	    disableSubComps();
	    ok = save(writer);
	    writer.close();
	}
	catch (IOException e) {
	    warn(e);
	}
	finally {
	    enableSubComps();
	}

	if (ok) {
	    setWriteTarget(file);
	}
    }

    /**
     * Loads the contents of the specified reader incrementally into
     * this component
     * @param reader the buffered reader to be loaded.
     * @return true if the loading was succeeded.
     */
    public boolean load(BufferedReader reader) {
	int incCount = isIncrementalLoad() ? INC_LINE_COUNT : 0;
	boolean loadOk = false;
	focused.setRichText(new RichText(focused.getRichText().getRichTextStyle()));
	focused.setCaretPosition(0);
	try {
	    TextBuffer buffer =
		new TextBuffer(focused.getRichText().getRichTextStyle().getTextStyle());
	    int lineCount = 0;
	    String line;
	    while ((line = reader.readLine()) != null) {
		if (incCount > 0 && ++lineCount > incCount) {
		    appendAnyway(buffer.toText(), false);
		    lineCount = 0;
		    buffer = new TextBuffer(
			focused.getRichText().getRichTextStyle().getTextStyle());
		}
		buffer.append(line).append(Text.LINE_SEPARATOR_CHAR);
	    }
	    if (buffer.length() > 0) {
		appendAnyway(buffer.toText(), false);
	    }
	    focused.setCaretPosition(0);
	    loadOk = true;
	}
	catch (Exception e) {
	    TextBuffer buffer =
		new TextBuffer(focused.getRichText().getRichTextStyle().getTextStyle());
	    buffer.append("--> " + e.getClass().getName() + " occurred");
	    buffer.append(Text.LINE_SEPARATOR_CHAR);
	    int len = focused.getRichText().length();
	    appendAnyway(buffer.toText(), false);
	    focused.select(len, len + buffer.length() - 1);
	}
	focused.clearUndo();
	textChanged = false;
	return loadOk;
    }

    /**
     * Saves the contents of this component into the specified writer.
     * @param writer the writer to save into.
     * @return true if the saving was succeeded.
     */
    public boolean save(Writer writer) {
	boolean saveOk = false;
	try {
	    TextBuffer buffer = new TextBuffer(focused.getTEXT());
	    buffer.writeTo(writer);
	    textChanged = false;
	    saveOk = true;
	}
	catch (Exception e) {
	    warn(e);
	}
	return saveOk;
    }

    /**
     * Prints the file.
     */
    public void print_file() {
	if(!isHTML()) {
	    File writeTarget = (File)writeTargets.get(focused);
	    print_file(writeTarget != null ? writeTarget.getPath() : null);
	}
	else {
	    URL url = getURL();
	    print_file(url != null ? url.toExternalForm() : null);
	}
    }

    protected void print_file(String header) {
	PrintJob job =
	    focused.getToolkit().getPrintJob(focused.getFrame(), getToolTip(A_PRINT), null);
	if (job == null)
	    return;
	try {
	    disableSubComps();
	    focused.print(job, header, true);
	}
	finally {
	    enableSubComps();
	}
    }


    //</TextEditorAPI>

    //<CodeEditorAPI>
    protected TextEditModel createCodeTextEditModel() {
	return new JavaSyntaxColoringModel();
    }

    /*
    protected Component createCodeTextComponent(TextEditModel model,
						int rows, int columns)
    {
	Component textComp = createTextComponent(model, rows, columns);

	super.setFont(getCodeModel().getBaseFont());
	setForeground(Color.black);
	setBackground(Color.white);
	setSelectionForeground(Color.black);
	setSelectionBackground(Color.lightGray);

	setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
	//setTextCaret(new TextCaret(TextCaret.HAT_CARET, true));
	//setCaretColor(Color.black);

	//setWordWrap(false);
	setSoftTab(true);
	setAutoIndentEnabled(true);
	setShowMatchEnabled(true);

	return textComp;
    }
    */

    protected JavaSyntaxColoringModel getCodeModel() {
	return (JavaSyntaxColoringModel)focused.getModel();
    }


    /**
     * Sets the font of this component.
     */
    public void setCodeFont(Font font) {
	getCodeModel().setBaseFont(font);
	focused.setFont(font);
    }

    /**
     * Returns the language mode that is one of "Java", "C", and "C++".
     * @see #isJavaMode()
     * @see #isCMode()
     * @see #isCPPMode()
     */
    public String getLangMode() {
	if (isCMode()) {
	    return "C";
	}
	else if (isCPPMode()) {
	    return "C++";
	}
	else { // isJavaMode()
	    return "Java";
	}
    }

    /**
     * Checks if this component colors the syntax of the Java language.
     * @see #setJavaMode()
     */
    public boolean isJavaMode() {
	return getCodeModel().isJavaMode();
    }

    /**
     * Makes this component colors the syntax of the Java language.
     * @see #isJavaMode()
     */
    public void setJavaMode() {
	if (isJavaMode())
	    return;
	String oldValue = getLangMode();
	getCodeModel().setJavaMode();
	focused.setRichText(focused.getRichText());
	PropertyChangeSupport change = focused.getPropertyChangeSupport();
	if (change != null) {
	    change.firePropertyChange(P_LANG_MODE, oldValue, getLangMode());
	}
    }

    /**
     * Checks if this component colors the syntax of the C language.
     * @see #setCMode()
     */
    public boolean isCMode() {
	return getCodeModel().isCMode();
    }

    /**
     * Makes this component colors the syntax of the C language.
     * @see #isCMode()
     */
    public void setCMode() {
	if (isCMode())
	    return;
	String oldValue = getLangMode();
	getCodeModel().setCMode();
	focused.setRichText(focused.getRichText());
	PropertyChangeSupport change = focused.getPropertyChangeSupport();
	if (change != null) {
	    change.firePropertyChange(P_LANG_MODE, oldValue, getLangMode());
	}
    }

    /**
     * Checks if this component colors the syntax of the C++ language.
     * @see #setCPPMode()
     */
    public boolean isCPPMode() {
	return getCodeModel().isCPPMode();
    }

    /**
     * Makes this component colors the syntax of the C++ language.
     * @see #isCPPMode()
     */
    public void setCPPMode() {
	if (isCPPMode())
	    return;
	String oldValue = getLangMode();
	getCodeModel().setCPPMode();
	focused.setRichText(focused.getRichText());
	PropertyChangeSupport change = focused.getPropertyChangeSupport();
	if (change != null) {
	    change.firePropertyChange(P_LANG_MODE, oldValue, getLangMode());
	}
    }

    /**
     * Tests if the syntax coloring is enabled.
     */
    public boolean isSyntaxColoringEnabled() {
	return getCodeModel().isSyntaxColoringEnabled();
    }

    /**
     * Enables or disables the syntax coloring.
     */
    public void setSyntaxColoringEnabled(boolean b) {
	boolean oldValue = isSyntaxColoringEnabled();
	if (oldValue == b)
	    return;
	getCodeModel().setSyntaxColoringEnabled(b);
	focused.setRichText(focused.getRichText());
	if (syntaxColorButton.getState() != b) {
	    syntaxColorButton.setState(b);
	}
	PropertyChangeSupport change = focused.getPropertyChangeSupport();
	if (change != null) {
	    change.firePropertyChange(P_SYNTAX_COLOR,
				      new Boolean(oldValue),
				      new Boolean(b));
	}
    }

    /**
     * Returns the font color for the normal tokens.
     */
    public Color getNormalColor() {
	return getCodeModel().getNormalColor();
    }

    /**
     * Sets the font color for the normal tokens.
     */
    public void setNormalColor(Color color) {
	Color c = getNormalColor();
	if ((c == null ? color == null : c.equals(color)))
	    return;
	getCodeModel().setNormalColor(color);
	setRichText(focused.getRichText());
    }

    /**
     * Returns the font color for the keyword tokens.
     */
    public Color getKeywordColor() {
	return getCodeModel().getKeywordColor();
    }

    /**
     * Sets the font color for the keyword tokens.
     */
    public void setKeywordColor(Color color) {
	Color c = getKeywordColor();
	if ((c == null ? color == null : c.equals(color)))
	    return;
	getCodeModel().setKeywordColor(color);
	focused.setRichText(focused.getRichText());
    }

    /**
     * Returns the font color for the constant tokens.
     */
    public Color getConstantColor() {
	return getCodeModel().getConstantColor();
    }

    /**
     * Sets the font color for the constant tokens.
     */
    public void setConstantColor(Color color) {
	Color c = getConstantColor();
	if ((c == null ? color == null : c.equals(color)))
	    return;
	getCodeModel().setConstantColor(color);
	focused.setRichText(focused.getRichText());
    }

    /**
     * Returns the font color for the comment tokens.
     */
    public Color getCommentColor() {
	return getCodeModel().getCommentColor();
    }

    /**
     * Sets the font color for the comment tokens.
     */
    public void setCommentColor(Color color) {
	Color c = getCommentColor();
	if ((c == null ? color == null : c.equals(color)))
	    return;
	getCodeModel().setCommentColor(color);
	focused.setRichText(focused.getRichText());
    }

    /**
     * Invoked when an item's state has been changed.
     * @see java.awt.event.ItemListener
     */
    /*
    public void itemStateChanged(ItemEvent e) {
	Object obj = e.getItem();
	if (obj == null || !(obj instanceof String)) {
	    super.itemStateChanged(e);
	    return;
	}

	String command = (String)checkboxMenuMap.get(obj);
	if (command == null) command = (String)obj;
	if (command.equals(I_SYNTAX_COLOR)) {
	    setSyntaxColoringEnabled(e.getStateChange() == ItemEvent.SELECTED);
	}
	else {
	    super.itemStateChanged(e);
	}
    }
    */


    /**
     * Creates a tool bar.
     * @param showToolBar         if true, then shows the tool bar initially;
     *                            otherwise hides.
     * @param openActionListener  the action listener that receives action
     *                            events from the open button in tool bar.
     * @param saveActionListener  the action listener that receives action
     *                            events from the save button in tool bar.
     * @param printActionListener the action listener that receives action
     *                            events from the print button in tool bar.
     */
    protected ToolBar createCodeToolBar(boolean showToolBar,
				    ActionListener openActionListener,
				    ActionListener saveActionListener,
				    ActionListener printActionListener)
    {
	Component[] file  = createFileComponents(openActionListener,
						 saveActionListener);
	Component[] print = createPrintComponents(printActionListener);
	Component[] find  = createFindComponents(true);
	Component[] edit  = createEditComponents();
	Component[] color = createSyntaxColorComponents();

	Component[][] bar = new Component[][]{ file, print, find, edit, color };
	return new ToolBar(new Component[][][]{ bar }, showToolBar);
    }

    protected Component[] createSyntaxColorComponents() {
	syntaxColorButton = createIconToggleButton(I_SYNTAX_COLOR);
	syntaxColorButton.addItemListener(this);
	syntaxColorButton.setState(isSyntaxColoringEnabled());
	return new Component[]{ syntaxColorButton };
    }

    protected Menu createCodeViewMenu() {
	Menu menu = createViewMenu();
	menu.addSeparator();
	menu.add(createLangModeMenu());
	return menu;
    }

    class LangModeSelection implements ActionListener, java.io.Serializable {
	public void actionPerformed(ActionEvent e) {
	    String command = e.getActionCommand();
	    if (command.equals("Java")) {
		setJavaMode();
	    }
	    else if (command.equals("C")) {
		setCMode();
	    }
	    else if (command.equals("C++")) {
		setCPPMode();
	    }
	}
    }

    protected Menu createLangModeMenu() {
	SelectionMenu menu = new SelectionMenu(getToolLabel(L_LANG_MODE));
	menu.addActionListener(new LangModeSelection());
	menu.add("Java", "Java", isJavaMode());
	menu.add("C",    "C",    isCMode());
	menu.add("C++",  "C++",  isCPPMode());
	return menu;
    }

    protected void setCodeWriteTarget(File file) {
	if (file != null) {
	    String name = file.getName();
	    for (Enumeration e = JavaSuffixes.elements(); e.hasMoreElements(); )
	    {
		if (name.endsWith((String)e.nextElement())) {
		    setWriteTarget(file);
		    setJavaMode();
		    return;
		}
	    }
	    for (Enumeration e = CSuffixes.elements(); e.hasMoreElements(); ) {
		if (name.endsWith((String)e.nextElement())) {
		    setWriteTarget(file);
		    setCMode();
		    return;
		}
	    }
	    for (Enumeration e = CPPSuffixes.elements(); e.hasMoreElements(); )
	    {
		if (name.endsWith((String)e.nextElement())) {
		    setWriteTarget(file);
		    setCPPMode();
		    return;
		}
	    }
	}
	setWriteTarget(file);
    }
    //</CodeEditorAPI>

    //<DocumentEditorAPI>
    class ListPSModifier
	implements ParagraphStyleModifier, java.io.Serializable
    {
	Visualizable heading;
	int headingSpace;

	ListPSModifier(Visualizable heading, int headingSpace) {
	    this.heading      = heading;
	    this.headingSpace = headingSpace;
	}

	public ParagraphStyle modify(ParagraphStyle pStyle) {
	    if (pStyle.getHeadingSpace() == headingSpace &&
		pStyle.getHeading()      == heading)
	    {
		return pStyle;
	    }
	    int listLevel = getListLevel(pStyle);
	    if (listLevel == 0) listLevel = 1;
	    return new ParagraphStyle(pStyle.getStyleName(),
				      pStyle.getAlignment(),
				      getListIndent(listLevel),
				      pStyle.getRightIndent(),
				      pStyle.getLineSpace(),
				      pStyle.getParagraphSpace(),
				      pStyle.getTabWidth(),
				      heading,
				      headingSpace,
				      pStyle.getBaseStyle());
	}
    }

    abstract class LeftIndentPSModifier
	implements ParagraphStyleModifier, java.io.Serializable
    {
	boolean increment;
	
	LeftIndentPSModifier(boolean increment) {
	    this.increment = increment;
	}
	
	public abstract ParagraphStyle modify(ParagraphStyle pStyle);
    }

    class DocumentLeftIndentPSModifier
	extends LeftIndentPSModifier
    {
	
	DocumentLeftIndentPSModifier(boolean increment) {
	    super(increment);
	}
	
	public ParagraphStyle modify(ParagraphStyle pStyle) {
	    int listLevel = getListLevel(pStyle);
	    if (increment) {
		++listLevel;
	    }
	    else { // decrease
		if (listLevel > 0)
		    --listLevel;
	    }
	    Visualizable heading = (listLevel==0 ? null : pStyle.getHeading());
	    int headingSpace = (listLevel == 0 ? 0 : pStyle.getHeadingSpace());
	    
	    return new ParagraphStyle(pStyle.getStyleName(),
				      pStyle.getAlignment(),
				      getListIndent(listLevel),
				      pStyle.getRightIndent(),
				      pStyle.getLineSpace(),
				      pStyle.getParagraphSpace(),
				      pStyle.getTabWidth(),
				      heading,
				      headingSpace,
				      pStyle.getBaseStyle());
	}
    }

    protected int getListLevel(ParagraphStyle pStyle) {
	ParagraphStyle baseStyle =
	    focused.getRichText().getRichTextStyle().getParagraphStyle();
	return (pStyle.getLeftIndent() - baseStyle.getLeftIndent())
								/ LIST_INDENT;
    }

    protected int getListIndent(int listLevel) {
	ParagraphStyle baseStyle =
	    focused.getRichText().getRichTextStyle().getParagraphStyle();
	return baseStyle.getLeftIndent() + (LIST_INDENT * listLevel);
    }

    protected void setDocumentWriteTarget(File file) {
	setWriteTarget(file, false);
    }

    protected void setWriteTarget(File file, boolean asObject) {
	setWriteTarget(file);
	setWriteAsObject(file, asObject);
    }

    protected void setWriteAsObject(File file, boolean asObject) {
	if(file == null)
	    return;
	if(asObject) {
	    if(saveAsObject == null)
		saveAsObject = new Hashtable();
	    saveAsObject.put(file, Boolean.TRUE);
	}
	else {
	    if(saveAsObject != null &&
	       saveAsObject.get(file) != null) {
		if(getEditorType() != DOCUMENT_EDITOR)
		    saveAsObject.remove(file);
		else
		    saveAsObject.put(file, Boolean.FALSE);
	    }
	}
    }

    protected boolean isWriteAsObject(File file)
    {
	if(file == null || saveAsObject == null)
	    return false;

	Boolean b = (Boolean)saveAsObject.get(file);
	if (b == null)
	    return false;

	return ((Boolean)saveAsObject.get(file)).booleanValue();
    }

    /**
     * Opens the specified file.
     */
    public void openDocumentFile(File file) {
	boolean ok = false;
	try {
	    InputStream input = new FileInputStream(file);
	    disableSubComps();
	    ok = loadAsObject(input);
	    input.close();

	    if (!ok) {
		open_file(file);
		return;
	    }
	}
	catch (IOException e) {
	    warn(e);
	}
	finally {
	    enableSubComps();
	}

	if (ok) {
	    setWriteTarget(file, true);
	}
    }

    /**
     * Saves the text into the specified file.
     */
    public void saveDocumentFileAs(File file) {
	saveDocumentFileAs(file, isWriteAsObject(file));
    }

    /**
     * Saves the text into the selected file.
     * @param serialize if true, saves the text as an object.
     */
    public void saveDocumentFileAs(boolean serialize) {
	File writeTarget = (File)writeTargets.get(focused);
	File file = getFileFromSaveDialog(
			getToolTip(A_SAVE),
			(writeTarget != null ? writeTarget.getParent() : null),
			(writeTarget != null ? writeTarget.getName() : null));
	if (file == null)
	    return;
	saveDocumentFileAs(file, serialize);
    }

    /**
     * Saves the text into the specified file.
     * @param file      the file to store into.
     * @param serialize if true, saves the text as an object.
     */
    public void saveDocumentFileAs(File file, boolean serialize) {
	if (!serialize) {
	    save_file_as(file);
	    return;
	}

	boolean ok = false;
	try {
	    OutputStream output = new FileOutputStream(file);
	    disableSubComps();
	    ok = saveAsObject(output);
	    output.close();
	}
	catch (IOException e) {
	    warn(e);
	}
	finally {
	    enableSubComps();
	}

	if (ok) {
	    setWriteTarget(file, true);
	}
    }

    /**
     * Loads the contents of the specified stream as object into
     * this component.
     * @param stream the stream to be loaded.
     * @return true if the loading was succeeded.
     */
    public boolean loadAsObject(java.io.InputStream stream) {
	boolean loadOk = false;
	try {
	    java.io.ObjectInput in = new java.io.ObjectInputStream(stream);
	    RichText richText = (RichText)in.readObject();
	    focused.setRichText(richText);
	    focused.clearUndo();
	    textChanged = false;
	    loadOk = true;
	}
	catch (Exception e) {
	    //warn(e);
	}
	return loadOk;
    }

    /**
     * Saves the contents of this component as object into the
     * specified stream.
     * @param stream the stream to save into.
     * @return true if the saving was succeeded.
     */
    public boolean saveAsObject(java.io.OutputStream stream) {
	boolean saveOk = false;
	try {
	    java.io.ObjectOutput out = new java.io.ObjectOutputStream(stream);
	    out.writeObject(focused.getRichText());
	    out.flush();
	    textChanged = false;
	    saveOk = true;
	}
	catch (Exception e) {
	    warn(e);
	}
	return saveOk;
    }

    //</DocumentEditorAPI>

    //<DocumentEditorAPI+HTMLEditorAPI>
    /**
     * Makes the selected text large.
     */
    public void make_font_large() {
	set_font_size_diff(2);
    }

    /**
     * Makes the selected text small.
     */
    public void make_font_small() {
	set_font_size_diff(-2);
    }

    /**
     * Makes the selected paragraph as a list.
     */
    public void make_list() {
	if(isHTML()) {
	    int range[] = getListStyleRange();
	    focused.modifyRangeParagraphStyle(new HTMLPSModifier(2), range[0], range[1]);
	    return;
	}
	focused.modifySelectionParagraphStyle(new ListPSModifier(LIST_HEADING, LIST_HEADING_SPACE));
    }

    /**
     * Clears the selected paragraph as a list.
     */
    public void clear_list() {
	if(isHTML()) {
	    int range[] = getListStyleRange();
	    focused.modifyRangeParagraphStyle(new HTMLPSModifier(1), range[0], range[1]);
	    return;
	}
	BasicPSModifier modifier = new BasicPSModifier();
	modifier.put(BasicPSModifier.HEADING, BasicPSModifier.NULL);
	modifier.put(BasicPSModifier.HEADING_SPACE, 0);
	focused.modifySelectionParagraphStyle(modifier);
    }

    /**
     * Increases the selected paragraph indentation.
     */
    public void increase_indent() {
	if(isHTML())
	    focused.modifySelectionParagraphStyle(new HTMLLeftIndentPSModifier(true));
	else
	    focused.modifySelectionParagraphStyle(new DocumentLeftIndentPSModifier(true));
    }

    /**
     * Decreases the selected paragraph indentation.
     */
    public void decrease_indent() {
	if(isHTML())
	    focused.modifySelectionParagraphStyle(new HTMLLeftIndentPSModifier(false));
	else
	    focused.modifySelectionParagraphStyle(new DocumentLeftIndentPSModifier(false));
    }
    //</DocumentEditorAPI+HTMLEditorAPI>

    //<HTMLEditorAPI>
    class DefaultLinkAction implements ActionListener, java.io.Serializable {
	public void actionPerformed(ActionEvent e) {
	    URL url;
	    try {
		//url = new URL(getURL(), e.getActionCommand());
		// JDK1.2 bug? workaround
		String spec = e.getActionCommand();
		int start = 0;
		int limit = spec.length();
		while ((limit > 0) && (spec.charAt(limit - 1) <= ' ')) {
		    limit--;	// eliminate trailing whitespace
		}
		while ((start < limit) && (spec.charAt(start) <= ' ')) {
		    start++;	// eliminate leading whitespace
		}
		if (start < limit) {
		    spec = spec.substring(start, limit);
		    if (spec.charAt(0) == '#') {
			String base = getURL().toExternalForm();
			int index = base.indexOf('#');
			if (index >= 0) {
			    base = base.substring(0, index);
			}
			url = new URL(base + spec);
		    }
		    else {
			url = new URL(getURL(), spec);
		    }
		}
		else {
		    // spec is empty
		    url = getURL();
		}
	    }
	    catch (MalformedURLException ex) {
		warn(ex);
		return;
	    }

	    goto_page(url);
	}

    }

    protected TextEditModel createDefaultTextEditModel() {
	return new HTMLTextEditModel((HTMLStyle)HTMLStyles.get(L_M_STYLE));
    }

    /**
     * Sets the foreground color of this text component to be the specified
     * color.
     */
    public void setForeground(Color color) {
	Color oldValue = focused.getForeground();
	if (color != null && getModel() != null) {
	    getHTMLText().setTextColor(color);
	}
	if ((oldValue == null ? color == null : oldValue.equals(color)))
	    return;
	focused.setForeground(color);
	PropertyChangeSupport change = focused.getPropertyChangeSupport();
	if (change != null) {
	    change.firePropertyChange(P_FOREGROUND, oldValue, color);
	}
    }

    /**
     * Sets the background color of this text component to be the specified
     * color.
     */
    public void setBackground(Color color) {
	Color oldValue = focused.getBackground();
	if (color != null && getModel() != null) {
	    getHTMLText().setBackgroundColor(color);
	}
	if ((oldValue == null ? color == null : oldValue.equals(color)))
	    return;
	focused.setBackground(color);
	PropertyChangeSupport change = focused.getPropertyChangeSupport();
	if (change != null) {
	    change.firePropertyChange(P_BACKGROUND, oldValue, color);
	}
    }

    /**
     * Sets the action listener for the link to the specified listener.
     */
    public void setLinkActionListener(ActionListener l) {
	if (l == null)
	    throw new NullPointerException();
	linkActionListener = l;
    }

    /**
     * Returns the html text of this text component.
     * @see #setHTMLText(jp.kyasu.graphics.html.HTMLText)
     */
    public HTMLText getHTMLText() {
	return (HTMLText)focused.getRichText();
    }

    /**
     * Sets the html text of this text component.
     * @see #getHTMLText()
     */
    public void setHTMLText(HTMLText htmlText) {
	if (htmlText == null)
	    throw new NullPointerException();
	focused.setBackground(htmlText.getBackgroundColor());
	focused.setForeground(htmlText.getTextColor());

	focused.setRichText(htmlText);
    }

    /**
     * Returns the html style of this text component.
     */
    public HTMLStyle getHTMLStyle() {
	return getHTMLText().getHTMLStyle();
    }

    /**
     * Sets the html style of this html document to be the specified style.
     */
    public void setHTMLStyle(HTMLStyle htmlStyle) {
	TextChange change = getHTMLText().setHTMLStyle(htmlStyle);
	getView().textModelChanged(new TextModelEvent(
					getModel(),
					TextModelEvent.TEXT_MODEL_EDITED,
					change));
    }

    /**
     * Returns the url of this text component.
     */
    public URL getURL() {
	return getHTMLText().getURL();
    }

    /**
     * Sets the url of this text component to be the specified url.
     */
    public void setURL(URL url) {
	HTMLText htmlText = getHTMLText();
	URL oldValue = htmlText.getURL();
	htmlText.setURL(url);
	if (urlField != null) {
	    urlField.setText(url == null ? "" : url.toExternalForm());
	}
	if ((oldValue == null ? url == null : oldValue.equals(url)))
	    return;
	PropertyChangeSupport change = focused.getPropertyChangeSupport();
	if (change != null) {
	    change.firePropertyChange(P_URL, oldValue, url);
	}
    }

    /**
     * Returns the title of this text component.
     */
    public String getTitle() {
	return getHTMLText().getTitle();
    }

    /**
     * Sets the title of this text component to be the specified string.
     */
    public void setTitle(String title) {
	HTMLText htmlText = getHTMLText();
	String oldValue = htmlText.getTitle();
	htmlText.setTitle(title);
	if ((oldValue == null ? title == null : oldValue.equals(title)))
	    return;
	PropertyChangeSupport change = focused.getPropertyChangeSupport();
	if (change != null) {
	    change.firePropertyChange(P_TITLE, oldValue, title);
	}
    }

    /*
    /*
     * Returns the background color of this text component.
     *
    public Color getBackgroundColor() {
	return getHTMLText().getBackgroundColor();
    }

    /*
     * Sets the background color of this text component to be the specified
     * color.
     *
    public void setBackgroundColor(Color color) {
	getHTMLText().setBackgroundColor(color);
    }

    /*
     * Returns the text color of this text component.
     *
    public Color getTextColor() {
	return getHTMLText().getTextColor();
    }

    /*
     * Sets the text color of this text component to be the specified color.
     *
    public void setTextColor(Color color) {
	getHTMLText().setTextColor(color);
    }
    */

    /**
     * Returns the link color of this text component.
     */
    public Color getLinkColor() {
	return getHTMLText().getLinkColor();
    }

    /**
     * Sets the link color of this text component to be the specified color.
     */
    public void setLinkColor(Color color) {
	getHTMLText().setLinkColor(color);
    }

    /**
     * Returns the names of all target anchors (references) in this html
     * document.
     */
    public String[] getAllAnchorNames() {
	return getHTMLText().getAllAnchorNames();
    }

    /**
     * Returns the index of the specified target anchor (reference) in this
     * html document.
     *
     * @param  name the name of the target anchor (reference).
     * @return the index of the target anchor (reference); or <code>-1</code>
     *         if the target anchor (reference) does not exist.
     */
    public int getAnchorIndex(String name) {
	return getHTMLText().getAnchorIndex(name);
    }

    /**
     * Disables sub components.
     * @see #enableSubComps()
     * @see #disableSubComps(boolean)
     */
    public void disableHTMLSubComps() {
	disableHTMLSubComps(true);
    }

    /**
     * Disables sub components.
     * @param withStop if true, updates the stop button state.
     * @see #disableSubComps()
     */
    public synchronized void disableHTMLSubComps(boolean withStop) {
	if (savedTextCursor != null)
	    return;

	if (withStop) {
	    stopButton.setEnabled(true);
	}
	disableTextSubComps();
    }

    /**
     * Enables sub components.
     * @see #disableSubComps()
     * @see #enableSubComps(boolean)
     */
    public void enableHTMLSubComps() {
	enableHTMLSubComps(true);
    }

    /**
     * Enables sub components.
     * @param withStop if true, updates the stop button state.
     * @see #enableSubComps()
     */
    public synchronized void enableHTMLSubComps(boolean withStop) {
	if (savedTextCursor == null)
	    return;

	if (withStop) {
	    stopButton.setEnabled(false);
	}
	enableTextSubComps();
    }

    /**
     * Inserts a link into the selected text.
     */
    public void insert_link() {
	if (focused.selectionIsCaret())
	    return;
	String urlString = Dialog.request(focused.getFrame(),
					  getToolLabel("requestURL"),
					  40);
	if (urlString.length() == 0)
	    return;
	insert_link(urlString);
    }

    /**
     * Inserts a link with the specified url string into the selected text.
     */
    public void insert_link(String urlString) {
	if (focused.selectionIsCaret())
	    return;
	ClickableTextAction action = new ClickableTextAction(urlString);
	action.addActionListener(linkActionListener);
	BasicTSModifier modifier = new BasicTSModifier();
	modifier.put(BasicTSModifier.CLICKABLE, action);
	modifier.put(BasicTSModifier.COLOR,     getLinkColor());
	modifier.put(BasicTSModifier.UNDERLINE, true);
	focused.modifySelectionTextStyle(modifier);
    }

    /**
     * Inserts a link with the specified url event into the selected text.
     */
    public void insert_link(ActionEvent e) {
	String urlString = e.getActionCommand();
	if (urlString.length() == 0)
	    return;
	insert_link(urlString);
    }

    /**
     * Deletes a link from the selected text.
     */
    public void delete_link() {
	if (focused.selectionIsCaret())
	    return;
	BasicTSModifier modifier = new BasicTSModifier();
	modifier.put(BasicTSModifier.CLICKABLE, BasicTSModifier.NULL);
	modifier.put(BasicTSModifier.COLOR,     BasicTSModifier.NULL);
	modifier.put(BasicTSModifier.UNDERLINE, BasicTSModifier.NULL);
	focused.modifySelectionTextStyle(modifier);
    }

    class FontSizeModifier implements TextStyleModifier, java.io.Serializable
    {
	boolean large;

	FontSizeModifier(boolean large) {
	    this.large = large;
	}

	public TextStyle modify(TextStyle tStyle) {
	    if (!(tStyle instanceof ModTextStyle))
		return tStyle;
	    int diff = 0;
	    FontModifier fmodifier = ((ModTextStyle)tStyle).getFontModifier();
	    if (fmodifier != null) {
		Object obj = fmodifier.get(FontModifier.SIZE_DIFF);
		if (obj != null && (obj instanceof Integer)) {
		    diff = ((Integer)obj).intValue();
		}
	    }
	    HTMLStyle htmlStyle = getHTMLStyle();
	    int index = htmlStyle.getHTMLFontIndex(diff);
	    if (large) {
		if (index == 7)
		    return tStyle;
		BasicTSModifier modifier = new BasicTSModifier();
		modifier.put(BasicTSModifier.SIZE_DIFF,
			     htmlStyle.getFontPointDifference(index + 1));
		return modifier.modify(tStyle);
	    }
	    else { // small
		if (index == 1)
		    return tStyle;
		BasicTSModifier modifier = new BasicTSModifier();
		modifier.put(BasicTSModifier.SIZE_DIFF,
			     htmlStyle.getFontPointDifference(index - 1));
		return modifier.modify(tStyle);
	    }
	}
    }

    class HTMLPSModifier
	implements ParagraphStyleModifier, java.io.Serializable
    {
	int listMode;
	ParagraphStyle newStyle;

	HTMLPSModifier(int listMode) {
	    this.listMode = listMode;
	    this.newStyle = null;
	}

	HTMLPSModifier(ParagraphStyle pStyle) {
	    this.listMode = 0;
	    this.newStyle = pStyle;
	}

	public ParagraphStyle modify(ParagraphStyle pStyle) {
	    HTMLStyle htmlStyle = getHTMLStyle();
	    int align = pStyle.getAlignment();
	    int bqLevel = htmlStyle.getBqIncrementLevel(pStyle);
	    int listLevel = htmlStyle.getListIncrementLevel(pStyle);
	    switch (listMode) {
	    case 0: // new non list style
		bqLevel += listLevel;
		listLevel = 0;
		pStyle = newStyle;
		break;
	    case 1: // un list
		if (isListStyle(pStyle)) {
		    bqLevel += listLevel;
		    listLevel = 0;
		    pStyle = htmlStyle.getDefaultParagraphStyle();
		}
		else {
		    return pStyle;
		}
		break;
	    case 2: // unordered list
		if (!("LI-UL".equals(pStyle.getStyleName()))) {
		    listLevel += bqLevel;
		    bqLevel = 0;
		    if (listLevel < 1) listLevel = 1;
		    pStyle = htmlStyle.getULIParagraphStyle(listLevel,
							    focused.getForeground());
		}
		break;
	    case 3: // ordered list
		if (!("LI-OL".equals(pStyle.getStyleName()))) {
		    listLevel += bqLevel;
		    bqLevel = 0;
		    if (listLevel < 1) listLevel = 1;
		    pStyle = htmlStyle.getOLIParagraphStyle(listLevel,
							    0, focused.getForeground());
		}
		break;
	    case 4: // description title
		if (!("DT".equals(pStyle.getStyleName()))) {
		    listLevel += bqLevel;
		    bqLevel = 0;
		    if (listLevel < 1) listLevel = 1;
		    pStyle = htmlStyle.getDTParagraphStyle(listLevel);
		}
		break;
	    case 5: // description definition
		if (!("DD".equals(pStyle.getStyleName()))) {
		    listLevel += bqLevel;
		    bqLevel = 0;
		    if (listLevel < 1) listLevel = 1;
		    ++listLevel;
		    pStyle = htmlStyle.getDDParagraphStyle(listLevel);
		}
		break;
	    default:
		return pStyle;
	    }
	    if (align != ParagraphStyle.LEFT || bqLevel > 0) {
		BasicPSModifier modifier = new BasicPSModifier();
		if (align != ParagraphStyle.LEFT) {
		    modifier.put(BasicPSModifier.ALIGNMENT, align);
		}
		if (bqLevel > 0) {
		    modifier.put(BasicPSModifier.LEFT_INDENT,
			     htmlStyle.getLeftIndentation(bqLevel, listLevel));
		    modifier.put(BasicPSModifier.RIGHT_INDENT,
			     htmlStyle.getRightIndentation(bqLevel, listLevel));
		}
		pStyle = pStyle.deriveStyle(modifier);
	    }
	    return pStyle;
	}
    }

    /**
     * Sets the paragraph style named by the specified name.
     */
    public void set_paragraph_style(String name) {
	if (name == null)
	    return;

	if (name.equals("LI")) {
	    int r[] = getListStyleRange();
	    focused.modifyRangeParagraphStyle(new HTMLPSModifier(2), r[0], r[1]);
	}
	else if (name.equals("DT")) {
	    int r[] = getListStyleRange();
	    focused.modifyRangeParagraphStyle(new HTMLPSModifier(4), r[0], r[1]);
	}
	else if (name.equals("DD")) {
	    int r[] = getListStyleRange();
	    focused.modifyRangeParagraphStyle(new HTMLPSModifier(5), r[0], r[1]);
	}
	else {
	    ParagraphStyle pStyle = getHTMLStyle().getParagraphStyle(name);
	    if (pStyle == null)
		return;
	    int r[] = getListStyleRange();
	    focused.modifyRangeParagraphStyle(new HTMLPSModifier(pStyle), r[0], r[1]);
	}
    }

    /**
     * Makes the selected paragraph as an ordered list.
     */
    public void make_ordered_list() {
	int range[] = getListStyleRange();
	focused.modifyRangeParagraphStyle(new HTMLPSModifier(3), range[0], range[1]);
    }

    /**
     * Clears the selected paragraph as an ordered list.
     */
    public void clear_ordered_list() {
	clear_list();
    }

    /**
     * Inserts an anchor.
     */
    public void insert_anchor() {
	String target = Dialog.request(focused.getFrame(),
				       getToolLabel("requestAnchor"));
	if (target.length() == 0)
	    return;
	insert_anchor(target);
    }

    /**
     * Inserts an anchor with the specified target string.
     */
    public void insert_anchor(String target) {
	Text text = new Text(new TextAttachment(new VAnchor(target)),
			     getInsertionStyle());
	focused.replaceSelection(text);
    }

    /**
     * Inserts an anchor with the specified target event into the selected text.
     */
    public void insert_anchor(ActionEvent e) {
	String target = e.getActionCommand();
	if (target.length() == 0)
	    return;
	insert_anchor(target);
    }

    /**
     * Goes to the page with the specified url.
     */
    public void goto_page(URL url) {
	if (url == null)
	    throw new NullPointerException();
	load(url, false);
    }

    /**
     * Goes to the page with the specified url string.
     */
    public void goto_page(String urlString) {
	URL url;
	try {
	    url = new URL(urlString);
	}
	catch (MalformedURLException e) {
	    warn(e);
	    return;
	}
	goto_page(url);
    }

    /**
     * Goes to the page with the specified url action.
     */
    public void goto_page(ActionEvent e) {
	goto_page(e.getActionCommand());
    }

    /**
     * Reloads the current page.
     */
    public void reload_page() {
	URL url = getURL();
	if (url == null || historyIndex < 0 || history.isEmpty()) {
	    goto_page(getToolLabel(L_KFC_URL));
	}
	load(url, true);
    }

    /**
     * Goes to the next page.
     */
    public void forward_page() {
	Stack history = (Stack)htmlHistories.get(focused);
	int historyIndex = ((Integer)htmlHistoryIndices.get(focused)).intValue();
	if (historyIndex < 0 || historyIndex == history.size() - 1) {
	    return;
	}
	updateLocationOfTextInHistory();
	++historyIndex;
	if (historyIndex == history.size() - 1) {
	    forwardButton.setEnabled(false);
	}
	if (historyIndex > 0) {
	    backwardButton.setEnabled(true);
	}
	gotoHistory(historyIndex);
    }

    /**
     * Goes to the previouse page.
     */
    public void backward_page() {
	Stack history = (Stack)htmlHistories.get(focused);
	int historyIndex = ((Integer)htmlHistoryIndices.get(focused)).intValue();
	if (historyIndex < 0 || historyIndex == 0) {
	    return;
	}
	updateLocationOfTextInHistory();
	--historyIndex;
	if (historyIndex == 0) {
	    backwardButton.setEnabled(false);
	}
	if (historyIndex < history.size() - 1) {
	    forwardButton.setEnabled(true);
	}
	gotoHistory(historyIndex);
    }

    /**
     * Stops the loading.
     */
    public void stop_loading() {
	closeLoadInputStream();
	Thread backgroundThread = 
	    (Thread) backgroundThreads.get(focused);
	if (backgroundThread != null) {
	    Thread thread = backgroundThread;
	    backgroundThreads.remove(focused);
	    backgroundThread = null;
	    thread.interrupt();
	    try { thread.join(); } // wait for backgroundThread to die
	    catch (InterruptedException e) {}
	}
    }

    /**
     * Edits the document property.
     */
    public void edit_document_property() {
	String title = getTitle();
	title = Dialog.request(focused.getFrame(),
			       getToolTip(A_DOC_TITLE) + ":",
			       (title == null ? "" : title),
			       40);
	if (title.length() == 0)
	    return;
	setTitle(title);
    }

    /**
     * Saves the text into the file.
     */
    public void saveHTMLFile() {
	// TODO
	File writeTarget = new File("edit.html");
	URL url = getURL();
	String file;
	if (url != null && (file = url.getFile()) != null) {
	    int index = file.lastIndexOf('/');
	    if (index >= 0 && index < file.length() - 1) {
		file = file.substring(index + 1, file.length());
		index = file.lastIndexOf('.');
		if (index >= 1) {
		    file = file.substring(0, index);
		    writeTarget = new File(file + ".html");
		}
	    }
	}

	saveHTMLFileAs(writeTarget);
    }

    /**
     * Saves the text into the specified file.
     */
    public void saveHTMLFileAs(File file) {
	boolean ok = false;
	try {
	    disableHTMLSubComps(false);
	    ok = saveAsHTML(file);
	}
	finally {
	    enableHTMLSubComps(false);
	}

	if (ok) {
	    setWriteTarget(file);
	}
    }

    /**
     * Saves the contents of this component as a HTML into the specified file.
     * @param file the file to be saved into.
     * @return true if the saving was succeeded.
     */
    public boolean saveAsHTML(File file) {
	boolean saveOk = false;
	try {
	    HTMLWriter htmlWriter = new HTMLWriter(getHTMLText());
	    htmlWriter.writeTo(file, getWriteEncoding());
	    textChanged = false;
	    saveOk = true;
	}
	catch (Exception e) {
	    warn(e);
	}
	return saveOk;
    }

    class LoadRunnable implements Runnable {
	URL url;
	boolean reload;

	LoadRunnable(URL url, boolean reload) {
	    this.url    = url;
	    this.reload = reload;
	}

	public void run() {
	    Exception ex = null;
	    InputStream loadInputStream = null;
	    try {
		disableHTMLSubComps();

		if (!reload && loadHistory(url))
		    return;

		URLConnection conn = url.openConnection();
		conn.getURL();
		conn.getContentType();
		loadInputStream = conn.getInputStream();
		if(htmlLoadInputStreams == null)
		    htmlLoadInputStreams = new Hashtable();
		htmlLoadInputStreams.put(focused, loadInputStream);
		

		load(conn, reload);
	    }
	    catch (IOException ie)       { ex = ie; }
	    catch (SecurityException se) { ex = se; }
	    finally {
		if(loadInputStream != null &&
		   htmlLoadInputStreams.get(focused) != null)
		    htmlLoadInputStreams.remove(focused);
		loadInputStream  = null;
		Thread backgroundThread = 
		    (Thread)backgroundThreads.get(focused);
		if(backgroundThread != null) {
		    backgroundThreads.remove(focused);
		    backgroundThread = null;
		}

		enableSubComps();
	    }

	    if (ex != null) {
		warn(ex);
	    }
	}
    }

    protected void load(URL url, boolean reload) {
	Thread backgroundThread =
	    (Thread) backgroundThreads.get(focused);
	if (backgroundThread != null)
	    return;
	backgroundThread = new Thread(new LoadRunnable(url, reload));
	backgroundThreads.put(focused, backgroundThread);
	try {
	    int p = Math.max(Thread.currentThread().getPriority() - 1,
			     Thread.MIN_PRIORITY);
	    backgroundThread.setPriority(p);
	}
	catch (SecurityException e) {}
	backgroundThread.start();
    }

    protected boolean loadHistory(URL url) {
	HistoryElement e = getHistoryElement(url);
	if (e == null)
	    return false;

	updateLocationOfTextInHistory();

	HTMLText htmlText = e.htmlText;
	htmlText.setURL(url);

	setHTMLText(htmlText);
	setURL(url);
	setTitle(htmlText.getTitle());
	setBackground(htmlText.getBackgroundColor());
	setForeground(htmlText.getTextColor());
	setLinkColor(htmlText.getLinkColor());

	textChanged = false;

	String ref = getURL().getRef();
	int index = -1;
	if (ref != null) {
	    index = getAnchorIndex(ref);
	}

	// enables at here for setCaretPosition() and pushToHistory()
	enableSubComps();

	focused.setCaretPosition((index >= 0 ? index : 0), true);

	pushToHistory(getHTMLText());

	return true;
    }

    protected void load(URLConnection conn, boolean reload) {
	URL url = conn.getURL();
	String contentType = conn.getContentType();

	updateLocationOfTextInHistory();

	setHTMLText(new HTMLText(getHTMLStyle()));
	setURL(url);
	setTitle(url.toExternalForm());
	setBackground(Color.white);
	setForeground(Color.black);
	setLinkColor(Color.blue);

	try {
	    if (contentType == null) {
		loadError(conn, "No content type");
	    }
	    if (contentType.equals("text/html")) {
		loadHTML(conn, url);
	    }
	    else if (contentType.equals("text/plain")) {
		loadText(conn);
	    }
	    else if (contentType.equals("image/gif")      ||
		     contentType.equals("image/jpeg")     ||
		     contentType.equals("image/x-bitmap") ||
		     contentType.equals("image/x-pixmap"))
	    {
		loadImage(conn, url);
	    }
	    else {
		loadError(conn, "Unknown content type");
	    }
	}
	catch (IOException e) {
	}

	closeLoadInputStream();

	textChanged = false;

	String ref = getURL().getRef();
	int index = -1;
	if (ref != null) {
	    index = getAnchorIndex(ref);
	}

	// enables at here for setCaretPosition() and pushToHistory()
	enableSubComps();

	focused.setCaretPosition((index >= 0 ? index : 0), true);

	if (reload) {
	    updateHistory(getHTMLText());
	}
	else {
	    pushToHistory(getHTMLText());
	}
    }

    protected void loadHTML(URLConnection conn, URL url) throws IOException {
	HTMLReader htmlReader = new HTMLReader(getHTMLStyle(),
					       linkActionListener);
	HTMLReaderTarget target =
		(isIncrementalLoad() ?
		 //TBD
		 //			(HTMLReaderTarget)new HTMLEditorTarget(this) :
		 (HTMLReaderTarget)new DefaultHTMLReaderTarget() :
			(HTMLReaderTarget)new DefaultHTMLReaderTarget());
	BufferedReader reader = new BufferedReader(
				new InputStreamReader(conn.getInputStream(),
						      getReadEncoding()),
				(32 * 1024));

	try {
	    URL oldURL = getURL();
	    String oldTitle = getTitle();

	    htmlReader.read(url, reader, target);
	    if (!isIncrementalLoad()) {
		HTMLText htmlText = target.getHTMLText();

		setHTMLText(htmlText);

		// fires PropertyChangeEvents
		URL newURL = htmlText.getURL();
		String newTitle = htmlText.getTitle();
		if (!oldURL.equals(newURL)) {
		    htmlText.setURL(oldURL);
		    setURL(newURL);
		}
		if (!oldTitle.equals(newTitle)) {
		    htmlText.setTitle(oldTitle);
		    setTitle(newTitle);
		}
		setBackground(htmlText.getBackgroundColor());
		setForeground(htmlText.getTextColor());
	    }
	}
	finally {
	    target.close();
	    reader.close();
	}
    }

    protected void loadText(URLConnection conn) throws IOException {
	BufferedReader reader = new BufferedReader(
				new InputStreamReader(conn.getInputStream(),
						      getReadEncoding()),
				(32 * 1024));

	try {
	    int incCount = (isIncrementalLoad() ? INC_LINE_COUNT : 0);

	    ParagraphStyle pStyle = getHTMLStyle().getParagraphStyle("PRE");
	    focused.setCaretPosition(0);
	    boolean editable = focused.isEditable();
	    try {
		focused.setEditable(true);
		focused.setSelectionParagraphStyle(pStyle);
	    }
	    finally {
		focused.setEditable(editable);
	    }

	    TextStyle tStyle = pStyle.getBaseStyle();
	    TextBuffer buffer = new TextBuffer();
	    buffer.setTextStyle(tStyle);
	    int lineCount = 0;
	    String line;
	    while ((line = reader.readLine()) != null) {
		InputStream loadInputStream = 
		    (InputStream)htmlLoadInputStreams.get(focused);
		if (loadInputStream == null)
		    throw new IOException();

		if (incCount > 0 && ++lineCount > incCount) {
		    appendAnyway(buffer.toText(), false);
		    lineCount = 0;
		    buffer = new TextBuffer();
		    buffer.setTextStyle(tStyle);
		}
		buffer.append(line).append(Text.LINE_BREAK_CHAR);
	    }
	    if (buffer.length() > 0) {
		appendAnyway(buffer.toText(), false);
	    }
	}
	finally {
	    reader.close();
	}
    }

    protected void loadImage(URLConnection conn, URL url) {
	Image image = focused.getToolkit().getImage(url);

	focused.setCaretPosition(0);
	appendAnyway(
		new Text(
		    new TextAttachment(new VImage(image, url)),
		    ((HTMLText)focused.getRichText()).getHTMLStyle().getDefaultParagraphStyle().getBaseStyle()),
		false);
    }

    protected void loadError(URLConnection conn, String errorMessage) {
	focused.setCaretPosition(0);
	appendAnyway(
		new Text(
		    errorMessage,
		    ((HTMLText)focused.getRichText()).getHTMLStyle().getDefaultParagraphStyle().getBaseStyle()),
		false);
    }

    protected HistoryElement getHistoryElement(URL url) {
	if (history.isEmpty()) {
	    return null;
	}
	for (Enumeration e = history.elements(); e.hasMoreElements(); ) {
	    HistoryElement elem = (HistoryElement)e.nextElement();
	    if (url.sameFile(elem.htmlText.getURL())) {
		return elem;
	    }
	}
	return null;
    }

    protected void pushToHistory(HTMLText htmlText) {
	if (historyIndex >= 0) {
	    while (historyIndex < history.size() - 1) {
		history.pop();
	    }
	}
	HistoryElement elem = new HistoryElement(
					htmlText,
					focused.getLocationOfText(),
					htmlText.getURL().getRef());
	history.push(elem);
	while (history.size() > MAX_HISTORY) {
	    history.removeElementAt(0);
	}
	historyIndex = history.size() - 1;
	forwardButton.setEnabled(false);
	if (historyIndex > 0) {
	    backwardButton.setEnabled(true);
	}
    }

    protected void updateHistory(HTMLText htmlText) {
	if (historyIndex < 0) {
	    return;
	}
	HistoryElement elem = new HistoryElement(
					htmlText,
					focused.getLocationOfText(),
					htmlText.getURL().getRef());
	history.setElementAt(elem, historyIndex);
    }

    protected void gotoHistory(int index) {
	historyIndex = index;
	HistoryElement elem = (HistoryElement)history.elementAt(historyIndex);
	HTMLText htmlText = elem.htmlText;
	URL url = htmlText.getURL();
	if (elem.ref == null) {
	    if (url.getRef() != null) {
		try {
		    htmlText.setURL(new URL(url.getProtocol(),
					    url.getHost(),
					    url.getPort(),
					    url.getFile()));
		}
		catch (MalformedURLException e) {}
	    }
	}
	else {
	    if (!elem.ref.equals(url.getRef())) {
		try {
		    htmlText.setURL(new URL(url.getProtocol(),
					    url.getHost(),
					    url.getPort(),
					    url.getFile() + "#" + elem.ref));
		}
		catch (MalformedURLException e) {}
	    }
	}

	URL oldURL = getURL();
	String oldTitle = getTitle();

	setHTMLText(htmlText);

	// fires PropertyChangeEvents
	URL newURL = htmlText.getURL();
	String newTitle = htmlText.getTitle();
	if (!oldURL.equals(newURL)) {
	    htmlText.setURL(oldURL);
	    setURL(newURL);
	}
	if (!oldTitle.equals(newTitle)) {
	    htmlText.setTitle(oldTitle);
	    setTitle(newTitle);
	}

	setForeground(htmlText.getTextColor());
	setBackground(htmlText.getBackgroundColor());
	setLinkColor(htmlText.getLinkColor());
	focused.setLocationOfText(elem.locationOfText);
    }

    protected void updateLocationOfTextInHistory() {
	if (historyIndex < 0 || history.isEmpty()) {
	    return;
	}
	HistoryElement elem = (HistoryElement)history.elementAt(historyIndex);
	elem.locationOfText = focused.getLocationOfText();
    }

    protected void closeLoadInputStream() {
	InputStream loadInputStream = 
	    (InputStream)htmlLoadInputStreams.get(focused);
	if (loadInputStream != null) {
	    try { loadInputStream.close(); } // I/O exception will occur.
	    catch (IOException e) {}
	    finally {
		htmlLoadInputStreams.remove(focused);
		loadInputStream = null;
	    }
	}
    }


    class URLAction implements ActionListener, java.io.Serializable {
	public void actionPerformed(ActionEvent e) {
	    goto_page(e);
	}
    }


    //</HTMLEditorAPI>
    

    //<RichTextEditorAPI>
    protected Component[] createRichTextFontComponents1() {
	boldButton = createIconToggleButton(I_BOLD);
	boldButton.addItemListener(this);
	italicButton = createIconToggleButton(I_ITALIC);
	italicButton.addItemListener(this);

	boldButton.setEnabled(false);
	italicButton.setEnabled(false);

	addCaretDisableComp(boldButton);
	addCaretDisableComp(italicButton);

	return new Component[]{ boldButton, italicButton };
    }

    protected Component[] createRichTextScriptComponents() {
	superscriptButton = createIconButton(I_SUPERSCRIPT);
	superscriptButton.addActionListener(this);
	normalscriptButton = createIconButton(I_NORMALSCRIPT);
	normalscriptButton.addActionListener(this);
	subscriptButton = createIconButton(I_SUBSCRIPT);
	subscriptButton.addActionListener(this);

	superscriptButton.setEnabled(false);
	normalscriptButton.setEnabled(false);
	subscriptButton.setEnabled(false);

	addCaretDisableComp(superscriptButton);
	addCaretDisableComp(normalscriptButton);
	addCaretDisableComp(subscriptButton);

	return new Component[]{ superscriptButton, normalscriptButton, subscriptButton };
    }

    protected Component[] createRichTextFontDecorationComponents() {
	underlineButton = createIconToggleButton(I_UNDERLINE);
	underlineButton.addItemListener(this);

	underlineButton.setEnabled(false);

	addCaretDisableComp(underlineButton);

	return new Component[]{ underlineButton };
    }

    protected Component[] createRichTextFontAttributeComponents() {
	ColorButton color = new ColorButton();
	color.addItemListener(this);
	color.setToolTipText(getToolTip("fontColor"));

	color.setEnabled(false);
	addCaretDisableComp(color);

	return new Component[]{ color };
    }

    protected Component[] createRichTextFontSizeComponents() {
	Button large = createIconButton(A_LARGE);
	large.addActionListener(this);
	Button small = createIconButton(A_SMALL);
	small.addActionListener(this);

	large.setEnabled(false);
	small.setEnabled(false);
	addCaretDisableComp(large);
	addCaretDisableComp(small);

	return new Component[]{ large, small };
    }


    protected Component[] createRichTextAlignmentComponents() {
	Button left = createIconButton(A_LEFT);
	left.addActionListener(this);
	Button center = createIconButton(A_CENTER);
	center.addActionListener(this);
	Button right = createIconButton(A_RIGHT);
	right.addActionListener(this);
	return new Component[]{ left, center, right };
    }

    /**
     * Sets the font name of the selected text to be the specified name.
     */
    public void set_font_name(String name) {
	if (name == null)
	    throw new NullPointerException();
	if (focused.selectionIsCaret())
	    return;
	BasicTSModifier modifier = new BasicTSModifier();
	modifier.put(BasicTSModifier.NAME, name);
	focused.modifySelectionTextStyle(modifier);
    }

    /**
     * Sets the font name of the selected text to be the selected name.
     * Selected item must be a <code>String</code>.
     */
    public void set_font_name(ItemEvent e) {
	if (e.getStateChange() != ItemEvent.SELECTED)
	    return;
	Object obj = e.getItem();
	if (obj != null && (obj instanceof String)) {
	    set_font_name((String)obj);
	}
    }

    /**
     * Sets the font size of the selected text to be the specified size.
     */
    public void set_font_size(int size) {
	if (focused.selectionIsCaret())
	    return;
	BasicTSModifier modifier = new BasicTSModifier();
	modifier.put(BasicTSModifier.SIZE, size);
	focused.modifySelectionTextStyle(modifier);
    }

    /**
     * Sets the font size of the selected text differed from the specified size.
     */
    public void set_font_size_diff(int size) {
	if (focused.selectionIsCaret())
	    return;
	if(isHTML()) {
	    focused.modifySelectionTextStyle(new FontSizeModifier(size > -1));
	}
	else {
	    BasicTSModifier modifier = new BasicTSModifier();
	    modifier.put(BasicTSModifier.SIZE_DIFF, size);
	    focused.modifySelectionTextStyle(modifier);
	}
    }

    /**
     * Sets the font size of the selected text to be the selected size.
     * Selected item must be an <code>Integer</code> or a <code>String</code>
     * such as "2", "+2", and "-2" .
     */
    public void set_font_size(ItemEvent e) {
	if (e.getStateChange() != ItemEvent.SELECTED)
	    return;
	Object obj = e.getItem();
	if (obj == null)
	    return;
	if (obj instanceof Integer) {
	    set_font_size(((Integer)obj).intValue());
	}
	else if (obj instanceof String) {
	    Object signedInt[] = parseSignedInt((String)obj);
	    if (signedInt == null)
		return;
	    String sign = (String)signedInt[0];
	    int val = ((Integer)signedInt[1]).intValue();
	    if ("+".equals(sign))
		set_font_size_diff(val);
	    else if ("-".equals(sign))
		set_font_size_diff(-val);
	    else
		set_font_size(val);
	}
    }

    /**
     * Sets the font color of the selected text to be the specified color.
     */
    public void set_font_color(Color color) {
	if (focused.selectionIsCaret())
	    return;
	BasicTSModifier modifier = new BasicTSModifier();
	if (color != null)
	    modifier.put(BasicTSModifier.COLOR, color);
	else
	    modifier.put(BasicTSModifier.COLOR, BasicTSModifier.NULL);
	focused.modifySelectionTextStyle(modifier);
    }

    /**
     * Sets the font color of the selected text to be the selected color.
     */
    public void set_font_color(ItemEvent e) {
	if (e.getStateChange() != ItemEvent.SELECTED)
	    return;
	Object obj = e.getItem();
	if (obj == null) {
	    set_font_color((Color)null);
	}
	else if (obj instanceof Color) {
	    set_font_color((Color)obj);
	}
    }

    /**
     * Makes or clears the selected text bold.
     */
    public void make_font_bold(boolean bold) {
	if(focused == null ||
	   focused.selectionIsCaret())
	    return;
	BasicTSModifier modifier = new BasicTSModifier();
	if (bold)
	    modifier.put(BasicTSModifier.BOLD, true);
	else
	    modifier.put(BasicTSModifier.BOLD, BasicTSModifier.NULL);
	focused.modifySelectionTextStyle(modifier);
    }

    /** Makes the selected text bold. */
    public void make_font_bold() {
	make_font_bold(true);
    }

    /** Clears the selected text bold. */
    public void clear_font_bold() {
	make_font_bold(false);
    }

    /**
     * Makes or clears the selected text italic.
     */
    public void make_font_italic(boolean italic) {
	if (focused.selectionIsCaret())
	    return;
	BasicTSModifier modifier = new BasicTSModifier();
	if (italic)
	    modifier.put(BasicTSModifier.ITALIC, true);
	else
	    modifier.put(BasicTSModifier.ITALIC, BasicTSModifier.NULL);
	focused.modifySelectionTextStyle(modifier);
    }

    /** Makes the selected text italic. */
    public void make_font_italic() {
	make_font_italic(true);
    }

    /** Clears the selected text italic. */
    public void clear_font_italic() {
	make_font_italic(false);
    }

    /**
     * Makes or clears the selected text underlined.
     */
    public void make_font_underlined(boolean underline) {
	if (focused.selectionIsCaret())
	    return;
	BasicTSModifier modifier = new BasicTSModifier();
	if (underline)
	    modifier.put(BasicTSModifier.UNDERLINE, true);
	else
	    modifier.put(BasicTSModifier.UNDERLINE, BasicTSModifier.NULL);
	focused.modifySelectionTextStyle(modifier);
    }

    /** Makes or clears the selected text underlined. */
    public void make_font_underlined() {
	make_font_underlined(true);
    }

    /** Clears the selected text underlined. */
    public void clear_font_underlined() {
	make_font_underlined(false);
    }

    /**
     * Makes or clears the selected text super/subsctipted..
     */
    public int set_font_script(int scriptLevel) {
	int level = focused.getSelectedTEXT().getTextStyleAt(0).getExtendedFont().getScriptLevel();
	if (!focused.selectionIsCaret()){
	    if(scriptLevel == NORMALSCRIPT)
		level = NORMALSCRIPT;
	    else{
		level = Math.abs(level) + Math.abs(scriptLevel);
		if(scriptLevel < 0)
		    level = -level;
	    }
	    BasicTSModifier modifier = new BasicTSModifier();
	    modifier.put(BasicTSModifier.SCRIPT, new Integer(level));
	    focused.modifySelectionTextStyle(modifier);
	}
	return level;
    }

    /** Clears the selected text super/subscripted */
    public int make_font_normalscripted() {
	return set_font_script(NORMALSCRIPT);
    }

    /** Makes the selected text superscripted. */
    public int make_font_superscripted() {
	return set_font_script(SUPERSCRIPT);
    }

    /** Clears the selected text superscripted. */
    public int clear_font_superscripted() {
	return set_font_script(NORMALSCRIPT);
    }

    /** Makes the selected text subscripted. */
    public int make_font_subscripted() {
	return set_font_script(SUBSCRIPT);
    }

    /** Clears the selected text subscripted. */
    public int clear_font_subscripted() {
	return set_font_script(NORMALSCRIPT);
    }

    /**
     * Clears the all styles of the selected text.
     */
    public void clear_font_styles() {
	if (focused.selectionIsCaret())
	    return;
	BasicTSModifier modifier = new BasicTSModifier();
	modifier.put(BasicTSModifier.BOLD,      BasicTSModifier.NULL);
	modifier.put(BasicTSModifier.ITALIC,    BasicTSModifier.NULL);
	modifier.put(BasicTSModifier.UNDERLINE, BasicTSModifier.NULL);
	modifier.put(BasicTSModifier.COLOR,     BasicTSModifier.NULL);
	modifier.put(BasicTSModifier.SCRIPT, ExtendedFont.NORMALSCRIPT);
	modifier.put(BasicTSModifier.CLICKABLE, BasicTSModifier.NULL);
	focused.modifySelectionTextStyle(modifier);
    }

    /**
     * Sets the alignment of the selected paragraph to be the specified style.
     * @see jp.kyasu.graphics.ParagraphStyle#LEFT
     * @see jp.kyasu.graphics.ParagraphStyle#CENTER
     * @see jp.kyasu.graphics.ParagraphStyle#RIGHT
     */
    public void set_alignment(int align) {
	switch (align) {
	case ParagraphStyle.LEFT:
	case ParagraphStyle.CENTER:
	case ParagraphStyle.RIGHT:
	    break;
	default:
	    throw new IllegalArgumentException("improper alignment: " + align);
	}
	BasicPSModifier modifier = new BasicPSModifier();
	modifier.put(BasicPSModifier.ALIGNMENT, align);
	focused.modifySelectionParagraphStyle(modifier);
    }

    /** Sets the alignment of the selected paragraph to be left. */
    public void align_to_left() {
	set_alignment(ParagraphStyle.LEFT);
    }

    /** Sets the alignment of the selected paragraph to be centered. */
    public void align_to_center() {
	set_alignment(ParagraphStyle.CENTER);
    }

    /** Sets the alignment of the selected paragraph to be right. */
    public void align_to_right() {
	set_alignment(ParagraphStyle.RIGHT);
    }

    /**
     * Inserts a horizontal line.
     */
    public void insert_hr() {
	TextAttachment ta = new TextAttachment(new VHRBorder());
	ta.setRatioToWidth(1.0f);
	insertTextAttachmentAsLine(ta);
    }

    /**
     * Inserts an image.
     */
    public void insert_image() {
	File writeTarget = null;
	if(writeTargets != null)
	    writeTarget = (File)writeTargets.get(focused);
	File file = getFileFromLoadDialog(
			getToolTip(A_IMAGE),
			(writeTarget != null ? writeTarget.getParent() : null),
			null);
	if (file == null)
	    return;
	VImage image = new VImage(file.getPath());
	if (image.getImage() == null) {
	    Dialog.warn(focused.getFrame(),
			file.getPath() + getToolLabel("fileNotImage"));
	    return;
	}
	insertVisualizable(image);
    }

    protected void insertTextAttachmentAsLine(TextAttachment ta) {
	TextBuffer buffer = new TextBuffer();
	buffer.setTextStyle(getInsertionStyle());
	buffer.append(Text.LINE_SEPARATOR_CHAR);
	buffer.append(ta);
	buffer.append(Text.LINE_SEPARATOR_CHAR);
	focused.replaceSelection(buffer.toText());
    }

    protected void insertVisualizable(Visualizable v) {
	focused.replaceSelection(new Text(new TextAttachment(v), getInsertionStyle()));
    }

    protected TextStyle getInsertionStyle() {
	return focused.getRichText().getRichTextStyle().getTextStyle();
    }

    protected Integer parseInt(String s) {
	if (s == null)
	    return null;
	try {
	    return new Integer(Integer.parseInt(s));
	}
	catch (NumberFormatException e) {
	    return null;
	}
    }

    protected Object[] parseSignedInt(String s) {
	if (s == null || s.length() == 0)
	    return null;
	char c = s.charAt(0);
	String sign = null;
	if (c == '+' || c == '-') {
	    s = s.substring(1, s.length());
	    sign = new String(new char[]{ c });
	}
	Integer val = parseInt(s);
	if (val == null)
	    return null;
	return new Object[]{ sign, val };
    }
    //</RichTextEditorAPI>

    
    /**
     * Invoked when an action occurs.
     * @see java.awt.event.ActionListener
     */
    public void actionPerformed(ActionEvent event)
    {
	Object source = event.getSource();
	//<TextEditor>
	if (source == openButton) {
	    open_file();
	}
	else if (source == saveButton) {
	    save_file();
	}
	else if (source == printButton) {
	    print_file();
	}
	else if (source == copyButton) {
	    focused.copy_clipboard();
	}
	else if (source == cutButton) {
	    focused.cut_clipboard();
	}
	else if (source == pasteButton) {
	    focused.paste_clipboard();
	}
	else if (source == undoButton) {
	    focused.undo();
	}
	else if (source == findButton) {
	    focused.find_word();
	}
	else if (source == gotoButton) {
	    focused.goto_line();
	}
	//</TextEditor>
	//<RichTextEditor>
	else if (source == boldButton) {
	    make_font_bold();
	}
	else if (source == italicButton) {
	    make_font_italic();
	}
	else if (source == underlineButton) {
	    make_font_underlined();
	}
	else if (source == superscriptButton) {
	    make_font_superscripted();
	}
	else if (source == normalscriptButton) {
	    make_font_normalscripted();
	}
	else if (source == subscriptButton) {
	    make_font_subscripted();
	}
	else if (source == largeButton) {
	    make_font_large();
	}
	else if (source == smallButton) {
	    make_font_small();
	}
	else if (source == leftAlignButton) {
	    align_to_left();
	}
	else if (source == centerAlignButton) {
	    align_to_center();
	}
	else if (source == rightAlignButton) {
	    align_to_right();
	}
	else if (source == imageButton) {
	    insert_image();
	}
	else if (source == hrButton) {
	    insert_hr();
	}
	//</RichTextEditor>
	//<DocumentEditor>
	if (source == incIndentButton) {
	    increase_indent();
	}
	else if (source == decIndentButton) {
	    decrease_indent();
	}
	else if (source == listButton) {
	    make_list();
	}
	else if (source == unlistButton) {
	    clear_list();
	}
	//</DocumentEditor>
	//<HTMLEditor>
	else if (source == linkButton) {
	    insert_link();
	}
	else if (source == orderdListButton) {
	    make_ordered_list();
	}
	else if (source == anchorButton) {
	    insert_anchor();
	}
	else if (source == forwardButton) {
	    forward_page();
	}
	else if (source == backwardButton) {
	    backward_page();
	}
	else if (source == stopButton) {
	    stop_loading();
	}
	else if (source == reloadButton) {
	    reload_page();
	}
	else if (source == docTitleButton) {
	    edit_document_property();
	}
	//</HTMLEditor>
	else {
	    String command = event.getActionCommand();
	    //</TextEditor>
	    if (command.equals(A_OPEN)) {
		open_file();
	    }
	    else if (command.equals(A_SAVE)) {
		save_file();
	    }
	    else if (command.equals(A_SAVE_AS)) {
		save_file_as();
	    }
	    else if (command.equals(A_PRINT)) {
		print_file();
	    }
	    else if (command.equals(A_COPY)) {
		focused.copy_clipboard();
	    }
	    else if (command.equals(A_CUT)) {
		focused.cut_clipboard();
	    }
	    else if (command.equals(A_PASTE)) {
		focused.paste_clipboard();
	    }
	    else if (command.equals(A_UNDO)) {
		focused.undo();
	    }
	    else if (command.equals(A_FIND)) {
		focused.find_word();
	    }
	    else if (command.equals(A_GOTO)) {
		focused.goto_line();
	    }
	    //</TextEditor>
	    //<RichTextEditor>
	    else if (command.equals(A_BOLD)) {
		make_font_bold();
	    }
	    else if (command.equals(A_ITALIC)) {
		make_font_italic();
	    }
	    else if (command.equals(A_UNDERLINE)) {
		make_font_underlined();
	    }
	    else if (command.equals(A_SUPERSCRIPT)) {
		make_font_superscripted();
	    }
	    else if (command.equals(A_NORMALSCRIPT)) {
		make_font_normalscripted();
	    }
	    else if (command.equals(A_SUBSCRIPT)) {
		make_font_subscripted();
	    }
	    else if (command.equals(A_LARGE)) {
		make_font_large();
	    }
	    else if (command.equals(A_SMALL)) {
		make_font_small();
	    }
	    else if (command.equals(A_CLEAR_STYLE)) {
		clear_font_styles();
	    }
	    else if (command.equals(A_LEFT)) {
		align_to_left();
	    }
	    else if (command.equals(A_CENTER)) {
		align_to_center();
	    }
	    else if (command.equals(A_RIGHT)) {
		align_to_right();
	    }
	    else if (command.equals(A_IMAGE)) {
		insert_image();
	    }
	    else if (command.equals(A_HR)) {
		insert_hr();
	    }
	    //</RichTextEditor>
	    //<DocumentEditor>
	    else if (command.equals(A_INC_INDENT)) {
		increase_indent();
	    }
	    else if (command.equals(A_DEC_INDENT)) {
		decrease_indent();
	    }
	    else if (command.equals(A_SAVE_AS_TEXT)) {
		saveDocumentFileAs(false);
	    }
	    else if (command.equals(A_SAVE_AS_OBJECT)) {
		saveDocumentFileAs(true);
	    }
	    else if (command.equals(A_LIST)) {
		make_list();
	    }
	    else if (command.equals(A_UNLIST)) {
		clear_list();
	    }
	    //</DocumentEditor>
	    //<HTMLEditor>
	    else if (command.equals(A_INC_INDENT)) {
		increase_indent();
	    }
	    else if (command.equals(A_DEC_INDENT)) {
		decrease_indent();
	    }
	    else if (command.equals(A_LINK)) {
		insert_link();
	    }
	    else if (command.equals(A_ULIST)) {
		make_list();
	    }
	    else if (command.equals(A_OLIST)) {
		make_ordered_list();
	    }
	    else if (command.equals(A_ANCHOR)) {
		insert_anchor();
	    }
	    else if (command.equals(A_FORWARD)) {
		forward_page();
	    }
	    else if (command.equals(A_BACKWARD)) {
		backward_page();
	    }
	    else if (command.equals(A_STOP)) {
		stop_loading();
	    }
	    else if (command.equals(A_RELOAD)) {
		reload_page();
	    }
	    else if (command.equals(A_DOC_TITLE)) {
		edit_document_property();
	    }
	    //</HTMLEditor>
	}
    }

    /**
     * Invoked when an item's state has been changed.
     * @see java.awt.event.ItemListener
     */
    public void itemStateChanged(ItemEvent event)
    {
	/*
	Object obj = event.getItem();
	if (obj == null || !(obj instanceof String)) {
	    return;
	}

	String command = (String)checkboxMenuMap.get(obj);
	if (command == null) command = (String)obj;
	boolean selected = (event.getStateChange() == ItemEvent.SELECTED);
	if (command.equals(I_WORD_WRAP)) {
	    setWordWrap(selected);
	}
	else if (command.equals(I_SOFT_TAB)) {
	    setSoftTab(selected);
	}
	else if (command.equals(I_AUTO_INDENT)) {
	    setAutoIndentEnabled(selected);
	}
	else if (command.equals(I_SHOW_MATCH)) {
	    setShowMatchEnabled(selected);
	}
	else if (command.equals(I_INCREMENTAL_LOAD)) {
	    setIncrementalLoad(selected);
	}
	*/

	Object source = event.getSource();
	Object item = event.getItem();

	if(source == boldButton) {
	    make_font_bold(boldButton.getState());
	}
	if(source == italicButton) {
	    make_font_italic(italicButton.getState());
	}
	/*
	if(source == fontNameChoice) {
	    //	    set_font_name(event);
	}
	else if(item == paragraphStyle) {
	    if (event.getStateChange() != ItemEvent.SELECTED)
		return;
	    Object obj = event.getItem();
	    if (obj != null && (obj instanceof String)) {
		String value = (String)PStyles.get((String)obj);
		if (value != null) {
		    //		    set_paragraph_style(value);
		}
	    }
	    return;
	}
	else if (source == syntaxColorButton) {
	    //	    setSyntaxColoringEnabled(event.getStateChange() == ItemEvent.SELECTED);
	}
	*/
	else if(source == charSet) {
	    if(focused != null) {
		int caretPosition = focused.getCaretPosition();
		focused.insert(new String(new char[] {charSet.getChar()}),
			       caretPosition);
	    }
	}
    }


    //FocusListener API
    /**
     * Invoked when a component gained focus
     * @see java.awt.event.FocusListener
     */
    public void focusGained(FocusEvent event)
    {
	Object source  = event.getSource();
	if(!(source instanceof TextComponent))
	    return;

	if(!event.isTemporary()) {
	    setTextComponent((TextComponent)source);
	}
	fireFocusGained(event);
    }

    /**
     * Invoked when a component lost focus
     * @see java.awt.event.FocusListener
     */
    public void focusLost(FocusEvent event)
    {
	if(!event.isTemporary()) {
	    lastFocused = focused;
	    setTextComponent(null);
	}

	fireFocusLost(event);
    }

    // FocusController API
    /**
     * Returns <CODE>Component</CODE> having the focus,
     * or null if no <CODE>Component</CODE> under this controller
     * has the focus.
     *
     * @return Component having the focus, or null if none
     */
    public Component getFocusedComponent()
    {
	return focused;
    }

    /**
     * Listens <CODE>FocusEvent</CODE> of <CODE>component</CODE>,
     * if it is not yet listened.
     *
     * @param component <CODE>Component</CODE> to be listened
     */
    public synchronized void listen(Component component)
    {
	if(component == null ||
	   focusListenants.contains(component))
	    return;

	component.addFocusListener(this);
	focusListenants.addElement(component);
    }

    /**
     * Unlistens <CODE>FocusEvent</CODE> of <CODE>component</CODE>,
     * if it is already listened by this <CODE>Editor</CODE>
     *
     * @param component <CODE>Component</CODE> to be unlistened
     */
    public synchronized void unlisten(Component component)
    {
	if(component == null ||
	   !(focusListenants.contains(component)))
	    return;

	toolBarStates.remove(component);
	component.removeFocusListener(this);
	focusListenants.remove(component);
    }

    /**
     * Adds <COCE>listner</CODE> to the list of
     * <CODE>FocusListener</CODE> listening this
     *
     * @param listner <CODE>FocusListener</CODE> to be added 
     */
    public synchronized void addFocusListener(FocusListener listener)
    {
	if(focusListeners == null)
	    focusListeners = new Vector();
	focusListeners.addElement(listener);
    }

    /**
     * Removes <COCE>listner</CODE> from the list of
     * <CODE>FocusListener</CODE> listening this
     *
     * @param listner <CODE>FocusListener</CODE> to be removed
     */
    public synchronized void removeFocusListener(FocusListener listener)
    {
	focusListeners.remove(listener);

	if(focusListeners.isEmpty())
	    focusListeners = null;
    }

    /**
     * Invokes <CODE>focusGained</CODE> of <CODE>ForcusListener</CODE>s
     * to dispatch <CODE>event</CODE> to them
     *
     * @param event <CODE>FocusEvent</CODE> to be dispatched to registered
     * <CODE>ForcusListener</CODE>s
     *
     * @see java.awt.event.FocusListener
     */
    public void fireFocusGained(FocusEvent event)
    {
	if(focusListeners == null)
	    return;

	Object[] listeners = null;
	synchronized(focusListeners) {
	    listeners = focusListeners.toArray();
	}
	for(int i = 0; i < listeners.length; i++) {
	    ((FocusListener)listeners[i]).focusGained(event);
	}
    }

    /**
     * Invokes <CODE>focusLost</CODE> of <CODE>ForcusListener</CODE>s
     * to dispatch <CODE>event</CODE> to them
     *
     * @param event <CODE>FocusEvent</CODE> to be dispatched to registered
     * <CODE>ForcusListener</CODE>s
     *
     * @see java.awt.event.FocusListener
     */
    public void fireFocusLost(FocusEvent event)
    {
	if(focusListeners == null)
	    return;
	
	Object[] listeners = null;
	synchronized(focusListeners) {
	    listeners = focusListeners.toArray();
	}
	for(int i = 0; i < listeners.length; i++) {
	    ((FocusListener)listeners[i]).focusLost(event);
	}
    }


    // Editor API
    /**
     * Returns <CODE>TextComponent</CODE> under edition
     *
     * @return TextComponent under edition
     */
    public TextComponent getTextComponent()
    {
	return focused;
    }

    /**
     * Returns <CODE>TextComponent</CODE> had focus previously
     *
     * @return TextComponent had focus previously
     */
    public TextComponent getLastFocused()
    {
	return lastFocused;
    }

    /**
     * Sets <CODE>textComponent</CODE> to be edited by this
     *
     * @param textComponent <CODE>TextComponent</CODE> to be edited
     */
    public synchronized void setTextComponent(TextComponent textComponent)
    {
	setTextComponent(textComponent, null);
    }

    /**
     * Sets <CODE>textComponent</CODE> to be edited by this
     *
     * @param textComponent <CODE>TextComponent</CODE> to be edited
     */
    public synchronized void setTextComponent(TextComponent textComponent,
					      Hashtable status)
    {
	if(focused == textComponent)
	    return;
	if(focused != null) {
	    Hashtable toolBarStatus = (Hashtable)toolBarStates.get(focused);
	    if(toolBarStatus == null) {
		toolBarStatus = new Hashtable();
		toolBarStates.put(focused, toolBarStatus);
	    }
	    //	    toolBarStatus.clear();

	    Enumeration e = toolBarComponents.elements();
	    while(e.hasMoreElements()){
		Component subcomponent = (Component)e.nextElement();
		if(subcomponent.isEnabled()) {
		    if(subcomponent instanceof ToggleButton) {
			toolBarStatus.put(subcomponent, 
					  new Boolean(((ToggleButton)subcomponent).getState()));
		    }
		    else if (subcomponent instanceof List) {
			toolBarStatus.put(subcomponent, 
					  ((List)subcomponent).getSelectedItem());
		    }
		    else if (subcomponent instanceof Button) {
			toolBarStatus.put(subcomponent, Boolean.TRUE);
		    }
		    /*
		    else if (component instanceof ColorButton) {
		    }
		    */
		}
	    }
	    focused.removeTextListener(this);
	    focused.removeTextPositionListener(this);
	}
	lastFocused = focused;
	focused = textComponent;
	if(focused != null) {
	    focused.addTextListener(this);
	    focused.addTextPositionListener(this);
	    Hashtable toolBarStatus = (Hashtable)toolBarStates.get(focused);
	    if(toolBarStatus == null) {
		if(status != null) {
		    toolBarStatus = (Hashtable)(status.clone());
		    toolBarStates.put(focused, toolBarStatus);
		}
	    }
	    Enumeration e = toolBarComponents.elements();
	    while(e.hasMoreElements()){
		Component subcomponent = (Component)e.nextElement();
		if(toolBarStatus == null) {
		    if(subcomponent instanceof ToggleButton)
			((ToggleButton)subcomponent).setState(false);
		    subcomponent.setEnabled(true);
		}
		else {
		    Object value = toolBarStatus.get(subcomponent);
		    if(value == null) {
			subcomponent.setEnabled(false);
			continue;
		    }
		    subcomponent.setEnabled(true);
		    if(subcomponent instanceof ToggleButton) {
			((ToggleButton)subcomponent).setState(((Boolean)value).booleanValue());
		    }
		    else if (subcomponent instanceof List) {
			((List)subcomponent).select((String)value);
		    }
		}
	    }
	    if(isHTML()) {
		if(htmlURLFields != null) {
		    urlField = (TextField)htmlURLFields.get(focused);
		    if(urlField == null) {
			urlField = new TextField(40);
			urlField.addActionListener(new URLAction());
			htmlURLFields.put(focused, urlField);
		    }
		}

		if(htmlHistories != null) {
		    history = (Stack)htmlHistories.get(focused);
		    if(history == null) {
			history = new Stack();
			htmlHistories.put(focused, history);
			historyIndex = -1;
		    }
		    else
			historyIndex = ((Integer)htmlHistoryIndices.get(focused)).intValue();
		}
	    }
	    else {
		urlField = null;
		history  = null;
		historyIndex = -1;
	    }
	}
    }

    public synchronized void saveToolBarStatus()
    {
	Hashtable toolBarStatus = (Hashtable)toolBarStates.get(focused);
	if(toolBarStatus == null) {
	    toolBarStatus = new Hashtable();
	    toolBarStates.put(focused, toolBarStatus);
	}
	
	toolBarStatus.clear();

	Enumeration e = toolBarComponents.elements();
	while(e.hasMoreElements()){
	    Component subcomponent = (Component)e.nextElement();
	    if(subcomponent.isEnabled()) {
		if(subcomponent instanceof ToggleButton) {
		    toolBarStatus.put(subcomponent, 
				      new Boolean(((ToggleButton)subcomponent).getState()));
		}
		else if (subcomponent instanceof List) {
		    toolBarStatus.put(subcomponent, 
				      ((List)subcomponent).getSelectedItem());
		}
		else if (subcomponent instanceof Button) {
		    toolBarStatus.put(subcomponent, Boolean.TRUE);
		}
		/*
		  else if (component instanceof ColorButton) {
		  }
		*/
	    }
	}
    }

    public void loadToolBarStatus() {
	loadToolBarStatus((Hashtable)toolBarStates.get(focused));

    }

    public synchronized void loadToolBarStatus(Hashtable toolBarStatus) {
	if(toolBarStatus == null)
	    return;

	if(toolBarStates.get(focused) != toolBarStatus)
	    toolBarStates.put(focused, toolBarStatus);

	Enumeration e = toolBarComponents.elements();
	while(e.hasMoreElements()){
	    Component subcomponent = (Component)e.nextElement();
	    if(toolBarStatus == null) {
		if(subcomponent instanceof ToggleButton)
		    ((ToggleButton)subcomponent).setState(false);
		subcomponent.setEnabled(true);
	    }
	    else {
		Object value = toolBarStates.get(subcomponent);
		if(value == null) {
		    subcomponent.setEnabled(false);
		    continue;
		}
		subcomponent.setEnabled(true);
		if(subcomponent instanceof ToggleButton) {
		    ((ToggleButton)subcomponent).setState(((Boolean)value).booleanValue());
		}
		else if (subcomponent instanceof List) {
		    ((List)subcomponent).select((String)value);
		}
	    }
	}
    }

    /**
     * Set focus to the last focused <CODE>TextComponent</CODE>
     */
    public void setLastFocus()
    {
	if(lastFocused != null)
	    lastFocused.requestFocus();
    }


    /**
     * Returns a <CODE>ToolBar</CODE> linked to this editor
     * or null if no <CODE>ToolBar</CODE> is linked to.
     *
     * @return ToolBar linked to this editor
     */
    public ToolBar getToolBar()
    {
	return toolBar;
    }

    /**
     * Set  <CODE>toolBar</CODE> to be used by this <CODE>Editor</CODE>
     *
     * @param toolBar <CODE>ToolBar</CODE> to be linked to this <CODE>Editor</CODE>
     */
    public void setToolBar(ToolBar toolBar)
    {
	if(this.toolBar == toolBar)
	    return;

	this.toolBar = toolBar;
    }

    /**
     * Returns a <CODE>Menu</CODE> linked to this editor
     * or null if no <CODE>Menu</CODE> is linked to.
     *
     * @return Menu linked to this editor
     */
    public MenuBar getMenuBar()
    {
	return menubar;
    }

    /**
     * Set  <CODE>menubar</CODE> to be used by this <CODE>Editor</CODE>
     *
     * @param menubar <CODE>MenuBar</CODE> to be linked to this <CODE>Editor</CODE>
     */
    public void setMenuBar(MenuBar menubar)
    {
	if(this.menubar == menubar)
	    return;

	this.menubar = menubar;
    }

    /**
     * Disables sub components.
     * @see #enableSubComps()
     */
    public void disableSubComps()
    {
	switch(getEditorType()) {
	case TEXT_EDITOR:
	case CODE_EDITOR:
	case DOCUMENT_EDITOR:
	    disableTextSubComps();
	    break;
	case HTML_EDITOR:
	    disableHTMLSubComps();
	    break;
	}
    }

    /**
     * Enables sub components.
     * @see #disableSubComps()
     */
    public void enableSubComps()
    {
	switch(getEditorType()) {
	case TEXT_EDITOR:
	case CODE_EDITOR:
	case DOCUMENT_EDITOR:
	    enableTextSubComps();
	    break;
	case HTML_EDITOR:
	    enableHTMLSubComps();
	    break;
	}
    }

    /**
     * Creates a <CODE>ToolBar</CODE> to be used by this <CODE>Editor</CODE>
     * depending on specified <CODE>editorType</CODE>.
     * The <CODE>ToolBar</CODE> is a graphical representaion of internal state of
     * the <CODE>TextComponent</CODE> under edition.  Subclasses of this class
     * may override this method.
     *
     * @param editorType          int specifying type of editor
     *
     * @return ToolBar newly created.
     */
    protected ToolBar createToolBar(int editorType)
    {
	return createToolBar(editorType, true);
    }

    /**
     * Creates a <CODE>ToolBar</CODE> to be used by this <CODE>Editor</CODE>
     * depending on specified <CODE>editorType</CODE>.
     * The <CODE>ToolBar</CODE> is a graphical representaion of internal state of
     * the <CODE>TextComponent</CODE> under edition.  Subclasses of this class
     * may override this method.
     *
     * @param editorType          int specifying type of editor
     *
     * @param showToolBar         if true, then shows the tool bar initially;
     *
     * @return ToolBar newly created.
     */
    protected ToolBar createToolBar(int editorType, boolean showToolBar)
    {
	return createToolBar(editorType, showToolBar, null, null, null);
    }


    /**
     * Creates a <CODE>ToolBar</CODE> to be used by this <CODE>Editor</CODE>
     * depending on specified <CODE>editorType</CODE>.
     * The <CODE>ToolBar</CODE> is a graphical representaion of internal state of
     * the <CODE>TextComponent</CODE> under edition.  Subclasses of this class
     * may override this method.
     *
     * @param editorType          int specifying type of editor
     *
     * @param showToolBar         if true, then shows the tool bar initially;
     *                            otherwise hides.
     * @param openActionListener  the action listener that receives action
     *                            events from the open button in tool bar.
     * @param saveActionListener  the action listener that receives action
     *                            events from the save button in tool bar.
     * @param printActionListener the action listener that receives action
     *                            events from the print button in tool bar.
     *
     * @return ToolBar newly created.
     */
    protected ToolBar createToolBar(int editorType,
				    boolean showToolBar,
				    ActionListener openActionListener,
				    ActionListener saveActionListener,
				    ActionListener printActionListener)
    {
	switch(editorType) {
	case CODE_EDITOR:
	    return createCodeToolBar(showToolBar,
				     openActionListener,
				     saveActionListener,
				     printActionListener);
	case DOCUMENT_EDITOR:
	    return createDocumentEditorToolBar(showToolBar,
				     openActionListener,
				     saveActionListener,
				     printActionListener);
	case HTML_EDITOR:
	    return createHTMLToolBar(showToolBar,
				     openActionListener,
				     saveActionListener,
				     printActionListener);
	case TEXT_EDITOR:
	default:
	    return createTextEditorToolBar(showToolBar,
				     openActionListener,
				     saveActionListener,
				     printActionListener);
	}
    }

    //<TextEditor>
    //File buttons
    Button openButton;
    Button saveButton;

    //Print Button
    Button printButton;

    //Find buttons
    protected Button findButton;
    protected Button gotoButton;

    //Copy buttons
    protected Button copyButton;
    protected Button cutButton;
    protected Button pasteButton;
    protected Button undoButton;
    //</TextEditor>

    //<CodeEditor>
    //</CodeEditor>

    //<RichTextEditor>
    //</RichTextEditor>

    //<DocumentEditor>
    //List button
    Button listButton;
    ToggleButton listToggleButton;
    Button unlistButton;
    ToggleButton orderedListToggleButton;
    Button decIndentButton;
    Button incIndentButton;    
    //</DocumentEditor>

    //<HTMLEditor>
    protected Button orderdListButton;
    protected Button docTitleButton;
    //</HTMLEditor>

    //Character chooser buttons
    protected CharacterSelectButton charSet;

    //Font choice and buttons
    protected Choice fontNameChoice;
    protected ToggleButton boldButton;
    protected ToggleButton italicButton;
    protected ToggleButton underlineButton;
    protected ColorButton colorButton;

    //Font size buttons
    protected Button largeButton;
    protected Button smallButton;

    //Script buttons
    protected Button superscriptButton; 
    protected Button normalscriptButton; 
    protected Button subscriptButton;

    //Alignemt buttons
    Button leftAlignButton;
    Button centerAlignButton;
    Button rightAlignButton;

    //Insertion buttons
    Button hrButton;
    Button imageButton;

    //Web navigation Components
    Button backwardButton;
    Button forwardButton;
    Button reloadButton;
    Button stopButton;

    //HTML paragraph style Component
    Choice paragraphStyle;

    //HTML authoring Component
    Button anchorButton;
    ToggleButton linkButton;

    //Document button
    Button documentTitleButton;

    //Syntac color button
    protected ToggleButton syntaxColorButton;

    //<TextEditor>
    /**
     * Creates a tool bar.
     * @param showToolBar         if true, then shows the tool bar initially;
     *                            otherwise hides.
     * @param openActionListener  the action listener that receives action
     *                            events from the open button in tool bar.
     * @param saveActionListener  the action listener that receives action
     *                            events from the save button in tool bar.
     * @param printActionListener the action listener that receives action
     *                            events from the print button in tool bar.
     */
    protected ToolBar createTextEditorToolBar(boolean showToolBar,
				    ActionListener openActionListener,
				    ActionListener saveActionListener,
				    ActionListener printActionListener)
    {
	Component[] file  = createFileComponents(openActionListener,
						 saveActionListener);
	Component[] print = createPrintComponents(printActionListener);
	Component[] find  = createFindComponents(true);
	Component[] edit  = createEditComponents();

	Component[][] bar = new Component[][]{ file, print, find, edit };
	return new ToolBar(new Component[][][]{ bar }, showToolBar);
    }

    protected Component[] createFileComponents(
					ActionListener openActionListener,
					ActionListener saveActionListener)
    {
	openButton = createIconButton(A_OPEN);
	openButton.addActionListener(openActionListener != null ?
					openActionListener : this);
	saveButton = createIconButton(A_SAVE);
	saveButton.addActionListener(saveActionListener != null ?
					saveActionListener : this);
	return new Component[]{ openButton, saveButton };
    }

    protected Component[] createPrintComponents(
					ActionListener printActionListener)
    {
	printButton = createIconButton(A_PRINT);
	printButton.addActionListener(printActionListener == null ?
					this : printActionListener);
	return new Component[]{ printButton  };
    }

    /**
     * Creates and returns <CODE>Component</CODE>s to be used
     * in find operation.  The subclass may override this method.
     *
     * @return array of <CODE>Component</CODE>s to be used in find operation
     */
    protected Component[] createFindComponents(boolean withGoto)
    {
	findButton = createIconButton(A_FIND);
	findButton .addActionListener(this);

	if (!withGoto)
	    return new Component[]{ findButton  };

	gotoButton = createIconButton(A_GOTO);
	gotoButton.addActionListener(this);
	return new Component[]{ findButton , gotoButton };
    }

    /**
     * Creates and returns <CODE>Component</CODE>s to be used
     * in edit operation.  The subclass may override this method.
     *
     * @return array of <CODE>Component</CODE>s to be used in edit operation
     */
    protected Component[] createEditComponents()
    {
	copyButton = createIconButton(A_COPY);
	copyButton.addActionListener(this);

	cutButton = createIconButton(A_CUT);
	cutButton.addActionListener(this);

	pasteButton = createIconButton(A_PASTE);
	pasteButton.addActionListener(this);

	undoButton = createIconButton(A_UNDO);
	undoButton.addActionListener(this);

	copyButton.setEnabled(false);
	cutButton.setEnabled(false);
	addCaretDisableComp(copyButton);
	addCaretDisableComp(cutButton);

	return new Component[]{ copyButton, cutButton, pasteButton, undoButton };
    }

    protected Button createButton(String command) {
	return createButton(command, command);
    }

    protected Button createButton(String label, String command) {
	Button b = new Button(label);
	return (Button)setupButton(b, command);
    }

    protected ToggleButton createToggleButton(String command) {
	return createToggleButton(command, command);
    }

    protected ToggleButton createToggleButton(String label, String command) {
	ToggleButton b = new ToggleButton(label);
	return (ToggleButton)setupButton(b, command);
    }

    protected Button createIconButton(String command) {
	VImage vimage = getIcon(command);
	Button b = new Button(new VActiveButton(vimage));
	return (Button)setupButton(b, command);
    }

    protected ToggleButton createIconToggleButton(String command) {
	VImage vimage = getIcon(command);
	ToggleButton b = new ToggleButton(new VActiveButton(vimage));
	return (ToggleButton)setupButton(b, command);
    }

    protected AbstractButton setupButton(AbstractButton b, String command)
    {
	return setupButton(b, command, getToolTip(command));
    }

    protected AbstractButton setupButton(AbstractButton b, String command, String tipString)
    {
	b.setActionCommand(command);
	b.setToolTipText(tipString);
	addSubComp(b);
	return b;
    }


    protected Menu createTextEditorInsertMenu() {
	return null;
    }

    protected Menu createTextEditorFormatMenu() {
	return null;
    }

    protected Menu createTextEditorEditMenu() {
	Menu menu = new Menu(getToolLabel(L_EDIT));
	MenuItem copy = createMenuItem(A_COPY, this);
	MenuItem cut  = createMenuItem(A_CUT,  this);
	menu.add(copy);
	menu.add(cut);
	menu.add(createMenuItem(A_PASTE, this));
	menu.addSeparator();
	menu.add(createMenuItem(A_FIND,  this));
	menu.addSeparator();
	menu.add(createMenuItem(A_UNDO,  this));

	copy.setEnabled(false);
	cut.setEnabled(false);
	addCaretDisableItem(copy);
	addCaretDisableItem(cut);

	return menu;
    }

    protected Menu createTextEditorViewMenu() {
	Menu menu = new Menu(getToolLabel(L_VIEW));
	menu.add(createFontMenu());
	menu.addSeparator();
	menu.add(createCheckboxMenuItem(I_WORD_WRAP, isWordWrap(), this));
	menu.add(createCheckboxMenuItem(I_SOFT_TAB, isSoftTab(), this));
	menu.add(createCheckboxMenuItem(I_AUTO_INDENT, isAutoIndentEnabled(),
					this));
	menu.add(createCheckboxMenuItem(I_SHOW_MATCH, isShowMatchEnabled(),
					this));
	menu.addSeparator();
	menu.add(createCheckboxMenuItem(I_INCREMENTAL_LOAD, isIncrementalLoad(),
					this));
	menu.addSeparator();
	menu.add(createReadCharSetMenu());
	menu.add(createWriteCharSetMenu());
	return menu;
    }

    class FontSelection implements ActionListener, java.io.Serializable {
	boolean isName;

	FontSelection(boolean isName) {
	    this.isName = isName;
	}

	public void actionPerformed(ActionEvent e) {
	    String command = e.getActionCommand();
	    Font font = focused.getFont();
	    if (isName) {
		if (command.equals(font.getName()))
		    return;
		font = new Font(command, font.getStyle(), font.getSize());
	    }
	    else { // size
		int size;
		try { size = Integer.parseInt(command); }
		catch (NumberFormatException ex) { return; }
		if (size == font.getSize())
		    return;
		font = new Font(font.getName(), font.getStyle(), size);
	    }
	    focused.setFont(font);
	}
    }

    protected Menu createTextEditorFontMenu() {
	Menu menu = new Menu(getToolLabel(L_FONT));
	Font font = focused.getFont();

	SelectionMenu name = new SelectionMenu(getToolLabel(L_FONT_NAME));
	name.addActionListener(new FontSelection(true));
	String names[] = { "Monospaced", "SansSerif", "Serif", "Dialog" };
	for (int i = 0; i < names.length; i++) {
	    String label = names[i];
	    boolean state = label.equalsIgnoreCase(font.getName());
	    name.add(label, label, state);
	}
	menu.add(name);

	SelectionMenu size = new SelectionMenu(getToolLabel(L_FONT_SIZE));
	size.addActionListener(new FontSelection(false));
	int sizes[] = { 8, 10, 12, 14, 16, 18, 20, 22, 24 };
	for (int i = 0; i < sizes.length; i++) {
	    String label = String.valueOf(sizes[i]);
	    boolean state = (sizes[i] == font.getSize());
	    size.add(label, label, state);
	}
	menu.add(size);

	return menu;
    }


    class CharSetSelection implements ActionListener, java.io.Serializable {
	boolean read;

	CharSetSelection(boolean read) {
	    this.read = read;
	}

	public void actionPerformed(ActionEvent e) {
	    String encoding = e.getActionCommand();
	    if (read) {
		setReadEncoding(encoding);
	    }
	    else {
		setWriteEncoding(encoding);
	    }
	}
    }

    protected CharSetMenu createReadCharSetMenu() {
	CharSetMenu read = new CharSetMenu(getToolLabel(L_READ_ENCODING),
					   getReadEncoding());
	read.addActionListener(new CharSetSelection(true));
	return read;
    }

    protected CharSetMenu createWriteCharSetMenu() {
	CharSetMenu write = new CharSetMenu(getToolLabel(L_WRITE_ENCODING),
					    getWriteEncoding());
	write.addActionListener(new CharSetSelection(false));
	return write;
    }

    protected MenuItem createMenuItem(String command, ActionListener l) {
	MenuItem mi = new MenuItem(getToolLabel(command));
	mi.setActionCommand(command);
	mi.addActionListener(l);
	return mi;
    }

    protected CheckboxMenuItem createCheckboxMenuItem(String command,
						      boolean state,
						      ItemListener l)
    {
	String label = getToolLabel(command);
	CheckboxMenuItem cmi = new CheckboxMenuItem(label, state);
	//cmi.setActionCommand(command);
	checkboxMenuMap.put(label, command);
	cmi.addItemListener(l);
	return cmi;
    }

    protected void addSubComp(Component component) {
	if (toolBarComponents == null)
	    toolBarComponents = new Vector();

	if(toolBarComponents.contains(component))
	    return;

	toolBarComponents.addElement(component);
    }

    protected void addCaretDisableComp(Component c) {
	if (caretDisableComps == null)
	    caretDisableComps = new Vector();
	caretDisableComps.addElement(c);
    }

    protected void addCaretDisableItem(MenuItem item) {
	if (caretDisableComps == null)
	    caretDisableComps = new Vector();
	caretDisableComps.addElement(item);
    }
    //</TextEditor>

    //<RichTextEditor>
    protected Component[] createFontComponents1() {
	boldButton = createIconToggleButton(I_BOLD);
	boldButton.addItemListener(this);
	italicButton = createIconToggleButton(I_ITALIC);
	italicButton.addItemListener(this);

	boldButton.setEnabled(false);
	italicButton.setEnabled(false);

	addCaretDisableComp(boldButton);
	addCaretDisableComp(italicButton);

	return new Component[]{ boldButton, italicButton };
    }

    protected Component[] createScriptComponents() {
	superscriptButton = createIconButton(I_SUPERSCRIPT);
	superscriptButton.addActionListener(this);
	normalscriptButton = createIconButton(I_NORMALSCRIPT);
	normalscriptButton.addActionListener(this);
	subscriptButton = createIconButton(I_SUBSCRIPT);
	subscriptButton.addActionListener(this);

	superscriptButton.setEnabled(false);
	normalscriptButton.setEnabled(false);
	subscriptButton.setEnabled(false);

	addCaretDisableComp(superscriptButton);
	addCaretDisableComp(normalscriptButton);
	addCaretDisableComp(subscriptButton);

	return new Component[]{ superscriptButton, normalscriptButton, subscriptButton };
    }

    protected Component[] createFontDecorationComponents() {
	underlineButton = createIconToggleButton(I_UNDERLINE);
	underlineButton.addItemListener(this);

	underlineButton.setEnabled(false);

	addCaretDisableComp(underlineButton);

	return new Component[]{ underlineButton };
    }

    protected Component[] createFontAttributeComponents() {
	ColorButton color = new ColorButton();
	color.addItemListener(this);
	color.setToolTipText(getToolTip("fontColor"));

	color.setEnabled(false);
	addCaretDisableComp(color);

	return new Component[]{ color };
    }

    protected Component[] createFontSizeComponents() {
	Button large = createIconButton(A_LARGE);
	large.addActionListener(this);
	Button small = createIconButton(A_SMALL);
	small.addActionListener(this);

	large.setEnabled(false);
	small.setEnabled(false);
	addCaretDisableComp(large);
	addCaretDisableComp(small);

	return new Component[]{ large, small };
    }


    protected Component[] createAlignmentComponents() {
	Button left = createIconButton(A_LEFT);
	left.addActionListener(this);
	Button center = createIconButton(A_CENTER);
	center.addActionListener(this);
	Button right = createIconButton(A_RIGHT);
	right.addActionListener(this);
	return new Component[]{ left, center, right };
    }
    //</RichTextEditor>

    //<DocumentEditor>
    /**
     * Creates a tool bar.
     * @param showToolBar         if true, then shows the tool bar initially;
     *                            otherwise hides.
     * @param openActionListener  the action listener that receives action
     *                            events from the open button in tool bar.
     * @param saveActionListener  the action listener that receives action
     *                            events from the save button in tool bar.
     * @param printActionListener the action listener that receives action
     *                            events from the print button in tool bar.
     */
    protected ToolBar createDocumentEditorToolBar(boolean showToolBar,
				    ActionListener openActionListener,
				    ActionListener saveActionListener,
				    ActionListener printActionListener)
    {
	Component name = createFontNameComponent();
	Component[] font = createFontComponents1();
	Component[] font1 = new Component[font.length + 1];
	font1[0] = name;
	System.arraycopy(font, 0, font1, 1, font.length);

	Component[] fontDecoration = createFontDecorationComponents();
	Component[] script = createScriptComponents();
	Component[] fontSize = createFontSizeComponents();
	Component[] fontAttribute = createFontAttributeComponents();

	Component[] align = createAlignmentComponents();
	Component[] list  = createListComponents();
	Component[][] bar1 = new Component[][]{ font1, 
						fontDecoration, script,
						fontSize, fontAttribute, 
						align, list };

	Component[] file   = createFileComponents(openActionListener,
						  saveActionListener);
	Component[] print  = createPrintComponents(printActionListener);
	Component[] find   = createFindComponents(false);
	Component[] edit   = createEditComponents();
	Component[] insert = createInsertComponents();

	Component[][] bar2 =
			new Component[][]{ file, print, find, edit, insert };
	return new ToolBar(new Component[][][]{ bar1, bar2 }, showToolBar);
    }

    class DocumentFontNameItemListener implements ItemListener, java.io.Serializable {
	public void itemStateChanged(ItemEvent e) {
	    set_font_name(e);
	}
    }

    protected Component createFontNameComponent() {
	Choice name = new Choice();
	name.addItem("SansSerif");
	name.addItem("Serif");
	name.addItem("Monospaced");
	name.addItem("Dialog");
	name.setToolTipText(getToolTip("fontName"));
	name.addItemListener(new DocumentFontNameItemListener());

	name.setEnabled(false);
	addSubComp(name);
	addCaretDisableComp(name);

	return name;
    }

    protected Component[] createListComponents() {

	listButton = createIconButton(A_LIST);
	listButton.addActionListener(this);
	unlistButton = createIconButton(A_UNLIST);
	unlistButton.addActionListener(this);

	decIndentButton  = createIconButton(A_DEC_INDENT);
	decIndentButton.addActionListener(this);
	incIndentButton = createIconButton(A_INC_INDENT);
	incIndentButton .addActionListener(this);

	return new Component[]{ listButton, unlistButton, decIndentButton, incIndentButton };
    }

    protected Component[] createInsertComponents() {
	Button hr = createIconButton(A_HR);
	hr.addActionListener(this);
	Button image = createIconButton(A_IMAGE);
	image.addActionListener(this);

	return new Component[]{ hr, image };
    }

    protected Menu createViewMenu() {
	Menu menu = new Menu(getToolLabel(L_VIEW));
	menu.add(createCheckboxMenuItem(I_INCREMENTAL_LOAD, isIncrementalLoad(),
					this));
	menu.addSeparator();
	menu.add(createReadCharSetMenu());
	menu.add(createWriteCharSetMenu());
	return menu;
    }

    protected Menu createInsertMenu() {
	Menu menu = new Menu(getToolLabel(L_INSERT));
	menu.add(createMenuItem(A_HR,    this));
	menu.add(createMenuItem(A_IMAGE, this));
	return menu;
    }

    protected Menu createFormatMenu() {
	Menu menu = new Menu(getToolLabel(L_FORMAT));

	Menu font = new Menu(getToolLabel(L_FONT_STYLE));
	font.add(createMenuItem(A_BOLD,       this));
	font.add(createMenuItem(A_ITALIC,     this));
	font.add(createMenuItem(A_UNDERLINE,  this));
	font.add(createMenuItem(A_SUPERSCRIPT,  this));
	font.add(createMenuItem(A_NORMALSCRIPT,  this));
	font.add(createMenuItem(A_SUBSCRIPT,  this));
	font.setEnabled(false);
	addCaretDisableItem(font);
	menu.add(font);

	MenuItem clear = createMenuItem(A_CLEAR_STYLE, this);
	clear.setEnabled(false);
	addCaretDisableItem(clear);
	menu.add(clear);

	menu.addSeparator();

	MenuItem large = createMenuItem(A_LARGE, this);
	MenuItem small = createMenuItem(A_SMALL, this);
	large.setEnabled(false);
	small.setEnabled(false);
	addCaretDisableItem(large);
	addCaretDisableItem(small);
	menu.add(large);
	menu.add(small);

	menu.addSeparator();

	Menu align = new Menu(getToolLabel(L_ALIGN));
	align.add(createMenuItem(A_LEFT,   this));
	align.add(createMenuItem(A_CENTER, this));
	align.add(createMenuItem(A_RIGHT,  this));
	menu.add(align);

	Menu list = new Menu(getToolLabel(L_LIST));
	list.add(createMenuItem(A_LIST,   this));
	list.add(createMenuItem(A_UNLIST, this));
	menu.add(list);

	menu.addSeparator();

	menu.add(createMenuItem(A_INC_INDENT, this));
	menu.add(createMenuItem(A_DEC_INDENT, this));

	return menu;
    }


    //</DocumentEditor>

    //<HTMLEditor>
    /**
     * Creates a tool bar.
     * @param showToolBar         if true, then shows the tool bar initially;
     *                            otherwise hides.
     * @param openActionListener  the action listener that receives action
     *                            events from the open button in tool bar.
     * @param saveActionListener  the action listener that receives action
     *                            events from the save button in tool bar.
     * @param printActionListener the action listener that receives action
     *                            events from the print button in tool bar.
     */
    protected ToolBar createHTMLToolBar(boolean showToolBar,
				    ActionListener openActionListener,
				    ActionListener saveActionListener,
				    ActionListener printActionListener)
    {
	Component web1[] = createWebComponents1();
	Component web2[] = createWebComponents2();
	Component[][] bar1 = new Component[][]{ web1, web2 };

	Component pstyle = createParagraphStyleComponent();
	Component name = createFontNameComponent();
	Component[] font = createFontComponents1();
	Component[] font1 = new Component[font.length + 2];
	font1[0] = pstyle;
	font1[1] = name;
	System.arraycopy(font, 0, font1, 2, font.length);

	//	Component[] font2 = createFontComponents2();
	Component[] fontDecoration = createFontDecorationComponents();
	//	Component[] script = createScriptComponents();
	Component[] fontSize = createFontSizeComponents();
	Component[] fontAttribute = createFontAttributeComponents();

	Component[] align = createAlignmentComponents();
	Component[] list  = createHTMLListComponents();
	//	Component[][] bar2 = new Component[][]{ font1, font2, align, list };
	Component[][] bar2 = new Component[][]{ font1, 
						fontDecoration,
						fontSize, fontAttribute, 
						align, list };

	Component[] file   = createFileComponents(openActionListener,
						  saveActionListener);
	Component[] print  = createPrintComponents(printActionListener);
	Component[] find   = createFindComponents(false);
	Component[] edit   = createEditComponents();
	Component[] doc    = createDocumentComponents();
	Component[] insert = createHTMLInsertComponents();

	Component[][] bar3 = new Component[][]{ file, print, find, edit, doc,
						insert };
	return new ToolBar(new Component[][][]{ bar1, bar2, bar3 },
			   showToolBar);
    }

    protected Component[] createWebComponents1() {
	backwardButton = createIconButton(A_BACKWARD);
	backwardButton.addActionListener(this);
	forwardButton = createIconButton(A_FORWARD);
	forwardButton.addActionListener(this);
	Button reload = createIconButton(A_RELOAD);
	reload.addActionListener(this);
	stopButton = createIconButton(A_STOP);
	stopButton.addActionListener(this);

	forwardButton.setEnabled(false);
	backwardButton.setEnabled(false);
	stopButton.setEnabled(false);
	toolBarComponents.removeElement(stopButton);

	return new Component[]{ backwardButton, forwardButton, reload,
				stopButton };
    }

    protected Component[] createWebComponents2() {
	Label label = new Label("URL");
	urlField = new TextField(40);
	urlField.addActionListener(new URLAction());

	addSubComp(urlField);

	return new Component[]{ label, urlField };
    }

    class PStyleItemListener implements ItemListener, java.io.Serializable {
	public void itemStateChanged(ItemEvent e) {
	    if (e.getStateChange() != ItemEvent.SELECTED)
		return;
	    Object obj = e.getItem();
	    if (obj != null && (obj instanceof String)) {
		String value = (String)PStyles.get((String)obj);
		if (value != null) {
		    set_paragraph_style(value);
		}
	    }
	}
    }

    protected Component createParagraphStyleComponent() {
	Choice pstyle = new Choice();
	String styles[] = { "normal", "h1", "h2", "h3", "h4", "h5", "h6",
			    "address", "pre", "li", "dd", "dt" };
	for (int i = 0; i < styles.length; i++) {
	    pstyle.addItem(getToolLabel(styles[i]));
	}
	pstyle.setToolTipText(getToolTip(L_PARA_STYLE));
	pstyle.addItemListener(new PStyleItemListener());

	addSubComp(pstyle);

	return pstyle;
    }

    class HTMLFontNameItemListener implements ItemListener, java.io.Serializable {
	String variable = getToolLabel(L_VARIABLE);
	String fixed    = getToolLabel(L_FIXED);

	public void itemStateChanged(ItemEvent e) {
	    if (e.getStateChange() != ItemEvent.SELECTED)
		return;
	    Object obj = e.getItem();
	    if (obj != null && (obj instanceof String)) {
		String s = (String)obj;
		if (s.equals(variable)) {
		    set_font_name("SansSerif");
		}
		else if (s.equals(fixed)) {
		    set_font_name("Monospaced");
		}
	    }
	}
    }

    protected Component createHTMLFontNameComponent() {
	Choice name = new Choice();
	name.addItem(getToolLabel(L_VARIABLE));
	name.addItem(getToolLabel(L_FIXED));
	name.setToolTipText(getToolTip(L_FONT_NAME));
	name.addItemListener(new HTMLFontNameItemListener());

	name.setEnabled(false);
	addSubComp(name);
	addCaretDisableComp(name);

	return name;
    }

    protected Component[] createHTMLListComponents() {
	listToggleButton = createIconToggleButton(I_ULIST);
	listToggleButton.addItemListener(this);
	orderedListToggleButton = createIconToggleButton(I_OLIST);
	orderedListToggleButton.addItemListener(this);
	decIndentButton = createIconButton(A_DEC_INDENT);
	decIndentButton.addActionListener(this);
	incIndentButton = createIconButton(A_INC_INDENT);
	incIndentButton.addActionListener(this);
	return new Component[]{ listToggleButton, orderedListToggleButton, decIndentButton, incIndentButton };
    }

    protected Component[] createDocumentComponents() {
	Button doc = createIconButton(A_DOC_TITLE);
	doc.addActionListener(this);

	return new Component[]{ doc };
    }

    protected Component[] createHTMLInsertComponents() {
	linkButton = createIconToggleButton(I_LINK);
	linkButton.addItemListener(this);
	Button anchor = createIconButton(A_ANCHOR);
	anchor.addActionListener(this);
	Button hr = createIconButton(A_HR);
	hr.addActionListener(this);
	Button image = createIconButton(A_IMAGE);
	image.addActionListener(this);

	linkButton.setEnabled(false);
	addCaretDisableComp(linkButton);

	return new Component[]{ linkButton, anchor, hr, image };
    }

    protected Menu createHTMLViewMenu() {
	Menu menu = new Menu(getToolLabel(L_VIEW));
	menu.add(createFontMenu());
	menu.add(createMenuItem(A_DOC_TITLE, this));
	menu.addSeparator();
	menu.add(createMenuItem(A_RELOAD, this));
	menu.addSeparator();
	menu.add(createCheckboxMenuItem(I_INCREMENTAL_LOAD, isIncrementalLoad(),
					this));
	menu.addSeparator();
	menu.add(createReadCharSetMenu());
	menu.add(createWriteCharSetMenu());
	return menu;
    }

    class HTMLFontSelection implements ActionListener, java.io.Serializable {
	public void actionPerformed(ActionEvent e) {
	    String command = e.getActionCommand();
	    HTMLStyle htmlStyle = (HTMLStyle)HTMLStyles.get(command);
	    if (htmlStyle == null)
		return;
	    if (htmlStyle == ((HTMLText)focused.getRichText()).getHTMLStyle())
		return;
	    setHTMLStyle(htmlStyle);
	}
    }

    protected Menu createHTMLFontMenu() {
	SelectionMenu menu = new SelectionMenu(getToolLabel(L_FONT));
	menu.addActionListener(new HTMLFontSelection());
	String names[] = { L_S_STYLE, L_M_STYLE, L_L_STYLE, L_VL_STYLE };
	HTMLStyle htmlStyle = ((HTMLText)focused.getRichText()).getHTMLStyle();
	for (int i = 0; i < names.length; i++) {
	    String command = names[i];
	    String label = getToolLabel(command);
	    boolean state = (htmlStyle == HTMLStyles.get(command));
	    menu.add(label, command, state);
	}
	return menu;
    }

    protected Menu createHTMLInsertMenu() {
	Menu menu = new Menu(getToolLabel(L_INSERT));
	MenuItem link = createMenuItem(A_LINK, this);
	link.setEnabled(false);
	addCaretDisableItem(link);
	menu.add(link);
	menu.add(createMenuItem(A_ANCHOR, this));
	menu.addSeparator();
	menu.add(createMenuItem(A_HR,     this));
	menu.add(createMenuItem(A_IMAGE,  this));
	return menu;
    }

    protected Menu createHTMLFormatMenu() {
	Menu menu = new Menu(getToolLabel(L_FORMAT));

	Menu font = new Menu(getToolLabel(L_FONT_STYLE));
	font.add(createMenuItem(A_BOLD,      this));
	font.add(createMenuItem(A_ITALIC,    this));
	font.add(createMenuItem(A_UNDERLINE, this));
	font.setEnabled(false);
	addCaretDisableItem(font);
	menu.add(font);

	MenuItem clear = createMenuItem(A_CLEAR_STYLE, this);
	clear.setEnabled(false);
	addCaretDisableItem(clear);
	menu.add(clear);

	menu.addSeparator();

	MenuItem large = createMenuItem(A_LARGE, this);
	MenuItem small = createMenuItem(A_LARGE, this);
	large.setEnabled(false);
	small.setEnabled(false);
	addCaretDisableItem(large);
	addCaretDisableItem(small);
	menu.add(large);
	menu.add(small);

	menu.addSeparator();

	menu.add(createPStyleMenu());

	Menu align = new Menu(getToolLabel(L_ALIGN));
	align.add(createMenuItem(A_LEFT,   this));
	align.add(createMenuItem(A_CENTER, this));
	align.add(createMenuItem(A_RIGHT,  this));
	menu.add(align);

	Menu list = new Menu(getToolLabel(L_LIST));
	list.add(createMenuItem(A_ULIST, this));
	list.add(createMenuItem(A_OLIST, this));
	menu.add(list);

	menu.addSeparator();

	menu.add(createMenuItem(A_INC_INDENT, this));
	menu.add(createMenuItem(A_DEC_INDENT, this));

	return menu;
    }

    class PStyleActionListener implements ActionListener, java.io.Serializable
    {
	public void actionPerformed(ActionEvent e) {
	    String value = (String)PStyles.get(e.getActionCommand());
	    if (value != null) {
		set_paragraph_style(value);
	    }
	}
    }

    protected Menu createPStyleMenu() {
	Menu menu = new Menu(getToolLabel(L_PARA_STYLE));
	ActionListener l = new PStyleActionListener();
	String styles[] = { "normal", "h1", "h2", "h3", "h4", "h5", "h6",
			    "address", "pre", "li", "dd", "dt" };
	for (int i = 0; i < styles.length; i++) {
	    String style = styles[i];
	    String label = getToolLabel(style);
	    MenuItem mi = new MenuItem(label);
	    mi.setActionCommand(label);
	    mi.addActionListener(l);
	    menu.add(mi);
	}
	return menu;
    }

    protected void insertHTMLTextAttachmentAsLine(TextAttachment ta) {
	TextBuffer buffer = new TextBuffer();
	buffer.setTextStyle(getInsertionStyle());
	buffer.append(Text.LINE_SEPARATOR_CHAR);
	buffer.append(ta);
	buffer.append(Text.LINE_SEPARATOR_CHAR);

	MultipleUndo mundo = new MultipleUndo();

	int start = focused.getSelectionStart();
	int end   = focused.getSelectionEnd();
	Undo undo = focused.getModel().replace(start, end, buffer.toText());
	if (undo != null) {
	    mundo.addUndo(undo);
	}

	ParagraphStyle pStyle = ((HTMLText)focused.getRichText()).getHTMLStyle().getParagraphStyle("P");
	int pos = focused.getCaretPosition();

	undo = getModel().setParagraphStyle(pos - 1, pos - 1, pStyle);
	if (undo != null) {
	    mundo.addUndo(undo);
	}

	focused.getController().setUndo(mundo);

	/*
	setCaretPosition(focused.getCaretPosition() - 1);
	setSelectionParagraphStyle(pStyle);
	setCaretPosition(focused.getCaretPosition() + 1);
	*/
    }

    protected int[] getListStyleRange() {
	int start = focused.getSelectionStart();
	HTMLText htmlText = getHTMLText();
	RunArray pstyles = htmlText.getParagraphStyleRuns();
	if (isListStyle((ParagraphStyle)pstyles.get(start))) {
	    int runStart = start - pstyles.getRunOffsetAt(start);
	    int runEnd   = start + pstyles.getRunLengthAt(start);
	    if (runEnd > htmlText.length())
		runEnd = htmlText.length();
	    return new int[]{ runStart, runEnd };
	}
	else {
	    int end = focused.getSelectionEnd();
	    return new int[]{ start, end };
	}
    }

    protected final boolean isListStyle(ParagraphStyle pStyle) {
	String styleName = pStyle.getStyleName();
	return (styleName != null && (styleName.equals("LI-UL") ||
				      styleName.equals("LI-OL") ||
				      styleName.equals("DT")    ||
				      styleName.equals("DD")));
    }

    protected String fileToURLString(File file) {
	Vector path = new Vector();
	File f = new File(file.getAbsolutePath());
	path.insertElementAt(f.getName(), 0);
	String name;
	while ((name = f.getParent()) != null) {
	    f = new File(name);
	    if (f.getName().length() == 0)
		break;
	    path.insertElementAt(f.getName(), 0);
	}
	StringBuffer buffer = new StringBuffer();
	buffer.append("file:");
	for (Enumeration e = path.elements(); e.hasMoreElements(); ) {
	    buffer.append('/').append((String)e.nextElement());
	}
	return buffer.toString();
    }


    /** Internal constant for serialization */
    static protected final String actionListenerK = "actionL".intern();

    private void writeObject(java.io.ObjectOutputStream s)
	throws java.io.IOException
    {
	s.defaultWriteObject();

	jp.kyasu.awt.ListenerSerializer.write(s,
					      actionListenerK,
					      linkActionListener);
	s.writeObject(null);
    }

    private void readObject(java.io.ObjectInputStream s)
	throws ClassNotFoundException, java.io.IOException
    {
	s.defaultReadObject();

	Object keyOrNull;
	while ((keyOrNull = s.readObject()) != null) {
	    String key = ((String)keyOrNull).intern();
	    if (key == actionListenerK)
		linkActionListener = (ActionListener)s.readObject();
	    else // skip value for unrecognized key
		s.readObject();
	}

	history      = new Stack();
	historyIndex = -1;
	InputStream loadInputStream = 
	    (InputStream)htmlLoadInputStreams.get(focused);
	htmlLoadInputStreams.remove(focused);
	loadInputStream  = null;
	Thread backgroundThread = 
	    (Thread)backgroundThreads.get(focused);
	if(backgroundThread == null)
	    return;

	backgroundThreads.remove(focused);
	backgroundThread = null;
    }

    //</HTMLEditor>

    /**
     * Creates and returns <CODE>Component</CODE>s to be used
     * in font style modification.  The subclass may override this method.
     * Users of this method includes <CODE>RichTextEditor</CODE>
     *
     * @return array of <CODE>Component</CODE>s to be used in font style modification
     */
    protected Component[] createFontComponents()
    {
	return createFontComponents(null);
    }

    /**
     * Creates and returns <CODE>Component</CODE>s to be used
     * in font style modification.  The subclass may override this method.
     * Users of this method includes <CODE>RichTextEditor</CODE>
     *
     * @return array of <CODE>Component</CODE>s to be used in font style modification
     */
    protected Component[] createFontComponents(Component nameChoice)
    {
	boldButton = createIconToggleButton(I_BOLD);
	boldButton.addItemListener(this);
	boldButton.setEnabled(false);
	addCaretDisableComp(boldButton);

	italicButton = createIconToggleButton(I_ITALIC);
	italicButton.addItemListener(this);
	italicButton.setEnabled(false);
	addCaretDisableComp(italicButton);

	if(nameChoice == null)
	    return new Component[]{ boldButton, italicButton };
	else
	    return new Component[]{ nameChoice, boldButton, italicButton };
    }

    /**
     * Creates and returns a <CODE>Component</CODE> to be used
     * in font name selection
     * The subclass may override this method.
     * Users of this method includes <CODE>DocumentEditor</CODE>
     *
     * @return array of <CODE>Component</CODE> to be used in font name selection
     */
    protected Component createFontNameComponent(int textType)
    {
	//	if((textType & HTML_TYPE) != 0)
	//return createFontNameComponent(htmlFontNameModel);
	return createFontNameComponent(documentFontNameModel);
    }

    /**
     * Creates and returns a <CODE>Component</CODE> to be used
     * in font name selection
     * The subclass may override this method.
     * Users of this method includes <CODE>DocumentEditor</CODE>
     *
     * @return array of <CODE>Component</CODE> to be used in font name selection
     */
    protected Component createFontNameComponent(TextListModel fontNamesModel)
    {
	fontNameChoice = new Choice(fontNamesModel);
	fontNameChoice.setToolTipText(getToolTip(L_FONT_NAME));
	fontNameChoice.addItemListener(this);

	fontNameChoice.setEnabled(false);
	addSubComp(fontNameChoice);
	addCaretDisableComp(fontNameChoice);

	return fontNameChoice;
    }

    /**
     * Creates and returns <CODE>Component</CODE>s to be used
     * in list handling
     * The subclass may override this method.
     * Users of this method includes <CODE>DocumentEditor</CODE>
     *
     * @return array of <CODE>Component</CODE>s to be used in list handling
     */
    /*
    protected Component[] createListComponents()
    {
	return  createListComponents(0);
    }
    */

    /**
     * Creates and returns <CODE>Component</CODE>s to be used
     * in list handling
     * The subclass may override this method.
     * Users of this method includes <CODE>DocumentEditor</CODE>
     *
     * @return array of <CODE>Component</CODE>s to be used in list handling
     */
    /*
    protected Component[] createListComponents(int textType)
    {
	decIndentButton  = createIconButton(A_DEC_INDENT);
	decIndentButton .addActionListener(this);
	incIndentButton  = createIconButton(A_INC_INDENT);
	incIndentButton .addActionListener(this);

	listButton = createIconButton(A_LIST);
	listButton.addActionListener(this);
	unlistButton  = createIconButton(A_UNLIST);
	unlistButton.addActionListener(this);
	
	return new Component[]{ listButton, unlistButton, decIndentButton, incIndentButton };
    }
    */


    /**
     * Creates and returns <CODE>Component</CODE>s to be used
     * in object insertion
     * The subclass may override this method.
     * Users of this method includes <CODE>DocumentEditor</CODE>
     *
     * @return array of <CODE>Component</CODE>s to be used in object insertion
     */
    protected Component[] createInsertComponents(int textType)
    {
	hrButton = createIconButton(A_HR);
	hrButton.addActionListener(this);
	imageButton  = createIconButton(A_IMAGE);
	imageButton.addActionListener(this);

	return new Component[]{ hrButton, imageButton };
    }


    protected Component[] createCharTableComponents()
    {
	charSet = (CharacterSelectButton)setupButton(new CharacterSelectButton(), "charset", "charset");
	charSet.addItemListener(this);

	return new Component[]{ charSet  };
    }


    /**
     * Returns the tool tip for the spcified name.
     */
    public static String getToolTip(String name) {
	String label = getToolLabel(name);
	if (label.length() > 3 && label.endsWith("...")) {
	    label = label.substring(0, label.length() - 3);
	}
	return label;
    }


    /**
     * Returns the tool label for the spcified name.
     */
    public static String getToolLabel(String name) {
	return EditorResources.getResourceString(name);
    }

    /**
     * Returns the icon for the spcified name.
     * <p>
     * Available icons are: 'anchor', 'backward', 'bold', 'center', 'color',
     * 'copy', 'cut', 'decindent', 'find', 'forward', 'goto', 'hr', 'image',
     * 'incindent', 'italic', 'java', 'large', 'left', 'link', 'list', 'new',
     * 'normalscript', 'olist', 'open', 'paste', 'preview', 'print', 
     * ''redo', 'reload', right', 'save', 'scolor', 'small',
     * 'subscript', 'superscript', 'stop', 'table', 'underline',
     * 'undo', and 'unlist'.
     */
    public VImage getIcon(String name) {
	String file = "icons/" + name + ".gif";
	return jp.kyasu.awt.AWTResources.getIcon(getClass(), file);
    }

    protected Menu createEditMenu()
    {
	Menu menu = new Menu(getToolLabel(L_EDIT));
	MenuItem copy = createMenuItem(A_COPY, this);
	MenuItem cut  = createMenuItem(A_CUT,  this);
	menu.add(copy);
	menu.add(cut);
	menu.add(createMenuItem(A_PASTE, this));
	menu.addSeparator();
	menu.add(createMenuItem(A_FIND,  this));
	menu.addSeparator();
	menu.add(createMenuItem(A_UNDO,  this));

	copy.setEnabled(false);
	cut.setEnabled(false);
	addCaretDisableItem(copy);
	addCaretDisableItem(cut);

	return menu;
    }

    
    protected Menu createFontMenu()
    {
	Menu menu = new Menu(getToolLabel(L_FONT));
	Font font = focused.getFont();

	SelectionMenu name = new SelectionMenu(getToolLabel(L_FONT_NAME));
	name.addActionListener(this);
	String names[] = { "Monospaced", "SansSerif", "Serif", "Dialog" };
	for (int i = 0; i < names.length; i++) {
	    String label = names[i];
	    boolean state = label.equalsIgnoreCase(font.getName());
	    name.add(label, label, state);
	}
	menu.add(name);

	SelectionMenu size = new SelectionMenu(getToolLabel(L_FONT_SIZE));
	size.addActionListener(this);
	int sizes[] = { 8, 10, 12, 14, 16, 18, 20, 22, 24 };
	for (int i = 0; i < sizes.length; i++) {
	    String label = String.valueOf(sizes[i]);
	    boolean state = (sizes[i] == font.getSize());
	    size.add(label, label, state);
	}
	menu.add(size);

	return menu;
    }

    //taken from HTMLEditor

    class HTMLLeftIndentPSModifier
	extends LeftIndentPSModifier
	//	implements ParagraphStyleModifier, java.io.Serializable
    {
	HTMLLeftIndentPSModifier(boolean increment) {
	    super(increment);
	}

	public ParagraphStyle modify(ParagraphStyle pStyle) {
	    HTMLStyle htmlStyle = ((HTMLText)focused.getRichText()).getHTMLStyle();
	    int align = pStyle.getAlignment();
	    int bqLevel = htmlStyle.getBqIncrementLevel(pStyle);
	    int listLevel = htmlStyle.getListIncrementLevel(pStyle);
	    boolean needToBqLevel = (bqLevel > 0);
	    String pStyleName = pStyle.getStyleName();
	    if ("LI-UL".equals(pStyleName)) {
		if (increment) { ++listLevel; }
		else { listLevel = Math.max(listLevel - 1, 1); }
		pStyle = htmlStyle.getULIParagraphStyle(listLevel,
							focused.getForeground());
	    }
	    else if ("LI-OL".equals(pStyleName)) {
		if (increment) { ++listLevel; }
		else { listLevel = Math.max(listLevel - 1, 1); }
		int index = htmlStyle.getOLIIndex(pStyle.getHeading());
		pStyle =
		    htmlStyle.getOLIParagraphStyle(listLevel, index,
						   focused.getForeground());
	    }
	    else if ("DT".equals(pStyleName)) {
		if (increment) { ++listLevel; }
		else { listLevel = Math.max(listLevel - 1, 0); }
		pStyle = htmlStyle.getDTParagraphStyle(listLevel);
	    }
	    else if ("DD".equals(pStyleName)) {
		if (increment) { ++listLevel; }
		else { listLevel = Math.max(listLevel - 1, 1); }
		pStyle = htmlStyle.getDDParagraphStyle(listLevel);
	    }
	    else {
		if (increment) { ++bqLevel; }
		else if (bqLevel > 0) { --bqLevel; }
		else {
		    return pStyle;
		}
		needToBqLevel = true;
	    }
	    if (align != ParagraphStyle.LEFT || needToBqLevel) {
		BasicPSModifier modifier = new BasicPSModifier();
		if (align != ParagraphStyle.LEFT) {
		    modifier.put(BasicPSModifier.ALIGNMENT, align);
		}
		if (needToBqLevel) {
		    modifier.put(BasicPSModifier.LEFT_INDENT,
			     htmlStyle.getLeftIndentation(bqLevel, listLevel));
		    modifier.put(BasicPSModifier.RIGHT_INDENT,
			     htmlStyle.getRightIndentation(bqLevel, listLevel));
		}
		pStyle = pStyle.deriveStyle(modifier);
	    }
	    return pStyle;
	}
    }

    /**
     * Increases the selected paragraph indentation.
     */
    public void increase_HTMLindent() {
	focused.modifySelectionParagraphStyle(new HTMLLeftIndentPSModifier(true));
    }

    /**
     * Decreases the selected paragraph indentation.
     */
    public void decrease_HTMLindent() {
	focused.modifySelectionParagraphStyle(new HTMLLeftIndentPSModifier(false));
    }

    public static TextEditModel getDefaultTextEditModel()
    {
	return new DefaultTextEditModel(RichTextStyle.DEFAULT_DOCUMENT_STYLE);
    }

    public static void main(String args[]) {
	jp.kyasu.awt.Frame f = new jp.kyasu.awt.Frame("TextEditor");
	jp.kyasu.awt.NativePanel p = new jp.kyasu.awt.NativePanel();
	EditAdaptor editor = new EditAdaptor(TEXT_EDITOR);
	p.add(editor.getToolBar(), java.awt.BorderLayout.NORTH);
	jp.kyasu.awt.SplitPanel sp = 
	    new jp.kyasu.awt.SplitPanel(0);
	jp.kyasu.awt.TextField tf = new jp.kyasu.awt.TextField(getDefaultTextEditModel());
	sp.add(tf);
	editor.listen(tf);
	jp.kyasu.awt.TextArea ta = new jp.kyasu.awt.TextArea(getDefaultTextEditModel());
	sp.add(ta);
	editor.listen(ta);
	p.add(sp, java.awt.BorderLayout.CENTER);
	f.add(p, java.awt.BorderLayout.CENTER);
	f.pack();
	f.setVisible(true);
    }
}

//<HTMLEditor>
/*
final
class HistoryElement {
    HTMLText htmlText;
    String ref;
    Point locationOfText;

    public HistoryElement(HTMLText htmlText, Point locationOfText, String ref)
    {
	if (htmlText == null || locationOfText == null)
	    throw new NullPointerException();
	this.htmlText       = htmlText;
	this.locationOfText = locationOfText;
	this.ref            = ref;
    }
}
*/
//</HTMLEditor>
