/*
 * ExtendedFont.java
 *
 * Copyright (c) 1997, 1998 Kazuki YASUMATSU.  All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */
/*
 * Copyright (c) 2000, 2002 Nozomi `James' Ytow
 * All rights reserved.
 */

package jp.kyasu.graphics;

import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Toolkit;
import java.util.Hashtable;

import jp.kyasu.graphics.GenericScript;

/**
 * The <code>ExtendedFont</code> class implements an extended font object.
 * The extended font has a color attribute, an underline attribute and
 * super/subscript attribute.
 * <p>
 * The extended font is immutable.
 *
 * @version 	25 Nov 1997
 * @author 	Kazuki YASUMATSU
 */
public class ExtendedFont implements java.io.Serializable {
    /** The font metrics of this font. */
    protected FontMetrics metrics;

    /** The color attribute of this font. */
    protected Color color;

    /** The underline attribute of this font. */
    protected boolean underline;

    /** The super/subscript attribute of this font and its constants. */
    protected GenericScript script;
    static public final int NORMALSCRIPT  = GenericScript.NORMALSCRIPT;
    static public final int SUPERSCRIPT   = GenericScript.SUPERSCRIPT;
    static public final int SUBSCRIPT     = GenericScript.SUBSCRIPT;

    static public final int BASELINE      = GenericScript.BASELINE;
    
    /** The small capital attribute of this font. */
    protected boolean smallCapital;

    /** The cashe for the font metrics. */
    static private final Hashtable FontMetricsCashe = new Hashtable();

    /**
     * Returns the font metrics in the cashe for the given font. If
     * the font metrics is not in the cashe, creates a font metrics and
     * puts it into the cache.
     */
    static protected synchronized FontMetrics getFontMetrics(Font font) {
	if (font == null)
	    throw new NullPointerException();
	FontMetrics fm = (FontMetrics)FontMetricsCashe.get(font);
	if (fm == null) {
	    fm = Toolkit.getDefaultToolkit().getFontMetrics(font);
	    FontMetricsCashe.put(font, fm);
	}
	return fm;
    }


    /**
     * Constructs an extended font with the specified name, style and size.
     *
     * @param name  the name of the font.
     * @param style the style of the font.
     * @param size  the point size of the font.
     */
    public ExtendedFont(String name, int style, int size) {
	this(name, style, size, null, false, NORMALSCRIPT);
    }

    /**
     * Constructs an extended font with the specified name, style, size
     * and color.
     *
     * @param name  the name of the font.
     * @param style the style of the font.
     * @param size  the point size of the font.
     * @param color the color of the font.
     */
    public ExtendedFont(String name, int style, int size, Color color) {
	this(name, style, size, color, false, NORMALSCRIPT);
    }

    /**
     * Constructs an extended font with the specified name, style, size
     * and underline.
     *
     * @param name      the name of the font.
     * @param style     the style of the font.
     * @param size      the point size of the font.
     * @param underline the font is underlined.
     */
    public ExtendedFont(String name, int style, int size, boolean underline) {
	this(name, style, size, null, underline, NORMALSCRIPT);
    }

    /**
     * Constructs an extended font with the specified name, style, size
     * and super/subscript.
     *
     * @param name        the name of the font.
     * @param style       the style of the font.
     * @param size        the point size of the font.
     * @param superscript the font is underlined.
     */
    public ExtendedFont(String name, int style, int size, int superscript) {
	this(name, style, size, null, false, superscript);
    }

    /**
     * Constructs an extended font with the specified name, style, size,
     * color, underline and super/subscript.
     *
     * @param name      the name of the font.
     * @param style     the style of the font.
     * @param size      the point size of the font.
     * @param color     the color of the font.
     * @param underline the font is underlined.
     * @param superscript the font is super/subscripted.
     */
    public ExtendedFont(String name, int style, int size,
			Color color, boolean underline, 
			int superscript)
    {
	this(name, style, size, color, underline, new GenericScript(null, superscript));
    }

    /**
     * Constructs an extended font with the specified name, style, size,
     * color, underline and super/subscript.
     *
     * @param name        the name of the font.
     * @param style       the style of the font.
     * @param size        the point size of the font.
     * @param color       the color of the font.
     * @param underline   the font is underlined.
     * @param script      the font is super/subscripted.
     */
    public ExtendedFont(String name, int style, int size,
			Color color, boolean underline, 
			GenericScript script, int scriptLevel)
    {
	this(name, style, size, color, underline, 
	     new GenericScript(script, scriptLevel));
    }

    /**
     * Constructs an extended font with the specified name, style, size,
     * color, underline and super/subscript.
     *
     * @param name        the name of the font.
     * @param style       the style of the font.
     * @param size        the point size of the font.
     * @param color       the color of the font.
     * @param underline   the font is underlined.
     * @param script      the font is super/subscripted.
     */
    public ExtendedFont(String name, int style, int size,
			Color color, boolean underline, 
			GenericScript script)
    {
	if (name == null)
	    throw new NullPointerException();
	metrics = getFontMetrics(new Font(name, style, size));
	this.color     = color;
	this.underline = underline;
	this.script = script;
    }

    /**
     * Constructs an extended font with the specified font.
     *
     * @param font      the font object.
     */
    public ExtendedFont(Font font) {
	this(font, null, false, NORMALSCRIPT);
    }

    /**
     * Constructs an extended font with the specified font and color.
     *
     * @param font      the font object.
     * @param color     the color of the font.
     */
    public ExtendedFont(Font font, Color color) {
	this(font, color, false, NORMALSCRIPT);
    }

    /**
     * Constructs an extended font with the specified font and underline.
     *
     * @param font      the font object.
     * @param underline the font is underlined.
     */
    public ExtendedFont(Font font, boolean underline) {
	this(font, null, underline, NORMALSCRIPT);
    }

    /**
     * Constructs an extended font with the specified font and super/subscript.
     *
     * @param font        the font object.
     * @param superscript the font is super/subscripted.
     */
    public ExtendedFont(Font font, int scriptLevel) {
	this(font, null, false, scriptLevel);
    }

    /**
     * Constructs an extended font with the specified font and super/subscript.
     *
     * @param font        the font object.
     * @param superscript the font is super/subscripted.
     */
    public ExtendedFont(Font font, GenericScript script) {
	this(font, null, false, script);
    }

    /**
     * Constructs an extended font with the specified font, color,
     * underline and super/subscirpt.
     *
     * @param font      the font object.
     * @param color     the color of the font.
     * @param underline the font is underlined.
     * @param superscript the font is super/subscripted.
     */
    public ExtendedFont(Font font, Color color, boolean underline,
			int superscript) {
	this(font, color, underline, new GenericScript(null, superscript));
    }

    /**
     * Constructs an extended font with the specified font, color,
     * underline and super/subscirpt.
     *
     * @param font        the font object.
     * @param color       the color of the font.
     * @param underline   the font is underlined.
     * @param superscript the font is super/subscripted.
     */
    public ExtendedFont(Font font, Color color, boolean underline,
			GenericScript script) {
	if (font == null)
	    throw new NullPointerException();
	metrics = getFontMetrics(font);
	this.color     = color;
	this.underline = underline;
	this.script = script;
    }

    /** Returns the platform specific family name of the font. */
    public String getFamily() { return getFont().getFamily(); }

    /** Returns the logical name of the font. */
    public String getName()   { return getFont().getName(); }

    /**
     * Returns the style of the font.
     * @see java.awt.Font#PLAIN
     * @see java.awt.Font#BOLD
     * @see java.awt.Font#ITALIC
     */
    public int getStyle()     { return getFont().getStyle(); }

    /** Returns the point size of the font. */
    public int getSize()      { return metrics.getFont().getSize(); }

    /** Returns true if the font is plain. */
    public boolean isPlain()  { return getFont().isPlain(); }

    /** Returns true if the font is bold. */
    public boolean isBold()   { return getFont().isBold(); }

    /** Returns true if the font is italic. */
    public boolean isItalic() { return getFont().isItalic(); }

    /** Returns true if the extended font is superscript. */
    public boolean isSuperscript()
    {
	if(script == null)
	    return false;
	return script.getScriptLevel() > 0;
    }

    /** Returns true if the extended font is subscript. */
    public boolean isSubscript()
    {
	if(script == null)
	    return false;
	return script.getScriptLevel() < 0;
    }

    /** Returns true if the extended font is neither super nor subscript. */
    public boolean isNormalscript()
    {
	if(script == null)
	    return true;
	return script.getScriptLevel() == 0;
    }


    /**
     * Returns the font object in this extended font.
     */
    public Font getFont() {
	if(script == null)
	return metrics.getFont();
	return script.getFont(metrics);
    }

    /**
     * Returns the font metrics for this extended font.
     */
    public FontMetrics getFontMetrics() {
	if(script == null)
	return metrics;
	return getFontMetrics(script.getFont(metrics));
    }

    /**
     * Returns the color of the extended font.
     */
    public Color getColor() {
	return color;
    }

    /**
     * Checks if this extended font is underlined.
     */
    public boolean isUnderline() {
	return underline;
    }

    /**
     * Checks if this extended font is small capitalized
     */
    public boolean isSmallCapital() {
	return smallCapital;
    }


    /**
     * Returns the superscirpt of the extended font.
     */
    public GenericScript getScript() {
	return script;
    }

    /**
     * Returns the superscirpt of the extended font.
     */
    public int getScriptLevel() {
	return script.getScriptLevel();
    }

    /**
     * Creates a new font by replicating this font with a new color
     * object associated with it.
     *
     * @param  color the color object for the new font.
     * @return a new font.
     */
    public ExtendedFont deriveFont(Color color) {
	return new ExtendedFont(getFont(), color);
    }

    /**
     * Creates a new font by replicating this font with a new underline
     * attribute associated with it.
     *
     * @param  underline the underline attribute for the new font.
     * @return a new font.
     */
    public ExtendedFont deriveFont(boolean underline) {
	return new ExtendedFont(getFont(), underline);
    }

    /**
     * Creates a new font by replicating this font with a new superscript
     * attribute associated with it.
     *
     * @param  superscipt the super/subscript attribute for the new font.
     * @return a new font.
     */
    public ExtendedFont deriveFont(int scriptLevel) {
	if(script == null)
	    return new ExtendedFont(getFont(), new GenericScript(null, NORMALSCRIPT));

	return new ExtendedFont(getFont(), script.deriveGenericScript(scriptLevel));
    }

    public int getBaselineOffset() {
	if(script == null || metrics == null)
	    return BASELINE;
	return script.getBaselineOffset(metrics);
    }

    public int getAdvanceOffset() {
	if(script == null || script.getScriptLevel() == NORMALSCRIPT 
	   || metrics == null || !metrics.getFont().isItalic())
	    return BASELINE;
	//	double advance = script.getBaselineOffset(metrics) * Math.tan(metrics.getFont().getItalicAngle());
	//	return advance > 0.0? (int)(advance + 0.5) : (int)(advance - 0.5);
	return (int)(script.getBaselineOffset(metrics) * Math.tan(metrics.getFont().getItalicAngle()));
    }


    /**
     * Returns a hashcode for this font.
     */
    public int hashCode() {
	int hash = getFont().hashCode();
	if (color != null) hash ^= color.hashCode();
	if (underline) hash ^= (new Boolean(true)).hashCode();
	hash ^= script.hashCode();
	return hash;
    }

    /**
     * Compares two objects for equality.
     */
    public boolean equals(Object anObject) {
	if (this == anObject)
	    return true;
	if (anObject == null)
	    return false;
	if (getClass() == anObject.getClass()) {
	    //return equalsStyle((ExtendedFont)anObject);
	    ExtendedFont exFont = (ExtendedFont)anObject;
	    return (metrics   == exFont.metrics   &&
		    underline == exFont.underline &&
		    script == exFont.script &&
		    (color == null ?
			exFont.color == null :
			color.equals(exFont.color)));
	}
	return false;
    }

    /**
     * Compares the styles of two fonts for equality.
     */
    public boolean equalsStyle(ExtendedFont exFont) {
	return (getStyle() == exFont.getStyle()    &&
		getSize()  == exFont.getSize()     &&
		getName().equals(exFont.getName()) &&
		underline == exFont.underline      &&
		script == exFont.script  &&
		(color == null ?
			exFont.color == null :
			color.equals(exFont.color)));
    }

    /**
     * Returns the string representation of this extended font.
     */
    public String toString() {
	StringBuffer buffer = new StringBuffer();
	buffer.append("name=" + getName() + ",style=");
	if (isBold()) {
	    buffer.append(isItalic() ? "bolditalic" : "bold");
	}
	else {
	    buffer.append(isItalic() ? "italic" : "plain");
	}
	buffer.append(",size=" + getSize());
	if (underline)
	    buffer.append(",underline=true");
	if (color != null)
	    buffer.append(",color=" + color);
	if (script != null)
	    buffer.append(",script=" + script);
	return getClass().getName() + "[" + buffer.toString() + "]";
    }


    private void readObject(java.io.ObjectInputStream s)
	throws java.io.IOException, ClassNotFoundException
    {
	s.defaultReadObject();
	// Make the metrics to be unique.
	metrics = getFontMetrics(getFont());
    }
}
