/*
 * V3DTabBorder.java
 *
 * Copyright (c) 2001 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */
/*
 * $Id: V3DTabBorder.java,v 1.2 2002/09/22 19:47:19 nozomi Exp $
 * $Log: V3DTabBorder.java,v $
 * Revision 1.2  2002/09/22 19:47:19  nozomi
 * KFC style copyright
 *
 * Revision 1.1.1.1  2002/01/16 12:33:32  ryo
 * initial import into CVS
 */

package jp.kyasu.graphics;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Insets;

/**
 * The <code>V3DTabBorder</code> class implements a 3-D highlighted
 * border for the buttons. The border has a style, the trigger style or
 * the toggle style. The border creates different visual presentations
 * according to the style.
 *
 * @see		jp.kyasu.graphics.VButton
 * @see		jp.kyasu.graphics.VButton#TRIGGER
 * @see		jp.kyasu.graphics.VButton#TOGGLE
 *
 * @version 	10 Nov 2001
 * @author 	Nozomi "James" Ytow
 */

public class V3DTabBorder
    extends V3DButtonBorder
{
    //tab placement holder
    int tabPlacement;

    //tab slope holder
    static int tabSlope = 4;

    /**
     * Constructs a raised border with the trigger style.
     */
    public V3DTabBorder() {
	this(true, VTab.TOP);
    }

    /**
     * Constructs a border with the trigger style, and the specified
     * boolean that determines the border to be raised or sunk.
     *
     * @param raised if true, the border is raised; otherwise, sunk.
     */
    public V3DTabBorder(boolean raised) {
	this(raised, VTab.TOP);
    }

    /**
     * Constructs a raised border with the specified style.
     *
     * @param style the style of the border.
     */
    public V3DTabBorder(int placement) {
	this(true, placement);
    }

    /**
     * Constructs a border with the specified boolean that determines
     * the border to be raised or sunk, and the specified style.
     *
     * @param raised if true, the border is raised; otherwise, sunk.
     * @param style  the style of the border.
     */
    public V3DTabBorder(boolean raised, int placement) {
	this(raised, placement, THICK);
    }

    /**
     * Constructs a border with the specified boolean that determines
     * the border to be raised or sunk, and the specified style.
     *
     * @param raised if true, the border is raised; otherwise, sunk.
     * @param style  the style of the border.
     * @param thin   if true, the border is thin.
     */
    public V3DTabBorder(boolean raised, int placement, boolean thin) {
	this(0, 0, raised, placement, thin);
    }

    /**
     * Constructs a raised border with the trigger style, and the
     * specified width and height.
     *
     * @param width  the width of the border.
     * @param height the height of the border.
     */
    public V3DTabBorder(int width, int height) {
	this(width, height, true, VTab.TOP);
    }

    /**
     * Constructs a border with the trigger style, the specified width
     * and height, and the specified boolean that determines the border
     * to be raised or sunk,
     *
     * @param width  the width of the border.
     * @param height the height of the border.
     * @param raised if true, the border is raised; otherwise, sunk.
     */
    public V3DTabBorder(int width, int height, boolean raised) {
	this(width, height, raised, VTab.TOP);
    }

    /**
     * Constructs a raised border with the specified width and height,
     * and the specified style.
     *
     * @param width  the width of the border.
     * @param height the height of the border.
     * @param style  the style of the border.
     */
    public V3DTabBorder(int width, int height, int placement) {
	this(width, height, true, placement);
    }

    /**
     * Constructs a border with the specified width and height,
     * the specified boolean that determines the border to be
     * raised or sunk, and the specified style.
     *
     * @param width  the width of the border.
     * @param height the height of the border.
     * @param raised if true, the border is raised; otherwise, sunk.
     * @param style  the style of the border.
     */
    public V3DTabBorder(int width, int height, boolean raised, int placement) {
	this(width, height, raised, placement, THICK);
    }

    /**
     * Constructs a border with the specified width and height,
     * the specified boolean that determines the border to be
     * raised or sunk, and the specified style.
     *
     * @param width  the width of the border.
     * @param height the height of the border.
     * @param raised if true, the border is raised; otherwise, sunk.
     * @param style  the style of the border.
     * @param thin   if true, the border is thin.
     */
    public V3DTabBorder(int width, int height, boolean raised, int placement,
			   boolean thin)
    {
	super(width, height, raised, VButton.TOGGLE, thin);
	setTabPlacement(placement);
    }


    /**
     * Returns the insets of this 3-D border.
     * @see jp.kyasu.graphics.VBorder#getInsets()
     */
    public Insets getInsets() {
	return new Insets(0, 0, 0, 0);
    }

    /**
     * Returns a clone of this pane border.
     */
    public Object clone()
    {
	V3DTabBorder vborder = (V3DTabBorder)(super.clone());
	vborder.setTabPlacement(tabPlacement);
	return vborder;
    }

    /**
     * Paints the raised 3-D border at the specified location,
     * with the specified dimension.
     */
    /*
    protected void paintRaised(Graphics g, int x, int y, int width, int height)
    {
	if (thin) {
	    paintBottomRight(g, x, y, width, height, Color.gray, null);
	    paintTopLeft(g, x, y, width, height, Color.white, null);
	}
	else {
	    super.paintRaised(g, x, y, width, height);
	}
    }
    */

    /**
     * Paints the sunk 3-D border at the specified location,
     * with the specified dimension.
     */
    protected void paintSunk(Graphics g, int x, int y, int width, int height)
    {
	/*
	if (thin) {
	    paintBottomRight(g, x, y, width, height, Color.white, null);
	    paintTopLeft(g, x, y, width, height, Color.gray, null);
	    return;
	}

	super.paintSunk(g, x, y, width, height);
	*/
	paintRaised(g, x, y, width, height);
    }

    /**
     * Paints the top and left corners at the specified location,
     * the specified dimension, and the specified colors.
     */
    protected void paintTopLeft(Graphics g,
				int x, int y, int width, int height,
				Color outer, Color inner)
    {
	Color originalColor = g.getColor();
	g.setColor(outer);
	int i = 0;
	int slopeOffset;
	switch(tabPlacement) {
	case VTab.TOP:
	case VTab.BOTTOM:
	    slopeOffset = height / tabSlope;

	    switch(tabPlacement) {
	    case VTab.TOP:
		//left
		g.drawLine(x, y + height,
			   x + slopeOffset, y);

		//top
		g.drawLine(x + slopeOffset, y,
			   x + width - slopeOffset, y);

		if (inner != null) {
		    g.setColor(inner);
		    //left
		    g.drawLine(x + 1, y + height,
			       x + slopeOffset + 1, y + 1);
		    
		    //top
		    g.drawLine(x + slopeOffset + 1, y + 1,
			       x + width - slopeOffset - 1, y + 1);
		}
		break;

	    case VTab.BOTTOM:
		//left
		g.drawLine(x, y,
			   x + slopeOffset, y + height - 1);

		//no top

		if (inner != null) {
		    g.setColor(inner);
		    //left
		    g.drawLine(x + 1, y, 
			       x + slopeOffset + 1, y + height - 1);
		    
		    //no top
		}
		break;
	    }
	break;

	case VTab.RIGHT:
	case VTab.LEFT:
	    slopeOffset = width / tabSlope;

	    switch (tabPlacement) {
	    case VTab.RIGHT:
		//top
		g.drawLine(x, y,
			   x + width, y + slopeOffset);

		//no left

		if (inner != null) {
		    g.setColor(inner);
		    //top
		    g.drawLine(x, y + 1,
			       x + width, y + slopeOffset + 1);

		    //left
		    g.drawLine(x + 1, y + slopeOffset,
			       x + 1, y + height - slopeOffset - 1);
		}
		break;


	    case VTab.LEFT:
		//top
		g.drawLine(x + width, y,
			   x , y + slopeOffset);

		//left
		g.drawLine(x, y + slopeOffset,
			   x, y + height - slopeOffset - 1);

		if (inner != null) {
		    g.setColor(inner);
		    //top
		    g.drawLine(x + width, y + 1,
			       x + 1, y + slopeOffset + 1);

		    //left
		    g.drawLine(x + 1, y + slopeOffset,
			       x + 1, y + height - slopeOffset - 1);
		}
		break;

	    }
	break;
	}

	g.setColor(originalColor);
    }

    /**
     * Paints the bottom and right corners at the specified location,
     * the specified dimension, and the specified colors.
     */
    protected void paintBottomRight(Graphics g,
				    int x, int y, int width, int height,
				    Color outer, Color inner)
    {
	Color originalColor = 	g.getColor();
	g.setColor(outer);
	int slopeOffset = 0;

	switch(tabPlacement) {
	case VTab.TOP:
	case VTab.BOTTOM:
	    slopeOffset = height / tabSlope;
	    switch(tabPlacement) {
	    case VTab.TOP:
		//no bottom

		//right
		g.drawLine(x + width - slopeOffset, y,
			   x + width, y + height - 1);

		if (inner != null) {
		    g.setColor(inner);
		    //no bottom

		    //right
		    g.drawLine(x + width - slopeOffset - 1, y,
			       x + width - 1, y + height - 1);
		}
		break;

	    case VTab.BOTTOM:
		//bottom
		g.drawLine(x + slopeOffset, y + height,
			   x + width - slopeOffset, y + height);

		//right
		g.drawLine(x + width - slopeOffset, y + height,
			   x + width, y);

		if (inner != null) {
		    g.setColor(inner);
		    //bottom
		    g.drawLine(x + slopeOffset + 1, y + height - 1,
			   x + width - slopeOffset + 1, y + height - 1);

		    //right
		    g.drawLine(x + width - slopeOffset - 1, y + height - 1,
			       x + width - 1, y);
		}

		break;
	    }
	    break;
	case VTab.RIGHT:
	case VTab.LEFT:
	    slopeOffset = width / tabSlope;
	    switch(tabPlacement) {
	    case VTab.RIGHT:
		//top
		g.drawLine(x, y,
			   x + width, y + slopeOffset);

		//right
		g.drawLine(x + width, y + slopeOffset,
			   x + width, y + height - slopeOffset);

		if (inner != null) {
		    g.setColor(inner);
		    //top
		    g.drawLine(x, y + 1,
			       x + width - 1, y + slopeOffset + 1);

		    //right
		    g.drawLine(x + width - 1, y + slopeOffset + 1,
			       x + width - 1, y + height - slopeOffset - 1);
		}
		break;
	    case VTab.LEFT:
		//top
		g.drawLine(x + width, y,
			   x, y + slopeOffset);

		//no right

		if (inner != null) {
		    g.setColor(inner);
		    //top
		    g.drawLine(x + width + 1, y + 1,
			       x + 1, y + slopeOffset + 1);

		    //no right
		}
		break;
	    }
	    break;
	}

	g.setColor(originalColor);
    }


    /**
     *
     * Returns placement of tab 
     *
     * @see setTabPlacement(int)
     */
    public int getTabPlacement()
    {
	return tabPlacement;
    }
        
    /**
     *
     * Returns tab placement which is one of:
     * <UL>
     * <LI>VTab.TOP 
     * <LI>VTab.BOTTOM 
     * <LI>VTab.LEFT 
     * <LI>VTab.RIGHT 
     * </UL>
     * Default placement is <code>VTab.TOP</code>
     *
     * @param placement placement of tab to the contents
     *
     * @exception IllegalArgumentException if tabPlacement is not any of above 
     * valid values.
     *
     * @see getTabPlacement()
     */
    public void setTabPlacement(int placement)
    {
	tabPlacement = placement;
    }

    /**
     *
     * Returns slope of tab 
     *
     * @see setTabSlope(int)
     */
    public int getTabSlope()
    {
	return tabSlope;
    }
        
    /**
     *
     * Returns tab slope
     * Default slpe is 4;
     *
     * @param placement placement of tab to the contents
     *
     * @exception IllegalArgumentException if tabPlacement is not any of above 
     * valid values.
     *
     * @see getTabPlacement()
     */
    public void setTabSlope(int slope)
    {
	tabSlope = slope;
    }

}
