/*
 * V3DBorder.java
 *
 * Copyright (c) 1997, 1998 Kazuki YASUMATSU.  All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */

package jp.kyasu.graphics;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.Rectangle;

import jp.kyasu.awt.Tab;

/**
 * The <code>V3DBorder</code> class implements a 3-D highlighted border.
 * The 3-D border has a raised state. The 3-D border creates different
 * visual presentations according to the raised state.
 *
 * @version 	02 May 1998
 * @author 	Kazuki YASUMATSU
 */
public class V3DTabbedPanelBorder
    extends V3DBorder
{
    protected Tab currentTab;

    /**
     * Constructs a raised 3-D border.
     */
    public V3DTabbedPanelBorder() {
	this(0, 0);
    }

    /**
     * Constructs a 3-D border with and the boolean that determines
     * the 3-D border to be raised or sunk.
     *
     * @param raised if true, the 3-D border is raised; otherwise, sunk.
     */
    public V3DTabbedPanelBorder(boolean raised) {
	this(0, 0, raised);
    }

    /**
     * Constructs a raised 3-D border with the specified width and height.
     *
     * @param width  the width of the 3-D border.
     * @param height the height of the 3-D border.
     */
    public V3DTabbedPanelBorder(int width, int height) {
	this(width, height, true);
    }

    /**
     * Constructs a 3-D border with the specified width and height,
     * and the boolean that determines the 3-D border to be raised or sunk.
     *
     * @param width  the width of the 3-D border.
     * @param height the height of the 3-D border.
     * @param raised if true, the 3-D border is raised; otherwise, sunk.
     */
    public V3DTabbedPanelBorder(int width, int height, boolean raised)
    {
	super(width, height);
	currentTab = null;
    }

    public void setTab(Tab tab)
    {
	currentTab = tab;
    }

    public Tab getTab()
    {
	return currentTab;
    }

    /**
     * Returns the insets of this 3-D border.
     * @see jp.kyasu.graphics.VBorder#getInsets()
     */
    public Insets getInsets() {
	return new Insets(0, 0, 0, 0);
    }

    /**
     * Returns a clone of this pane border.
     */
    public Object clone()
    {
	V3DTabbedPanelBorder vborder = (V3DTabbedPanelBorder)super.clone();
	vborder.currentTab = currentTab;
	return vborder;
    }

    /**
     * Paints the 3-D border at the specified location,
     * with the specified dimension.
     * @see jp.kyasu.graphics.VBorder#paint(java.awt.Graphics, int, int, int, int)
     */
    public void paint(Graphics g, int x, int y, int width, int height) {
	Rectangle bounds = currentTab.getBounds();
	int tabPlacement = currentTab.getTabPlacement();
	Color tabColor  = currentTab.getButtonBackground();
	Color save = g.getColor();
	if (raised) {
	    paintRaised(g, x, y, width, height, tabColor, bounds, tabPlacement);
	}
	else {
	    paintSunk(g, x, y, width, height, tabColor, bounds, tabPlacement);
	}
	g.setColor(save);
    }

    /**
     * Paints the raised 3-D border at the specified location,
     * with the specified dimension.
     */
    protected void paintRaised(Graphics g, 
			       int x, int y, int width, int height,
			       Color tabColor,
			       Rectangle bounds, int tabPlacement)
    {
	paintBottomRight(g, x, y, width, height,
			 Color.black, Color.gray, tabColor,
			 bounds, tabPlacement);
	paintTopLeft(g, x, y, width, height,
		     Color.lightGray, Color.white, tabColor,
		     bounds, tabPlacement);
    }

    /**
     * Paints the sunk 3-D border at the specified location,
     * with the specified dimension.
     */
    protected void paintSunk(Graphics g,
			     int x, int y, int width, int height,
			     Color tabColor,
			     Rectangle bounds, int tabPlacement)
    {
	paintTopLeft(g, x, y, width, height,
		     Color.gray, Color.black, tabColor,
		     bounds, tabPlacement);
	paintBottomRight(g, x, y, width, height, 
			 Color.white, Color.lightGray, tabColor,
			 bounds, tabPlacement);
    }


    /**
     * Paints the top and left corners at the specified location,
     * the specified dimension, and the specified colors.
     */
    protected void paintTopLeft(Graphics g,
				int x, int y, int width, int height,
				Color outer, Color inner, Color tabColor,
				Rectangle bounds, int tabPlacement)
    {
	g.setColor(outer);
	int i = 0;
	switch(tabPlacement) {
	case Tab.TOP:
	    //left
	    g.drawLine(x, y,
		       x + width, y);
	    //		       x, y + height - 1);
	    //top
	    /*
	    g.drawLine(x, y,
		       bounds.x, y);
	    //	    g.setColor(tabColor);
	    g.drawLine(bounds.x, y,
		       bounds.x + bounds.width, y);
	    g.setColor(outer);
	    g.drawLine(bounds.x + bounds.width, y,
		       x + width, y);
	    */
	    if (inner != null) {
		x++;
		y++;
		g.setColor(inner);
		//left
		g.drawLine(x, y,
			   x, y + height - 2);
		//top
		g.drawLine(x, y,
			   bounds.x - 1, y);
		g.setColor(tabColor);
		g.drawLine(bounds.x, y,
			   bounds.x + bounds.width, y);
		g.setColor(inner);
		g.drawLine(bounds.x + bounds.width + 1, y,
			   x + width - 2, y);
		x--;
		y--;
	    }
	    break;
	case Tab.LEFT:
	    //top
	    g.drawLine(x, y,
		       x + width, y);

	    //left
	    g.drawLine(x, y,
		       x, bounds.y);
	    g.setColor(tabColor);
	    g.drawLine(x, bounds.y,
		       x, bounds.y + bounds.height);
	    g.setColor(outer);
	    g.drawLine(x, bounds.y + bounds.height,
		       x, y + height);

	    if (inner != null) {
		x++;
		y++;
		g.setColor(inner);
		//top
		g.drawLine(x, y,
			   x + width, y);
		//left
		g.drawLine(x, y,
			   x, bounds.y - 1);

		g.setColor(tabColor);

		g.drawLine(x, bounds.y,
			   x, bounds.y + bounds.height);
		g.setColor(inner);
		g.drawLine(x, bounds.y + bounds.height + 1,
			   x, y + height - 2);
		
		y--;
		x--;
	    }
	    break;
	default: //either BOTTOM or RIGHT
	    g.drawLine(x, y,
		       x + width - 1, y);
	    g.drawLine(x, y,
		       x, y + height - 1);

	    if (inner != null) {
		g.setColor(inner);
		x++;
		y++;
		g.drawLine(x,             y,
			   x + width - 3, y);
		g.drawLine(x, y,
			   x, y + height - 3);
		x--;
		y--;
	    }
	    break;
	}
    }

    /**
     * Paints the bottom and right corners at the specified location,
     * the specified dimension, and the specified colors.
     */
    protected void paintBottomRight(Graphics g,
				    int x, int y, int width, int height,
				    Color outer, Color inner, Color tabColor,
				    Rectangle bounds, int tabPlacement)
    {
	g.setColor(outer);
	int i = 0;
	switch(tabPlacement) {
	case Tab.BOTTOM:
	    //right
	    g.drawLine(x + width, y,
		       x + width, y + height);
	    //bottom
	    g.drawLine(x + width, y + height,
		       bounds.x + bounds.width,  y + height);
	    g.setColor(tabColor);
	    g.drawLine(bounds.x + bounds.width, y + height,
		       bounds.x,  y + height);
	    g.setColor(outer);
	    g.drawLine(bounds.x, y + height,
		       x,  y + height);

	    if (inner != null) {
		x--;
		y--;
		g.setColor(inner);
		//right
		g.drawLine(x + width, y,
			   x + width, y + height);
		//bottom
		g.drawLine(x + width, y + height,
			   bounds.x + bounds.width,  y + height);
		g.setColor(tabColor);
		g.drawLine(bounds.x + bounds.width, y + height,
			   bounds.x,  y + height);
		g.setColor(inner);
		g.drawLine(bounds.x, y + height,
			   x,  y + height);
		y++;
		x++;
	    }
	    break;
	case Tab.RIGHT:
	    //bottom
	    y += height;
	    g.drawLine(x, y,
		       x + width, y);
	    y -= height;
	    //right
	    x += width;
	    g.drawLine(x, y,
		       x,  bounds.y);
	    g.setColor(tabColor);
	    g.drawLine(x, bounds.y,
		       x, bounds.y + bounds.height);
	    g.setColor(outer);
	    g.drawLine(x, bounds.y + bounds.height,
		       x,  y + height);
	    x -= width;

	    if (inner != null) {
		x--;
		y--;
		g.setColor(inner);

		//bottom
		y += height;
		g.drawLine(x, y,
			   x + width, y);
		y -= height;
		//right
		x += width;
		g.drawLine(x, y,
			   x,  bounds.y);
		g.setColor(tabColor);
		g.drawLine(x, bounds.y,
			   x, bounds.y + bounds.height);
		g.setColor(inner);
		g.drawLine(x, bounds.y + bounds.height,
			   x,  y + height);
		x -= width;

		y++;
		x++;
	    }
	    break;
	default:
	    //right
	    g.drawLine(x + width, y,
		       x + width, y + height);
	    //bottom
	    g.drawLine(x + width, y + height,
		       x,  y + height);
	    if (inner != null) {
		x--;
		y--;
		g.setColor(inner);
		//right
		g.drawLine(x + width, y,
			   x + width, y + height);
		//bottom
		g.drawLine(x + width, y + height,
			   x + 2, y + height);
		y++;
		x++;
	    }
	    break;
	}
    }
}
