/*
 * VTab.java
 *
 * Copyright (c) 2001 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */
/*
 * $Id: VTab.java,v 1.2 2002/09/22 19:47:19 nozomi Exp $
 * $Log: VTab.java,v $
 * Revision 1.2  2002/09/22 19:47:19  nozomi
 * KFC style copyright
 *
 * Revision 1.1.1.1  2002/01/16 12:33:32  ryo
 * initial import into CVS
 */

package jp.kyasu.graphics;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.Point;

/**
 * The <code>VTab</code> class implements the visual button that
 * acts as a button model. The button creates different visual presentations
 * according to the style and the state.
 *
 * @see 	jp.kyasu.awt.AbstractButton
 * @see 	jp.kyasu.awt.Button
 * @see 	jp.kyasu.awt.ToggleButton
 *
 * @version 	10 Nov 2001
 * @author 	Nozomi "James" Ytow
 */
public class VTab
    extends VButton
{
    //tab placement constants
    public static final int TOP = 1;
    public static final int LEFT = 2;
    public static final int BOTTOM = 3;
    public static final int RIGHT = 4;

    /**
     * Constructs a trigger button with the <code>false</code> state.
     */
    public VTab() {
	this(false);
    }

    /**
     * Constructs a trigger button with the specified state.
     *
     * @param state the state of the button.
     */
    public VTab(boolean state) {
	this(state, TOP);
    }

    /**
     * Constructs a button with the <code>false</code> state,
     * and the specified style.
     *
     * @param style the style of teh button.
     */
    public VTab(int placement) {
	this(false, placement);
    }

    /**
     * Constructs a button with the specified state and the specified style.
     *
     * @param state the state of teh button.
     * @param style the style of teh button.
     */
    public VTab(boolean state, int placement) {
	this("", state, placement);
    }

    /**
     * Constructs a trigger button with the <code>false</code> state,
     * and the specified string.
     *
     * @param str the string.
     */
    public VTab(String str) {
	this(str, false, TOP);
    }

    /**
     * Constructs a trigger button with the specified string and the
     * specified state.
     *
     * @param str   the string.
     * @param state the state of the button.
     */
    public VTab(String str, boolean state) {
	this(str, state, TOP);
    }

    /**
     * Constructs a button with the <code>false</code> state,
     * the specified string, and the specified style.
     *
     * @param str   the string.
     * @param style the style of the button.
     */
    public VTab(String str, int placement) {
	this(str, false, placement);
    }

    /**
     * Constructs a button with the specified string, the specified state,
     * and the specified style.
     *
     * @param str   the string.
     * @param state the state of teh button.
     * @param style the style of teh button.
     */
    public VTab(String str, boolean state, int placement) {
	this(new Text(str), state, placement);
    }

    /**
     * Constructs a trigger button with the <code>false</code> state,
     * and the specified text.
     *
     * @param text the text.
     */
    public VTab(Text text) {
	this(text, false, TOP);
    }

    /**
     * Constructs a trigger button with the specified text and the
     * specified state.
     *
     * @param text  the text.
     * @param state the state of the button.
     */
    public VTab(Text text, boolean state) {
	this(text, state, TOP);
    }

    /**
     * Constructs a button with the <code>false</code> state,
     * the specified text, and the specified style.
     *
     * @param text  the text.
     * @param style the style of the button.
     */
    public VTab(Text text, int placement) {
	this(text, false, placement);
    }

    /**
     * Constructs a button with the specified text, the specified state,
     * and the specified style.
     *
     * @param text  the text.
     * @param state the state of teh button.
     * @param style the style of teh button.
     */
    public VTab(Text text, boolean state, int placement) {
	this(new VText(text), state, placement);
    }

    /**
     * Constructs a trigger button with the <code>false</code> state,
     * and the specified visual object.
     *
     * @param visualizable the visual object.
     */
    public VTab(Visualizable visualizable) {
	this(visualizable, false, TOP);
    }

    /**
     * Constructs a trigger button with the specified visual object
     * and the specified state.
     *
     * @param visualizable the visual object.
     * @param state        the state of the button.
     */
    public VTab(Visualizable visualizable, boolean state) {
	this(visualizable, state, TOP);
    }

    /**
     * Constructs a button with the <code>false</code> state,
     * the specified visual object, and the specified style.
     *
     * @param visualizable the visual object.
     * @param style        the style of the button.
     */
    public VTab(Visualizable visualizable, int placement) {
	this(visualizable, false, placement);
    }

    /**
     * Constructs a button with the specified visual object,
     * the specified state, and the specified style.
     *
     * @param visualizable the visual object.
     * @param state        the state of teh button.
     * @param style        the style of teh button.
     */
    public VTab(Visualizable visualizable, boolean state, int placement) {
	this(visualizable, true, false, state, placement);
    }

    /**
     * Constructs a button with the specified visual object,
     * states, and style.
     *
     * @param visualizable the visual object.
     * @param enabled      the enabled state of teh button.
     * @param focused      the focused state of teh button.
     * @param state        the state of teh button.
     * @param style        the style of teh button.
     */
    protected VTab(Visualizable visualizable, boolean enabled,
		      boolean focused, boolean state, int placement)
    {
	this(visualizable, enabled, focused, state, placement,
	     new V3DTabBorder(!state, placement));
    }

    /**
     * Constructs a button with the specified visual object,
     * states, style, and border.
     *
     * @param visualizable the visual object.
     * @param enabled      the enabled state of teh button.
     * @param focused      the focused state of teh button.
     * @param state        the state of teh button.
     * @param style        the style of teh button.
     * @param border       the border of teh button.
     */
    protected VTab(Visualizable visualizable, boolean enabled,
		      boolean focused, boolean state, int placement,
		      V3DButtonBorder border)
    {
	super(visualizable, enabled, focused, state, TOGGLE, border);
	//	setTabPlacement(placement);
    }


    protected VBorder createFocusedBorder()
    {
	return new VDashedTabBorder();
    }

    /**
     * Creates a new button by replicating this button with a new visual
     * object associated with it.
     *
     * @param  visualizable the visual object for the new button.
     * @return a new button.
     */
    public VLabel deriveLabel(Visualizable visualizable) {
	if (visualizable == null)
	    throw new NullPointerException();
	return new VTab(visualizable, enabled, focused, state, getTabPlacement());
    }

    /**
     * Sets the style of the button to the specified style.
     *
     * @param style the style of the button.
     * @see   #TRIGGER
     * @see   #TOGGLE
     */
    public void setStyle(int style)
    {
	//Always TOGGLE
	return;
    }

    /**
     * Returns the size of the visual content with the focused (dashed line)
     * border.
     */
    protected Dimension getFocusedSize() {
	Dimension d = visualizable.getSize();
	Insets insets = focusedBorder.getInsets();
	int width  = d.width  + (insets.left + insets.right);
	int height = d.height + (insets.top + insets.bottom);
	int slope = getTabSlope() / 2 ;
	switch(getTabPlacement()) {
	case TOP:
	case BOTTOM:
	    width += height / slope;
	    break;
	case RIGHT:
	case LEFT:
	    height += width / slope;
	    break;
	}
	return new Dimension(width, height);
    }

    /**
     * Paints the label at the specified location with the component.
     *
     * @param g      the graphics.
     * @param p      the location.
     * @param comp   the component used to make the disabled presentation.
     * @param raised if false, paints the visual content at the sunk location.
     */
    /*
    protected void paint(Graphics g, Point p, Component comp, boolean raised)
    {
	super.paint(g, p, comp, !raised);
    }
    */

    /**
     * Paints the button at the specified location with the component.
     *
     * @param g    the graphics.
     * @param p    the location.
     * @param comp the component used to make the disabled presentation.
     */
    public void paint(Graphics g, Point p, Component comp) {
	Dimension d = getSize();
	int width  = d.width  - (insets.left + insets.right);
	int height = d.height - (insets.top + insets.bottom);
	int xOffset = 0;
	int yOffset = 0;
	int slope = getTabSlope();
	switch(getTabPlacement()) {
	case TOP:
	case BOTTOM:
	    xOffset = height / slope;
	    break;
	case RIGHT:
	case LEFT:
	    yOffset = width / slope;
	    break;
	}
	if (width > 0 && height > 0) {
	    super.paint(g,
			new Point(p.x + xOffset + insets.left, p.y + yOffset + insets.top),
			comp,
			state);
	}
	if (comp != null) {
	    d = comp.getSize();
	    paintBorder(g, 0, 0, d.width, d.height);
	}
	else {
	    paintBorder(g, p.x, p.y, d.width, d.height);
	}
    }

    /**
     *
     * Returns placement of tab 
     *
     * @see setTabPlacement(int)
     */
    public int getTabPlacement()
    {
	return ((V3DTabBorder)border).getTabPlacement();
    }
        
    /**
     *
     * Returns tab placement which is one of:
     * <UL>
     * <LI>VTab.TOP 
     * <LI>VTab.BOTTOM 
     * <LI>VTab.LEFT 
     * <LI>VTab.RIGHT 
     * </UL>
     * Default placement is <code>VTab.TOP</code>
     *
     * @param placement placement of tab to the contents
     *
     * @exception IllegalArgumentException if tabPlacement is not any of above 
     * valid values.
     *
     * @see getTabPlacement()
     */
    public void setTabPlacement(int placement)
    {
	if(placement != getTabPlacement())
	    ((V3DTabBorder)border).setTabPlacement(placement);
    }

    /**
     *
     * Returns placement of tab 
     *
     * @see setTabPlacement(int)
     */
    public int getTabSlope()
    {
	return ((V3DTabBorder)border).getTabSlope();
    }
        
    /**
     *
     * Returns tab placement which is one of:
     * <UL>
     * <LI>VTab.TOP 
     * <LI>VTab.BOTTOM 
     * <LI>VTab.LEFT 
     * <LI>VTab.RIGHT 
     * </UL>
     * Default placement is <code>VTab.TOP</code>
     *
     * @param placement placement of tab to the contents
     *
     * @exception IllegalArgumentException if tabPlacement is not any of above 
     * valid values.
     *
     * @see getTabPlacement()
     */
    public void setTabSlope(int slope)
    {
	if(slope != getTabSlope())
	    ((V3DTabBorder)border).setTabSlope(slope);
    }

    /**
     * Fills label.  It should be override in subclasses where label is 
     * not rectangular.
     * Added by Nozomi `James' Ytow to support non-rectangular label e.g. tab
     *
     * @param g Graphics to be paint
     * @param left x origin of reagin to be paint
     * @param top  y origin of reagin to be paint
     * @param width width to be paint
     * @param height height to be paint
     * 
     * @version 	11 Nov. 2001
     * @author 	Nozomi `James' Ytow
     */
    public void fillLabel(Graphics g, int left, int top, int width, int height)
    {
	int x[] = new int[4];
	int y[] = new int[4];
	int offset = getTabSlope();
	int tabPlacement = getTabPlacement();
	switch(tabPlacement) {
	case TOP:
	case BOTTOM:
	    offset = height / offset;
	    break;
	case RIGHT:
	case LEFT:
	    offset = width / offset;
	    break;
	default:
	    offset = 0;
	}

	switch(tabPlacement) {
	case TOP:
	    x[0] = left;
	    y[0] = top + height;
	    x[1] = left + offset;
	    y[1] = top;
	    x[2] = left + width - offset;
	    y[2] = top;
	    x[3] = left + width;
	    y[3] = top  + height;
	    break;
	case BOTTOM:
	    x[0] = left;
	    y[0] = top;
	    x[1] = left + width;
	    y[1] = top;
	    x[2] = left + width - offset;
	    y[2] = top  + height;
	    x[3] = left + offset;
	    y[3] = top  + height;
	    break;
	case RIGHT:
	    x[0] = left;
	    y[0] = top;
	    x[1] = left + width;
	    y[1] = top + offset;
	    x[2] = left + width;
	    y[2] = top  + height - offset;
	    x[3] = left;
	    y[3] = top  + height;
	    break;
	case LEFT:
	    x[0] = left;
	    y[0] = top + offset;
	    x[1] = left + width;
	    y[1] = top;
	    x[2] = left + width;
	    y[2] = top  + height;
	    x[3] = left;
	    y[3] = top  + height - offset;
	    break;
	default:
	    x[0] = left;
	    y[0] = top;
	    x[1] = left + width;
	    y[1] = top;
	    x[2] = left + width;
	    y[2] = top  + height;
	    x[3] = left;
	    y[3] = top  + height;
	}
    	g.fillPolygon(x,  y, 4);
    }

}
