/*
 * TextPositionInfo.java
 *
 * Copyright (c) 1997, 1998 Kazuki YASUMATSU.  All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */
/*
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 */
/*
 * $Id: TextPositionInfo.java,v 1.3 2002/08/01 04:17:18 nozomi Exp $
 * $Log: TextPositionInfo.java,v $
 * Revision 1.3  2002/08/01 04:17:18  nozomi
 * change comment style
 *
 * Revision 1.2  2002/06/23 20:25:33  nozomi
 * modify to support ExtendedFont
 *
 */

package jp.kyasu.graphics.text;

import jp.kyasu.graphics.ParagraphStyle;
import jp.kyasu.graphics.TextStyle;

import java.awt.FontMetrics;

/**
 * The <code>TextPositionInfo</code> class provides an information of
 * position in <code>TextLayout</code>.
 *
 * @version 	01 Aug 2002
 * @author 	Kazuki YASUMATSU
 * @author 	Nozomi `James' Ytow
 */
public class TextPositionInfo extends TextLineInfo {
    /** The index of text at this position. */
    public int textIndex;

    /** The index of line at this position. */
    public int lineIndex;

    /** The x position. */
    public int x;

    /**
     * The slant angle of current position and its constant for vertical.
     * It is necessary to support slanted font.
     */
    public float slantAngle;
    static public final float VERTICAL = 0.0f;

    /**
     * The baseline offset for super/subscript, addscript and ruby
     */
    public int baselineOffset;

    /**
     * The advance offset for slanted super/subscript, addscript and ruby.
     * It is a combination of slant ans generic script, and hence it can 
     * be calcurated as <code>baselineOffset * Math.tan(slantAngle)</code>.
     * It is stored here, however, to avoid dispersion of the calculation.
     * It is calculated by an <code>ExtendedFont</code> object.
     */
    public int advanceOffset;

    /**
     * The size of font in real (for scaled script)
     */
    public int size;

    /**
     * Constructs an empty text position information.
     */
    public TextPositionInfo() {
	super();
	textIndex = 0;
	lineIndex = 0;
	x         = 0;
	slantAngle     = 0.0f;
	baselineOffset = 0;
	advanceOffset  = 0;
	size           = 0;
    }

    /**
     * Constructs a text position information with the specified informations.
     *
     * @param textIndex       the index of text.
     * @param lineIndex       the index of line.
     * @param x               the x position.
     * @param y               the y position.
     * @param lineBegin       the beginning text index.
     * @param lineEnd         the ending text index.
     * @param remainWidth     the remaining width.
     * @param lineHeight      the height.
     * @param lineSkip        the height plus space after line and paragraph.
     * @param baseline        the baseline.
     * @param paragraphStyle  the paragraph style.
     * @param isParagraphMark <code>true</code>when end of a paragraph.
     */
    public TextPositionInfo(int textIndex, int lineIndex, int x, int y,
			    int lineBegin, int lineEnd, int remainWidth,
			    int lineHeight, int lineSkip, int baseline,
			    ParagraphStyle paragraphStyle,
			    boolean isParagraphMark)
    {
	this(textIndex, lineIndex, x, y, 
	     lineBegin, lineEnd, remainWidth,
	     lineHeight, lineSkip, baseline,
	     paragraphStyle, isParagraphMark, 0.0f, 0, 0);
    }

    /**
     * Constructs a text position information with the specified informations.
     *
     * @param textIndex       the index of text.
     * @param lineIndex       the index of line.
     * @param x               the x position.
     * @param y               the y position.
     * @param lineBegin       the beginning text index.
     * @param lineEnd         the ending text index.
     * @param remainWidth     the remaining width.
     * @param lineHeight      the height.
     * @param lineSkip        the height plus space after line and paragraph.
     * @param baseline        the baseline.
     * @param paragraphStyle  the paragraph style.
     * @param isParagraphMark <code>true</code>when end of a paragraph.
     * @param slantAngle      the slant angle
     * @param baselineOffset  the baseline offset
     * @param advanceOffset   the advance offset
     */
    public TextPositionInfo(int textIndex, int lineIndex, int x, int y,
			    int lineBegin, int lineEnd, int remainWidth,
			    int lineHeight, int lineSkip, int baseline,
			    ParagraphStyle paragraphStyle,
			    boolean isParagraphMark,
			    float slantAngle,
			    int baselineOffset,
			    int advanceOffset)
    {
	super(lineBegin, lineEnd, remainWidth, lineHeight, lineSkip, baseline,
	      y, paragraphStyle, isParagraphMark);
	this.textIndex = textIndex;
	this.lineIndex = lineIndex;
	this.x         = x;
	this.slantAngle     = slantAngle;
	this.baselineOffset = baselineOffset;
	this.advanceOffset  = advanceOffset;
	this.size           = lineHeight - lineSkip;
    }

    /**
     * Constructs a text position information with the specified text line
     * information and other informations.
     *
     * @param textIndex the index of text.
     * @param lineIndex the index of line.
     * @param x         the x position.
     * @param lineInfo  the text line information.
     */
    public TextPositionInfo(int textIndex, int lineIndex, int x,
			    TextLineInfo lineInfo)
    {
	this(textIndex, lineIndex, x, 0.0f, 0, 0, lineInfo);
    }

    /**
     * Constructs a text position information with the specified text line
     * information and other informations.
     *
     * @param textIndex  the index of text.
     * @param lineIndex  the index of line.
     * @param x          the x position.
     * @param slantAngle the slant angle of this position.
     * @param baselineOffset the baseline offset for super/subscript etc.
     * @param advanceOffset the advance offset for slanted super/subscript etc.
     * @param lineInfo   the text line information.
     */
    public TextPositionInfo(int textIndex, int lineIndex, int x,
			    float slantAngle,
			    int baselineOffset,
			    int advanceOffset,
			    TextLineInfo lineInfo)
    {
	super(lineInfo);
	this.textIndex = textIndex;
	this.lineIndex = lineIndex;
	this.x         = x;
	this.slantAngle = slantAngle;
	this.baselineOffset = baselineOffset;
	this.advanceOffset = advanceOffset;
	this.size       = this.lineHeight;
	if(paragraphStyle != null)
	    this.size -= this.paragraphStyle.getLineSpace();
    }

    /**
     * Constructs a text position information with the same information
     * as the specified text position information.
     *
     * @param posInfo the text position information.
     */
    public TextPositionInfo(TextPositionInfo posInfo) {
	super(posInfo);
	this.textIndex = posInfo.textIndex;
	this.lineIndex = posInfo.lineIndex;
	this.x         = posInfo.x;
	this.slantAngle = posInfo.slantAngle;
	this.baselineOffset = posInfo.baselineOffset;
	this.advanceOffset = posInfo.advanceOffset;
	this.size       = posInfo.size;
    }

    /**
     * Returns a string representation of the parameters.
     */
    public String paramString() {
	StringBuffer buffer = new StringBuffer();
	buffer.append("textIndex=" + textIndex);
	buffer.append(",lineIndex=" + lineIndex);
	buffer.append("," + super.paramString());
	buffer.append(",x=" + x);
	buffer.append(",size=" + size);
	if(slantAngle != 0.0f)
	    buffer.append(",slant=" + slantAngle);
	if(baselineOffset != 0)
	    buffer.append(",baselineOffset=" + baselineOffset);
	if(advanceOffset != 0)
	    buffer.append(",advanceOffset=" + advanceOffset);
	return buffer.toString();
    }

    /**
     * Sets slant angle and size valuse given by <CODE>metrics</CODE>.
     *
     * @param metrics <CODE>FontMetrics</CODE> to adapt to
     */
    public void adaptTo(FontMetrics metrics) {
	if(metrics != null && metrics.getFont() != null) {
	    slantAngle = - metrics.getFont().getItalicAngle();
	    size = metrics.getFont().getSize();
	}
    }

    /**
     * Sets slant angle, size, baseline offset and advane offset
     * valuse given by <CODE>testStyle</CODE>.
     *
     * @param textStyle <CODE>TextStyle</CODE> to adapt to
     */
    public void adaptTo(TextStyle textStyle) {
	if(textStyle != null && textStyle.getExtendedFont() != null){
	    adaptTo(textStyle.getExtendedFont().getFontMetrics());
	    baselineOffset = textStyle.getExtendedFont().getBaselineOffset();
	    advanceOffset  = textStyle.getExtendedFont().getAdvanceOffset();
	}
    }

}
