/*
 * CompareAdapter.java
 *
 * Copyright (c) 1997, 1998 Kazuki YASUMATSU.  All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee or royalty is hereby
 * granted, provided that both the above copyright notice and this
 * permission notice appear in all copies of the software and
 * documentation or portions thereof, including modifications, that you
 * make.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE NO
 * REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED. BY WAY OF EXAMPLE,
 * BUT NOT LIMITATION, COPYRIGHT HOLDERS MAKE NO REPRESENTATIONS OR
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR
 * THAT THE USE OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY
 * THIRD PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.
 * COPYRIGHT HOLDERS WILL BEAR NO LIABILITY FOR ANY USE OF THIS SOFTWARE
 * OR DOCUMENTATION.
 */
/*
 * Copyright (c) 2002 Nozomi `James' Ytow
 * All rights reserved.
 */
/*
 * $Id: CompareAdapter.java,v 1.2 2002/09/09 05:39:25 nozomi Exp $
 * $Log: CompareAdapter.java,v $
 * Revision 1.2  2002/09/09 05:39:25  nozomi
 * add comparison of Object[]s and Strings
 *
 */

package jp.kyasu.util;

/**
 * The adapter for the comparison operations.
 * The methods in this class are empty; this class is provided as a
 * convenience for easily creating comparers by extending this class
 * and overriding only the methods of interest.
 *
 * @see 	jp.kyasu.util.Comparer
 * @see 	jp.kyasu.util.Sorter
 *
 * @version 	09 Sep 2002
 * @author 	Kazuki YASUMATSU
 * @author 	Nozomi `James' Ytow
 */
public abstract class CompareAdapter
    implements Comparer
{
    /**
     * Constructs a <CODE>CompareAdapter</CODE>
     * which compares arrays sequentially
     */
    public CompareAdapter() { this(null); }

    /**
     * Constructs a <CODE>CompareAdapter</CODE>
     * which compares arrays in specified <CODE>order</CODE>.
     * If arrays to be compared to is longer than <CODE>order</CODE>,
     * sequential comparison will be used for remaining part.
     *
     * @param order an array of <CODE>int</CODE> specifying comparison order
     */
    public CompareAdapter(int[] order)
    {
	setSearchOrder(order);
    }

    /** the array of <CODE>int</CODE> retaining comparison order */
    protected int[] searchOrder;

    /**
     * Returns the array of <CODE>int</CODE> retaining comparison order
     *
     * @return int[] retaining comparison order
     */
    public int[] getSearchOrder()
    {
	return searchOrder;
    }

    /**
     * Sets <CODE>order</CODE> to specify comparison sequence.
     * The <CODE>order</CODE> may be null to use default comparison order.
     *
     * @param order an array of <CODE>int</CODE> retaining comparison order,
     * or null to use default comparison order
     */
    public void setSearchOrder(int[] order)
    {
	searchOrder = order;
    }

    /**
     * Sets search order to default, i.e. in simple sequence
     */
    public void setDefaultSearchOrder()
    {
	setSearchOrder(null);
    }


    /** Returns &lt;0 if x &lt; y, etc. */
    public int compare(boolean x, boolean y) {
	return (x == y ? 0 : (x ? -1 : 1));
    }

    /** Returns &lt;0 if x &lt; y, etc. */
    public int compare(byte x, byte y) {
	return (x == y ? 0 : (x < y ? -1 : 1));
    }

    /** Returns &lt;0 if x &lt; y, etc. */
    public int compare(char x, char y) {
	return (x == y ? 0 : (x < y ? -1 : 1));
    }

    /** Returns &lt;0 if x &lt; y, etc. */
    public int compare(short x, short y) {
	return (x == y ? 0 : (x < y ? -1 : 1));
    }

    /** Returns &lt;0 if x &lt; y, etc. */
    public int compare(int x, int y) {
	return (x == y ? 0 : (x < y ? -1 : 1));
    }

    /** Returns &lt;0 if x &lt; y, etc. */
    public int compare(long x, long y) {
	return (x == y ? 0 : (x < y ? -1 : 1));
    }

    /** Returns &lt;0 if x &lt; y, etc. */
    public int compare(float x, float y) {
	return (x == y ? 0 : (x < y ? -1 : 1));
    }

    /** Returns &lt;0 if x &lt; y, etc. */
    public int compare(double x, double y) {
	return (x == y ? 0 : (x < y ? -1 : 1));
    }

    /** Returns &lt;0 if x &lt; y, etc. */
    public int compare(Object x, Object y)
    {
	if((x instanceof Object[]) &&
	   (y instanceof Object[]))
	    return compare((Object[])x, (Object[])y);

	if((x instanceof String) &&
	   (y instanceof String))
	    return compare((String)x, (String)y);

	return compare(x.hashCode(), y.hashCode());
    }

    /** Returns &lt;0 if x &lt; y, etc. */
    public int compare(String x, String y)
    {
	int lx = x.length();
	int ly = y.length();
	int i = 0;
	while(i < lx && i < ly && Character.toLowerCase(x.charAt(i)) == Character.toLowerCase(y.charAt(i)))
	    i++;
	
	if(i < lx && i < ly) //different char
	    return compare(Character.toLowerCase(x.charAt(i)), Character.toLowerCase(y.charAt(i)));

	//no difference, so compare length
	return compare(lx, ly);

    }

    /**
     * Returns &lt;0 if x &lt; y, etc.
     * It compared length of arrays at first.
     * If the lengths are same, then elements of arrays will be compared
     * until finding difference or to last elements of arrays.
     * The comparison order can be specified by <CODE>searchOrder</CODE>.
     *
     * @param x an array of <CODE>Object</CODE>s to be compared
     * @param y an array of <CODE>Object</CODE>s to be compared
     *
     * @return int less than zero if <CODE>x</CODE> is "smaller" than <CODE>y</CODE>,
     * greater than zero if inverse, or zero if they are the same
     */
    public int compare(Object[] x, Object[] y)
    {
	int result = x.length - y.length;
	if(result == 0) {
	    result = compare(x, y, 0);
	}
	return result;
    }

    /**
     * Returns &lt;0 if x &lt; y, etc.
     *
     * @param x an array of <CODE>Object</CODE>s to be compared
     * @param y an array of <CODE>Object</CODE>s to be compared
     * @param index of elements to be compared in these arrays
     *
     * @return int less than zero if <CODE>index</CODE>th element
     * of <CODE>x</CODE> is "smaller" than that of <CODE>y</CODE>,
     * greater than zero if inverse, or zero if they are the same
     */
    protected int compare(Object[] x, Object[] y, int index)
    {
	int result = 0;

	if(searchOrder == null || index >= searchOrder.length ) {
	    result = compare(x[index], y[index]);
	    if(result == 0) {
		if(index < x.length - 1) {
		    index++;
		    result = compare(x[index], y[index]);
		}
	    }
	}
	else {
	    result = compare(x[searchOrder[index]], y[searchOrder[index]]);
	    if(result == 0) {
		if(index < x.length - 1) {
		    index++;
		    if(index < searchOrder.length)
			result = compare(x[searchOrder[index]], y[searchOrder[index]]);
		}
	    }
	}

	return result;

    }

}
