/*
 *
 * Author.cxx: an implementation of persistent Author class
 * for a Nomenclature Heuristic Model.
 * It may be called also N3, Nomenclature Netowrok Navigator
 * or more simply, MkII.
 *
 * Copyright (c) 1999 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: Author.cxx,v 1.11 1999/12/05 18:23:45 nozomi Exp $
 *	$Log: Author.cxx,v $
 *	Revision 1.11  1999/12/05 18:23:45  nozomi
 *	improve merge code
 *	
 *	Revision 1.10  1999/09/23 06:27:52  nozomi
 *	resolve method using Resolver class
 *	
 *	Revision 1.9  1999/09/05 18:21:13  nozomi
 *	name parsing backed to previous one
 *	
 *	Revision 1.8  1999/09/01 00:46:56  nozomi
 *	modify name parsing
 *	
 *	Revision 1.7  1999/08/23 06:29:24  nozomi
 *	fix "senior" handling only for single author
 *	
 *	Revision 1.6  1999/08/09 17:22:36  nozomi
 *	spell correction
 *	
 *	Revision 1.5  1999/03/24 11:03:34  nozomi
 *	Author(const string&) implemented
 *	
 *	Revision 1.4  1999/03/23 14:18:18  nozomi
 *	parseName(const char*) bug fix to handle `-' in name.
 *	
 *	Revision 1.3  1999/03/21 13:44:17  nozomi
 *	merge(Author*) add
 *	
 *	Revision 1.2  1999/03/15 16:42:35  nozomi
 *	gcc 2.7 support macro
 *	
 *	Revision 1.1  1999/03/14 18:15:55  nozomi
 *	minor modification in pointer resolve
 *	
 *	Revision 1.0.0  1999/03/14 02:15:48  nozomi
 *	Initial version, though aka MkII
 *	
 *
 */

#include "d_common.h"
#include "d_Object.h"
#include "d_Ref.h"
#include "Date.h"
#include "Author.h"
#include "NamedField.h"
#include "NameRecord.h"
#include "Publication.h"
#include "Annotation.h"
#include <ctype.h>
#if (__GNUC_MINOR__ > 7)
#include <cctype>
#else
#include <cctype.h>
#endif

//constructor
Author::Author(void)
  : d_Object(_classname), affiliations(), publications(), 
    //    birthD(),deathD(),
    _life(),
    firstPubl(Date::invalidYear), lastPubl(Date::invalidYear)
{
  parseName("");
}

Author::Author(const Author& a)
  : d_Object(_classname), 
    affiliations(a.affiliations), publications(a.publications),
    //    birthD(a.birthD),deathD(a.deathD),
    _life(a._life),
    firstPubl(a.firstPubl), lastPubl(a.lastPubl)
{
  _title=new_strdup::strdup(a._title);
  _firstname=new_strdup::strdup(a._firstname); 
  _middlename=new_strdup::strdup(a._middlename);
  _surname=new_strdup::strdup(a._surname);
  _epithet=new_strdup::strdup(a._epithet);
}

Author::Author(const NamedField &nf, const char* fn)
  : d_Object(_classname), affiliations(), publications(), 
    //    birthD(), deathD(), 
    _life(),
    firstPubl(Date::invalidYear), lastPubl(Date::invalidYear)
{*this = Author(&nf, fn);}

Author::Author(const char* a, int y)
  : d_Object(_classname), 
    affiliations(), publications(),
    //    birthD(),deathD(y),
    _life(Date(y), Date(y)),
    firstPubl(y), lastPubl(y)
{
  if(y != Date::invalidYear){
    //    birthD.year(y-Author::earliestPublicationAt);
    _life.from().year(y-Author::earliestPublicationAt);
  }
  parseName(a);
}

Author::Author(const string& s)
  : d_Object(_classname), 
    affiliations(), publications(),
    //    birthD(),deathD(),
    _life(),
    firstPubl(), lastPubl()
{
  int l = s.length();
  if(l == 0) {parseName("");return;}
  int head = 0;
  if(!strncmp(s.c_str(), "Author::", 8))
    head = 8;
  int i;
  for(i = head; i < l && '_' != s[i]; i++){}
  parseName(string(s, head, i-head).c_str());
  i++;
  head = i;
  if(i < l && '_' != s[i]){
    for(; i < l && '_' != s[i]; i++){}
    //    birthD.year(atoi(string(s, head, i-head).c_str()));
    _life.from().year(atoi(string(s, head, i-head).c_str()));
  }
  i++;
  if(i < l && '_' != s[i]){
    head = i;
    for(; i < l && '_' != s[i]; i++){}
    l = atoi(string(s, head, i-head).c_str());
    firstPubl = lastPubl = l;
    //    deathD.year(l);
    _life.to().year(l);
  }
}

Author::Author(const char* a, Date b, Date d)
  : d_Object(_classname), 
    affiliations(), publications(),
    //    birthD(b),deathD(d),
    _life(b, d),
    firstPubl(Date::invalidYear), lastPubl(Date::invalidYear)
{parseName(a);}

//destructor
Author::~Author(void)
{
  delete [] _firstname; delete [] _middlename; delete [] _surname;
  delete [] _title; delete [] _epithet;
}

string Author::fullname(char c) const
{
  string s;
  if(strlen(_surname)){ s += _surname; }
  if(strlen(_epithet)){ s += c; s += _epithet; }
  if(strlen(_firstname)){ s += ","; s += c; s += _firstname;}
  if(strlen(_middlename)){ s += c;s += _middlename;}
  if(strlen(_title)){ s += ","; s += c; s += _title;}
  return s;
}

#if 0
string Author::fullname(void) const
{
  string s(_surname);
  if(strlen(_epithet)){ s += " "; s += _epithet; }
  if(strlen(_firstname)){ s += ", "; s += _firstname;}
  if(strlen(_middlename)){ s += " ";s += _middlename;}
  if(strlen(_title)){ s += ", "; s += _title;}
  return s;
}
#endif

string Author::abbrname(char c) const
{
  string s;
  if(strlen(_surname)){ s += _surname; }
  if(strlen(_epithet)){ s += c; s += _epithet; }
  if(strlen(_firstname)){ s += ","; s += c; s += *_firstname; s+= ".";}
  if(strlen(_middlename)){ s += c;s += *_middlename; s+= ".";}
  if(strlen(_title)){ s += ","; s += c; s += _title;}
  return s;
}

string Author::initializedName(void) const
{
  string s;
  if(strlen(_surname)){ s += _surname; }
  if(strlen(_epithet)){ s += _epithet; }
  if(strlen(_firstname)){ s += *_firstname;}
  if(strlen(_middlename)){ s += *_middlename;}
  if(strlen(_title)){ s += _title;}
  return s;
}

void Author::surname(const char* s)
{ delete []_surname; _surname = new_strdup::strdup(s);}

Author& Author::operator= (const Author& a)
{
  affiliations = a.affiliations;
  publications = a.publications;
  //  birthD = a.birthD;
  //  deathD = a.deathD;
  _life = a._life;
  firstPubl = a.firstPubl;
  lastPubl = a.lastPubl;
  _title=new_strdup::strdup(a._title);
  _firstname=new_strdup::strdup(a._firstname); 
  _middlename=new_strdup::strdup(a._middlename);
  _surname=new_strdup::strdup(a._surname);
  _epithet=new_strdup::strdup(a._epithet);
  return *this;
}

static string cutWSatBothEnds(const char*);

Author::Author(const char *s)
  : d_Object(_classname), affiliations(), publications(),
    //    birthD(), deathD(), 
    _life(),
    firstPubl(Date::invalidYear), lastPubl(Date::invalidYear)
{
  int l;
  if(NULL == s || 0 == (l = strlen(s))){parseName("");return;}

  const char *f;
  int i;
  if(!strncmp(s, "Author::", 8)){
     s += 8; l-=8;
  }

  for(i = 0, f = s; i < l && (isspace(*f) || isalnum(*f) || '.' == *f || ',' == *f || '-' == *f || '\'' == *f); f++, i++){}
  if(i == l){parseName(s);return;}

  const char separator= *f;
  const char *tmp = f - 1;

  //skip heading ws
  for(i = 0, f = s; i < l && isspace(*f); f++, i++){}
  //skip tailing ws
  for(i = 0; i < tmp - f && isspace(*tmp); tmp--, i++){}
  int al = tmp + 1 - f;
  parseName(f, al);
  for(i = tmp - s, f = tmp; i < l && separator != *f++; i++){}
  for(; i < l && isspace(*f);f++, i++){}
  affiliations.push_back(Affiliation(f));
  return;
} 

  
Author::Author(const NamedField *nf, const char* fn)
  : d_Object(_classname), affiliations(), publications(),
    _title(NULL),_firstname(NULL),
    _middlename(NULL), _surname(NULL), _epithet(NULL), 
    //    birthD(), deathD(), 
    _life(),
    firstPubl(Date::invalidYear), lastPubl(Date::invalidYear)
{_Author(nf, fn);}

void Author::_Author(const NamedField *nf, const char* fn)
{
  if(strcmp(nf->entryName(), fn)) {*this = Author("");return;}

  const char* s;
  if(strlen(s = nf->strcmp(fn))){*this = Author(s); return;}

  if(NULL != (s = nf->strcasecmp("name"))) parseName(s);
  else if(NULL != (s = nf->strcasecmp("fullname"))) parseName(s);
  else{
    if(NULL == (s = nf->strcasecmp("surname"))) s ="";
    _surname = new_strdup::strdup(s);
    if(NULL == (s = nf->strcasecmp("firstname"))) s ="";
    _firstname = new_strdup::strdup(s);
    if(NULL == (s = nf->strcasecmp("middlename"))) s ="";
    _middlename = new_strdup::strdup(s);
    if(NULL == (s = nf->strcasecmp("title"))) s ="";
    _title = new_strdup::strdup(s);
    if(NULL == (s = nf->strcasecmp("epithet"))) s ="";
    _epithet = new_strdup::strdup(s);
  }
  if(NULL != nf->strcasecmp("Affiliation")){
    list<NamedField>::const_iterator tmp;
    for(tmp = nf->subEntries.begin(); tmp != nf->subEntries.end(); ++tmp){
      if(tmp->strcasecmp("Affiliation")){
	affiliations.push_back(Affiliation(&*tmp));
      }
    }
  }

  if(NULL != (s = nf->strcasecmp("firstPublication")) && strlen(s)) firstPubl = atoi(s);
  if(NULL != (s = nf->strcasecmp("lastPublication")) && strlen(s)) lastPubl = atoi(s);
  //  if(NULL != (s = nf->strcasecmp("birth")) && strlen(s)) birthD = Date(s);
  //  if(NULL != (s = nf->strcasecmp("death")) && strlen(s)) deathD = Date(s);
  if(NULL != (s = nf->strcasecmp("birth")) && strlen(s)) _life.from(Date(s));
  if(NULL != (s = nf->strcasecmp("death")) && strlen(s)) _life.to(Date(s));
  if(NULL != nf->strcasecmp("Publication")){
    list<NamedField>::const_iterator tmp;
    for(tmp = nf->subEntries.begin(); tmp != nf->subEntries.end(); ++tmp)
      {
	int y = Date::invalidYear;
	if(NULL != (s = tmp->strcasecmp("Publication"))){
	  if(strlen(s)){
	    d_Ref<Publication> dp;
	    string str("");
	    if(strncmp(s, "Publication::", 13))
	      str += "Publication::";
	    str += s;
	    dp.persistentID(str);
	    publications.push_back(dp);
	    while(*s && '_' != *s){s++;} //skip until end of author
	    if(*s) s++;
	    while(*s && '_' != *s){s++;} //skip until end of title
	    if(*s) s++;
	    if(*s) y = atoi (s);
	  }
	  else{
	    Publication *p = new Publication(&*tmp);
	    publications.push_back(d_Ref<Publication>(p));
	    y = p->year();
	  }
	  if(y != Date::invalidYear){
	    if(firstPubl == Date::invalidYear || y < firstPubl) firstPubl = y;
	    if(lastPubl == Date::invalidYear || y > lastPubl) lastPubl = y;
	    //	    int yr = birthD.year();
	    int yr = _life.from().year();
	    if(Date::invalidYear == yr || (/*Date::invalidYear != yr &&*/ yr > y))
	      //	      birthD.year(y - earliestPublicationAt);
	      _life.from().year(y - earliestPublicationAt);
	    //	    yr = deathD.year();
	    yr = _life.to().year();
	    if(Date::invalidYear == yr || (/*Date::invalidYear != yr &&*/ yr < y))
	      //	      deathD.year(y);
	      _life.to().year(y);
	  }
	}
      }
  }
}


void Author::parseName(const char* s,int lim)
{
  int l;
  if(NULL == s || 0 == (l = strlen(s))){
  _empty:
    _surname = strdup();
  _surnameOnly:
    _title = strdup();
    _firstname = strdup();
    _middlename = strdup();
    _epithet = strdup();
    return;
  }
  if(lim)
    l = lim;
  
  int head;

  string name(cutWSatBothEnds(s));

  int tail = name.length();
  if(tail == 0)
    goto _empty;

  int words = 1; //because non nil

  //count words
  for(head = 1; head < tail; head++){
    if (isalpha(name[head]) && 
	(isspace(name[head -1 ])|| ispunct(name[head-1]))){
      words++;
    }
  }

  int nameComponents = words;

  if(words == 1){  //only surname
    _surname = strdup(name);
    goto _surnameOnly;
  }

  string surNm(""), firstNm(""), middleNm(""), titlNm(""), modNm("");

  if(string *word = new string[words]){
    
    head = 0;
    tail = name.length();

    int wordTail = head;
    int commas = 0;
    int commaAt = 0;
    int pre = -1;
    int post = -1;
    int title = -1;
    int i;

    for (i = 0; i < words ; i++){
      while(wordTail < tail 
	    && !isspace(name[wordTail]) 
	    && (!ispunct(name[wordTail]) || name[wordTail] == '.' 
		|| name[wordTail] == '-' || name[wordTail] == '\''
		|| name[wordTail] == '`')
	    )
	{wordTail++;};
      
      word[i] = string(name, head, wordTail-head);

      if(wordTail < tail && name[wordTail] == ','){
	commas++;
	if(commaAt == 0)
	  commaAt = i + 1;
	if(wordTail < tail - 1)
	  wordTail++;
      }

      
      if(pre < 0 && 
	 (
	  (word[i].length() == 3 &&
	   (word[i].find("von") == 0 || word[i].find("van") == 0 ||
	    word[i].find("VON") == 0 || word[i].find("VAN") == 0 ||
	    word[i].find("Von") == 0 || word[i].find("Van") == 0
	    )
	   )
	  ||
	  (word[i].length() == 2 &&
	   (word[i].find("Le") == 0 || word[i].find("le") == 0 ||
	    word[i].find("LE") == 0
	    )
	   )
	  )
	 )
	{
	  pre = i;
	  nameComponents--;
	}


      if(
	 (word[i].length() == 3 &&
	  (word[i].find("Sir") == 0 || word[i].find("SIR") == 0 || word[i].find("sir") == 0 )
	  )
	 ||
	 (word[i].length() == 4 &&
	  (word[i].find("Load") == 0 || word[i].find("LOAD") == 0 || word[i].find("load") == 0)
	  )
	 )
	{
	  if(titlNm.length())
	    titlNm += " ";
	  titlNm += word[i];
	  title = i;
	  nameComponents--;
	}

      if(post < 0 && 
	  (word[i].find("jr") == 0 || word[i].find("sen") == 0 ||
	   word[i].find("JR") == 0 || word[i].find("SEN") == 0 ||
	   word[i].find("Jr") == 0 ||  word[i].find("Sen") == 0||
	   word[i].find("jr.") == 0 || word[i].find("sen.") == 0 ||
	   word[i].find("JR.") == 0 || word[i].find("SEN.") == 0 ||
	   word[i].find("Jr.") == 0 ||  word[i].find("Sen.") == 0))
	{
	  post = i;
	  nameComponents--;
	}

      while(wordTail < tail && 
	    (isspace(name[wordTail]) || ispunct(name[wordTail]))
	    )
	{ wordTail++;};
      head = wordTail;
    }
    if(commas == 0){/* surname is last */
#if 0
      if(pre < 0){
	if(post < 0)
	  pre = words - 1;
	else{
	  pre = post - 1;
	}
      }
      for(i = pre; i < words; i++)
#else
      if(post < 0)
	post = words;
      else{
	modNm = word[post];
      }
      if(pre < 0){
	if(post < 0)
	  pre = words - 1;
	else{
	  if(post != 0)
	    pre = post - 1;
	  else{
	    pre = post + 1;
	    post = words;
	  }
	}
      }

      for(i = pre; i < post; i++)
#endif
	{
	  if(surNm.length())
	    surNm += " ";
	  surNm += word[i];
	}

      if(nameComponents > 1){
	i = 0;
	if (title > -1)
	  i = title + 1;
	firstNm = word[i++];
	if(nameComponents > 2)
	  for(; i < pre; i++){
	    if(middleNm.length())
	      middleNm += " ";
	    middleNm += word[i];
	  }
#if 0
	if(post > -1)
	  modNm = word[post];
#endif
      }
    }
    else{/* surname beforer */
      if(pre > commaAt){
	surNm = word[pre];
	words = pre;
      }

      i = 0;
      if(title>=0)
	i = title + 1;
      if(post > -1 ){
	modNm = word[post];
	commaAt--;
      }
      for (int j = 0; i < commaAt;i++,j++)
	{
	  if(j)
	    surNm += " ";
	  surNm += word[i];
	}
      if(post > -1 ) i++;
      if(nameComponents > 1){
	firstNm = word[i++];
	for (; i < words; i++){
	  if(middleNm.length() && word[i].length())
	    middleNm += " ";
	  middleNm += word[i];
	}
      }
    }
    delete[] word;
  }
  _surname = strdup(surNm);
  _title = strdup(titlNm);
  _firstname = strdup(firstNm);
  _middlename = strdup(middleNm);
  _epithet = strdup(modNm);
}

static string cutWSatBothEnds(const char* s)
{
  int head = 0;
  string f(s);
  int tail = f.length() - 1;
  if(
     !isspace(f[0]) && !isspace(f[tail])
     && (!ispunct(f[0]) || f[0] == '.')
     && (!ispunct(f[tail]) || f[tail] == '.')
     )
    return f;
  
  if(isspace(f[tail])){
    while(head < tail &&
	  (isspace(f[tail])
	   ||(ispunct(f[tail]) && f[tail] != '.')
	   )
	  ){ tail--;};
  }
  
  if(isspace(f[head]))
    while(head < tail && 
	  (isspace(f[head]) || ispunct(f[head]))
	  ) { head++;};
  if(head == tail && !f[tail]){
    return string("");
  }
  
  return string (f, head, tail - head + 1);
  
}

// identity check. Need improvement
bool operator==(Author& a, Author& b)
{
  if(&a == &b)  //apparent equivalence
    return true;

  if(a.fullname() == b.fullname())
    {
      return true;  //too simple!
    }

  return false;
}

ostream& operator<<(ostream & s, Author& a)
{
  s << "Author = {\n";
  s << " persistentID = {" <<  a.persistentID() << "}\n";
  if(strlen(a._title)) s << " title = {" << a._title << "}\n";
  if(strlen(a._firstname)) s << " firstname = {" << a._firstname << "}\n";
  if(strlen(a._middlename)) s << " middlename = {" << a._middlename << "}\n";
  if(strlen(a._surname)) s << " surname = {" << a._surname << "}\n"; 
  if(strlen(a._epithet)) s << " epithet = {" << a._epithet << "}\n";
  //  if(a.birthD.year() != Date::invalidYear) s << " birth = {" << a.birthD << "}\n";
  //  if(a.deathD.year() != Date::invalidYear) s << " death = {" << a.deathD << "}\n";
  if(a._life.from().year() != Date::invalidYear) s << " birth = {" << a._life.from() << "}\n";
  if(a._life.to().year() != Date::invalidYear) s << " death = {" << a._life.to()<< "}\n";
  if(!a.affiliations.empty())
    {
      list<Affiliation>::const_iterator tmp;
      for(tmp = a.affiliations.begin(); tmp != a.affiliations.end(); ++tmp)
	s << " Affiliation = {" << *tmp << "}\n"; 
    }
  if(a.firstPubl != Date::invalidYear) s << " firstPublication = {" << a.firstPubl << "}\n";
  else s << "% firstPublication = {" << a.firstPubl << "}\n";
  if(a.lastPubl != Date::invalidYear) s << " lastPublication = {" << a.lastPubl << "}\n";
  if(!a.publications.empty())
    {
      list<d_Ref<Publication> >::const_iterator tmp;
      for(tmp = a.publications.begin(); tmp != a.publications.end(); ++tmp)
	s << " Publication = {" << tmp->persistentID() << "}\n"; 
    }
  s << "}\n";
  return s;
}

#include <stdio.h> 

string Author::persistentID(void) const
{
  string s("Author::");
  s += fullname();
  s += "_";
  //if(birthD.year() != Date::invalidYear){s+= string(birthD);}
  //  s+= string(birthD);
  s+= string(_life.from());
  s += "_";
  //  if(deathD.year() != Date::invalidYear){char y[5]; sprintf(y, "%d", deathD.year()); s+= y;}
  //if(deathD.year() != Date::invalidYear){s+= string(deathD);}
  //  s+= string(deathD);
  s+= string(_life.to());
  return s;
}

void Author::birthYear(int y)
{ 
  if(Date::invalidYear == y) return;
  //  int yr = birthD.year();
  //  if(Date::invalidYear == yr || y < yr) birthD.year(y);
  int yr = _life.from().year();
  if(Date::invalidYear == yr || y < yr) _life.from().year(y);
}

void Author::deathYear(int y)
{
  if(Date::invalidYear == y) return;
  //  int yr = deathD.year();
  //  if(Date::invalidYear == yr || y > yr) deathD.year(y);
  int yr = _life.to().year();
  if(Date::invalidYear == yr || y > yr) _life.to().year(y);
}


void Author::year(int y)
{
  if(y == Date::invalidYear) return;

  if(firstPubl == Date::invalidYear || y < firstPubl) firstPubl = y;
  if(lastPubl == Date::invalidYear || y > lastPubl) lastPubl = y;
  /*
  int yr = deathD.year();
  if(Date::invalidYear == yr || y > yr) deathD.year(y);
  yr = birthD.year();
  y -= earliestPublicationAt;
  if(Date::invalidYear == yr || y < yr) birthD.year(y);
  */
  int yr = _life.to().year();
  if(Date::invalidYear == yr || y > yr) _life.to().year(y);
  yr = _life.from().year();
  y -= earliestPublicationAt;
  if(Date::invalidYear == yr || y < yr) _life.from().year(y);
}

Author* Author::resolve(Resolver &r)
{
  string s(persistentID());

  hash_map<string, list<Author*>, Hash<string> >::iterator auiter = r.au.find(s);
  if(auiter != r.au.end()){
    list<Author*>::iterator iter = auiter->second.begin();
    list<Author*>::iterator end = auiter->second.end();
    while(iter != end){
      if(this == *iter)
	return this;
      else if((*iter)->merge(this))
      	return *iter;
      ++iter;
    }
  }

  r.au[s].push_back(this);
  r.newAuthors.push_back(this);

  hash_map<string, list<d_Ref<Author>*>, Hash<string> >::iterator diter = r.uau.find(s);

  if(diter != r.uau.end()){
    register list<d_Ref<Author>*>::iterator iter = diter->second.begin();
    register list<d_Ref<Author>*>::iterator end = diter->second.end();
    register list<d_Ref<Author>*>::iterator current;
    while(iter != end){
      current = iter;
      ++iter;
      if(s == (*current)->persistentID() && NULL == (*current)->ptr()){
	(*current)->ptr(this);
	diter->second.erase(current);
      }
    }
  }

  bool ret = true;

  if(!publications.empty()){
    list<d_Ref<Publication> >::iterator iter = publications.begin();
    list<d_Ref<Publication> >::iterator end = publications.end();
    while(iter != end){
      Publication *p = iter->ptr();
      if(NULL != p){
	Publication *q = p->resolve(r);
	if(q != p){
	  iter->ptr(q);
	}
      }
      else
	(*iter).resolve(r.pub, r.upub);
      ++iter;
    }
  }

  return this;
}

Author* Author::resolve(hash_map<string, list<NameRecord*>, Hash<string> > &nr,
			hash_map<string, list<Annotation*>, Hash<string> > &an,
			hash_map<string, list<Citation*>, Hash<string> > &ci,
			hash_map<string, list<Publication*>, Hash<string> > &pub,
			hash_map<string, list<Author*>, Hash<string> > &au,
			hash_map<string, list<d_Ref<NameRecord>*>, Hash<string> > &unr,
			hash_map<string, list<d_Ref<Annotation>*>, Hash<string> > &uan,
			hash_map<string, list<d_Ref<Citation>*>, Hash<string> > &uci,
			hash_map<string, list<d_Ref<Publication>*>, Hash<string> > &upub,
			hash_map<string, list<d_Ref<Author>*>, Hash<string> > &uau)
{
#if 1
  hash_map<string, list<NRnode*>, Hash<string> > nrn;
  Resolver r(nr, an, ci, pub, au, nrn);
  Author *ret = resolve(r);
  unr = r.unr;
  uan = r.uan;
  uci = r.uci;
  upub = r.upub;
  uau = r.uau;
  return ret;
#else
  string s(persistentID());

  hash_map<string, list<Author*>, Hash<string> >::iterator auiter = au.find(s);
  if(auiter != au.end()){
    list<Author*>::iterator iter = auiter->second.begin();
    list<Author*>::iterator end = auiter->second.end();
    while(iter != end){
      if(this == *iter)
	return NULL;
      else if((*iter)->merge(this))
      	return *iter;
      ++iter;
    }
  }
  au[s].push_back(this);


  hash_map<string, list<d_Ref<Author>*>, Hash<string> >::iterator diter = uau.find(s);

  if(diter != uau.end()){
    list<d_Ref<Author>*>::iterator iter = diter->second.begin();
    list<d_Ref<Author>*>::iterator end = diter->second.end();
    while(iter != end){
      if(s == (*iter)->persistentID() && NULL == (*iter)->ptr())
	(*iter)->ptr(this);
      ++iter;
    }
  }

  bool ret = true;

  if(!publications.empty()){
    list<d_Ref<Publication> >::iterator iter = publications.begin();
    list<d_Ref<Publication> >::iterator end = publications.end();
    while(iter != end){
      if(NULL != iter->ptr()){
	Publication *p = (*iter)->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau);
	if(NULL != p){
	  //delete *iter!
	}
      }
      else
	(*iter).resolve(pub, upub);
      ++iter;
    }
  }

  return NULL;
#endif
}


bool Author::merge(Author *a)
{
  static const bool refuse = false;
  bool ret = false;
  if(this == a)
    return refuse;
  if(persistentID() == a->persistentID()){
    Author empty;
    if(persistentID().length() < empty.persistentID().length())
      return refuse;

    //must merge...
    affiliations.sort();
    a->affiliations.sort();
    affiliations.merge(a->affiliations);

    publications.sort();
    a->publications.sort();
    publications.merge(a->publications);
    return !refuse;
  }
  if(fullname() == a->fullname()){
    //    Period p(birthD, deathD);
    //    Period ap(a->birthD, a->deathD);
    //    if(p.exclusive(ap))
    if(_life.exclusive(a->_life))
      return refuse;
    return !refuse;
  }

  return ret;
}

int epithetCode(const char* e)
{
  int ret = strlen(e);
  if(ret != 0){
    if(0 == strncasecmp(e, "sen", 3))
      ret = 0;
    else if(0 == strncasecmp(e, "jr", 2))
      ret = 2;
  }
  else
    ret = 1;
  return ret;
}

bool operator<(const Author &a1, const Author &a2)
{

  int strc = strcasecmp(a1._surname, a2._surname);
  if(strc != 0)
    return strc < 0;

  strc = strcasecmp(a1._epithet, a2._epithet);
  if(strc != 0)
    return epithetCode(a1._epithet) < epithetCode(a2._epithet);

  strc = strcasecmp(a1.fullname().c_str(), a2.fullname().c_str());
  if(strc != 0)
    return strc < 0;
  /*
  if(a1.birthD != a2.birthD)
    return a1.birthD < a2.birthD;

  return a1.deathD < a2.deathD;
  */
  if(a1._life.from() != a2._life.from())
    return a1._life.from() < a2._life.from();

  return a1._life.to() < a2._life.to();
}
