/*
 *
 * Author.h: a definition of persistent Author class
 * for a Nomenclature Heuristic Model.
 * It may be called also N3, Nomenclature Netowrok Navigator
 * or more simply, MkII.
 *
 * Copyright (c) 1999 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: Author.h,v 1.6 1999/12/05 18:20:25 nozomi Exp $
 *	$Log: Author.h,v $
 *	Revision 1.6  1999/12/05 18:20:25  nozomi
 *	improve redundant include guard
 *	
 *	Revision 1.5  1999/09/23 06:28:15  nozomi
 *	resolve method using Resolver class
 *	
 *	Revision 1.4  1999/08/23 06:41:47  nozomi
 *	add abbrname method
 *	fullname is inline
 *	
 *	Revision 1.3  1999/03/24 11:04:01  nozomi
 *	Auhtor(const string&) add
 *	
 *	Revision 1.2  1999/03/21 13:44:52  nozomi
 *	merge(Author*) add
 *	
 *	Revision 1.1  1999/03/15 16:42:46  nozomi
 *	gcc 2.7 support macro
 *	
 *	Revision 1.0.0  1999/03/14 02:15:55  nozomi
 *	Initial version, though aka MkII
 *	
 *
 */

//
// The class Author contains information to identify author of Publication.
// It is prepared to be used the third alternative naivigation thread, 
// but not yet used extensively mainly because too many information is
// required to use.  It has parsed author name, Date of birth and death,
// year of first and last publication and list of Publications.
// The auther thread may be handled like biological entity via classes
// NameRecord, Annotation, Citation and Publication because name of
// an author appears like nomenclature name.  Publication like Who's who
// may be regarded as revision work.  However, current implementation of
// the NHM does not use this approach for three reasons: 1. the NHM is
// at phase of notion proving implemaentation, and hence the netowaork
// navigation should be restricted one region until its usufulness proven;
// 2. author thread is an backup threading and it is not (yet) improtant 
// as Annotation thread is, and 3. avoid unnecessary overhead derived from
// over generalization, i.e. an practical reason.
//

#if (!defined(_INCLUDED_AUTHOR_H))
#define _INCLUDED_AUTHOR_H

#include "d_Object.h"
#include "d_Ref.h"
#include "Period.h"
#include "Date.h"
#include "Affiliation.h"
#include "NamedField.h"

#if (__GNUC_MINOR__ > 7)
#include <string>
#include <list>
#else
#include <string.h>
#include <list.h>
#endif

class Publication;
class NameRecord;
class Annotation;
class Citation;
class Resolver;

class Author:public d_Object
{
public:
  static const int earliestPublicationAt = 10; //years old.... XXX

  //constructors
  Author();
  Author(const Author&);
  Author& operator= (const Author&);

  Author(const char* s);
  Author(const string &s);  //construct from persistentID
  Author(const char* a, int y);
  Author(const char* a, Date b, Date d);
  Author(const NamedField* nf, const char* fn = "Author");
  Author(const NamedField& nf, const char* fn = "Author");

  //destructor
  ~Author(void);

  //I/O
  friend istream& operator>>(istream & s, Author& a);
  friend ostream& operator<<(ostream & s, Author& a);
  ostream& printTo(ostream & os) const;

  //accessors
  const char* surname(void) const;
  string fullname(void) const;
  string fullname(char c) const;
  string abbrname(void) const;
  string abbrname(char c) const;
  string initializedName(void) const;
  Date birth(void) const;
  Date death(void) const;
  Period life(void) const;
  int birthYear(void) const;
  int deathYear(void) const;
  int firstPublicationYear(void) const;
  int lastPublicationYear(void) const;
  //  vector<Publication*>* publications(void) const;

  //manipulators
  void surname(const char*);
  void fullname(const char*);
  Date birth(const Date &);
  Date death(const Date &);
  Period life(const Period &);
  void year(const int);
  void birthYear(const int);
  void deathYear(const int);
  int first(const int);
  int last(const int);

  Author* resolve(Resolver &);
  Author* resolve(hash_map<string, list<NameRecord*>, Hash<string> > &nr,
		  hash_map<string, list<Annotation*>, Hash<string> > &an,
		  hash_map<string, list<Citation*>, Hash<string> > &ci,
		  hash_map<string, list<Publication*>, Hash<string> > &pub,
		  hash_map<string, list<Author*>, Hash<string> > &au,
		  hash_map<string, list<d_Ref<NameRecord>*>, Hash<string> > &unr,
		  hash_map<string, list<d_Ref<Annotation>*>, Hash<string> > &uan,
		  hash_map<string, list<d_Ref<Citation>*>, Hash<string> > &uci,
		  hash_map<string, list<d_Ref<Publication>*>, Hash<string> > &upub,
		  hash_map<string, list<d_Ref<Author>*>, Hash<string> > &uau);

  bool merge(Author *a);

  string persistentID(void) const;

  list<Affiliation> affiliations;
  list<d_Ref<Publication> > publications;

  friend bool operator<(const Author &, const Author &);

private:
  static const char * const _classname = "Author";
  char* _title;
  char* _firstname;
  char* _middlename;
  char* _surname;
  char* _epithet; // Jr., Sen,  etc.
  //  Date birthD, deathD;
  Period _life;
  int firstPubl, lastPubl;

  void parseName(const char* s, int lim = 0);
  void _Author(const NamedField *nf, const char* fn);
};

inline ostream& Author::printTo(ostream & os) const { return os << *(Author*)this; }

#include "Publication.h"
#include "NameRecord.h"
#include "Annotation.h"
#include "Citation.h"
#include "Resolver.h"

//I/O
inline istream& operator>>(istream & s, Author &a)
{ NamedField nf; s >> nf; a = Author(nf); return s;}

//accessors
inline const char* Author::surname(void) const {return _surname;}

inline Date Author::birth(void) const
{ return _life.from()/*birthD*/;}

inline Date Author::death(void) const
{ return _life.to()/*deathD*/;}

inline int Author::birthYear(void) const
{ return /*birthD*/ _life.from().year();}

inline int Author::deathYear(void) const
{ return /*deathD*/_life.to().year();}

inline int Author::firstPublicationYear(void) const
{ return firstPubl;}

inline int Author::lastPublicationYear(void) const
{ return lastPubl;}

inline string Author::fullname(void) const
{ return fullname(' ');}

inline void Author::fullname(const char* f)
{ parseName(f);}

inline string Author::abbrname(void) const
{ return abbrname(' ');}

inline Date Author::birth(const Date &b)
{ Date r = /*birthD*/_life.from(); /*birthD = b*/_life.from(b); return r;}

inline Date Author::death(const Date &b)
{ Date r = /*deathD*/_life.to(); /*deathD = b*/_life.to(b); return r;}

//inline bool operator!=(Author& a, Author& b){ return !(a == b); }

#endif
