/*
 *
 * Date.h: a definition of Date class
 * for a Nomenclature Heuristic Model.
 * It may be called also N3, Nomenclature Netowrok Navigator
 * or more simply, MkII.
 *
 * Copyright (c) 1999 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: Date.h,v 1.2 1999/12/05 18:20:37 nozomi Exp $
 *	$Log: Date.h,v $
 *	Revision 1.2  1999/12/05 18:20:37  nozomi
 *	improve redundant include guard
 *	
 *	Revision 1.1  1999/03/15 16:42:48  nozomi
 *	gcc 2.7 support macro
 *	
 *	Revision 1.0.0  1999/03/14 02:16:27  nozomi
 *	Initial version, though aka MkII
 *	
 *
 */

// Date class is simple date class.  Which regards Date::invalidYear as
// invalid year value, etc.

#if (!defined (__DATE_H))
#define __DATE_H
#include "d_common.h"
#include "NamedField.h"
#if (__GNUC_MINOR__ > 7)
#include <string>
#else
#include <string.h>
#endif

//assumptions:
//a month never longer than 63 days
//a year nver longer than 15 months
//none of year, month and date have value 0 as real value;
//     it is used as an `exception' value.

// Year: may be modified as in Stroustrup, 1997, 11.7.1

class Date{  //should be derived from persistend class

public:
  static const int Date::invalidYear = 0;
  static const int Date::invalidMonth = 0;
  static const int Date::invalidDay = 0;

  // constructors
  Date(int y= invalidYear, int m = invalidMonth, int d = invalidDay);
  Date(const char*, int lim = 0);
  Date(NamedField& nf, const char* s = "Date");
  Date(NamedField* nf, const char* s = "Date");
  Date(const Date&);
  Date& operator= (const Date&);
  operator string() const;

  // I/O
  friend istream& operator>>(istream & s, Date &d);
  friend ostream& operator<<(ostream & s, const Date &d);

  //accessors
  int year(void) const;
  int month(void) const;
  int day(void) const;

  //manipulators
  int year(int);
  int month(int);
  int day(int);

  //boolians
  bool valid(void) const;
  bool invalid(void) const;

  friend bool operator == (Date a, Date b);
  friend bool operator != (Date a, Date b);
  friend bool operator >= (Date a, Date b);
  friend bool operator <= (Date a, Date b);
  friend bool operator > (Date a, Date b);
  friend bool operator < (Date a, Date b);

private:
  int yr;
  int mon;
  int dy;
  static const int Date::bitD = 6;
  static const int Date::bitMD = 10;
  void _Date(const char*, int lim = 0);
  void _Date(NamedField*, const char*);
};

inline Date::Date(int y, int m, int d):yr(y), mon(m), dy(d){}

inline Date::Date(const Date& d):yr(d.yr), mon(d.mon), dy(d.dy){}

inline Date & Date::operator= (const Date& d)
{yr = d.yr; mon = d.mon; dy = d.dy; return *this;}

inline Date::Date(const char* s, int lim)
  :yr(invalidYear), mon(invalidMonth), dy(invalidDay){_Date(s, lim);}

inline Date::Date(NamedField &nf, const char* fn)
  :yr(invalidYear), mon(invalidMonth), dy(invalidDay) {_Date(&nf, fn);}

inline Date::Date(NamedField *nf, const char* fn)
  :yr(invalidYear), mon(invalidMonth), dy(invalidDay) {_Date(nf, fn);}

inline istream& operator>>(istream & s, Date &d)
{ NamedField nf; s >> nf; d = Date(nf); return s;}

inline int Date::year(void) const { return yr; }

inline int Date::month(void) const { return mon; }

inline int Date::day(void) const { return dy; }

inline bool operator == (Date a, Date b)
{ return a.year() == b.year() && a.month() == b.month() && a.day() == b.day();}

inline bool operator != (Date a, Date b) { return !(a == b); }

inline bool operator < (Date a, Date b) 
{ return ((a.year() << Date::bitMD) | (a.month() << Date::bitD) | 
	  a.day()) < ((b.year() << Date::bitMD) | ( b.month() << Date::bitD) | b.day());}

inline bool operator > (Date a, Date b)
{
  return ((a.year() << Date::bitMD) | (a.month() << Date::bitD) | a.day()) > ((b.year() << Date::bitMD) | ( b.month() << Date::bitD) | b.day());
}

inline bool operator <= (Date a, Date b) { return !(a > b); }

inline bool operator >= (Date a, Date b) { return !(a < b); }


inline int Date::year(int y) 
{ 
  int ret = yr;
  yr = y;
  return ret;
}

inline int Date::month(int m)
{
  int ret = mon;
  mon = m;
  return ret;
}

inline int Date::day(int d)
{
  int ret = dy;
  dy = d;
  return ret;
}

inline bool Date::valid(void) const
{
  if(yr != Date::invalidYear || mon != Date::invalidMonth || dy != Date::invalidDay)
    return true;
  else
    return false;
}
inline bool Date::invalid(void) const
{
  return !valid();
}

#endif
