/*
 *
 * NameRecord.cxx: an implementation of persistent NameRecord class
 * for a Nomenclature Heuristic Model.
 * It may be called also N3, Nomenclature Netowrok Navigator
 * or more simply, MkII.
 *
 * Copyright (c) 1999 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: NameRecord.cxx,v 1.20 1999/12/05 18:24:21 nozomi Exp $
 *	$Log: NameRecord.cxx,v $
 *	Revision 1.20  1999/12/05 18:24:21  nozomi
 *	improve merge code
 *	
 *	Revision 1.19  1999/09/20 14:18:21  nozomi
 *	modification relating to data addition
 *	
 *	Revision 1.18  1999/08/31 12:14:19  nozomi
 *	modifications relating change in Annotation (from set to list)
 *	
 *	Revision 1.17  1999/08/18 13:36:39  nozomi
 *	methods year and page.
 *	d_Ref<T> safer.
 *	
 *	Revision 1.16  1999/08/16 08:21:29  nozomi
 *	persistentID() calling bug fix
 *	
 *	Revision 1.15  1999/08/16 06:57:05  nozomi
 *	PID convertor name changed
 *	
 *	Revision 1.14  1999/08/16 06:52:56  nozomi
 *	persistentID format varsion 2
 *	
 *	Revision 1.13  1999/08/09 17:23:56  nozomi
 *	spell correction
 *	
 *	Revision 1.12  1999/08/09 07:30:34  nozomi
 *	correct spelling in an output field
 *	
 *	Revision 1.11  1999/08/09 05:49:43  nozomi
 *	lowerNR reading code modified
 *	
 *	Revision 1.10  1999/08/08 11:36:17  nozomi
 *	delete unused codes
 *	
 *	Revision 1.9  1999/08/08 04:18:00  nozomi
 *	Methods top and bottom were implemented to support hierarhy navigation
 *	
 *	Revision 1.8  1999/08/06 07:38:35  nozomi
 *	add a constructor
 *	
 *	Revision 1.7  1999/03/21 22:33:26  nozomi
 *	_mergeAnnotation() modified to check NULL ptr()
 *	
 *	Revision 1.6  1999/03/20 19:56:08  nozomi
 *	generic persistentID trial
 *	
 *	Revision 1.5  1999/03/20 16:09:48  nozomi
 *	Data merge code (trial).
 *	
 *	Revision 1.4  1999/03/15 13:14:27  nozomi
 *	volume and number inconsistency removed.
 *	input record sensitivity hopefully reduced.
 *	
 *	Revision 1.3  1999/03/15 11:47:02  nozomi
 *	_authorWithYear() fix
 *	
 *	Revision 1.2  1999/03/15 11:17:16  nozomi
 *	persistentID format was changed
 *	
 *	Revision 1.1  1999/03/14 18:06:01  nozomi
 *	fix persistentID(void) related
 *	
 *	Revision 1.0  1999/03/14 02:17:18  nozomi
 *	Initial version, though aka MkII
 *	
 *
 */

#include "d_Ref.h"
#include "NameRecord.h"
#include "NamedField.h"
#include "Annotation.h"
#include <stdlib.h>
#include <stdio.h>

//local utilities
//static Publication* _Publication(const string& a, const string& y);
static string _surname(list<d_Ref<Author> >::const_iterator da);
static string _surnames(const Citation& c);
static int _year(const Citation& c);

NameRecord::NameRecord(const NamedField *nf, const char* fn)
  : d_Object(_classname),
    annotation(),
    lowerNR(),
    _genericName(strdup()),
    _specificName(strdup()),
    _hierarchy(strdup()),
    _authority(), _recorder(),
    _type(), _typeOf(), _higherNR()/*,  _anyUse()*/
{_NameRecord(nf, fn);}

NameRecord::NameRecord(const NamedField &nf, const char* fn)
  : d_Object(_classname),
    annotation(),
    lowerNR(),
    _genericName(strdup()),
    _specificName(strdup()),
    _hierarchy(strdup()),
    _authority(), _recorder(),
    _type(), _typeOf(), _higherNR()/*,  _anyUse()*/
{_NameRecord(&nf, fn);}

NameRecord::~NameRecord(void)
{
  delete[] _genericName; delete[] _specificName; delete[] _hierarchy;
}

istream& operator>>(istream & s, NameRecord &r)
{ 
  NamedField nf;
  s >> nf; 

  //cleaning up lists
  r.lowerNR.clear();
  r.annotation.clear();

  r._NameRecord(&nf, "NameRecord"); 

  return s;
}

void NameRecord::_NameRecord(const char* s)
{
  if(0 ==  *s) return;

  int l = strlen(s);
  int i = 0;
  const char* f = s;
  int words = 0;

  while(*f){
    while(*f && isspace(*f)) f++;
    words++;
    while(*f && !isspace(*f)) f++;
    while(*f && isspace(*f)) f++;
  }

  if(0 == words) return;

  char separator = 0;
  for(i = 0, f = s; i < l && !separator; f++, i++)
    {
      if(!isspace(*f) && !isalnum(*f))
	{
	  switch(*f){
	  case '.': case ',': case ':': case ';':
	  case '(': case '<': case '[': case '{':
	  case ')': case '>': case ']': case '}':
	  case '&':
	    break;
	  default:
	    separator = *f;
	  }
	}
    }

  f = s;
  while(*f && isspace(*f)) f++;
  if(0 == separator){
    /* if(1 != words) throw something */
    //optimistic
    delete [] _specificName;
    _specificName = new_strdup::strdup(f);
    return;
  }

  while(*f){
    if(separator == *f)
      words++;
  }

  for(int j = 0; j < words; j++){
    const char* e = f;
    while(separator != *e) e++;
    char* dst = new char[e - f + 1];
    strncpy(dst, f, e-f);
    f = e + 1;
    switch(j)
      {
      case 4:
	//recorder
	break;
      case 3:
	//authority
	break;
      case 2:
	delete [] _specificName;
	_specificName = dst;
	break;
      case 1:
	delete [] _genericName;
	_genericName = dst;
	break;
      case 0:
	delete [] _hierarchy;
	_hierarchy = dst;
	break;
      }
  }
}
/*
struct _annotationEntries{const char* str; size_t n;};
static _annotationEntries _ae[] = {
  {"refer", 3},      // 1 to 1
  {"revise", 3},     // 1 to 1
  {"synonymize", 3}, // 1 to n
  {"homonimyze", 3}, // n to n
  {"part", 4},       // n to 1
  {"equivate", 2},   // n to zero 
  {"nov", 3},        // 1 to zero 
  {NULL,0}
};
*/
struct _strEntrieds{const char* fn; char** f;int n;};
struct _hier{const char* fn; d_Ref<NameRecord> *d;};

void NameRecord::_NameRecord(const NamedField *nf, const char* fn)
{
  if(strcmp(nf->entryName(), fn)) {_NameRecord("");return;}
  const char* s;
  if(strlen(s = nf->strcmp(fn))){_NameRecord(s); return;}

  delete[] _genericName; delete[] _specificName; delete[] _hierarchy;

  _strEntrieds _nf[]= {
    {"group", &_hierarchy, 5},
    {"generic", &_genericName, 5},
    {"specific", &_specificName, 5},
    {NULL,NULL}
  };
  
  _strEntrieds *_pnf = _nf;
  
  while(_pnf->fn){
    if(NULL != (s = nf->strncasecmp(_pnf->fn, _pnf->n))){
      *(_pnf->f) = new_strdup::strdup(s);
    }
    else
      *(_pnf->f) = strdup();
    _pnf++;
  }

  Resolver r;

  if(NULL != (s = nf->strcasecmp("authority"))){
    list<NamedField>::const_iterator tmp;
    bool found = false;
    for(tmp = nf->subEntries.begin(); !found && tmp != nf->subEntries.end(); ++tmp){
      if(!strcasecmp(tmp->entryName(), "authority")){
#if 0
	_authority = ;new Citation(*tmp, "authority");
#else
	Citation *p = new Citation(*tmp, "authority");

	Citation *q = p->resolve(r);
	if(q != p)
	  delete p;
	_authority = q;
#endif
	found = true;
      }
    }
  }

  if(NULL != (s = nf->strcasecmp("recorder"))){
    list<NamedField>::const_iterator tmp;
    bool found = false;
    for(tmp = nf->subEntries.begin(); !found && tmp != nf->subEntries.end(); ++tmp)
      if(!strcasecmp(tmp->entryName(), "recorder")){
#if 0
	_recorder = new Citation(*tmp, "recorder");
	_recorder->namerecords.push_back(this);
#else
	Citation *p = new Citation(*tmp, "recorder");
	p->namerecords.push_back(this);

	Citation *q = p->resolve(r);
	if(q != p)
	  delete p;
	_recorder = q;
#endif

	found = true;
      }
  }

  _hier _h[] = {
    {"type", &_type},
    {"typeOf", &_typeOf},
    {"higher", &_higherNR},
    {NULL,NULL}
  };

  _hier *h = _h;
  while(h->fn){ //types
    if(NULL != (s = nf->strcasecmp(h->fn))){
      list<NamedField>::const_iterator tmp;
      bool found = false;
      for(tmp = nf->subEntries.begin(); !found && tmp != nf->subEntries.end(); ++tmp){
	if(!strcmp(tmp->entryName(), h->fn)){
	  found = true;
	  s = tmp->strcasecmp(h->fn);
	  if(strlen(s)){
	    if(!strncmp(s, "NameRecord::", 12)){
	      //	      NameRecord dnr(s);
	      //	      (h->d)->persistentID(dnr.persistentID());
	      
	      //(h->d)->persistentID(s);
	      (h->d)->persistentID(NameRecord::persistentID(s));
	    }
	    else{
	      *(h->d) = new NameRecord(s);
	    }
	  }
	  else{
	    *(h->d) = new NameRecord(*tmp);
	  }
	}
      }
    }
    h++;
  }

  //lower groups
  if(NULL != (s = nf->strncasecmp("lower",5))){
    list<NamedField>::const_iterator tmp;
    for(tmp = nf->subEntries.begin(); tmp != nf->subEntries.end(); ++tmp){
#if 0
      if(NULL != (s = tmp->strcasecmp("lowerGroup"))){
	if('_' == *s){
	  d_Ref<NameRecord> rnr;
	  rnr.persistentID(s);
	  lowerNR.push_back(rnr);
	}
	else{
	  lowerNR.push_back(d_Ref<NameRecord>(new NameRecord(*tmp)));
	}
      }
#else
      if(!strcmp(tmp->entryName(), "lower")){
	s = tmp->strcasecmp("lower");
	if(strlen(s)){
	  if(!strncmp(s, "NameRecord::", 12)){
	    d_Ref<NameRecord> rnr;
	    //NameRecord dnr(s);
	    //rnr.persistentID(dnr.persistentID());
	    //rnr.persistentID(s);
	    rnr.persistentID(NameRecord::persistentID(s));
	    lowerNR.push_back(rnr);
	  }
	  else{
	    lowerNR.push_back(new NameRecord(s));
	  }
	}	  
	else{
	  lowerNR.push_back(d_Ref<NameRecord>(new NameRecord(*tmp)));
	}
      }
#endif
    }
  }

  //annotations
  const Annotation::annoTuple *anno = Annotation::tuple;
  while(anno->str){
    list<NamedField>::const_iterator tmp;
    for(tmp = nf->subEntries.begin(); tmp != nf->subEntries.end(); ++tmp){
      if(!strncmp(tmp->entryName(), anno->str, anno->n)){
	s = tmp->contents();
	if(NULL != s && !strncmp(s, "Annotation::", 12)){
	    d_Ref<Annotation> *da = new d_Ref<Annotation>;
	    da->persistentID(s);
	    annotation.push_back(*da);
	  }
	else{
	  annotation.push_back(d_Ref<Annotation>(new Annotation(this, *tmp, anno->str, anno->n)));
	}
      }
    }
    anno++;
  }
} 

ostream& operator<<(ostream & s, NameRecord& nr)
{
  s << "NameRecord = {\n";
  s << "%"<<&nr<<endl;
  s << " persistentID = {" <<  nr.persistentID() << "}\n";
#if 0
  s << "% persistentID = {" <<  nr.persistentID(3) << "}\n";
#endif

  if(strlen(nr._hierarchy)) s << " group = {" << nr._hierarchy << "}\n";
  if(strlen(nr._genericName)) s << " genericname = {" << nr._genericName << "}\n";
  if(strlen(nr._specificName)) s << " specificname = {" << nr._specificName << "}\n";
#if 0
  if(nr._authority.publication()->persistentID().length()){
    s << " authority = {";
    if((nr._authority).page()){
      s << "\n  Publication = {" << ((nr._authority).publication())->persistentID() << "}\n";
      s << "  page = {" << (nr._authority).page() << "}\n ";
    }
    else
      s << ((nr._authority).publication())->persistentID();
    s << "}\n";
  }
#else
  if(!!nr._authority && ((nr._authority)->persistentID()).length()){
    s << " authority = {" << (nr._authority)->persistentID() << "}\n";
  }
#endif
#if 0
  if(((nr._recorder).publication()->persistentID()).length()){
    s << " recorder = {";
    string str;
    if((nr._recorder).page()){
	s << "\n  Publication = {" << ((nr._recorder).publication())->persistentID() << "}\n";
	s << "  page = {" << (nr._recorder).page() << "}\n ";
    }
    else
      s << ((nr._recorder).publication())->persistentID();
    s << "}\n";
  }
#else
  if(!!nr._recorder && ((nr._recorder)->persistentID()).length()){
    s << " recorder = {" << (nr._recorder)->persistentID() << "}\n";
  }
  //  if(nr._recorder.persistentID().length()){
  //      s << " recorder = {" << nr._recorder.persistentID() << "}\n";
  //  }
#endif  
  if(0 != (nr._type).persistentID().length()){
    s << " type = {" << (nr._type).persistentID() << "}\n";
  }
  if(0 != (nr._typeOf).persistentID().length()){
    s << " typeOf = {" << (nr._typeOf).persistentID() << "}\n";
  }
  if(0 != (nr._higherNR).persistentID().length()){
    if(NULL == (nr._higherNR).ptr())
      s << "%unsolved\n";
    s << " higher = {" << (nr._higherNR).persistentID() << "}\n";
  }

  if(!nr.lowerNR.empty())
    {
      list<d_Ref<NameRecord> >::const_iterator tmp;
      for(tmp = (nr.lowerNR).begin(); tmp != (nr.lowerNR).end(); ++tmp){
	s << " lower = {" << tmp->persistentID() << "}\n";
      }
    }

  if(!nr.annotation.empty())
    {
      list<d_Ref<Annotation> >::const_iterator tmp;
      for(tmp = (nr.annotation).begin(); tmp != (nr.annotation).end(); ++tmp){
	if(NULL != tmp->ptr())
	  {
	  s << " " << (*tmp)->annotation() << " = {" << tmp->persistentID() << "}\n";
	  s << flush;
	  }
	else{
	  string str(tmp->persistentID());
	  int l = str.length();
	  int i = 0, j;
	  while(i < l && ':' != str[i]){i++;}
	  j = i += 2;
	  while(j < l && '_' != str[j]){j++;}
	  s << " " << string(str, i, j - i ) << " = {" << str << "}\n";
	s << flush;
	}
      }
    }

  s << "}\n";

  return s;
}

static string _surname(list<d_Ref<Author> >::const_iterator da)
{
  if(da->ptr())
    return (*da)->surname();
  string s(da->persistentID(), 8 /* strlen("Author::") */);
  int i;
  int l = s.length();
  for(i = 0; '_' != s[i] && '.' != s[i] && ',' != s[i] && !isspace(s[i]) && i < l; i++){}
  return string(s, 0, i);
}
#if 0
static string _surnames(const Citation& c)
{
  string s;
  if(0 == c.publication()->persistentID().length()) return string("");
  if(c.publication()->ptr()){
    string s;
    if(((*c.publication()))->authorlist.empty())
      { s += (*(c.publication()))->authors();}
    else{
      list<d_Ref<Author> >::const_iterator cia = (*(c.publication()))->authorlist.begin();
      s += _surname(cia);
      while( ++cia != (*(c.publication()))->authorlist.end()){
	s += " ";
	s += _surname(cia);
	//      s += (*cia)->surname();
      }
    }
    return s;
  }
  else{
    string s(c.publication()->persistentID());
    int i = 0;
    int l = s.length();
    while(i < l && ':' != s[i]){i++;}
    if(i == l) return string("");
    i += 2;
    int head = i;
    while(i < l && (isalnum(s[i]) || isspace(s[i]))){i++;}
    if(i == l) return string("");
    return string(s, head, i - head);
  }
}
#else
static string _surnames(const Citation& c)
{
  string s("");
  //  if(0 == c.publication()->persistentID().length()) return string("");
  if(0 == c.persistentID().length()){
    return s;
  }
  //  if((c.publication())->ptr()){
  if(c.publication().ptr()){
    //    string s;
    //    if(((*c.publication()))->authorlist.empty())
    if(c.publication()->authorlist.empty())
      { s += c.publication()->authors();}
    else{
      list<d_Ref<Author> >::const_iterator cia = c.publication()->authorlist.begin();
      list<d_Ref<Author> >::const_iterator cie = c.publication()->authorlist.end();
      s += _surname(cia);
      while( ++cia != cie){
	s += " ";
	s += _surname(cia);
	//      s += (*cia)->surname();
      }
    }
    return s;
  }
  else{
    //    string s(c.publication()->persistentID());
    string s(c.persistentID());
    int i = 0;
    int l = s.length();
    while(i < l && ':' != s[i]){i++;}
    if(i == l) return string("");
    i += 2;
    int head = i;
    while(i < l && (isalnum(s[i]) || isspace(s[i]))){i++;}
    if(i == l) return string("");
    return string(s, head, i - head);
  }
}
#endif

static int _year(const Citation& c)
{
  //  if(c.publication()->ptr())
  if(c.publication().ptr())
    return c.publication()->year();

  string s(c.publication()->persistentID());
  int i = 0;
  int l = s.length();
  while(i < l && ':' != s[i]){i++;}
  if(i == l) return Date::invalidYear;
  i += 2;
  int head = i;
  while(i < l && (isalnum(s[i]) || isspace(s[i]))){i++;}
  if(i == l) return Date::invalidYear;
  if('_' != s[i])
    while(i < l && '_' != s[i]){i++;}
  i++;
  while(i < l && '_' != s[i]){i++;}
  if(i == l) return Date::invalidYear;
  i++;
  head = i;
  while(i < l && '_' != s[i]){i++;}
  if(i == l) return Date::invalidYear;
  return atoi(string(s, head, i - head).c_str());
}

string NameRecord::_publicationStr(const d_Ref<Citation>& c) const
{
  string s("");
  string p("");  // for page

  // get publiction's PID directry if available
  if(NULL != c.ptr() && NULL != (c->publication()).ptr()){
    //    s = c->publication()->persistentID();
    s = c->publication().persistentID();
  }
  else{
    if(NULL != c.ptr()){
      s = c->persistentID();
    }
    else{
      s = c.persistentID();
    }
  }
  // if it is empty.... 
  int l = s.length();
  if(0 == l) return string("_______");
  register int i = 0;

  //peal things
  if(NULL == c.ptr() || NULL == (c->publication()).ptr())
    {
      i = 10; // strlen(Citation::);
      
      //skip appearance
      while(i < l && '_' != s[i]){++i;} ++i;
      
      int head = i;
      //skip page
      while(i < l && '_' != s[i]){++i;}
      //      ++i;
      if(i < l)
	p = string(s, head, i - head); 
    }
  // skip type info
  i += 13; //strlen("Publication::");
  
  string str(s, i);
  str += "_";
  if(0 == p.length()){
    if(NULL != c.ptr()){
      /*
      if(0 != c->page()){
	char pg[10];
	sprintf(pg, "%d", c->page());
	p= pg;
      }
      */
      p = c->page();
    }
    else{
      s = c.persistentID();
      l = s.length();
      i = 10; // strlen(Citation::);
      
      //skip appearance
      while(i < l && '_' != s[i]){++i;} ++i;
      int head = i;
      while(i < l && '_' != s[i]){++i;}
      if(i < l)
	p = string(s, head, i - head);
    }
  }
  str += p;
  return str;
}

string NameRecord::_authorWithYear(const d_Ref<Citation>& c) const
{
  if(NULL != c.ptr() && NULL != (c->publication()).ptr())
    {
      string ret(_surnames(*(c.ptr())));
      ret += "_";
      //      int yr = _year(*(c.ptr()));
      int yr = c->publication()->year();
      if(yr != Date::invalidYear)
	{ char y[5]; sprintf(y, "%d", yr); ret += y; }
      return ret;
    }
  
  string s("");
#if 1
  if(NULL != c.ptr()){
    //    s = "Citation::__";
    s == c->publication().persistentID();
  }
  else
#endif
    s = c.persistentID();
  int l = s.length();
  if(0 == l) return string("_");

  register int i = 0;

  string str("");
#if 1
  if(NULL == c.ptr())
#endif    
  {
    i = 10; // strlen(Citation::);

    //skip appearance
    while(i < l && '_' != s[i]){++i;} ++i;

    //skip page
    while(i < l && '_' != s[i]){++i;}
    ++i;
  }

  // skip type info
  i += 13; //strlen("Publication::");

  //save begininng
  int head = i;

  //search authors end
  while(i < l &&  '_' != s[i]){++i;}
  str += string(s, head, i - head);

  //save year begininng
  head = i;
  ++i;
  while(i < l &&  '_' != s[i]){++i;}
  
  str += string(s, head, i - head);
  return str;
}

string NameRecord::_authorWithYear(const Citation& c) const
{
  //  if(NULL == c.publication()->ptr()) return string("_");
  //  if(0 == c.publication()->persistentID().length()) return string("_");
  if(0 == c.persistentID().length()) return string("_");
  string ret(_surnames(c));
  //  if(c.publication()->ptr()){
  if(c.publication().ptr()){
#if 0
    string ret;
    if((*(c.publication()))->authorlist.empty())
      { ret += (*(c.publication()))->authors();}
    else{
      list<d_Ref<Author> >::const_iterator cia = (*(c.publication()))->authorlist.begin();
      ret += _surname(cia);
      while( ++cia != (*(c.publication()))->authorlist.end()){
	ret += " ";
	ret += _surname(cia);
	//      ret += (*cia)->surname();
      }
    }
#else
#endif
    ret += "_";
    //    int yr = (*(c.publication()))->year();
    //    int yr = c.publication()->year();
    int yr = _year(c);
    if(yr != Date::invalidYear)
      { char y[5]; sprintf(y, "%d", yr); ret += y; }
    return ret;
  }
  else{
#if 1
    //    string s(c.publication()->persistentID());
    string s(c.persistentID());
    int i = 0;
    int l = s.length();
    while(i < l && ':' != s[i]){i++;}
    if(i == l) return string("_");
    i += 2;
    int head = i;
    while(i < l && (isalnum(s[i]) || isspace(s[i]))){i++;}
    if(i == l) return string("_");
#if 0
    string ret(s, head, i - head);
#endif
    ret += '_';
    if('_' != s[i])
      while(i < l && '_' != s[i]){i++;}
    i++;
    while(i < l && '_' != s[i]){i++;}
    if(i == l) return ret;
    i++;
    head = i;
    while(i < l && '_' != s[i]){i++;}
    if(i == l) return ret;
    ret += string(s, head, i - head);
    return ret;
#else
#endif
  }
}

/*
 *
 * Naming, naming, naming!  Nomenclature is a system of naming!
 *
 */
/*
 *
 * output formats of NameRecord::persistentID():
 *
 * ver. 0: NameRecord::group_generic_specific_author_yearA_recoder_yearR
 * ver. 1: NameRecord::group_generic_specific_author_yearA_recorderPublicationStr
 * ver. 2: NameRecord::group_generic_specific_recorderPublicationStr
 * ver. 3: NameRecord::group_generic_specific_authorityPublicationStr_rrecorderPublicationStr
 *
 * Note on ver.3 format:
 * Ver.1 format seemed rather redundant because a pair of NAME and CITATION is sufficient
 * to unique identification of name.  Hence authority part was eliminated from ver. 2.
 * However, it was found that a citation can contain two or more usage of a single name 
 * like NAME sensu X, NAME sensu Y and so on.  Here X or Y is conventionally citation
 * specifier as "Cavalier-Smith 1996", but it must not be confused as citation specifier.
 * It is a semantics specifier of NAME.  The persistentIDs of NameRecord can be summarized
 * as:
 * 
 * ver. 0: NAME_Authority_recorder
 * ver. 1: NAME_AuthorityCITATION_recorderCITATION
 * ver. 2: NAME_CITATION
 * ver. 3: NAME_CITATION;
 *         NAME = NAME(sensu ver.2)_semanticsSpecifier(AuthorityCIATION)
 *
 * Even though a pair of author and year has been used in taxonomy as the semantics specifier,
 * it is insufficient for Nomencurator because its uniqueness is insufficient, e.g. Hill 1996c.
 * Combination of author name and year with citation specifier (e.g. 1996 + c) is sufficient in
 * ordinary publications, but it is ambiguous for Nomencurator.  Even identifier of a Publication
 * is insufficient because a Publication may contain multiple usage of the name.  Appropriate 
 * portion of the original publication may be sufficient to specify what encoded by the NameRecord,
 * but it can be ambiguous.  Page where the description appeared has been used, but single page
 * may contain multiple usage.  As a compromise, page-like-string (e.g. page + figure number) can 
 * be used as precise specifier of CITATION.
 *
 */

string NameRecord::_trinityName(void) const
{
  string s(_classname);  
  s += "::";
  s += _hierarchy;
  s += "_"; s += _genericName;
  s += "_";  s += _specificName;
  return s;
}

string NameRecord::_persistentID(int version) const
{
  //for core part of NAME
  string s(_trinityName());
  //NAME semantics specifier (sensu version 3)
  switch(version){
  case 2:
    //empty
    break;
  default:
    s += "_";
    switch(version){
    case 0: case 1:
      s += _authorWithYear(_authority);
    case 3:
      s += _publicationStr(_authority);
      break;
    }
    break;
  }

  // CITATION part
  s += "_";
  switch(version){
  case 0:
    s += _authorWithYear(_recorder);
    break;
  default:
    s += _publicationStr(_recorder);
    break;
  }
  return s;
}

#if 0
string NameRecord::persistentID(void) const
{
  string s(_classname);
  s += "::";
  s += _hierarchy;
  s += "_"; s += _genericName;
  s += "_"; s += _specificName;
  s += "_"; 
#if 0  //no need anymore?
  s += _authorWithYear(_authority);
  s += "_";
#endif
#if 1 //new format
  /*
  if(NULL == _recorder.ptr())
    cerr << _recorder.persistentID()<<endl;
    */
  s += _publicationStr(_recorder);
#else
  s += _authorWithYear(_recorder);
#endif
  return s;
}
#endif

NameRecord::NameRecord(void)
  : d_Object(_classname),
    annotation(),
    lowerNR(),
    _genericName(strdup()),
    _specificName(strdup()),
    _hierarchy(strdup()),
    _authority(), _recorder(),
    _type(), _typeOf(), _higherNR()/*,  _anyUse()*/{}

NameRecord::NameRecord(const char *s)
  : d_Object(_classname),
    annotation(),
    lowerNR(),
    _genericName(strdup()),
    _specificName(strdup()),
    _hierarchy(strdup()),
    _authority(), _recorder(),
    _type(), _typeOf(), _higherNR()/*,  _anyUse()*/{_NameRecord(s);}

NameRecord::NameRecord(const string& s)
  : d_Object(_classname),
    annotation(),
    lowerNR(),
    _genericName(strdup()),
    _specificName(strdup()),
    _hierarchy(strdup()),
    _authority(), _recorder(),
    _type(), _typeOf(), _higherNR()/*,  _anyUse()*/{_NameRecord(s);}


NameRecord::NameRecord(const NameRecord& nr)
  : d_Object(_classname),
    annotation(nr.annotation),
    lowerNR(nr.lowerNR),
    _genericName(new_strdup::strdup(nr._genericName)),
    _specificName(new_strdup::strdup(nr._specificName)),
    _hierarchy(new_strdup::strdup(nr._hierarchy)),
    _authority(nr._authority), _recorder(nr._recorder),
    _type(nr._type), _typeOf(nr._typeOf), 
    _higherNR(nr._higherNR)/*,  _anyUse()*/ {}

NameRecord& NameRecord::operator=(const NameRecord& nr)
{
  annotation = nr.annotation;
  _genericName = new_strdup::strdup(nr._genericName);
  _specificName = new_strdup::strdup(nr._specificName);
  _hierarchy = new_strdup::strdup(nr._hierarchy);
  _authority = nr._authority;
  _recorder = nr._recorder;
  _type = nr._type;
  _typeOf = nr._typeOf;
  _higherNR = nr._higherNR;
#if 1
  lowerNR = nr.lowerNR;
  /*_anyUse = nr._anyUse;*/
#endif
  return *this;}

NameRecord::NameRecord
(Publication& r, 
 const char * specific, const char * generic, const char * group) 
  : d_Object(_classname),
    annotation(),
    lowerNR(),
    _genericName(new_strdup::strdup(generic)),
    _specificName(new_strdup::strdup(specific)),
    _hierarchy(new_strdup::strdup(group)),
    //    _authority(&a), _recorder(&r),
    _authority(new Citation), _recorder(new Citation(&r)),
    _type(), _typeOf(), _higherNR()/*,  _anyUse()*/
{
  _recorder->namerecords.push_back(d_Ref<NameRecord>(this));
}

NameRecord::NameRecord
(Publication& r, 
 const char * specific, const char * generic, const char * group, 
 Publication& a)
  : d_Object(_classname),
    annotation(),
    lowerNR(),
    _genericName(new_strdup::strdup(generic)),
    _specificName(new_strdup::strdup(specific)),
    _hierarchy(new_strdup::strdup(group)),
    //    _authority(&a), _recorder(&r),
    _authority(new Citation(&a)), _recorder(new Citation(&r)),
    _type(), _typeOf(), _higherNR()/*,  _anyUse()*/
{
  _recorder->namerecords.push_back(d_Ref<NameRecord>(this));
}

NameRecord::NameRecord
(Citation& r, 
 const char * specific, const char * generic, const char * group, 
 Citation& a)
  : d_Object(_classname),
    annotation(),
    lowerNR(),
    _genericName(new_strdup::strdup(generic)),
    _specificName(new_strdup::strdup(specific)),
    _hierarchy(new_strdup::strdup(group)),
    _authority(&a), _recorder(&r),
    _type(), _typeOf(), _higherNR()/*,  _anyUse()*/
{
  r.namerecords.push_back(d_Ref<NameRecord>(this));
}

NameRecord::NameRecord
(Citation *r, 
 const char * specific, const char * generic, const char * group, 
 Citation *a)
  : d_Object(_classname),
    annotation(),
    lowerNR(),
    _genericName(new_strdup::strdup(generic)),
    _specificName(new_strdup::strdup(specific)),
    _hierarchy(new_strdup::strdup(group)),
    _authority(a), _recorder(r),
    _type(), _typeOf(), _higherNR()/*,  _anyUse()*/
{
  r->namerecords.push_back(d_Ref<NameRecord>(this));
}

NameRecord::NameRecord
(d_Ref<Citation> &r, 
 const char * specific, const char * generic, const char * group, 
 d_Ref<Citation> &a)
  : d_Object(_classname),
    annotation(),
    lowerNR(),
    _genericName(new_strdup::strdup(generic)),
    _specificName(new_strdup::strdup(specific)),
    _hierarchy(new_strdup::strdup(group)),
    _authority(a), _recorder(r),
    _type(), _typeOf(), _higherNR()/*,  _anyUse()*/
{
  r->namerecords.push_back(d_Ref<NameRecord>(this));
}

#if 0
inline NameRecord::NameRecord
(const char* recorder, int recorded, 
 const char* specific, const char* generic, const char* group, 
 const char* authorizer, int authorized)
  : d_Object(_classname), 
    annotation(),
    lowerNR(),
    _genericName(new_strdup::strdup(generic)),
    _specificName(new_strdup::strdup(specific)),
    _hierarchy(new_strdup::strdup(group)),
    _authority(new Publication(authorizer, authorized)), //XXXX
    _recorder(new Publication(recorder, recorded)), //XXXX
    _type(), _typeOf(), _higherNR()/*,  _anyUse()*/{}


#endif

string NameRecord::generic(void) const {if(this == NULL) return string("");return string(_genericName);}
string NameRecord::specific(void) const {if(this != NULL) return string(_specificName); return string("");}
string NameRecord::name(void) const 
{ string s(_genericName); if(s.length() > 0) s += " "; s+= _specificName; return s;}

void NameRecord::generic(const char* s)
{ delete [] _genericName; _genericName = new_strdup::strdup(s);}

void NameRecord::specific(const char* s)
{ delete [] _specificName; _specificName = new_strdup::strdup(s);}

void NameRecord::group(const char* s)
{ delete [] _hierarchy; _hierarchy = new_strdup::strdup(s);}

void NameRecord::generic(const string &s)
{ delete [] _genericName; _genericName = strdup(s);}

void NameRecord::specific(const string &s)
{ delete [] _specificName; _specificName = strdup(s);}

void NameRecord::group(const string &s)
{ delete [] _hierarchy; _hierarchy = strdup(s);}

bool NameRecord::_weakcmp(const string& s) const
{
  string word[7];
  int l = s.length();
  int i;
  for(i = 0; i < l && ':' != s[i]; i++){}
  word[0] = string(s, 0, i);
  i += 2;
  int head;
  int j;
  for(j = 1; j < 7 && i < l; j++){
    for(head = i; i < l && '_' != s[i]; i++){}
    word[j] = string(s, head, i - head);
    i++; 
  }
  word[6] += string(s, i, l - i);

  if(word[0] != "NameRecord") return false;
  if(word[3] != _specificName) return false;
  if(word[2] != _genericName) return false;
  //  if(word[6] != _authorWithYear(_recorder)) return false;
  if(word[6] != _authorWithYear(*_recorder)) return false;
  return true;
}

bool NameRecord::_weakcmp(const NameRecord& nr) const
{
  if(strcmp(nr._specificName, _specificName)) return false;
  if(strcmp(nr._genericName, _genericName)) return false;
  //  if(_authorWithYear(_recorder) != _authorWithYear(nr._recorder)) return false;
  if(_authorWithYear(*_recorder) != _authorWithYear(*(nr._recorder))) return false;
  return true;
}

#if 0
static Publication* _Publication(const string& a, const string& y)
{
  int yr = Date::invalidYear;
  if(0 != y.length()) yr = atoi(y.c_str());
  return new Publication(a.c_str(), yr);
}
#endif

#if 0
bool NameRecord::eq(const NameRecord &nr, int skip, int ignore) const
{
  bool ret = true;
  if(this == &nr) return ret;

  if(skip < 1){
    int mask = 0;
    if(skip < -5) skip = -5;
    switch(skip){
    case -5: 
      mask |= 0x40; //continue to the next case
    case -4:
      mask |= 0x02; //continue to the next case
    case -3:
      mask |= 0x01; //continue to the next case
    case -2:
      mask |= 0x08; //continue to the next case
    case -1:
      mask |= 0x10;
      break;
    }
    skip = mask;
  }
  skip = ~skip;
  
  if(ret && (skip&0x10))
    {
      int y = _year(_authority);
      int ny = _year(nr._authority);
      if(y != Date::invalidYear && ny != Date::invalidYear && y != ny)
	ret &= false;
    }
  if(ret && (skip&8))
    {
      string s(_surnames(_authority));
      string ns(_surnames(nr._authority));
      if(0 != s.length() && 0 != ns.length() && s != ns)
	ret &= false;
    }
  if(ret && (skip&1))
    {
      if(0 != strlen(_hierarchy) && 0 != strlen(nr._hierarchy) &&
	 strcmp(_hierarchy, nr._hierarchy))
	ret &= false;
    }
  
  if(ret && (skip&2))
    {
      if(0 != strlen(_genericName) && 0 != strlen(nr._genericName) &&
	 strfcmp(_genericName, nr._genericName, ignore))
	ret &= false;
    }
  
  if(ret && (skip&0x40))
    {
      int y = _year(_recorder);
      int ny = _year(nr._recorder);
      if(y != Date::invalidYear && ny != Date::invalidYear && y != ny)
	ret &= false;
    }
  if(ret && (skip&0x20))
    {
      string s(_surnames(_recorder));
      string ns(_surnames(nr._recorder));
      if(0 != s.length() && 0 != ns.length() && s != ns)
	ret &= false;
    }
  return ret;
}

const d_Ref<NameRecord> *NameRecord::find(const list<d_Ref<NameRecord> > &lnr, int skip, int ignore) const
{
  d_Ref<NameRecord> *ret = NULL;
  if(!lnr.empty()){
    list<d_Ref<NameRecord> >::const_iterator iter;
    NameRecord *nr;
    for(iter = lnr.begin();!ret && iter = lnr.!end(); ++iter){
      if(iter->persistentID().length()){
	if(NULL != iter->tr()) nr = &*iter;
	else nr = new NameRecord(iter->persistentID().length());
	if(eq(*nr))
	  ret = iter;
	if(NULL == iter->ptr()) delete nr;
      }
    }
  }
  return ret;
}
#endif

NameRecord* NameRecord::resolve(Resolver & r)
{

  //  string s(NameRecord::persistentID(persistentID().c_str()));
  string s(persistentID());
  string g(_classname);
  g += "::";
  g += _hierarchy;
  g += "_"; g += _genericName;
  g += "_"; g += _specificName;
  g += "_"; 

  string a(g);
  
  if(_currentPIDversion != 2){
    g += _authorWithYear(_authority);
    g += "_";
    a += "__";
    if(_currentPIDversion > 2){
      g += "______";
      a += "______";
    }
  }

  g += _authorWithYear(_recorder);
  a += _authorWithYear(_recorder);
  if(_currentPIDversion > 0){
    g += "______";
    a += "______";
  }

  //  cerr << g << endl;
  //  cerr << a << endl;

  char *c = s.end();
  while('_' != *--c){}
  string woPage(s, 0, c+1-s.begin());

  //  cerr << "strings prepared"<<endl;
  //  cerr << s <<endl;
  //  cerr << g <<endl;
  //  cerr << a <<endl;
  //  cerr << woPage <<endl;

  bool yet = true;
  hash_map<string, list<NameRecord*>, Hash<string> >::iterator niter = r.nr.find(s);
  hash_map<string, list<NameRecord*>, Hash<string> >::iterator nend = r.nr.end();
  if(niter != nend || 
     nend != (niter = r.nr.find(woPage)) ||
     nend != (niter = r.nr.find(g)) ||
     nend != (niter = r.nr.find(a))){
    list<NameRecord*>::iterator iter = niter->second.begin();
    list<NameRecord*>::iterator end = niter->second.end();

    while(iter != end && yet){
      string str((*iter)->persistentID());
      if(this == *iter)
	return this;
      //consider
      if((*iter)->merge(this)){
	return *iter;
      }
      ++iter;
    }
  }

  &(r.nr[s]); //?
  r.nr[s].push_back(this);
  r.newNameRecords.push_back(this);

  hash_map<string, list<d_Ref<NameRecord>*>, Hash<string> >::iterator uiter = r.unr.find(s);

  /*
   * Find pointers to this object from unsolved pointer list.
   * If found, set pointer and remove it from unsolved pointer list.
   * If solved pointer is containd in the list, removed it also.
   * 
   */
  if(uiter != r.unr.end()){
    register list<d_Ref<NameRecord>*>::iterator iter = uiter->second.begin();
    register list<d_Ref<NameRecord>*>::iterator end  = uiter->second.end();
    register list<d_Ref<NameRecord>*>::iterator current  = iter;
    while(iter != end){

      /*
       * make a copy pointer to current iterator,
       * and then increment itertor iter to point next pointer
       * BEFORE possible erase of current object.
       */
      current = iter;  //copy
      ++iter;          //then increment before possible erase()

      if(NULL == (*current)->ptr()){ //if unresolved
	string istr(NameRecord::persistentID((*current)->persistentID().c_str()));

	/*
	 * If string matched to one of the patterns, set pointer
	 * and then remove from the unsolved list.
	 */
	if((s == istr) || (a == istr) || (woPage == istr) || (g == istr)){ //generic...
	  (*current)->ptr(this);
	  uiter->second.erase(current);
	}
      }
      else  //parhaps not happen
	uiter->second.erase(current);
    }
  }
  else{
    //    cerr << "it was not found in unresolved pointers"<< endl;
  }

  NameRecord emptyNR;
  size_t empty = emptyNR.persistentID().length();

  if(NULL != _type.ptr()){
    NameRecord *q = _type->resolve(r);
    if(_type.ptr() != q){
      _type.ptr(q);
    }
  }
  else if(_type.persistentID().length() > empty){
    _type.resolve(r.nr, r.unr);
  }

  if(NULL != _typeOf.ptr()){
    NameRecord *q = _typeOf->resolve(r);
    if(_typeOf.ptr() != q){
      _typeOf.ptr(q);
    }
  }
  else if (_typeOf.persistentID().length() > empty){
    _typeOf.resolve(r.nr, r.unr);
  }

  if(NULL != _higherNR.ptr()){
    NameRecord *q = _higherNR->resolve(r);
    if(_higherNR.ptr() != q){
      _higherNR.ptr(q);
    }
  }
  else if (_higherNR.persistentID().length() > empty){
    _higherNR.resolve(r.nr, r.unr);
  }

  if(!lowerNR.empty()){
    list<d_Ref<NameRecord> >::iterator iter = lowerNR.begin();
    list<d_Ref<NameRecord> >::iterator end = lowerNR.end();
    while(iter != end){
      if(NULL != iter->ptr()){
	NameRecord *q =(*iter)->resolve(r);
	if(iter->ptr() != q){
	  iter->ptr(q);
	}
      }
      else if (iter->persistentID().length() > empty)
	iter->resolve(r.nr, r.unr);
      ++iter;
    }
  }

  Annotation emptyAnnotation;
  empty = emptyAnnotation.persistentID().length();

  if(!annotation.empty()){
    list<d_Ref<Annotation> >::iterator iter = annotation.begin();
    list<d_Ref<Annotation> >::iterator end= annotation.end();
    while(iter != end){
      if(NULL != iter->ptr()){
	Annotation *q = (*iter)->resolve(r);
	if(iter->ptr() != q){
	  iter->ptr(q);
	}
      }
      else if (iter->persistentID().length() > empty)
	iter->resolve(r.an, r.uan);
      ++iter;
    }
  }


  Citation emptyCitation;
  empty = emptyCitation.persistentID().length();

  if(NULL != _recorder.ptr()){
    Citation *q = _recorder.ptr()->resolve(r);
    if(_recorder.ptr() != q){
      _recorder.ptr(q);
    }
  }
  else if (_recorder.persistentID().length() > empty){
    _recorder.resolve(r.ci, r.uci);
  }

  if(NULL != _authority.ptr()){
    Citation *q = _authority.ptr()->resolve(r);
    if(_authority.ptr() != q){
      _authority.ptr(q);
    }
  }
  else if (_authority.persistentID().length() > empty){
    _authority.resolve(r.ci, r.uci);
  }

  return this;
}

NameRecord* NameRecord::resolve(hash_map<string, list<NameRecord*>, Hash<string> > &nr,
				hash_map<string, list<Annotation*>, Hash<string> > &an,
				hash_map<string, list<Citation*>, Hash<string> > &ci,
				hash_map<string, list<Publication*>, Hash<string> > &pub,
				hash_map<string, list<Author*>, Hash<string> > &au,
				hash_map<string, list<d_Ref<NameRecord>*>, Hash<string> > &unr,
				hash_map<string, list<d_Ref<Annotation>*>, Hash<string> > &uan,
				hash_map<string, list<d_Ref<Citation>*>, Hash<string> > &uci,
				hash_map<string, list<d_Ref<Publication>*>, Hash<string> > &upub,
				hash_map<string, list<d_Ref<Author>*>, Hash<string> > &uau)
{
#if 1
  hash_map<string, list<NRnode*>, Hash<string> > nrn;
  Resolver r(nr, an, ci, pub, au, nrn);
  NameRecord* ret = resolve(r);
  unr = r.unr; uan = r.uan; uci = r.uci; upub = r.upub; uau = r.uau;
  return ret;

#else
  //  cerr << "resolve in:"<<_higherNR.ptr()<<endl;
  string s(NameRecord::persistentID(persistentID().c_str()));
  string g(_classname);
  g += "::";
  g += _hierarchy;
  g += "_"; g += _genericName;
  g += "_"; g += _specificName;
  g += "_"; 
#if 0
  g += _authorWithYear(_authority);
  g += "_";
#endif
  g += _authorWithYear(_recorder);
  g += "______";
  char *c = s.end();
  while('_' != *--c){}
  string woPage(s, 0, c+1-s.begin());

  bool yet = true;
  hash_map<string, list<NameRecord*>, Hash<string> >::iterator niter = nr.find(s);
  hash_map<string, list<NameRecord*>, Hash<string> >::iterator nend = nr.end();
  if(niter != nend || 
     nend != (niter = nr.find(woPage)) ||
     nend != (niter = nr.find(g))){
    list<NameRecord*>::iterator iter = niter->second.begin();
    list<NameRecord*>::iterator end = niter->second.end();

    while(iter != end && yet){
      string str((*iter)->persistentID());
      if(this == *iter){
	return this /*NULL*/;
      }
      //      yet = !(*iter)->merge(this);
      else if((*iter)->merge(this)){
      	return *iter;
      }
      ++iter;
    }
  }

  nr[s].push_back(this);

  hash_map<string, list<d_Ref<NameRecord>*>, Hash<string> >::iterator uiter = unr.find(s);

  if(uiter != unr.end()){
    list<d_Ref<NameRecord>*>::iterator iter = uiter->second.begin();
    list<d_Ref<NameRecord>*>::iterator end = uiter->second.end();
    while(iter != end){
      if(NULL == (*iter)->ptr()){
	string istr(NameRecord::persistentID((*iter)->persistentID().c_str()));
	if((s == istr) || (woPage == istr) || (g == istr)) //generic...
	  (*iter)->ptr(this);
      }
      ++iter;
    }
  }

  if(NULL != _type.ptr()){
    NameRecord *p = _type->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau);
    if(NULL != p){
      //delete _type.pter()!
    }
  }
  else{
    _type.resolve(nr, unr);
  }

  if(NULL != _typeOf.ptr()){
    NameRecord *p = _typeOf->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau);
    if(NULL != p){
      //delete _typeOf.ptr()!
    }
  }
  else if (_typeOf.persistentID().length() > 0){
    _typeOf.resolve(nr, unr);
  }

  if(NULL != _higherNR.ptr()){
    NameRecord *p = _higherNR->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau);
    if(NULL != p){
      //delete _higherNR.ptr()!
    }
  }
  else if (_higherNR.persistentID().length() > 0){
    _higherNR.resolve(nr, unr);
  }

  if(!lowerNR.empty()){
    list<d_Ref<NameRecord> >::iterator iter = lowerNR.begin();
    list<d_Ref<NameRecord> >::iterator end = lowerNR.end();
    while(iter != end){
      if(NULL != iter->ptr()){
	NameRecord *p =(*iter)->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau);
	if(NULL != p){
	  //delete *iter!
	}
      }
      else if (iter->persistentID().length() > 0)
	iter->resolve(nr, unr);
      ++iter;
    }
  }

  if(!annotation.empty()){
    list<d_Ref<Annotation> >::iterator iter = annotation.begin();
    list<d_Ref<Annotation> >::iterator end= annotation.end();
    while(iter != end){
      if(NULL != iter->ptr()){
	Annotation *p = (*iter)->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau);
	if(NULL != p){
	  //delete *iter!
	}
      }
      else if (iter->persistentID().length() > 0)
	iter->resolve(an, uan);
      ++iter;
    }
  }

  if(NULL != _recorder.ptr()){
    Citation *p = _recorder->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau);
    if(NULL != p){
      //delete _recorder.ptr()!
    }
  }
  else if (_recorder.persistentID().length() > 0){
    _recorder.resolve(ci, uci);
  }

  if(NULL != _authority.ptr()){
    Citation *p = _authority->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau);
    if(NULL != p){
      //delete _authority.ptr()!
    }
  }
  else if (_authority.persistentID().length() > 0){
    _authority.resolve(ci, uci);
  }
  return this /*NULL*/;
#endif
}


static const char *_groups[] = {
  "superdomain", "domain", "subdomain",
  "superkingdom", "kingom", "subkingom", 
  "superphylum", "phylum", "subphylum", 
  "superclass", "class", "subclass",
  "superorder", "order", "suborder",
  "superfamily", "family", "subfamily", 
  "supergenus", "genus", "subgenus",
  "superspecies", "species", "species", "variant",
  NULL
};

static int _encodeGroup(const string &grp)
{
  const char **g = _groups;
  while(*g && strcasecmp(grp.c_str(), *g)){g++;}
  if(*g) return (g - _groups + 1);
  else return 0;
}

bool NameRecord::lt_by_group::operator()(NameRecord& r1, NameRecord& r2)
{ return _encodeGroup(r1.group()) < _encodeGroup(r2.group()); }

bool NameRecord::gt_by_group::operator()(NameRecord& r1, NameRecord& r2)
{ return _encodeGroup(r1.group()) > _encodeGroup(r2.group()); }

bool NameRecord::lt_by_name::operator()(NameRecord& r1, NameRecord& r2)
{ return (r1.name()) < (r2.name()); }

bool NameRecord::gt_by_name::operator()(NameRecord& r1, NameRecord& r2)
{ return (r1.name()) > (r2.name()); }

bool NameRecord::lt_by_generic::operator()(NameRecord& r1, NameRecord& r2)
{ return (r1.generic()) < (r2.generic()); }

bool NameRecord::gt_by_generic::operator()(NameRecord& r1, NameRecord& r2)
{ return (r1.generic()) > (r2.generic()); }

bool NameRecord::lt_by_year::operator()(const NameRecord& r1, const NameRecord& r2){
  int y1 = Date::invalidYear;
  if ((r1.recorder()->publication()).ptr())
    y1 = (r1.recorder()->publication())->year();
  int y2 = Date::invalidYear;
  if ((r2.recorder()->publication()).ptr())
    y2 = (r2.recorder()->publication())->year();
  if(Date::invalidYear != y1 && Date::invalidYear != y2)
    if(y1 != y2) return y1 < y2;

  y1 = Date::invalidYear;
  if ((r1.authority()->publication()).ptr())
    y1 = (r1.authority()->publication())->year();
  y2 = Date::invalidYear;
  if ((r2.authority()->publication()).ptr())
    y2 = (r2.authority()->publication())->year();
  if(Date::invalidYear != y1 && Date::invalidYear != y2)
    if(y1 != y2) return y1 < y2;
  if(y1 != y2){
    if(Date::invalidYear == y1) return false;
    else return true;
  }

  if(r1.generic() != r2.generic()) 
    return r1.generic() < r2.generic();

  if(r1.name() != r2.name()) 
    return r1.name() < r2.name();

  return r1.persistentID() < r2.persistentID();
}

bool NameRecord::gt_by_year::operator()(NameRecord& r1, NameRecord& r2)
{
  int y1 = Date::invalidYear;
  int y2 = Date::invalidYear;

  if (r1.recorder().ptr())
#if 1
    if(r1.recorder()->publication().ptr())
      y1 = r1.recorder()->publication()->year();
#else
      y1 = _year(*(r1.recorder()));
#endif
  if (r2.recorder().ptr())
#if 1
    if(r2.recorder()->publication().ptr())
      y2 = r2.recorder()->publication()->year();
#else
      y2 = _year(*(r2.recorder()));
#endif
  if(Date::invalidYear != y1 && Date::invalidYear != y2)
    if(y1 != y2) return y1 > y2;

  y1 = Date::invalidYear;
  if (r1.authority().ptr())
      y1 = _year(*(r1.authority()));
  y2 = Date::invalidYear;
  if (r2.authority().ptr())
    y2 = _year(*(r2.authority()));

  if(Date::invalidYear != y1 && Date::invalidYear != y2)
    if(y1 != y2) return y1 > y2;
  if(y1 != y2){
    if(Date::invalidYear == y1)
      return false;
    else
      return true;
  }

  if(r1.generic() != r2.generic()) 
    return r1.generic() < r2.generic();
  
  if(r1.name() != r2.name()) 
    return r1.name() < r2.name();

  return r1.persistentID() < r2.persistentID();

}

bool NameRecord::gt::operator()(NameRecord& r1, NameRecord& r2)
{ 
  int g1 = _encodeGroup(r1.group());
  int g2 = _encodeGroup(r2.group());
  if (g1 != g2)
    return g1 < g2;

  NameRecord::gt_by_year op;
  return op(r1, r2);
}

bool NameRecord::lt::operator()(NameRecord& r1, NameRecord& r2)
{ 
  int g1 = _encodeGroup(r1.group());
  int g2 = _encodeGroup(r2.group());
  if (g1 != g2)
    return g1 < g2;

  NameRecord::lt_by_year op;
  return op(r1, r2);
}

bool NameRecord::_mergeNames(NameRecord* from)
{
  bool ret = false;
  if(0 == strlen(_genericName)){
    _genericName = new_strdup::strdup(from->_genericName);
    ret |= true;
  }
  if(0 == strlen(_specificName)){
    _specificName = new_strdup::strdup(from->_specificName);
    ret |= true;
  }
  if(0 == strlen(_hierarchy)){
    _hierarchy = new_strdup::strdup(from->_hierarchy);
    ret |= true;
  }

  return ret;
}

static bool merge_dNR(d_Ref<NameRecord> &to, d_Ref<NameRecord> &from)
{
  bool ret = false;
  if(to.ptr() != from.ptr()){
    ret |= true;
    if(NULL == to.ptr())
      to.ptr(from.ptr());
    else if(NULL == from.ptr())
      from.ptr(to.ptr());
    else
      ret &= to->merge(from.ptr());
  }
  return ret;
}

bool NameRecord::_mergetypes(NameRecord* from)
{
  bool ret = false;
  /*
  ret |= merge_dNR(_type, from->_type);
  ret |= merge_dNR(_typeOf, from->_typeOf);
  ret |= merge_dNR(_higherNR, from->_higherNR);
  */
  _type.merge(from->_type);
  _typeOf.merge(from->_typeOf);
  _higherNR.merge(from->_higherNR);

  if(lowerNR.empty() || from ->lowerNR.empty()){
    ret |= true;
    lowerNR.sort();
    from->lowerNR.sort();
    lowerNR.merge(from->lowerNR);
    list<d_Ref<NameRecord> >::iterator iter = lowerNR.begin();
    list<d_Ref<NameRecord> >::iterator end = lowerNR.end();
    while(iter != end){
      from->lowerNR.push_back(*iter);
      ++iter;
    }
  }
  return ret;
}

static bool _mergeAnnotations(NameRecord* to, NameRecord* from)
{
  //  return false;
  bool ret = false;
  if(to == from) return ret;
  string t(to->persistentID());
  if(t != from->persistentID()) return ret;
  list<d_Ref<Annotation> >::iterator iter = to->annotation.begin();
  list<d_Ref<Annotation> >::iterator end = to->annotation.end();
  list<d_Ref<Annotation> >::iterator fiter;
  list<d_Ref<Annotation> >::iterator fend = from->annotation.end();
  while(iter != end){
    fiter = from->annotation.begin();
    while(fiter != fend){
      list<d_Ref<NameRecord> >::iterator siter = (*fiter)->from.begin();
      list<d_Ref<NameRecord> >::iterator send = (*fiter)->from.end();
      while(siter != send){
	if(siter->persistentID() == t){
	  ret |= true;
	  siter->ptr(to);
	}
	//	else if(NULL != siter->ptr())
	//	  ((*iter)->from).insert(d_Ref<NameRecord>(siter->ptr()));
	++siter;
      }
      siter = (*fiter)->to.begin();
      send = (*fiter)->to.end();
      while(siter != send){
	if(siter->persistentID() == t){
	  ret |= true;
	  siter->ptr(to);
	}
	//	else if(NULL != siter->ptr())
	//	  ((*iter)->from).insert(d_Ref<NameRecord>(siter->ptr()));
	++siter;
      }
      ret |= (*iter)->merge(fiter->ptr());
      ++fiter;
    }
    ++iter;
  }
  return ret;
}

bool NameRecord::_mergeCitations(d_Ref<Citation> &first, d_Ref<Citation> &second)
{
  static const bool refuse = false;

  Citation *c = first.ptr();
  Citation *nc = second.ptr();
  Citation empty;
  if(c == nc && c == NULL &&
    (first.persistentID() != second.persistentID()
      || first.persistentID().length() <= empty.persistentID().length()))
    return refuse;

  if(c != nc && c != NULL && nc != NULL && refuse == c->merge(nc))
    return refuse;

  if(c != nc){
    if(c == NULL)
      first.ptr(nc);
    else if(nc == NULL)
      second.ptr(c);
  }
  return !refuse;
}

bool NameRecord::_mergeRecorders(NameRecord* n)
{
#if 1
  static const bool refuse = false;

  Citation *c = _recorder.ptr();
  Citation *nc = n->_recorder.ptr();
  Citation empty;
  if(c == nc && c == NULL &&
    (_recorder.persistentID() != n->_recorder.persistentID()
      || _recorder.persistentID().length() <= empty.persistentID().length()))
    return refuse;

  if(c != nc && c != NULL && nc != NULL && refuse == nc->merge(c))
    return refuse;

  if(c != nc){
    if(c == NULL)
      _recorder.ptr(nc);
    else if(nc == NULL)
      n->_recorder.ptr(c);
  }
  return !refuse;

#else
  string mine("");
  string yours("");
  if(NULL != _recorder.ptr())
    mine = _recorder.ptr()->persistentID();
  else
    mine = _recorder.persistentID();
  if(NULL !=  (n->_recorder).ptr())
    yours = (n->_recorder).ptr()->persistentID();
  else
    yours = (n->_recorder).persistentID();
  if(mine == yours
     && 0 != mine.length()) {
    return _mergeCitations(_recorder, n->_recorder);
  }
  return false;
#endif
}  

bool NameRecord::_mergeAuthorities(NameRecord* n)
{
  static const bool refuse = false;

  if(refuse != _mergeCitations(_authority, n->_authority))
     return !refuse;

  /*
   * didn't perfect match
   */

  string authY = _authorWithYear(_authority);
  string nAuthY = n->_authorWithYear(_authority);

  if(authY != nAuthY && authY != "_" && nAuthY != "_")
    return refuse;

  /*
   * try to merge even if empty....
   */

  Citation *a = _authority.ptr();
  Citation *na = n->_authority.ptr();

  if(a == na && a == NULL)
    return !refuse;

  if(NULL == a || authY == "_")
    _authority.ptr(na);
  else if(NULL == na || nAuthY == "_")
    n->_authority.ptr(a);

  return !refuse;
}  
#if 0
bool _mergeCitations(d_Ref<Citation> &m, d_Ref<Citation> &y)
{
  Citation* mine = m.ptr();
  Citation* yours = y.ptr();

  if(NULL == yours){
    if(NULL == mine){
      mine = new Citation(m.persistentID());
      m.ptr(mine);
    }
    y.ptr(mine);
  }
  else if(NULL == mine){
    m.ptr(yours);
  }
  else{
    //    if(mine->oid() < yours->oid())
          return mine->merge(yours);
    //    else
    //      return yours->merge(mine);
  }
  return true;
}
#endif

/*
 * NameRecord merge code.
 *
 * NameRecord should not be merged if:
 * 0. identical object
 * 1. specific names are different
 * 2. recorders are obvoiusly different
 * 3. authority (sensu) is obvoiusly different
 * For these cases, merge fucntion must refuse
 * to merge (regurn false).
 * These refuseve cases are handled at first.
 *
 * The remaining fields may be only incomplete,
 * hence they must be handled very carefully
 *
 */
// NameRecord merge code.
// NameRecord should not be merged if...
//  the same object
//  names do no match at all
// NameRecord should be merged if...
//  points identical Citation as _record
//  persistendID()s of _record match
//  persistentID()s of _record->publication() match

bool NameRecord::merge(NameRecord* n)
{
  static const bool refuse = false;

  /*
   * case zero, identical object
   */
  if(this == n) return refuse;

  /*
   * name matching test
   */ 
  int namematch = 0;

  /*
   * Refuse to marge if specific name does not match;
   * other components may be only incomplete
   */

  if(strcmp(_specificName, n->_specificName))
    return refuse;
  else
    namematch |= 1;
  if(!strcmp(_genericName, n->_genericName))
    namematch |= 2;
  if(!strcmp(_hierarchy, n->_hierarchy))
    namematch |= 4;

  bool ret = refuse;

  /*
   * refuse if recorders do not match
   * otherwise, merge them
   */
  if(refuse == _mergeCitations(_recorder, n->_recorder))
    return refuse;


  if(_authority.ptr() != n->_authority.ptr() && 
     refuse == _mergeAuthorities(n))
    return refuse;

  NameRecord *primary = this;
  NameRecord *secondary = n;

  if(namematch != 7){ //otherwise nothing to do on name
    if(!(namematch & 2)){
      if(strlen(_genericName) != 0 &&
	 strlen(n->_genericName) != 0)
	return refuse;
      if(_genericName == NULL || strlen(_genericName) == 0){
#if 1
	//	if(_genericName != NULL)
	  //	delete [] _genericName;
	_genericName = strdup(n->_genericName);
#else
	char* tmp = _genericName;
	_genericName = n->_genericName;
	n->_genericName = tmp;
#endif
      }
    }
    if(!(namematch & 4)){
      if(strlen(_hierarchy) != 0 &&
	 strlen(n->_hierarchy) != 0)
	return refuse;
      if(_hierarchy == NULL || strlen(_hierarchy) == 0){
#if 1
	//	if(_hierarchy != NULL)
	  //	  delete [] _hierarchy;
	_hierarchy = strdup(n->_hierarchy);
#else
	char* tmp = _hierarchy;
	_hierarchy = n->_hierarchy;
	n->_hierarchy = tmp;
#endif
      }
    }
  }
  //    return refuse;
  return !refuse;
#if 1

  NameRecord *nHigher = n->_higherNR.ptr();
  if(NULL == nHigher){
    if(_higherNR.ptr() != NULL)
      n->_higherNR.ptr(_higherNR.ptr());
    else
    _higherNR.merge(n->_higherNR);
  }
  else if(_higherNR.ptr() == NULL){
    _higherNR.ptr(n->_higherNR.ptr());
    
  }
  else{
    if(_higherNR.ptr()->merge(nHigher)){
      n->_higherNR.ptr(_higherNR.ptr());
      register list<d_Ref<NameRecord> >::iterator i = nHigher->lowerNR.begin();
      register list<d_Ref<NameRecord> >::iterator e = nHigher->lowerNR.end();
      while(i != e){
	if(i->ptr() == n)
	  i->ptr(this);
	i++;
      }
    }
  }

  NameRecord *nType = n->_type.ptr();
  _type.merge(n->_type);
  _typeOf.merge(n->_typeOf);
  lowerNR.sort();
  n->lowerNR.sort();
  lowerNR.merge(n->lowerNR);
  lowerNR.unique();
#endif
  return !refuse;
}


NameRecord* NameRecord::top(void)
{
  NameRecord *pnr = this;
  NameRecord *upper = this;
  while(NULL != (upper = pnr->_higherNR.ptr()))
    pnr = upper;

  return pnr;
}

list<NameRecord*> NameRecord::bottom(void)
{
  list<NameRecord*> ret;
  if(lowerNR.empty()){
    ret.push_back(this);
    return ret;
  }
  list<d_Ref<NameRecord> >::iterator iter = lowerNR.begin();
  list<d_Ref<NameRecord> >::iterator end  = lowerNR.end();
  while(iter != end){
    //    ret.merge((*iter).ptr()->bottom());
    list<NameRecord*> tmp((*iter).ptr()->bottom());
    ret.merge(tmp);
    ++iter;
  }
  return ret;
}

/*
 * How many underscores in the PID?
 * All has at least two for the name trinity. and...
 * ver. 0: _author_year_author_year hence 6
 * ver. 1: two for _author_yare and 8 for citation hence 12 = 2+2+8
 * ver. 2: 8 for citaiton hence 10 = 2+8
 * ver. 3: 8 for each citaiton hence 18 = 2+8+8
 *
 */
static int _underscoreNum[] = {6, 12, 10, 18, 0};
static int maxversion = 3;

int NameRecord::_underscores(int version) const
{
  if(version < 0 || version > maxversion)
    return 0;
  else
    return _underscoreNum[version];
}

int NameRecord::pidVersion(const char *s)
{
  const char *end  = s + strlen(s);
  int cnt = 0;
  while(s != end){
    if('_' == *s++)
      cnt++;
  }
  int version = 0;
  while(cnt != _underscoreNum[version] && _underscoreNum[version] != 0)
    version++;
  if(_underscoreNum[version] == 0)
    version = -1;
  return version;
}
/*
 * output formats of NameRecord::persistentID():
 *
 * ver. 0: NameRecord::group_generic_specific_author_yearA_recoder_yearR
 * ver. 1: NameRecord::group_generic_specific_author_yearA_recorderPublicationStr
 * ver. 2: NameRecord::group_generic_specific_recorderPublicationStr
 * ver. 3: NameRecord::group_generic_specific_authorityPublicationStr_rrecorderPublicationStr
 *
 */

string* NameRecord::_pidParse(const char* str)
{
  char *s = new_strdup::strdup(str);
  int pidv = pidVersion(s);

  int words = _underscoreNum[pidv] + 1;

  string* word = new string[words];

  int j;
  for(j = 0; j < words; j++){
    word[j] = "";
  }
  int l = strlen(s);
  int i;
  for(i = 0; i < l && ':' != s[i]; i++){}
  i += 2;
  int head;

  for(j = 0; i < l && j < words; j++){
    for(head = i; i < l && '_' != s[i]; i++){}
    if(i!= head)
      word[j] = string(s+head, i - head);
    i++;
  }

  delete [] s;
  return word;
}

string NameRecord::persistentID(const char *s, int castTo)
{
  int pidv = pidVersion(s);

  //  if(pidv == 2)
  //  if(pidv == _currentPIDversion)
  if(pidv == castTo)
    return string(s);
  if(pidv < 0)
    return string("");

  int words = _underscoreNum[pidv] + 1;
#if 1
  string *word = _pidParse(s);
#else
  string word[words];
  int j;
  for(j = 0; j < words; j++){
    word[j] = "";
  }
  int l = strlen(s);
  int i;
  for(i = 0; i < l && ':' != s[i]; i++){}
  i += 2;
  int head;
#if 0
  for(j = 0; j < 7; j++){
    for(head = i; i < l && '_' != s[i]; i++){}
    if(i!= head)
      word[j] = string(s+head, i - head);
    if(i < l)
      i++;
  }
  
  if(pidv == 1){
    for(j = 7; i < l && j < words; j++){
      for(head = i; i < l && '_' != s[i]; i++){}
      if(i!= head)
	word[j] = string(s+head, i - head);
      i++;
    }
  }
#else
  for(j = 0; i < l && j < words; j++){
    for(head = i; i < l && '_' != s[i]; i++){}
    if(i!= head)
      word[j] = string(s+head, i - head);
    i++;
  }
#endif  
#endif

  //trinity
  string ret(_classname);
  ret += "::";
  for(int j = 0; j < 3; j++){
    ret += word[j];
    ret += "_";
  }

  int recorderIndex = 3;

  //authority
  string auth("");
  switch(pidv){
  case 2:
    auth += "__";
    break;
  default:
    while(recorderIndex < 5){
      auth += word[recorderIndex++];
      auth += "_";
    }
    if(pidv > 2)
      recorderIndex += 6; //skip more 6 words
  }

  if(castTo > 2){
    if(pidv > 2){
      for(int j = 5; j < recorderIndex; j++){
	auth += word[j];
	auth += "_";
      }
    }
    else{
      auth += "______";  //spacer for 6 words and next
    }
  }

  //recorder
  string rec("");
  for(int j = 0;  j < 2; j++){
    ret += word[recorderIndex++];
    ret += "_";
  }
  if(castTo > 0){
    if(pidv != 0){
      while(recorderIndex < words){
	ret += word[recorderIndex++];
	if(recorderIndex < words)
	  ret += "_";
      }
    }
    else
      ret += "_____";
  }

  if(castTo != 2)
    ret += auth;
  
  ret += rec;

  delete[] word;

  return ret;
}

//persistentID parser 
void NameRecord::_NameRecord(const string& str)
{

  int version = pidVersion(str.c_str());
  if(version < 0)
    return;

  int words = _underscoreNum[version] + 1;
  string *word = _pidParse(str.c_str());

  //trinity
  if(word[0].length()){ delete [] _hierarchy; _hierarchy = strdup(word[0]); }
  if(word[1].length()){ delete [] _genericName; _genericName = strdup(word[1]); }
  if(word[2].length()){ delete [] _specificName; _specificName = strdup(word[2]); }

  int recIndex = version == 2? 3:(version < 2 ? 5:11);
  Citation *pc = new Citation;
  string record("Publication::"); record += word[recIndex];
  recIndex++;
  if(version == 0) {
    record += "_";
    record += word[recIndex]; 
    record += "_____"; 
    recIndex += 6; 
  }
  else{  
    int i = recIndex;
    recIndex += 6;
    while(i < recIndex){
      record += "_";
      record += word[i];
      ++i;
    }
  }

  pc->publication(record);

  if(version != 0 && word[recIndex].length()){
    pc->page(word[recIndex].c_str());
  }

  _recorder = pc;

  //authority

  if(version > 1 || word[3] != word[5] || word[4] != word[6]){
    pc = new Citation;
    string auth("Publication::"); 
    if(version == 2)
      auth += "_";
    else
      auth += word[3];auth += "_"; auth += word[4]; 

    if(version < 3)
      auth += "_____"; 
    else{
      int i = 5;
      while(i < 10){
	auth += "_";
	auth += word[i++];
      }
      pc->page(word[10].c_str());
    }
    pc->publication(auth);
  }

  _authority = pc;

  delete [] word;
}


void NameRecord::page(int p)
{
  if(NULL != _recorder.ptr())
    _recorder->page(p);
}

void NameRecord::page(const char *p)
{
  if(NULL != _recorder.ptr())
    _recorder->page(p);
}

string NameRecord::page(void) const
{
  if(NULL != _recorder.ptr())
    return _recorder->page();
  return string("");
}

void NameRecord::year(int y)
{
  if(NULL != _recorder.ptr())
    _recorder->year(y);
}

int NameRecord::year(void) const {
  if(NULL != _recorder.ptr())
    return _recorder->year();
  else
    return 0;
}

bool operator<(const NameRecord& r1, const NameRecord& r2)
{
  int g1 = _encodeGroup(r1._hierarchy);
  int g2 = _encodeGroup(r2._hierarchy);
  if (g1 != g2)
    return g1 < g2;

  int strc = strcasecmp(r1._genericName, r2._genericName);
  if(strc != 0)
    return strc < 0;

  strc = strcasecmp(r1._specificName, r2._specificName);
  if(strc != 0)
    return strc < 0;

  NameRecord::lt_by_year op;
  return op(r1, r2);
}


bool operator==(const NameRecord& nr1, const NameRecord& nr2)
{

  if(&nr1 == &nr2) return true;

  if(nr1.persistentID() == nr2.persistentID()) return true;

  return false;
}
