/*
 *
 * NamedField.h: a definition of class handling structured text
 * for a Nomenclature Heuristic Model.
 * It may be called also N3, Nomenclature Netowrok Navigator
 * or more simply, MkII.
 *
 * Copyright (c) 1999 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: NamedField.h,v 1.2 1999/12/05 18:20:58 nozomi Exp $
 *	$Log: NamedField.h,v $
 *	Revision 1.2  1999/12/05 18:20:58  nozomi
 *	improve redundant include guard
 *	
 *	Revision 1.1  1999/03/15 16:42:55  nozomi
 *	gcc 2.7 support macro
 *	
 *	Revision 1.0.0  1999/03/14 02:17:38  nozomi
 *	Initial version, though aka MkII
 *	
 *
 */

// NamedField class is used as `interface' between istream and 
// non-trivial classes of NHM.  It is used generally as:
//
// istream& operator>>(istream& s, Someclass &c)
// { NamedField nf; s >> nf; if(s) c = Someclass(nf); return s;}
// 
// Genaral expected format of input is:
//
// fieldname = {block}
//
// each block may be nested like:
//
// outer = {
//  nested1 = {
//   nested2 = {
//    nest2content = {some text}
//    nest2content = {another text}
//    nest2anothercontent = {yet another text}
//   }
//   nest1content = {text again}
//   nest1another = {you may continute to 
//                    next line; heading blank will be ignored exect one}
//   nest1yet = {so you can't break neither in a word nor white spaces should not be shrinked}
//  }
// }
//
// Ordinary line-oriented comments would be accepted, but some may be used as directive like
// #inlucde in future. 

#if (!(defined _INCLUDED_NAMEDFIELD_H))
#define _INCLUDED_NAMEDFIELD_H

#if (__GNUC_MINOR__ > 7)
#if (!defined(_INCLUDED_IOSTREAM))
#include <iostream>
#define _INCLUDED_IOSTREAM
#endif
#if (!defined(_INCLUDED_LIST))
#include <list>
#define _INCLUDED_LIST
#endif
#if (!defined(_INCLUDED_STRING))
#include <string>
#define _INCLUDED_STRING
#endif
#if (!defined(_INCLUDED_CSTRING))
#include <cstring>
#define _INCLUDED_CSTRING
#endif
#else
#if (!defined(_INCLUDED_IOSTREAM))
#include <iostream.h>
#define _INCLUDED_IOSTREAM
#endif
#if (!defined(_INCLUDED_LIST))
#include <list.h>
#define _INCLUDED_LIST
#endif
#if (!defined(_INCLUDED_STRING))
#include <string.h>
#define _INCLUDED_STRING
#endif
#if (!defined(_INCLUDED_CSTRING))
#include <cstring.h>
#define _INCLUDED_CSTRING
#endif
#endif

class NamedField
{
  static const int defaultLength = 256;
public:
  NamedField(char* e = "", char* c = "");
  NamedField(const NamedField &);
  NamedField &  operator=(const NamedField &);
  ~NamedField(void);

  // I/O
  friend istream& operator>>(istream & s, NamedField &e);

  //for test purpose only
  friend ostream& operator<<(ostream & s, const NamedField &e);

  const char* entryName(void) const;
  const char* contents(void) const;
  const char* contents(const char*) const;
  const char* strcmp(const char*) const;
  const char* strncmp(const char*, size_t) const;
  const char* strmcmp(const char*, size_t) const;
  const char* strcasecmp(const char*) const;
  const char* strncasecmp(const char*, size_t) const;
  const char* strmcasecmp(const char*, size_t) const;

  friend const char* strcmp(NamedField&, const char*);
  friend const char* strncmp(NamedField&, const char*, size_t);
  friend const char* strmcmp(NamedField&, const char*, size_t);
  friend const char* strcasecmp(NamedField&, const char*);
  friend const char* strncasecmp(NamedField&, const char*, size_t);
  friend const char* strmcasecmp(NamedField&, const char*, size_t);
  friend const char* strcmp(const char*, NamedField&);
  friend const char* strncmp(const char*, NamedField&, size_t);
  friend const char* strmcmp(const char*, NamedField&, size_t);
  friend const char* strcasecmp(const char*, NamedField&);
  friend const char* strncasecmp(const char*, NamedField&, size_t);
  friend const char* strmcasecmp(const char*, NamedField&, size_t);

private:
  const char* _cmp(int (*cmpfunc)(const char *, const char *),  const char* s) const;
  const char* _ncmp(int (*cmpfunc)(const char *, const char *, size_t),  const char* s, size_t n) const;

  char* _entryName;
  char* _contents;

public:
  list<NamedField> subEntries;
};

inline NamedField::NamedField(char* e, char* c)
  :_entryName(new char[strlen(e) + 1]), _contents(new char[strlen(c) + 1]), subEntries()
{strcpy(_entryName, e);strcpy(_contents, c);}

inline NamedField::~NamedField(void)
{ delete[] _entryName;delete[] _contents;}

inline const char* NamedField::entryName(void) const
{return _entryName;}

inline const char* NamedField::contents(void) const
{return _contents;}

inline const char* NamedField::contents(const char* s) const
{return _cmp(::strcmp, s);}

#include "d_common.h"  //for strmcmp, strmcasecmp

inline const char* NamedField::strcmp(const char* s) const
{return _cmp(::strcmp, s);}

inline const char* NamedField::strncmp(const char* s, size_t n) const
{return _ncmp(::strncmp, s, n);}

inline const char* NamedField::strmcmp(const char* s, size_t m) const
{return _ncmp(::strmcmp, s, m);}
  //{return _ncmp(_strmcmp::strmcmp, s, m);}

inline const char* NamedField::strcasecmp(const char* s) const
{return _cmp(::strcasecmp, s);}

inline const char* NamedField::strncasecmp(const char* s, size_t n) const
{return _ncmp(::strncasecmp, s, n);}

inline const char* NamedField::strmcasecmp(const char* s, size_t m) const
{return _ncmp(::strmcasecmp, s, m);}
  //{return _ncmp(_strmcmp::strmcasecmp, s, m);}


inline const char* strcmp(NamedField& nf, const char* s)
{return nf.strcmp(s);}

inline const char* strncmp(NamedField& nf, const char* s, size_t n)
{return nf.strncmp(s, n);}

inline const char* strmcmp(NamedField& nf, const char* s, size_t m)
{return nf.strmcmp(s, m);}

inline const char* strcasecmp(NamedField& nf, const char* s)
{return nf.strcasecmp(s);}

inline const char* strncasecmp(NamedField& nf, const char* s, size_t n)
{return nf.strncasecmp(s, n);}

inline const char* strmcasecmp(NamedField& nf, const char* s, size_t m)
{return nf.strmcasecmp(s, m);}

inline const char* strcmp(const char* s, NamedField& nf)
{return nf.strcmp(s);}

inline const char* strncmp(const char* s, NamedField& nf, size_t n)
{return nf.strncmp(s, n);}

inline const char* strmcmp(const char* s, NamedField& nf, size_t m)
{return nf.strmcmp(s, m);}

inline const char* strcasecmp(const char* s, NamedField& nf)
{return nf.strcasecmp(s);}

inline const char* strncasecmp(const char* s, NamedField& nf, size_t n)
{return nf.strncasecmp(s, n);}

inline const char* strmcasecmp(const char* s, NamedField& nf, size_t m)
{return nf.strmcasecmp(s, m);}

#endif /* (!(defined (_INCLUDED_NAMEDFIELD_H))*/
