/*
 *
 * Nomencurator.cxx: an implementation of navigational collection of persistent classes
 * for a Nomenclature Heuristic Model.
 * It may be called also N3, Nomenclature Netowrok Navigator
 * or more simply, MkII.
 *
 * Copyright (c) 1999 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: Nomencurator.cxx,v 1.22 1999/12/05 18:25:17 nozomi Exp $
 *	$Log: Nomencurator.cxx,v $
 *	Revision 1.22  1999/12/05 18:25:17  nozomi
 *	class NHM renamed to Nomencurator
 *	
 *	Revision 1.21  1999/09/23 06:34:40  nozomi
 *	resolve method using Resolver class
 *	
 *	Revision 1.20  1999/09/20 14:19:23  nozomi
 *	data merge/data add support
 *	
 *	Revision 1.19  1999/08/31 12:15:53  nozomi
 *	some changed from set to list
 *	
 *	Revision 1.18  1999/08/26 07:20:43  nozomi
 *	add NRnode resolver
 *	
 *	Revision 1.17  1999/08/23 07:48:16  nozomi
 *	add a table of Publcations hashed by an author name
 *	
 *	Revision 1.16  1999/08/16 08:02:02  nozomi
 *	Support NameRecord::persistentID() version 2
 *	
 *	Revision 1.15  1999/08/15 01:28:11  nozomi
 *	hierarchy resonstruction modified to eliminate reference multiplication
 *	
 *	Revision 1.14  1999/08/13 13:06:28  nozomi
 *	NRnode hierarchy reconstruction fix
 *	
 *	Revision 1.13  1999/08/10 07:40:34  nozomi
 *	NRnode hierarchy reconstruction
 *	
 *	Revision 1.12  1999/08/08 11:34:01  nozomi
 *	canges in NRnode handling
 *	
 *	Revision 1.11  1999/08/07 10:05:05  nozomi
 *	Fix hash table bug
 *	
 *	Revision 1.10  1999/08/07 08:54:28  nozomi
 *	hash table for persistentID
 *	search also objects already held (operator >>)
 *	
 *	Revision 1.9  1999/08/06 07:36:57  nozomi
 *	Pointer resolve bug fix
 *	
 *	Revision 1.8  1999/03/21 13:46:35  nozomi
 *	Citatioin handling add
 *	
 *	Revision 1.7  1999/03/20 19:55:04  nozomi
 *	minor change in deduce
 *	
 *	Revision 1.6  1999/03/17 18:35:29  nozomi
 *	delete debug code
 *	
 *	Revision 1.5  1999/03/17 18:29:18  nozomi
 *	support server mode
 *	
 *	Revision 1.4  1999/03/15 16:42:39  nozomi
 *	gcc 2.7 support macro
 *	
 *	Revision 1.3  1999/03/15 13:16:27  nozomi
 *	modify verbose messages
 *	
 *	Revision 1.2  1999/03/14 08:30:26  nozomi
 *	verbose mode supports two or more levels
 *	
 *	Revision 1.1  1999/03/14 07:24:09  nozomi
 *	verbose parameter add
 *	
 *	Revision 1.0  1999/03/14 02:16:51  nozomi
 *	Initial version, though aka MkII
 *	
 *
 */

#include "Nomencurator.h"
#if (__GNUC_MINOR__ > 7)
#include <functional>
#else
#include <functional.h>
#endif

//void unifyHashMappedList(hash_map<string, list<NRnode*>, Hash<string> > &h);
template <class T>
void unifyHashMappedList(hash_map<string, list<T*>, Hash<string> > &h);

void unifyHashMappedList(hash_map<string, list<NRnode*>, Hash<string> > &h);

template <class T>
void unifyHashMappedList(hash_map<string, list<T>, Hash<string> > &h);

#if 0
template <class T> template <class StrictWeakOrdering>
void unifyHashMappedList(hash_map<string, list<T*>, Hash<string> > &h, StrictWeakOrdering bf);
#endif

template <class T>
void unifyHashMappedList(hash_map<string, list<T*>, Hash<string> > &h, binary_function<T, T, bool> bf);

template <class T> template <class StrictWeakOrdering>
void unifyHashMappedList(hash_map<string, list<T>, Hash<string> > &h, StrictWeakOrdering bf);

//hash function.  Improve it!
size_t Hash<string>::operator ()(const string &key) const
{
  size_t res = 0;
  string::const_iterator p = key.begin();
  string::const_iterator end = key.end();
  while(p != end) res = (res<<1)^*p++;
  return res;
}

//default constructor.  Nothing special.
Nomencurator::Nomencurator(void)
  : _verbose(0), 
    _nr(), _auth(), _cite(), _pub(), _anno(), _nodes(),
    _hnr(), _name(), _generic(), _specific(), _hauth(), _hauthorByName(), 
    _hpub(), _hashPubByAuthor(),
    _ptr(), _ptrNameRecord(), _ptrAuthor(), _ptrCitation(),
    _ptrPublication(), _ptrAnnotation(), _ptrNRnode(), _ptrNRnodeNR()
{}

//copy constructor.  Nothing special.
Nomencurator::Nomencurator(const Nomencurator &n)
  : _verbose(0),
    _nr(n._nr), _auth(n._auth), _cite(n._cite), _pub(n._pub), _anno(n._anno), _nodes(n._nodes),
    _hnr(n._hnr), _name(n._name), _generic(n._generic), _specific(n._specific), _hauth(n._hauth), 
    _hauthorByName(n._hauthorByName), _hpub(n._hpub), _hashPubByAuthor(n._hashPubByAuthor),
    _ptr(n._ptr), _ptrNameRecord(n._ptrNameRecord), _ptrAuthor(n._ptrAuthor), _ptrCitation(n._ptrCitation),
    _ptrPublication(n._ptrPublication), _ptrAnnotation(n._ptrAnnotation), _ptrNRnode(n._ptrNRnode), _ptrNRnodeNR(n._ptrNRnodeNR)
{}

//copy operator.  Nothing special.
Nomencurator& Nomencurator::operator=(const Nomencurator& n)
{
  _nr = n._nr; _auth = n._auth; _cite = n._cite; _pub = n._pub; _anno = n._anno; _nodes = n._nodes;
  _verbose = n._verbose;
#if 0
  list<d_Ref<NameRecord> >::iterator nriter;
  list<d_Ref<Author> >::iterator auiter;
  list<d_Ref<Publication> >::iterator pubiter;

  if(!_nr.empty())
    for(nriter = _nr.begin();nriter != _nr.end();++nriter){
      //      _hnr.insert(_hnr.begin(),make_pair(nriter->persistentID(),&**nriter));
      _hnr[nriter->persistentID()] = &**nriter;
      if(nriter->ptr()){
	string n((*nriter)->generic());
	_generic[n].push_back(&**nriter);
	_specific[(*nriter)->specific()].push_back(&**nriter);
	n += " ";
	n += (*nriter)->specific();
	_name[n].push_back(&**nriter);
      }
      }
  
  if(!_pub.empty())
    for(pubiter = _pub.begin();pubiter != _pub.end();++pubiter){
      //_hpub.insert(_hpub.begin(),make_pair(pubiter->persistentID(),&**pubiter));
      _hpub[pubiter->persistentID()] = &**pubiter;
      _hashPubByAuthor[pubiter->authorWithYear()].push_back(&**pubiter);
    }
  
  if(!_auth.empty())
    for(auiter = _auth.begin();auiter != _auth.end();++auiter){
      //      _hauth.insert(_hauth.begin(),make_pair(auiter->persistentID(),&**auiter));
      _hauth[auiter->persistentID()] = &**auiter;
      }
#endif
  return *this;
}

//destructor
Nomencurator::~Nomencurator(void)
{
  int i = 0;
  int cnt = 0;
  //clear NRnodes if there are.

  list <NRnode*> nrndel;
  while(!_nodes.empty()){
    nrndel.push_back(_nodes.begin()->ptr());
    _nodes.pop_front();
    cnt++;
  }

  if(!nrndel.empty()){
    i = 0;
    nrndel.sort();
    nrndel.unique();
    while(!nrndel.empty()){
      NRnode* p = *(nrndel.begin());
      nrndel.pop_front();
      if(NULL != p){
	string s(p->persistentID());
	  delete p;
	  i++;
	  if(_verbose > 1){
	    cerr << "NRnode[" << i <<"/"<< cnt <<"] " << p << " deleted";
	  if(_verbose > 2)
	    cerr << " " << s;
	  cerr <<"\n";
	}
      }
    }
    if(_verbose)
      cerr << this << ": NRnode [" << --i << "/"<< --cnt <<"] deletion done\n";
  }


  list<Annotation*> andel;
  //clear Annotations if there are.
  cnt = 0;
  while(!_anno.empty()){
    andel.push_back(_anno.begin()->ptr());
    _anno.pop_front();
    cnt++;
  }
  if(!andel.empty()){
    andel.sort();
    andel.unique();
    i = 0;
    while(!andel.empty()){
      Annotation* p = *(andel.begin());
      andel.pop_front();
      if(NULL != p){
	string s(p->persistentID());
	delete p;
	i++;
	if(_verbose > 1) {
	  cerr << "Annotation[" << i <<"] " << p << " deleted";
	  if(_verbose > 2)
	    cerr << " " << s;
	  cerr <<"\n";
	}
      }
    }
    if(_verbose)
      cerr << this << ": Annotation [" << --i << "/"<< --cnt <<"] deletion done\n";
  }

  //clear NameRecords if there are.
  list <NameRecord*> nrdelete;
  cnt = 0;
  while(!_nr.empty()){
    nrdelete.push_back(_nr.begin()->ptr());
    _nr.pop_front();
    cnt++;
  }

  if(!nrdelete.empty()){
    i = 0;
    nrdelete.sort();
    nrdelete.unique();
    while(!nrdelete.empty()){
      NameRecord *p = *(nrdelete.begin());
      nrdelete.pop_front();
      if(NULL != p){
	string s(p->persistentID());
	delete p;
	i++;
	if(_verbose > 1){
	  cerr << "NameRecord[" << i <<"] " << p << " deleted";

	  if(_verbose > 2)
	    cerr << " " << s;

	  cerr <<"\n";
	}
      }
    }
    if(_verbose)
      cerr << this << ": NameRecord [" << --i << "/"<< --cnt <<"] deletion done\n";
  }


  list<Citation*> citedel;
  //clear Citations if there are.
  cnt = 0;
  while(!_cite.empty()){
    citedel.push_back(_cite.begin()->ptr());
    _cite.pop_front();
    cnt++;
  }
  if(!citedel.empty()){
    citedel.sort();
    citedel.unique();
    i = 0;
    while(!citedel.empty()){
      Citation* p = *(citedel.begin());
      citedel.pop_front();
      if(NULL != p){
	string s(p->persistentID());
	delete p;
	i++;
	if(_verbose > 1){
	  cerr << "Citation[" << i <<"] " << p << " deleted";
	  if(_verbose > 2)
	    cerr << " " << s;
	  cerr <<"\n";
	}
      }
    }
    if(_verbose)
      cerr << this << ": Citation [" << --i << "/"<< --cnt <<"] deletion done\n";
  }
  
  //clear Publications if there are.
  list <Publication*> pubdel;
  cnt = 0;
  while(!_pub.empty()){
    pubdel.push_back(_pub.begin()->ptr());
    _pub.pop_front();
    cnt++;
  }
  if(!pubdel.empty()){
    i = 0;
    pubdel.sort();
    pubdel.unique();
    while(!pubdel.empty()){
      Publication *p = *(pubdel.begin());
      pubdel.pop_front();
      if(NULL != p){
	string s(p->persistentID());
	delete p;
	i++;
	if(_verbose > 1 ){
	  cerr << "Publication[" << i <<"] " <<p << " deleted";
	  if(_verbose > 2)
	    cerr << " " << s;
	  cerr <<"\n";
	}
      }
    }
    if(_verbose)
      cerr << this << ": Publication [" << --i << "/"<< --cnt <<"] deletion done\n";
  }

  list<Author*> authdel;
  //clear Authors if there are.
  cnt = 0;
  while(!_auth.empty()){
    authdel.push_back(_auth.begin()->ptr());
    _auth.pop_front();
    cnt++;
  }
  if(!authdel.empty()){
    authdel.sort();
    authdel.unique();
    i = 0;
    while(!authdel.empty()){
      Author* p = *(authdel.begin());
      authdel.pop_front();
      if(NULL != p){
	string s(p->persistentID());
	delete p;
	i++;
	if(_verbose > 1) {
	  cerr << "Author[" << i <<"] " << p << " deleted";
	  if(_verbose > 2)
	    cerr << " " << s;
	  cerr <<"\n";
	}
      }
    }
    if(_verbose)
      cerr << this << ": Author [" << --i << "/"<< --cnt <<"] deletion done\n";
  }
}

bool Nomencurator::clear(void)
{
  //clear NRnodes if there are.
  if(!_nodes.empty())
    _nodes.clear();

  //clear NameRecords if there are.
  if(!_nr.empty())
    _nr.clear();

  //clear Citations if there are.
  if(!_cite.empty())
    _cite.clear();
  
  //clear Publications if there are.
  if(!_pub.empty())
    _pub.clear();

  //clear Authors if there are.
  if(!_auth.empty())
    _auth.clear();
  
  //clear Annotations if there are.
  if(!_anno.empty())
    _anno.clear();

  if(!_hnr.empty())
    _hnr.clear();

  if(!_name.empty())
    _name.clear();

  if(!_generic.empty())
    _generic.clear();

  if(!_specific.empty())
    _specific.clear();

  if(!_hauth.empty())
    _hauth.clear();

  if(!_hauthorByName.empty())
    _hauthorByName.clear();

  if(!_hpub.empty())
    _hpub.clear();

  if(!_hashPubByAuthor.empty())
    _hashPubByAuthor.clear();

  return true;
}

//Reading.  It might be better to move latter part to another function.
//It may be shared...but by whom?   Data adding function, parhaps.
istream& operator>> (istream &s, Nomencurator& n)
{
  int readin = 0;
  int nrread = 0;
  int authread = 0;
  int pubread = 0;
  int citeread = 0;
  int annoread = 0;
  int nrnread = 0;

  hash_map<string, list<NameRecord*>, Hash<string> > nr;
  hash_map<string, list<Annotation*>, Hash<string> > an;
  hash_map<string, list<Citation*>, Hash<string> > ci;
  hash_map<string, list<Publication*>, Hash<string> > pub;
  hash_map<string, list<Author*>, Hash<string> > au;
  hash_map<string, list<NRnode*>, Hash<string> > nrn;
  hash_map<string, list<d_Ref<NameRecord>*>, Hash<string> > unr;
  hash_map<string, list<d_Ref<Annotation>*>, Hash<string> > uan;
  hash_map<string, list<d_Ref<Citation>*>, Hash<string> > uci;
  hash_map<string, list<d_Ref<Publication>*>, Hash<string> > upub;
  hash_map<string, list<d_Ref<Author>*>, Hash<string> > uau;
  hash_map<string, list<d_Ref<NRnode>*>, Hash<string> > unrn;

  Resolver r;

  while(s){
    {
      NamedField nf;
      s >> nf;
      if(s){
	if(n._verbose > 3)
	  cerr << nf;
	const char* s = nf.entryName();
	if(!strcmp(s, "NameRecord")){
	  NameRecord *p = new NameRecord(nf);
	  if(n._verbose > 1)
	    cerr << *p;
	  nrread++;
	  if(p/*NULL*/ != p->resolve(r))
	    delete p;
	}
	else if(!strcmp(s, "NRnode")){
	  nrnread++;
	  NRnode *p = new NRnode(nf);
	  if(n._verbose > 1)
	    cerr << *p;
	  NRnode *q = p->resolve(r);
	  if(q != p)
	    delete p;
	}
	else if(!strcmp(s, "Citation")){
	  Citation *p = new Citation(&nf);
	  citeread++;
	  if(n._verbose > 1)
	    cerr << *p;
	  Citation *q = p->resolve(r);
	  if(q != p)
	    delete p;
	}
	else if(!strcmp(s, "Publication")){
	  Publication *p = new Publication(&nf);
	  pubread++;
	  if(n._verbose > 1)
	    cerr << *p << flush;
	  Publication *q = p->resolve(r);
	  if(q != p)
	    delete p;
	}
	else if(!strcmp(s, "Author")){
	  Author* p = new Author(nf);
	  readin++;
	  authread++;
	  if(n._verbose > 1)
	    cerr << *p;
	  //	  Author *r = p->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau);
	  Author *q = p->resolve(r);
	  if(q != p)
	    delete p;
	  //	  else
	  //	    r.newAuthors.push_back(p);
	}
	else{
	  const Annotation::annoTuple *t = Annotation::tuple;
	  while(t->str){
	    if(!strncmp(s, t->str, t->n)){
	      Annotation* p = new Annotation(nf, t->str);
	      annoread++;
	      if(n._verbose > 1)
		cerr << *p;
	      //	      Annotation *r = p->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau);
	      Annotation *q = p->resolve(r);
	      if(q != p)
		delete p;
	      //	      else
	      //		r.newAnnotations.push_back(p);
	      break;
	    }
	    ++t;
	  }
	}
      }
    }
  }

#if 1
  {
    int cnt = 0;
    hash_map<string, list<NameRecord*>, Hash<string> >::iterator iter = r.nr.begin();
    hash_map<string, list<NameRecord*>, Hash<string> >::iterator end = r.nr.end();
    list<NameRecord*>::iterator i;
    list<NameRecord*>::iterator e;
    while(iter != end){
	i = (iter->second).begin();
	e = (iter->second).end();
	while(i != e){
	  cnt++;
	  i++;
	}
	iter++;
    }
  }
#if 0
  n._resolve(nr, an, ci, pub, au, nrn,
	     unr, uan, uci, upub, uau, unrn);
#else
  n._resolve(r);

  if(!r.newNameRecords.empty()){
    register list<NameRecord*>::iterator i = r.newNameRecords.begin();
    register list<NameRecord*>::iterator e = r.newNameRecords.end();
    while(i != e){
      n._nr.push_back(*i);
      i++;
    }
  }

  if(!r.newAnnotations.empty()){
    register list<Annotation*>::iterator i = r.newAnnotations.begin();
    register list<Annotation*>::iterator e = r.newAnnotations.end();
    while(i != e){
      n._anno.push_back(*i);
      i++;
    }
  }

  if(!r.newCitations.empty()){
    register list<Citation*>::iterator i = r.newCitations.begin();
    register list<Citation*>::iterator e = r.newCitations.end();
    while(i != e){
      n._cite.push_back(*i);
      i++;
    }
  }
  if(!r.newPublications.empty()){
    register list<Publication*>::iterator i = r.newPublications.begin();
    register list<Publication*>::iterator e = r.newPublications.end();
    while(i != e){
      n._pub.push_back(*i);
      i++;
    }
  }
  if(!r.newAuthors.empty()){
    register list<Author*>::iterator i = r.newAuthors.begin();
    register list<Author*>::iterator e = r.newAuthors.end();
    while(i != e){
      n._auth.push_back(*i);
      i++;
    }
  }
  if(!r.newNRnodes.empty()){
    register list<NRnode*>::iterator i = r.newNRnodes.begin();
    register list<NRnode*>::iterator e = r.newNRnodes.end();
    while(i != e){
      n._nodes.push_back(*i);
      i++;
    }
  }

#endif
#else
  int authdeduced = 0;
  int pubdeduced = 0;
  int citededuced = 0;
  int nrdeduced = 0;
  int nrndeduced = 0;
  int newDeduce = 0;

  //Publicatoin contains NameRecord, Author, Annotation
  //NameRecord contains  NameRecord, Publication, Annotation
  //Annotation contains  NameRecord
  //Author contains Publication
  //Therefore loop need to close are:
  // Pub -> Auth -> Pub, twice
  // Pub -> NR -> Pub, twice
  // Pub -> NR -> Anno -> NR -> Pub, four times
  // Anno -> NR -> Anno, twice
  // use limit magicnumber four, not forever.

  string str;
  int looplimit = 4;
  do{
    newDeduce = 0;

    // sequence sensitive?
   if(!unr.empty()) {
     hash_map<string, list<d_Ref<NameRecord>*>, Hash<string> >::iterator diter = unr.begin();
     hash_map<string, list<NameRecord*>, Hash<string> >::iterator nend = nr.end();
     hash_map<string, list<NameRecord*>, Hash<string> >::iterator ptrnend = n._ptrNameRecord.end();
     hash_map<string, list<NameRecord*>, Hash<string> >::iterator nfnd;
     while(diter != unr.end()){
       register list<d_Ref<NameRecord>* >::iterator iter = diter->second.begin();
       register list<d_Ref<NameRecord>* >::iterator current;
       while(iter != diter->second.end()){
	 current = iter;
	 iter++;
	 if(!(*current)->ptr()){
	   string s((*current)->persistentID());
	   if(s.length()){
	     nfnd = nr.find(s);
	     if(nend == nfnd){
	       nfnd = n._ptrNameRecord.find(s);
	       if(ptrnend == nfnd)
		 nfnd = nend;
	     }
	     
	     if(nfnd != nend){
	       list<NameRecord*>::iterator niter = nfnd->second.begin();
	       list<NameRecord*>::iterator _nend = nfnd->second.end();
	       while(niter != _nend){
		 if((*niter)->persistentID() == s){
		   (*current)->ptr(*niter);
		   break;
		 }
		 ++niter;
	       }
	     }
	     else{  
	       NameRecord *p = new NameRecord(s);
	       NameRecord *q = p->resolve(r);
	       if(q == p){
		 nrdeduced++;
		 newDeduce++;
		 if(n._verbose)
		   cerr << "%deduce " << 4 - looplimit << ":"<< newDeduce << " " << p->persistentID() << "\n";
	       }
	       else
		 delete p;
	       (*current)->ptr(q);
	     }
	   }
	 }
       }
       ++diter;
     }
   }

   if(!unrn.empty()) {
     hash_map<string, list<d_Ref<NRnode>*>, Hash<string> >::iterator diter = unrn.begin();
     hash_map<string, list<NRnode*>, Hash<string> >::iterator nend = nrn.end();
     hash_map<string, list<NRnode*>, Hash<string> >::iterator ptrnend = n._ptrNRnode.end();
     hash_map<string, list<NRnode*>, Hash<string> >::iterator nfnd;

     while(diter != unrn.end()){
       register list<d_Ref<NRnode>* >::iterator iter = diter->second.begin();
       register list<d_Ref<NRnode>* >::iterator current;
       while(iter != diter->second.end()){
	 current = iter;
	 iter++;
	 if(!(*current)->ptr()){
	   string s((*current)->persistentID());
	   if(s.length()){
	     nfnd = nrn.find(s);
	     if(nend == nfnd){
	       nfnd = n._ptrNRnode.find(s);
	       if(ptrnend == nfnd)
		 nfnd = nend;
	     }
	     
	     if(nfnd != nend){
	       list<NRnode*>::iterator niter = nfnd->second.begin();
	       list<NRnode*>::iterator _nend = nfnd->second.end();
	       while(niter != _nend){
		 if((*niter)->persistentID() == s){
		   (*current)->ptr(*niter);
		   break;
		 }
		 ++niter;
	       }
	     }
	     else{  
	       NRnode *p = new NRnode(s);
	       NRnode *q = p->resolve(r);
	       if(q == p){
		 str = s;
		 nrndeduced++;
		 newDeduce++;
		 if(n._verbose)
		   cerr << "%deduce " << 4 - looplimit << ":"<< newDeduce << " " << p->persistentID() << "\n";
	       }
	       else
		 delete p;
	       (*current)->ptr(q);
	     }
	   }
	 }
       }
       ++diter;
     }
   }
   
   
   if(!uci.empty()) {
     hash_map<string, list<d_Ref<Citation>*>, Hash<string> >::iterator diter = uci.begin();
     hash_map<string, list<Citation*>, Hash<string> >::iterator cend = ci.end();
     hash_map<string, list<Citation*>, Hash<string> >::iterator pcend = n._ptrCitation.end();
     hash_map<string, list<Citation*>, Hash<string> >::iterator cfnd;
     while(diter != uci.end()){
       list<d_Ref<Citation>* >::iterator iter = diter->second.begin();
       while(iter != diter->second.end()){
	 if(!(*iter)->ptr()){
	   string s((*iter)->persistentID());
	   if(s.length()){
	     cfnd = ci.find(s);
	     if(cend == cfnd){
	       cfnd = n._ptrCitation.find(s);
	       if(pcend == cfnd)
		 cfnd = cend;
	     }
	     if(cend != cfnd){
	       list<Citation*>::iterator citer = cfnd->second.begin();
	       list<Citation*>::iterator _cend = cfnd->second.end();
	       while(citer != _cend){
		 if((*citer)->persistentID() == s){
		   (*iter)->ptr(*citer);
		   break;
		 }
		 ++citer;
	       }
	     }
	     else{
	       Citation *p = new Citation(s);
	       if(NULL == p->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau)){
		 str = s;
		 citededuced++;
		 newDeduce++;
		 if(n._verbose)
		   cerr << "%deduce " << 4 - looplimit << ":"<< newDeduce << " " << p->persistentID() << "\n";
	       }
	       else
		 delete p;
	     }
	   }
	 }
	 ++iter;
       }
       ++diter;
     }
   }
   
    if(!upub.empty())
      {
	hash_map<string, list<d_Ref<Publication>*>, Hash<string> >::iterator diter = upub.begin();
	hash_map<string, list<Publication*>, Hash<string> >::iterator pend = pub.end();
	hash_map<string, list<Publication*>, Hash<string> >::iterator ppend = n._ptrPublication.end();
	hash_map<string, list<Publication*>, Hash<string> >::iterator pfnd;
	while(diter != upub.end()){
	  list<d_Ref<Publication>* >::iterator iter = diter->second.begin();
	  while(iter != diter->second.end()){
	    if(!(*iter)->ptr()){
	      string s((*iter)->persistentID());
	      if(s.length()){
		pfnd = pub.find(s);
		if(pfnd == pend){
		  pfnd = n._ptrPublication.find(s);
		  if(pfnd == ppend)
		    pfnd = pend;
		}
		if(pfnd != pend){
		  list<Publication*>::iterator piter = pfnd->second.begin();
		  list<Publication*>::iterator _pend = pfnd->second.end();
		  while(piter != _pend){
		    if((*piter)->persistentID() == s){
		      (*iter)->ptr(*piter);
		      break;
		    }
		    ++piter;
		  }
		}
		else{
		  Publication *p = new Publication(s);
		  if(NULL == p->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau)){
		    str = s;
		    pubdeduced++;
		    newDeduce++;
		    if(n._verbose)
		      cerr << "%deduce " << 4 - looplimit << ":"<< newDeduce << " " << p->persistentID() << "\n";
		  }
		  else
		    delete p;
		}
	      }
	    }
	    ++iter;
	  }
	  ++diter;
	}
      }

    if(!uau.empty())
      {
	hash_map<string, list<d_Ref<Author>*>, Hash<string> >::iterator diter = uau.begin();
	hash_map<string, list<Author*>, Hash<string> >::iterator auend = au.end();
	hash_map<string, list<Author*>, Hash<string> >::iterator pauend = n._ptrAuthor.end();
	hash_map<string, list<Author*>, Hash<string> >::iterator aufnd;
	while(diter != uau.end()){
	  list<d_Ref<Author>* >::iterator iter = diter->second.begin();
	  while(iter != diter->second.end()){
	    if(!(*iter)->ptr()){
	      string s((*iter)->persistentID());
	      if(s.length()){
		aufnd = au.find(s);
		if(aufnd == auend){
		  aufnd = n._ptrAuthor.find(s);
		  if(aufnd == pauend)
		    aufnd = auend;
		}
		if(aufnd != auend){
		  list<Author*>::iterator auiter = aufnd->second.begin();
		  list<Author*>::iterator _auend = aufnd->second.end();
		  while(auiter != _auend){
		    if((*auiter)->persistentID() == s){
		      (*iter)->ptr(*auiter);
		      break;
		    }
		    ++auiter;
		  }
		}
		else{
		  int l = s.length();
		  if(l > 0){
		    int i;
		    for(i =0 ; i < l && '_' != s[i]; i++){}
		    if(i != l)
		      l = i;
		    Author *p = new Author(s);
		    if(NULL == p->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau)){
		      str = s;
		      authdeduced++;
		      newDeduce++;
		      if(n._verbose)
			cerr << "%deduce " << 4 - looplimit << ":"<< newDeduce << " " << p->persistentID() << "\n";
		    }
		    else
		      delete p;
		  }
		}
	      }
	    }
	    ++iter;
	  }
	  ++diter;
	}
      }

#if 0
    //more deduce
    list<string> ls;
    for(p = n._nr.begin();p != n._nr.end();++p){
      if(((*p)->authorityStr() != (*p)->recorderStr())
	 && ((*p)->annotation).empty()){
	string s("NameRecord::");
	s += (*p)->group(); s+="_";
	s += (*p)->generic(); s+="_";
	s += (*p)->specific(); s+="_";
	s += (*p)->authorityStr(); s+="_";
	s += (*p)->authorityStr();
	ls.push_back(s);
      }
    }
    if(!ls.empty())
      {
	ls.sort(); ls.unique();
	list<string>::iterator lsiter;
	for(lsiter = ls.begin();lsiter != ls.end();++lsiter){
	  tnr.push_back(new NameRecord(*lsiter));
	  ++g;
	}
	ls.clear();tnr.sort();
	n._nr.merge(tnr);n._nr.unique(d_Ref_unify<NameRecord>());
      }
#endif
  } while(newDeduce && --looplimit);

  if(n._verbose > 2)
    cerr << "% Deduce phase over\n";
  if(n._verbose && looplimit == 0 && 0 != newDeduce)
    cerr << "% Deduce not yet converge. " << newDeduce 
	 << " object" << (newDeduce >1? "s were":" was") << " tried to deduce in the last loop.\n"
	 << "% Data may contain mistake" << (newDeduce >1? "s":"") << ".  Last deduce on: " 
	 << str << "\n";
#if 0 //1
  // name change propagation
  if(!an.empty()){
    list<d_Ref<Annotation> >::iterator ani = n._anno.begin();
    list<d_Ref<Annotation> >::iterator ane = n._anno.end();
    string ref("refer"); //XXX
    while(ani != ane){
      if((*ani)->annotation() == ref){
	NameRecord *to = ((*ani)->to.begin())->ptr(); //only one, because ref
	NameRecord *from = ((*ani)->from.begin())->ptr(); //only one, because ref
	if(!(to->lowerNR.empty()) && to->specific() != from->specific()){
	  list<d_Ref<NameRecord> >::iterator toiter = to->lowerNR.begin();
	  list<d_Ref<NameRecord> >::iterator toend = to->lowerNR.end();
	  while(toiter != toend){
	    NameRecord *tonr = toiter->ptr();
	    list<d_Ref<NameRecord> >::iterator friter = from->lowerNR.begin();
	    list<d_Ref<NameRecord> >::iterator frend = from->lowerNR.end();
	    while(friter != frend && tonr != friter->ptr()){ ++friter; }
	    if(friter == frend){  // not found
	      NameRecord *prop = new NameRecord(*tonr);
	      prop->recorder()->publication(from->recorder()->publication().ptr());
	      prop->recorder()->page(from->recorder()->page());
	      prop->higher(from);
	      prop->generic(from->specific());
	      Annotation *pan = new Annotation;
	      pan->to.push_back(tonr);
	      pan->from.push_back(prop);
	      pan->annotation("propagate");
	      prop->annotation.clear();
	      prop->annotation.push_back(pan);
	      if(NULL != &*(prop->typeOf())){
		prop->typeOf(from);
		if(NULL != &*(from->typeOf()))
		  from->type(prop);
	      }
	    }
	    ++toiter;
	  }
	}
      }
      ++ani;
    }
  }
#endif

  if(!ci.empty())
    {
      hash_map<string, list<Citation*>, Hash<string> >::iterator iter = ci.begin();
      hash_map<string, list<Citation*>, Hash<string> >::iterator end = ci.end();
      list<Citation*>::iterator i;
      list<Citation*>::iterator e;
      while(iter != end){
	i = (iter->second).begin(); e = (iter->second).end();
	while(i != e){
	  n._cite.push_back(*i);
	  n._ptrCitation[(*i)->persistentID()].push_back(*i);
	  n._ptr[(*i)->persistentID()].push_back(*i);
	  ++i;
	}
	++iter;
      }
    }

  if(!au.empty())
    {
      hash_map<string, list<Author*>, Hash<string> >::iterator iter = au.begin();
      hash_map<string, list<Author*>, Hash<string> >::iterator end = au.end();
      list<Author*>::iterator i;
      list<Author*>::iterator e;
      while(iter != end){
	i = (iter->second).begin(); e = (iter->second).end();
	while(i != e){
	  n._auth.push_back(*i);
	  strint pid = (*i)->persistentID();
	  n._hauth[pid] = *i;
	  n._ptrAuthor[pid].push_back(*i);
	  n._ptr[pid].push_back(*i);
	  Author* a = *i;
	  n._hauthorByName[a->surname()].push_back(*i);
	  n._hauthorByName[a->fullname()].push_back(*i);
	  n._hauthorByName[a->initializedName()].push_back(*i);
	  ++i;
	}
	++iter;
      }
    }

  if(!pub.empty())
    {
      hash_map<string, list<Publication*>, Hash<string> >::iterator iter = pub.begin();
      hash_map<string, list<Publication*>, Hash<string> >::iterator end = pub.end();
      list<Publication*>::iterator i, e;
      while(iter != end){
	i = (iter->second).begin(); e = (iter->second).end();
	while(i != e){
	  n._pub.push_back(*i);
	  n._hpub[(*i)->persistentID()] = *i;
	  n._hashPubByAuthor[(*i)->authorWithYear()].push_back(*i);
	  n._ptr[(*i)->persistentID()].push_back(*i);
	  n._ptrPublication[(*i)->persistentID()].push_back(*i);
	  ++i;
	}
	++iter;
      }
    }


  if(!nr.empty())
    {
      hash_map<string, list<NameRecord*>, Hash<string> >::iterator iter = nr.begin();
      hash_map<string, list<NameRecord*>, Hash<string> >::iterator end = nr.end();
      list<NameRecord*>::iterator i;
      list<NameRecord*>::iterator e;
      while(iter != end){
	i = (iter->second).begin();
	e = (iter->second).end();
	while(i != e){
	  n._nr.push_back(*i);
	  n._ptr[(*i)->persistentID()].push_back(*i);
	  n._ptrNameRecord[(*i)->persistentID()].push_back(*i);
#if 0
	  n._nodes.push_back(*i);
	  NRnode *pnrn = &*(--(n._nodes.end()));
	  if(NULL == pnrn->resolve(nrn, nr, unrn, unr))
	    n._hnr[(*i)->persistentID()] = pnrn;
	  //	  n._hnr[(*i)->persistentID()] = &*(--(n._nodes.end()));
	  else
	    delete pnrn;
#else
	  NRnode *pnrn = new NRnode(*i);
	  //	  if(NULL != pnrn->resolve(nrn, nr, unrn, unr))
	  if(pnrn != pnrn->resolve(r))
	    delete pnrn;
#endif
	  ++i;
	}
	++iter;
      }


      if(!nrn.empty()){
	hash_map<string, list<NRnode*>, Hash<string> >::iterator iter = nrn.begin();
	hash_map<string, list<NRnode*>, Hash<string> >::iterator end  = nrn.end();
	list<NRnode*>::iterator i;
	list<NRnode*>::iterator e;
	while(iter != end){
	  i = (iter->second).begin();
	  e = (iter->second).end();
	  while(i != e){
	    n._nodes.push_back(*i);
	    n._hnr[(*i)->namerecord.persistentID()] = *i;
	    //	    n._ptr[(*i)->persistentID()].push_back(*i);
	    n._ptrNRnode[(*i)->persistentID()].push_back(*i);
	    n._ptrNRnodeNR[(*i)->namerecord.persistentID()].push_back(*i);
#if 1
	    //handling equiv....at where?
	    NameRecord *pnr = (*i)->namerecord.ptr();
	    if(NULL != pnr){
	      list<d_Ref<Annotation> >::iterator ai = pnr->annotation.begin();
	      list<d_Ref<Annotation> >::iterator ae = pnr->annotation.end();
	      bool equiv = false;
	      while(ai != ae && !(equiv = !strncmp((*ai)->annotation(), "equiv", 5))){
		++ai;
	      }
	      if(equiv){
		list<d_Ref<NameRecord> >::iterator niter = (*ai)->from.begin();
		list<d_Ref<NameRecord> >::iterator nrend = (*ai)->from.end();
		while(niter != nrend){
		  NameRecord *_pnr = niter->ptr();
		  if(NULL != _pnr && _pnr != pnr){
		    string nm(_pnr->generic());
		    n._generic[nm].push_back(*i);
		    n._specific[_pnr->specific()].push_back(*i);
		    nm += " ";
		    nm += _pnr->specific();
		    n._name[nm].push_back(*i);
		  }
		  niter++;
		}
#if 0
		niter = (*ai)->to.begin();
		nrend = (*ai)->to.end();
		while(niter != nrend){
		  NameRecord *_pnr = niter->ptr();
		  if(NULL != _pnr && _pnr != pnr){
		    string nm(_pnr->generic());
		    n._generic[nm].push_back(*i);
		    n._specific[_pnr->->specific()].push_back(*i);
		    nm += " ";
		    nm += _pnr->specific();
		    n._name[nm].push_back(*i);
		    niter++;
		  }
		}
#endif
	      }
	    }
#endif
	    ++i;
	  }
	  ++iter;
	}

   if(!unrn.empty())
      {
        hash_map<string, list<d_Ref<NRnode>*>, Hash<string> >::iterator diter = unrn.begin();
	hash_map<string, list<NRnode*>, Hash<string> >::iterator nend = nrn.end();
	hash_map<string, list<NRnode*>, Hash<string> >::iterator ptrnend = n._ptrNRnode.end();
	hash_map<string, list<NRnode*>, Hash<string> >::iterator nfnd;
	while(diter != unrn.end()){
	  list<d_Ref<NRnode>* >::iterator iter = diter->second.begin();
	  while(iter != diter->second.end()){
	    if(NULL == (*iter)->ptr()){
	      string s((*iter)->persistentID());
	      if(s.length()){
		nfnd = nrn.find(s);
		if(nend == nfnd){
		  nfnd = n._ptrNRnode.find(s);
		  if(ptrnend == nfnd)
		    nfnd = nend;
		}

		if(nfnd != nend){
		  list<NRnode*>::iterator niter = nfnd->second.begin();
		  list<NRnode*>::iterator _nend = nfnd->second.end();
		  while(niter != _nend){
		    if((*niter)->persistentID() == s){
		      (*iter)->ptr(*niter);
		      break;
		    }
		    ++niter;
		  }
		}
		else{  
		  NRnode *p = new NRnode(s);
		  //		  if(NULL == p->resolve(nrn, nr, unrn, unr)){
		  NRnode *q = p->resolve(r);
		  if(p == q/*p->resolve(r)*/){
		    str = s;
		    nrndeduced++;
		    newDeduce++;
		    if(n._verbose)
		      cerr << "%deduce " << 4 - looplimit << ":"<< newDeduce << " " << p->persistentID() << "\n";
		  }
		  else
		    delete p;
		  (*iter)->ptr(q);
		}
	      }
	    }
	    ++iter;
	  }
	  ++diter;
	}
      }

      if(!an.empty())
	{
	  hash_map<string, list<Annotation*>, Hash<string> >::iterator iter = an.begin();
	  hash_map<string, list<Annotation*>, Hash<string> >::iterator end = an.end();
	  list<Annotation*>::iterator ai;
	  list<Annotation*>::iterator ae;
	  while(iter != end){
	    ai = (iter->second).begin();
	    ae = (iter->second).end();
	    
	    while(ai != ae){
	      n._anno.push_back(*ai);
	      n._ptr[(*ai)->persistentID()].push_back(*ai);
	      n._ptrAnnotation[(*ai)->persistentID()].push_back(*ai);
	      list<d_Ref<NameRecord> >::iterator niter = (*ai)->to.begin();
	      list<d_Ref<NameRecord> >::iterator nrend = (*ai)->to.end();
	      while(niter != nrend){
		//n._hnr[niter->persistentID()]->annotations.push_back(*ai);
		string niterPID(NameRecord::persistentID(niter->persistentID().c_str()));
		n._hnr[niterPID]->annotations.push_back(*ai);

		list<d_Ref<NameRecord> >::iterator fiter = (*ai)->from.begin();
		list<d_Ref<NameRecord> >::iterator fend = (*ai)->from.end();
		while(fiter != fend){
		  if(niter->ptr() != fiter->ptr()){
		    //n._hnr[niter->persistentID()]->from.insert(n._hnr[fiter->persistentID()]);
		    string fiterPID(NameRecord::persistentID(fiter->persistentID().c_str()));
		    n._hnr[niterPID]->from.push_back(n._hnr[fiterPID]);
		  }
		  ++fiter;
		}
		++niter;
	      }
	      niter = (*ai)->from.begin();
	      nrend = (*ai)->from.end();
	      if(ai != ae)
		while(niter != nrend){
		  list<d_Ref<NameRecord> >::iterator titer = (*ai)->to.begin();
		  list<d_Ref<NameRecord> >::iterator tend = (*ai)->to.end();
		  NameRecord tmpnr(niter->persistentID());
		  string niterPID(tmpnr.persistentID());
		  while(titer != tend){
		    if(niter->ptr() != titer->ptr()){
		      //n._hnr[niter->persistentID()]->to.insert(n._hnr[titer->persistentID()]);
		      string titerPID(NameRecord::persistentID(titer->persistentID().c_str()));
		      n._hnr[niterPID]->to.push_back(n._hnr[titerPID]);
		    }
		    ++titer;
		  }
		  ++niter;
		}
	      ++ai;
	    }
	    ++iter;
	  }
	}
    }

  hash_map<string, NRnode*, Hash<string> >::const_iterator iter = n._hnr.begin();
  hash_map<string, NRnode*, Hash<string> >::const_iterator end = n._hnr.end();
  while(iter != end){
    //    n._ptrNRnode[(iter->second)->persistentID()].push_back(iter->second);
    //    n._ptr[(iter->second)->persistentID()].push_back(iter->second);
    string nm((iter->second)->namerecord->generic());
    if(nm.length()){
      n._generic[nm].push_back(iter->second);
      nm += " ";
    }
    n._specific[(iter->second)->namerecord->specific()].push_back(iter->second);
    nm += (iter->second)->namerecord->specific();
    n._name[nm].push_back(iter->second);
    
    ++iter;
  }

  //sort and unify
  unifyHashMappedList(n._name/*, NRnode::gt()*/);
  unifyHashMappedList(n._generic);
  unifyHashMappedList(n._specific);
  unifyHashMappedList(n._ptrNRnode);
  unifyHashMappedList(n._ptrNRnodeNR);
  unifyHashMappedList(n._ptrNameRecord);
#endif
  return s;
}

ostream& operator<< (ostream &s, const Nomencurator& n)
{
  list<d_Ref<Publication> >::const_iterator pubiter;

  int pc = 0;
  s << "%Authors\n";

  list<d_Ref<Author> > auth(n._auth);
  //n._auth.sort();
  auth.sort();
  //  list<d_Ref<Author> >::const_iterator auiter = n._auth.begin();
  list<d_Ref<Author> >::const_iterator auiter = auth.begin();
  //list<d_Ref<Author> >::const_iterator auend  = n._auth.end();
  list<d_Ref<Author> >::const_iterator auend  = auth.end();
  while(auiter != auend){
    if(auiter->ptr()){
      s << "%" << ++pc << "\n";
      s << *(*auiter)<< "\n";
    }
    ++auiter;
  }

  pc = 0;
  s << "\n%Publications\n";
  for(pubiter = n._pub.begin();pubiter != n._pub.end(); ++pubiter){
    s << "%" << ++pc << "\n";
    if(pubiter->ptr())
      s << *(*pubiter)<< "\n";
  }

  pc = 0;
  list<d_Ref<NameRecord> >::const_iterator nriter = n._nr.begin();
  list<d_Ref<NameRecord> >::const_iterator nrend = n._nr.end();
  if(nriter != nrend)
    s << "\n%NameRecords\n";
  while(nriter != nrend){
    if(nriter->ptr()){
    s << "%" << ++pc << "\n";
    s << *(*nriter) << "\n";
    }
    ++nriter;
  }

  pc = 0;
  list<d_Ref<Annotation> >::const_iterator aniter = n._anno.begin();
  list<d_Ref<Annotation> >::const_iterator anend = n._anno.end();
  if(aniter != anend)
    s << "\n%Annotations\n";
  while(aniter != anend){
    if(aniter->ptr()){
      s << "%" << ++pc << "\n";
      s << *(*aniter)<< "\n";
    }
    ++aniter;
  }

  pc = 0;
  list<d_Ref<Citation> >::const_iterator citer = n._cite.begin();
  list<d_Ref<Citation> >::const_iterator cend = n._cite.end();
  if(citer != cend)
    s << "\n%Citations\n";
  while(citer != cend){
    if(citer->ptr()){
      s << "%" << ++pc << "\n";
      s << *(*citer)<< "\n";
    }
    ++citer;
  }
#if 0
  pc = 0;
  s << "\n%NRnodes\n";
  
  //  list<NRnode>::const_iterator nodi = n._nodes.begin();
  //  list<NRnode>::const_iterator node = n._nodes.end();
  list<NRnode*>::const_iterator nodi = n._nodes.begin();
  list<NRnode*>::const_iterator node = n._nodes.end();
  while(nodi != node){
    s << "%" << ++pc << "\n";
    if(aniter->ptr()){
      s << *(*nodi) << "\n";
    }
    ++nodi;
  }
#endif
  return s;
}

struct queryType{
  const char *const type;
  const int length;
  list<NRnode*> (Nomencurator::* search)(const string&) const;
};

struct d_ObjectQueryType{
    const char *const type;
    const int length;
    list<d_Object*> (Nomencurator::* search)(const string&) const;
};

static queryType queries[] = {
  {"generic", 5, &Nomencurator::generic},
  {"specific", 5, &Nomencurator::specific},
  {"name", 4, &Nomencurator::name},
  {"ptr", 3, &Nomencurator::queryNRnodeByPtr},
#if 0
  {"NRnode", 4, &Nomencurator::_queryByNRnode},
  {"NameRecord", 4, &Nomencurator::_queryByNameRecord},
  {"Publication", 4, &Nomencurator::_queryByPublication},
  {"author", 4, &Nomencurator::_queryByAuthor},
#endif
  {"", 0, NULL},
};

#if 0
static d_ObjectQueryType objectQueries[] = {
  {"generic", 5, &Nomencurator::generic},
  {"specific", 5, &Nomencurator::specific},
  {"name", 4, &Nomencurator::name},
  {"ptr", 3, &Nomencurator::queryNRnodeByPtr},
#if 0
  {"NRnode", 4, &Nomencurator::_queryByNRnode},
  {"NameRecord", 4, &Nomencurator::_queryByNameRecord},
  {"Publication", 4, &Nomencurator::_queryByPublication},
  {"Author", 4, &Nomencurator::_queryByAuthor},
  {"Object", 4, &Nomencurator::_queryByAuthor},
#endif
  {"", 0, NULL},
};
#endif

list<NRnode*> Nomencurator::query(const char *s)
{
  if(NULL == s || *s == 0)
    return list<NRnode*>();
  int words = 1;
  while(isspace(*s)) s++;
  register const char *head = s;
  const char *end = head + strlen(head);
  while(head < end){
    if(isspace(*head++))
      words++;
  }

  head = s;
  if(words > 1){
    while(!isspace(*head)) head++;
    while(isspace(*head)) head++;
  }

  register const char *p = head;
  while(p < end && (*p != ':' || *(p+1) != ':')) p++;

  if(p != end)
    return list<NRnode*>(queryNRnodeByPtr(head));
  
  if(words > 1){
    int i = 0;
    while(queries[i].length && strncmp(s, queries[i].type, queries[i].length))
      ++i;
    if(queries[i].length){
      return (this->*(queries[i].search))(head);
    }
  }
  list<NRnode*> res;
  int i = 0;

  while(queries[i].length){
    list<NRnode*> tmp((this->*(queries[i].search))(head));
    tmp.sort(NRnode::gt());
    res.merge(tmp);
    ++i;
  }

  return res;
}

template <class T> void unifyHashMappedList(hash_map<string, list<T>, Hash<string> > &h)
{
  hash_map<string, list<T>, Hash<string> >::iterator i = h.begin();
  hash_map<string, list<T>, Hash<string> >::iterator e = h.end();
  while(i != e){
    i->second.sort();
    i->second.unique();
    i++;
  }
}

template <class T> void unifyHashMappedList(hash_map<string, list<T*>, Hash<string> > &h)
{
  hash_map<string, list<T*>, Hash<string> >::iterator i = h.begin();
  hash_map<string, list<T*>, Hash<string> >::iterator e = h.end();
  while(i != e){
    i->second.sort();
    i->second.unique();
    i++;
  }
}

void unifyHashMappedList(hash_map<string, list<NRnode*>, Hash<string> > &h)
{
  hash_map<string, list<NRnode*>, Hash<string> >::iterator i = h.begin();
  hash_map<string, list<NRnode*>, Hash<string> >::iterator e = h.end();
  while(i != e){
    i->second.sort(/*NRnode::gt()*/);
    i->second.unique();
    i++;
  }
}


//template <class T> template <class StrictWeakOrdering> void unifyHashMappedList(hash_map<string, list<T*>, Hash<string> > &h, StrictWeakOrdering bf)
template <class T> void unifyHashMappedList(hash_map<string, list<T*>, Hash<string> > &h, binary_function<T, T, bool> bf)
{
  hash_map<string, list<T*>, Hash<string> >::iterator i = h.begin();
  hash_map<string, list<T*>, Hash<string> >::iterator e = h.end();
  while(i != e){
    i->second.sort(bf);
    i->second.unique();
    i++;
  }
}

template <class T> template <class StrictWeakOrdering> void unifyHashMappedList(hash_map<string, list<T>, Hash<string> > &h, StrictWeakOrdering bf)
{
  hash_map<string, list<T>, Hash<string> >::iterator i = h.begin();
  hash_map<string, list<T>, Hash<string> >::iterator e = h.end();
  while(i != e){
    i->second.sort(bf);
    i->second.unique();
    i++;
  }
}

list<NRnode*> Nomencurator::name(const string &s)
{ return _name[s];}

list<NRnode*> Nomencurator::generic(const string &s)
{ return list<NRnode*>(_generic[s]);}


list<NRnode*> Nomencurator::specific(const string &s)
{ return list<NRnode*>(_specific[s]);}

list<d_Object*> Nomencurator::ptr(const string &s)
{return list<d_Object*>(_ptr[string(s)]);}

list<NameRecord*> Nomencurator::ptrNameRcord(const string &s)
{return list<NameRecord*>(_ptrNameRecord[s]);}

list<Author*> Nomencurator::ptrAuthor(const string &s)
{return list<Author*>(_ptrAuthor[s]);}

list<Author*> Nomencurator::author(const string &s)
{return list<Author*>(_hauthorByName[s]);}

list<Citation*> Nomencurator::ptrCitation(const string &s)
{return list<Citation*>(_ptrCitation[s]);}

list<Publication*> Nomencurator::ptrPublication(const string &s)
{return list<Publication*>(_ptrPublication[s]);}

list<Annotation*> Nomencurator::ptrAnnotation(const string &s)
{return list<Annotation*>(_ptrAnnotation[s]);}

list<NRnode*> Nomencurator::queryNRnodeByPtr(const string &s)
{ return list<NRnode*>(_ptrNRnode[s]);}

list<NRnode*> Nomencurator::queryNRnodeByNRPtr(const string &s)
{ return list<NRnode*>(_ptrNRnodeNR[s]);}

//Database merger.  Do we need transuction manager?
bool Nomencurator::merge(Nomencurator& n)
{
  //added item holder to solve references
  hash_map<string, list<NameRecord*>, Hash<string> > &nr = (_ptrNameRecord);
  hash_map<string, list<Annotation*>, Hash<string> > &an = (_ptrAnnotation);
  hash_map<string, list<Citation*>, Hash<string> > &ci = (_ptrCitation);
  hash_map<string, list<Publication*>, Hash<string> > &pub = (_ptrPublication);
  hash_map<string, list<Author*>, Hash<string> > &au = (_ptrAuthor);
  hash_map<string, list<NRnode*>, Hash<string> > &nrn = (_ptrNRnode);

  hash_map<string, list<d_Ref<NameRecord>*>, Hash<string> > unr;
  hash_map<string, list<d_Ref<Annotation>*>, Hash<string> > uan;
  hash_map<string, list<d_Ref<Citation>*>, Hash<string> > uci;
  hash_map<string, list<d_Ref<Publication>*>, Hash<string> > upub;
  hash_map<string, list<d_Ref<Author>*>, Hash<string> > uau;
  hash_map<string, list<d_Ref<NRnode>*>, Hash<string> > unrn;

  Resolver r(_ptrNameRecord, _ptrAnnotation, _ptrCitation, _ptrPublication,
	     _ptrAuthor, _ptrNRnode);

  bool duplicate = false;
  while(!n._nr.empty()){
    NameRecord *p = n._nr.begin()->ptr();
    n._nr.pop_front();
    if(NULL != p){
      if(p/*NULL*/ != p->resolve(r))
	 {
	   duplicate = true;
	   //delete p; //?
	 }
    }
  }

  while(!n._nodes.empty()){
    NRnode* p = n._nodes.begin()->ptr();
    n._nodes.pop_front();
    if(NULL != p){
      //if(NULL != p->resolve(nrn, nr, unrn, unr))
      if(p != p->resolve(r))
	{
	  duplicate = true;
	  //	delete p; //?
	}
    }
  }

  while(!n._cite.empty()){
    Citation *p = n._cite.begin()->ptr();
    n._cite.pop_front();
    if(NULL != p){
      //      if(NULL != p->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau))
      if(p/*NULL*/ != p->resolve(r))
	{
	duplicate = true;
	//	delete p; //?
      }
    }
  }

  while(!n._pub.empty()){
    Publication *p = n._pub.begin()->ptr();
    n._pub.pop_front();
    if(NULL != p){
      //if(NULL != p->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau))
      if(p/*NULL*/ != p->resolve(r))
	{
	//	delete p; //?
	duplicate = true;
      }
    }
  }

  while(!n._auth.empty()){
    Author* p = n._auth.begin()->ptr();
    n._auth.pop_front();
    if(NULL != p){
      //      if(NULL != p->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau))
      if(p/*NULL*/ != p->resolve(r))
	{
	//	delete p; //?
	duplicate = true;
      }
    }
  }

  while(!n._anno.empty()){
    Annotation* p = n._anno.begin()->ptr();
    n._anno.pop_front();
    if(NULL != p){
      //      if(NULL != p->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau))
      if(p/*NULL*/ != p->resolve(r))
	{
	//delete p; //?
	duplicate = true;
      }
    }
  }

  bool ret = _resolve(r);

  if(!r.newNameRecords.empty()){
    register list<NameRecord*>::iterator i = r.newNameRecords.begin();
    register list<NameRecord*>::iterator e = r.newNameRecords.end();
    while(i != e){
      _nr.push_back(*i);
      i++;
    }
  }

  if(!r.newAnnotations.empty()){
    register list<Annotation*>::iterator i = r.newAnnotations.begin();
    register list<Annotation*>::iterator e = r.newAnnotations.end();
    while(i != e){
      _anno.push_back(*i);
      i++;
    }
  }

  if(!r.newCitations.empty()){
    register list<Citation*>::iterator i = r.newCitations.begin();
    register list<Citation*>::iterator e = r.newCitations.end();
    while(i != e){
      _cite.push_back(*i);
      i++;
    }
  }
  if(!r.newPublications.empty()){
    register list<Publication*>::iterator i = r.newPublications.begin();
    register list<Publication*>::iterator e = r.newPublications.end();
    while(i != e){
      _pub.push_back(*i);
      i++;
    }
  }
  if(!r.newAuthors.empty()){
    register list<Author*>::iterator i = r.newAuthors.begin();
    register list<Author*>::iterator e = r.newAuthors.end();
    while(i != e){
      _auth.push_back(*i);
      i++;
    }
  }
  if(!r.newNRnodes.empty()){
    register list<NRnode*>::iterator i = r.newNRnodes.begin();
    register list<NRnode*>::iterator e = r.newNRnodes.end();
    while(i != e){
      _nodes.push_back(*i);
      i++;
    }
  }

  n.clear();
  return duplicate; //ret;
}

bool Nomencurator::_resolve(Resolver &r)
{
  int authdeduced = 0;
  int pubdeduced = 0;
  int citededuced = 0;
  int nrdeduced = 0;
  int nrndeduced = 0;
  int newDeduce = 0;

  //Publicatoin contains NameRecord, Author, Annotation
  //NameRecord contains  NameRecord, Publication, Annotation
  //Annotation contains  NameRecord
  //Author contains Publication
  //Therefore loop need to close are:
  // Pub -> Auth -> Pub, twice
  // Pub -> NR -> Pub, twice
  // Pub -> NR -> Anno -> NR -> Pub, four times
  // Anno -> NR -> Anno, twice
  // use limit magicnumber four, not forever.

  string str;
  int looplimit = 4;
  int count = 0;
  int ded = 0;
  do{
    newDeduce = 0;
    
    // sequence sensitive?
    
    if(!r.unrn.empty()) {
      hash_map<string, list<d_Ref<NRnode>*>, Hash<string> >::iterator diter = r.unrn.begin();
      hash_map<string, list<NRnode*>, Hash<string> >::iterator nend = r.nrn.end();
      hash_map<string, list<NRnode*>, Hash<string> >::iterator ptrnend = _ptrNRnode.end();
      hash_map<string, list<NRnode*>, Hash<string> >::iterator nfnd;
      while(diter != r.unrn.end()){
	register list<d_Ref<NRnode>* >::iterator iter = diter->second.begin();
	register list<d_Ref<NRnode>* >::iterator current;
	while(iter != diter->second.end()){
	  current = iter;
	  iter++;
	  if(!(*current)->ptr()){
	    string s((*current)->persistentID());
	    if(s.length()){
	      nfnd = r.nrn.find(s);
	      if(nend == nfnd){
		nfnd = _ptrNRnode.find(s);
		if(ptrnend == nfnd)
		  nfnd = nend;
	      }
	      
	      if(nfnd != nend){
		list<NRnode*>::iterator niter = nfnd->second.begin();
		list<NRnode*>::iterator _nend = nfnd->second.end();
		while(niter != _nend){
		  if((*niter)->persistentID() == s){
		    (*current)->ptr(*niter);
		    break;
		  }
		  ++niter;
		}
	      }
	      else{  
		NRnode *p = new NRnode(s);
		NRnode *q = p->resolve(r);
		if(q != p)
		  delete p;
		else {
		  str = s;
		  nrndeduced++;
		  newDeduce++;
		  if(_verbose)
		    cerr << "%deduce " << 4 - looplimit << ":"<< newDeduce << " " << p->persistentID() << "\n";
		}
	      }
	    }
	  }
	}
	++diter;
      }
    }
    
   //if unsolved NameRecords exist...
   if(!r.unr.empty()) {
     hash_map<string, list<d_Ref<NameRecord>*>, Hash<string> >::iterator diter = r.unr.begin();
     hash_map<string, list<NameRecord*>, Hash<string> >::iterator nend = r.nr.end();
     hash_map<string, list<NameRecord*>, Hash<string> >::iterator ptrnend = _ptrNameRecord.end();
     hash_map<string, list<NameRecord*>, Hash<string> >::iterator nfnd;
     
     while(diter != r.unr.end()){
       register list<d_Ref<NameRecord>* >::iterator iter = diter->second.begin();
       register list<d_Ref<NameRecord>* >::iterator current;
       while(iter != diter->second.end()){
	 current = iter;
	 iter++;
	 if(NULL != (*current)->ptr()){
	   string s((*current)->ptr()->persistentID());
	   
	   if(s.length()){
	     nfnd = r.nr.find(s);
	     if(nfnd == r.nr.end()){
	       nfnd = _ptrNameRecord.find(s);
	       if(nfnd == _ptrNameRecord.end())
		 nfnd = r.nr.end();
	     }
	     
	     if(nfnd != r.nr.end()){ //found
	       list<NameRecord*>::iterator niter = nfnd->second.begin();
	       list<NameRecord*>::iterator _nend = nfnd->second.end();
	       while(niter != _nend){
		 //		 if((*current)->ptr() != *current &&
		 if((*current)->ptr() != (*current)->ptr() &&
		    (*niter)->persistentID() == s){
		   (*current)->ptr(*niter);
		   break;
		 }
		 ++niter;
	       }
	     }
	     else{  
	       NameRecord *p = new NameRecord(s);
	       NameRecord *q = p->resolve(r);
	       if(q == p){
		 ded++;
		 r.newNameRecords.push_back(p);
		 str = s;
		 nrdeduced++;
		 newDeduce++;
		 if(_verbose)
		   cerr << "%deduce " << 4 - looplimit << ":"<< newDeduce << " " << p->persistentID() << "\n";
	       }
	       else
		 delete p;
	       //	       (*current)->ptr(q);
	     }
	   }
	 }
	 else{
	   string s((*current)->persistentID());
	   
	   if(s.length()){
	     NameRecord *p = new NameRecord(s);
	     NameRecord *q = p->resolve(r);
	     if(q != p)
	       delete p;
	     else {
	       str = s;
	       nrdeduced++;
	       newDeduce++;
	       if(_verbose)
		 cerr << "%deduce " << 4 - looplimit << ":"<< newDeduce << " " << p->persistentID() << "\n";
	     }
	   }
	 }
       }
       ++diter;
     }
   }

#if 0
   if(!r.unrn.empty()) {

     hash_map<string, list<d_Ref<NRnode>*>, Hash<string> >::iterator diter = r.unrn.begin();
     hash_map<string, list<NRnode*>, Hash<string> >::iterator nend = r.nrn.end();
     hash_map<string, list<NRnode*>, Hash<string> >::iterator ptrnend = _ptrNRnode.end();
     hash_map<string, list<NRnode*>, Hash<string> >::iterator nfnd;
     while(diter != r.unrn.end()){
       list<d_Ref<NRnode>* >::iterator iter = diter->second.begin();
       while(iter != diter->second.end()){
	 if(NULL == (*iter)->ptr()){
	   string s((*iter)->persistentID());

	   if(s.length()){
	     nfnd = r.nrn.find(s);
	     if(nend == nfnd){
	       nfnd = _ptrNRnode.find(s);
	       if(ptrnend == nfnd)
		 nfnd = nend;
	     }
	     
	     if(nfnd != nend){
	       list<NRnode*>::iterator niter = nfnd->second.begin();
	       list<NRnode*>::iterator _nend = nfnd->second.end();
	       while(niter != _nend){
		 if((*niter)->persistentID() == s){
		   (*iter)->ptr(*niter);
		   break;
		 }
		 ++niter;
	       }
	     }//end of find
	     else{  
	       NRnode *p = new NRnode(s);
	       NRnode *q = p->resolve(r);
	       if(p != q){
		 (*iter)->ptr(q);
		 delete p;
	       }
	       else {
		 (*iter)->ptr(p);
		 r.newNRnodes.push_back(p);
		 str = s;
		 nrndeduced++;
		 newDeduce++;
		 if(_verbose)
		   cerr << "%deduce " << 4 - looplimit << ":"<< newDeduce << " " << p->persistentID() << "\n";
	       }
	     }
	   }
	 }
	 ++iter;
       }
       ++diter;
     }
   }
   
#endif

   if(!r.uci.empty()) {
     hash_map<string, list<d_Ref<Citation>*>, Hash<string> >::iterator diter = r.uci.begin();
     hash_map<string, list<Citation*>, Hash<string> >::iterator cend = r.ci.end();
     hash_map<string, list<Citation*>, Hash<string> >::iterator pcend = _ptrCitation.end();
     hash_map<string, list<Citation*>, Hash<string> >::iterator cfnd;
     while(diter != r.uci.end()){
       register list<d_Ref<Citation>* >::iterator iter = diter->second.begin();
       register list<d_Ref<Citation>* >::iterator current;
       while(iter != diter->second.end()){
	 current = iter;
	 ++iter;
	 if(!(*current)->ptr()){
	   string s((*current)->persistentID());
	   if(s.length()){
	     cfnd = r.ci.find(s);
	     if(cend == cfnd){
	       cfnd = _ptrCitation.find(s);
	       if(pcend == cfnd)
		 cfnd = cend;
	     }
	     if(cend != cfnd){
	       list<Citation*>::iterator citer = cfnd->second.begin();
	       list<Citation*>::iterator _cend = cfnd->second.end();
		  while(citer != _cend){
		    if((*citer)->persistentID() == s){
		      (*current)->ptr(*citer);
		      break;
		    }
		    ++citer;
		  }
	     }
	     else{
	       Citation *p = new Citation(s);
	       Citation *q = p->resolve(r);
	       if(q != p)
		 delete p;
	       else{
		 str = s;
		 citededuced++;
		 newDeduce++;
		 if(_verbose)
		   cerr << "%deduce " << 4 - looplimit << ":"<< newDeduce << " " << p->persistentID() << "\n";
	       }
	     }
	   }
	 }
       }
       ++diter;
     }
   }
   
   if(!r.upub.empty()) {
       
     hash_map<string, list<d_Ref<Publication>*>, Hash<string> >::iterator diter = r.upub.begin();
     hash_map<string, list<Publication*>, Hash<string> >::iterator pend = r.pub.end();
     hash_map<string, list<Publication*>, Hash<string> >::iterator ppend = _ptrPublication.end();
     hash_map<string, list<Publication*>, Hash<string> >::iterator pfnd;

     while(diter != r.upub.end()){
       register list<d_Ref<Publication>* >::iterator iter = diter->second.begin();
       register list<d_Ref<Publication>* >::iterator current;
       while(iter != diter->second.end()){
	 current = iter;
	 iter++;
	 if(!(*current)->ptr()){
	   string s((*current)->persistentID());
	   if(s.length()){
	     pfnd = r.pub.find(s);
	     if(pfnd == pend){
	       pfnd = _ptrPublication.find(s);
	       if(pfnd == ppend)
		 pfnd = pend;
	     }
	     if(pfnd != pend){
	       list<Publication*>::iterator piter = pfnd->second.begin();
	       list<Publication*>::iterator _pend = pfnd->second.end();
	       while(piter != _pend){
		 if((*piter)->persistentID() == s){
		   (*current)->ptr(*piter);
		   break;
		 }
		 ++piter;
	       }
	     }
	     else{
	       Publication *p = new Publication(s);
	       Publication *q = p->resolve(r);
	       if(q != p)
		 delete p;
	       else{
		 pubdeduced++;
		 newDeduce++;
		 if(_verbose)
		   cerr << "%deduce " << 4 - looplimit << ":"<< newDeduce << " " << p->persistentID() << "\n";
	       }
	     }
	   }
	 }
       }
       ++diter;
     }
   }

   if(!r.uau.empty()) {
     hash_map<string, list<d_Ref<Author>*>, Hash<string> >::iterator diter = r.uau.begin();
     hash_map<string, list<Author*>, Hash<string> >::iterator auend = r.au.end();
     hash_map<string, list<Author*>, Hash<string> >::iterator pauend = _ptrAuthor.end();
     hash_map<string, list<Author*>, Hash<string> >::iterator aufnd;
     while(diter != r.uau.end()){
       register list<d_Ref<Author>* >::iterator iter = diter->second.begin();
       register list<d_Ref<Author>* >::iterator current;
       while(iter != diter->second.end()){
	 current = iter;
	 iter++;
	 if(!(*current)->ptr()){
	   string s((*current)->persistentID());
	   if(s.length()){
	     aufnd = r.au.find(s);
	     if(aufnd == auend){
	       aufnd = _ptrAuthor.find(s);
	       if(aufnd == pauend)
		 aufnd = auend;
	     }
	     if(aufnd != auend){
	       list<Author*>::iterator auiter = aufnd->second.begin();
	       list<Author*>::iterator _auend = aufnd->second.end();
	       while(auiter != _auend){
		 if((*auiter)->persistentID() == s){
		   (*current)->ptr(*auiter);
		   break;
		 }
		 ++auiter;
	       }
	     }
	     else{
	       int l = s.length();
	       if(l > 0){
		 int i;
		 for(i =0 ; i < l && '_' != s[i]; i++){}
		 if(i != l)
		   l = i;
		 Author *p = new Author(s);
		 Author *q = p->resolve(r);
		 
		 if(q != p)
		   delete p;
		 else{
		   authdeduced++;
		   newDeduce++;
		   if(_verbose)
		     cerr << "%deduce " << 4 - looplimit << ":"<< newDeduce << " " << p->persistentID() << "\n";
		 }
	       }
	     }
	   }
	 }
       }
       ++diter;
     }
   }
   
#if 0
    //more deduce
    list<string> ls;
    for(p = _nr.begin();p != _nr.end();++p){
      if(((*p)->authorityStr() != (*p)->recorderStr())
	 && ((*p)->annotation).empty()){
	string s("NameRecord::");
	s += (*p)->group(); s+="_";
	s += (*p)->generic(); s+="_";
	s += (*p)->specific(); s+="_";
	s += (*p)->authorityStr(); s+="_";
	s += (*p)->authorityStr();
	ls.push_back(s);
      }
    }
    if(!ls.empty())
      {
	ls.sort(); ls.unique();
	list<string>::iterator lsiter;
	for(lsiter = ls.begin();lsiter != ls.end();++lsiter){
	  tnr.push_back(new NameRecord(*lsiter));
	  ++g;
	}
	ls.clear();tnr.sort();
	_nr.merge(tnr);_nr.unique(d_Ref_unify<NameRecord>());
      }
#endif
  } while(newDeduce && --looplimit);

  if(_verbose > 2)
    cerr << "% Deduce phase over\n";
  if(_verbose && looplimit == 0 && 0 != newDeduce)
    cerr << "% Deduce not yet converge. " << newDeduce 
	 << " object" << (newDeduce >1? "s were":" was") << " tried to deduce in the last loop.\n"
	 << "% Data may contain mistake" << (newDeduce >1? "s":"") << ".  Last deduce on: " 
	 << str << "\n";

#if 0 //1
  // name change propagation
  if(!r.an.empty()){
    list<d_Ref<Annotation> >::iterator ani = _anno.begin();
    list<d_Ref<Annotation> >::iterator ane = _anno.end();
    string ref("refer"); //XXX
    while(ani != ane){
      if((*ani)->annotation() == ref){
	NameRecord *to = ((*ani)->to.begin())->ptr(); //only one, because ref
	NameRecord *from = ((*ani)->from.begin())->ptr(); //only one, because ref
	if(!(to->lowerNR.empty()) && to->specific() != from->specific()){
	  list<d_Ref<NameRecord> >::iterator toiter = to->lowerNR.begin();
	  list<d_Ref<NameRecord> >::iterator toend = to->lowerNR.end();
	  while(toiter != toend){
	    NameRecord *tonr = toiter->ptr();
	    list<d_Ref<NameRecord> >::iterator friter = from->lowerNR.begin();
	    list<d_Ref<NameRecord> >::iterator frend = from->lowerNR.end();
	    while(friter != frend && tonr != friter->ptr()){ ++friter; }
	    if(friter == frend){  // not found
	      NameRecord *prop = new NameRecord(*tonr);
	      prop->recorder()->publication(from->recorder()->publication().ptr());
	      prop->recorder()->page(from->recorder()->page());
	      prop->higher(from);
	      prop->generic(from->specific());
	      Annotation *pan = new Annotation;
	      pan->to.push_back(tonr);
	      pan->from.push_back(prop);
	      pan->annotation("propagate");
	      prop->annotatioclear();
	      prop->annotatiopush_back(pan);
	      if(NULL != &*(prop->typeOf())){
		prop->typeOf(from);
		if(NULL != &*(from->typeOf()))
		  from->type(prop);
	      }
	    }
	    ++toiter;
	  }
	}
      }
      ++ani;
    }
  }
#endif


  /*
   *
   * Create hash tables of records (including deduced records).
   *
   */

  /*
   * Author objects are stored in _hauth, _ptr and _ptrAuthor
   */
  if(!r.au.empty()) {
    hash_map<string, list<Author*>, Hash<string> >::iterator iter = r.au.begin();
    hash_map<string, list<Author*>, Hash<string> >::iterator end = r.au.end();
    list<Author*>::iterator i;
    list<Author*>::iterator e;
    while(iter != end){
      i = (iter->second).begin(); e = (iter->second).end();
      while(i != e){
	string s((*i)->persistentID());
	_hauth[s] = *i;
	_ptrAuthor[s].push_back(*i);
	_ptr[s].push_back(*i);
	Author* a = *i;
	_hauthorByName[a->surname()].push_back(*i);
	_hauthorByName[a->fullname()].push_back(*i);
	_hauthorByName[a->initializedName()].push_back(*i);
	++i;
      }
      ++iter;
    }
  }

  /*
   * Hash tables of Publication object
   */
  if(!r.pub.empty()) {
    hash_map<string, list<Publication*>, Hash<string> >::iterator iter = r.pub.begin();
    hash_map<string, list<Publication*>, Hash<string> >::iterator end = r.pub.end();
    list<Publication*>::iterator i, e;
    while(iter != end){
      i = (iter->second).begin(); e = (iter->second).end();
      while(i != e){
	string s((*i)->persistentID());
	_hpub[s] = *i;
	_hashPubByAuthor[(*i)->authorWithYear()].push_back(*i);
	_ptr[s].push_back(*i);
	_ptrPublication[s].push_back(*i);
	++i;
      }
      ++iter;
    }
  }

  int ncnt = 0;
  if(!r.nr.empty()) {
    
    hash_map<string, list<NameRecord*>, Hash<string> >::iterator iter = r.nr.begin();
    hash_map<string, list<NameRecord*>, Hash<string> >::iterator end = r.nr.end();
    list<NameRecord*>::iterator i;
    list<NameRecord*>::iterator e;
    while(iter != r.nr.end()/*end*/){
      i = (iter->second).begin();
      e = (iter->second).end();
      while(i != /*e*/(iter->second).end()){
	string s((*i)->persistentID());
	_ptr[s].push_back(*i);
	_ptrNameRecord[s].push_back(*i);
#if 0
	_nodes.push_back(*i);
	NRnode *pnrn = &*(--(_nodes.end()));
	pnrn->resolve(r);
	_hnr[(*i)->persistentID()] = pnrn;
	//	  _hnr[(*i)->persistentID()] = &*(--(_nodes.end()));
#else
	NRnode *pnrn = new NRnode(*i);
	NRnode *res = pnrn->resolve(r);
	if(pnrn != res){
	  //	    delete pnrn;
	}
	else{ //?
	  //	  _hnr[(*i)->persistentID()] = pnrn;
	  ncnt++;
	}
	//	  else
	//	    r.newNRnodes.push_back(pnrn);
#endif
	count++;
	++i;
      }
      ++iter;
    }
    
    if(!r.unrn.empty()) {
      hash_map<string, list<d_Ref<NRnode>*>, Hash<string> >::iterator diter = r.unrn.begin();
      hash_map<string, list<NRnode*>, Hash<string> >::iterator nend = r.nrn.end();
      hash_map<string, list<NRnode*>, Hash<string> >::iterator ptrnend = _ptrNRnode.end();
      hash_map<string, list<NRnode*>, Hash<string> >::iterator nfnd;
      while(diter != r.unrn.end()){
	list<d_Ref<NRnode>* >::iterator iter = diter->second.begin();
	while(iter != diter->second.end()){
	  if(NULL == (*iter)->ptr()){
	    string s((*iter)->persistentID());
	    if(s.length()){
	      nfnd = r.nrn.find(s);
	      if(r.nrn.end() == nfnd){
		nfnd = _ptrNRnode.find(s);
		if(_ptrNRnode.end() == nfnd)
		  nfnd = r.nrn.end();
	      }
	      
	      if(nfnd != r.nrn.end()){
		list<NRnode*>::iterator niter = nfnd->second.begin();
		list<NRnode*>::iterator _nend = nfnd->second.end();
		while(niter != _nend){
		  if((*niter)->persistentID() == s){
		    (*iter)->ptr(*niter);
		    break;
		  }
		  ++niter;
		}
	      }
	      else{  
		NRnode *p = new NRnode(s);
		NRnode *q = p->resolve(r);
		if(p != q){
		  (*iter)->ptr(q);
		  delete p;
		}
		else {
		  (*iter)->ptr(p);
		  str = s;
		  nrndeduced++;
		  newDeduce++;
		  if(_verbose)
		    cerr << "%deduce " << 4 - looplimit << ":"<< newDeduce << " " << p->persistentID() << "\n";
		}
	      }
	    }
	  }
	  ++iter;
	}
	++diter;
      }
    }
    
    if(!r.nrn.empty()){
      hash_map<string, list<NRnode*>, Hash<string> >::iterator iter = r.nrn.begin();
      hash_map<string, list<NRnode*>, Hash<string> >::iterator end  = r.nrn.end();
      list<NRnode*>::iterator i;
      list<NRnode*>::iterator e;
      while(iter != /*end*/r.nrn.end()){
	i = (iter->second).begin();
	e = (iter->second).end();
	while(i != e){
	  //r.newNRnodes.push_back(*i);
	  (*i)->resolve(r);
	  string ns((*i)->persistentID());
	  string s("");
#if 0
	  if(NULL != (*i)->namerecord.ptr())
	    s = ((*i)->namerecord)->persistentID();
	  else{
	    s = ((*i)->namerecord).persistentID();
	  }
#else
	  if(NULL == (*i)->namerecord.ptr()){
	    NameRecord *nr = new NameRecord(((*i)->namerecord).persistentID());
	    NameRecord *p = nr->resolve(r);
	    if(nr/*NULL*/ != p){
	      (*i)->namerecord.ptr(p);
	      delete nr;
	    }
	    else
	      (*i)->namerecord.ptr(nr);
	  }
	  s = ((*i)->namerecord)->persistentID();
#endif
	  _hnr[s] = (*i);
	  _ptr[ns].push_back(*i);
	  _ptrNRnode[ns].push_back(*i);
	  _ptrNRnodeNR[s].push_back(*i);
	  
	  string nm((*i)->namerecord.ptr()->generic());
	  if(nm.length()){
	    _generic[nm].push_back(*i);
	    nm += " ";
	  }
	  _specific[(*i)->namerecord->specific()].push_back(*i);
	  nm += (*i)->namerecord->specific();
	  _name[nm].push_back(*i);
	  
#if 1
	  //handling equiv....at where?
	  NameRecord *pnr = (*i)->namerecord.ptr();
	  if(NULL != pnr){
	    list<d_Ref<Annotation> >::iterator ai = pnr->annotation.begin();
	    list<d_Ref<Annotation> >::iterator ae = pnr->annotation.end();
	    bool equiv = false;
	    
	    while(ai != ae && equiv) {
	      if(NULL != ai -> ptr())
		equiv = !strncmp((*ai)->annotation(), "equiv", 5);
	      ++ai;
	    }
	    
	    if(equiv){
	      list<d_Ref<NameRecord> >::iterator niter = (*ai)->from.begin();
	      list<d_Ref<NameRecord> >::iterator nrend = (*ai)->from.end();
	      while(niter != nrend){
		NameRecord *_pnr = niter->ptr();
		if(NULL != _pnr && _pnr != pnr){
		  string nm(_pnr->generic());
		  _generic[nm].push_back(*i);
		  _specific[_pnr->specific()].push_back(*i);
		  nm += " ";
		  nm += _pnr->specific();
		  _name[nm].push_back(*i);
		}
		niter++;
	      }
#if 0
	      niter = (*ai)->to.begin();
	      nrend = (*ai)->to.end();
	      while(niter != nrend){
		NameRecord *_pnr = niter->ptr();
		if(NULL != _pnr && _pnr != pnr){
		  string nm(_pnr->generic());
		  _generic[nm].push_back(*i);
		  _specific[_pnr->->specific()].push_back(*i);
		  nm += " ";
		  nm += _pnr->specific();
		  _name[nm].push_back(*i);
		  niter++;
		}
	      }
#endif
	    }
	  }
#endif
	  ++i;
	}
	++iter;
      }
      
#if 1
      if(!r.unrn.empty()) {
	hash_map<string, list<d_Ref<NRnode>*>, Hash<string> >::iterator diter = r.unrn.begin();
	hash_map<string, list<NRnode*>, Hash<string> >::iterator nend = r.nrn.end();
	hash_map<string, list<NRnode*>, Hash<string> >::iterator ptrnend = _ptrNRnode.end();
	hash_map<string, list<NRnode*>, Hash<string> >::iterator nfnd;
	while(diter != r.unrn.end()){
	  list<d_Ref<NRnode>* >::iterator iter = diter->second.begin();
	  while(iter != diter->second.end()){
	    if(NULL == (*iter)->ptr()){
	      string s((*iter)->persistentID());
	      if(s.length()){
		nfnd = r.nrn.find(s);
		if(nend == nfnd){
		  nfnd = _ptrNRnode.find(s);
		  if(ptrnend == nfnd)
		    nfnd = nend;
		}
		
		if(nfnd != nend){
		  list<NRnode*>::iterator niter = nfnd->second.begin();
		  list<NRnode*>::iterator _nend = nfnd->second.end();
		  while(niter != _nend){
		    if((*niter)->persistentID() == s){
		      (*iter)->ptr(*niter);
		      break;
		    }
		    ++niter;
		  }
		}
		else{  
		  NRnode *p = new NRnode(s);
		  NRnode *q = new NRnode(s);
		  if(p != p->resolve(r)){
		    delete p;
		  }
		  else {
		    str = s;
		    nrndeduced++;
		    newDeduce++;
		    if(_verbose)
		      cerr << "%deduce " << 4 - looplimit << ":"<< newDeduce << " " << p->persistentID() << "\n";
		  }
		  //		  (*iter)->ptr(q);
		}
	      }
	    }
	    ++iter;
	  }
	  ++diter;
	}
      }
#endif
      //hierarchy reconstruct
      //hopefully it is already solved by constructor of NRnode....
      list<d_Ref<NRnode> >::iterator nri = _nodes.begin();
      list<d_Ref<NRnode> >::iterator nre = _nodes.end();
      while(nri != nre){
	NameRecord *pnr = nri->ptr()->namerecord->higher().ptr();
	if(NULL != pnr){
	  NRnode *pnrn = _hnr[pnr->persistentID()];
	  if(NULL != pnrn){
	    if(NULL == nri->ptr()->higher.ptr())
	      (*nri)->higher.ptr(pnrn);
	    list <d_Ref<NRnode> >::iterator dni = pnrn->lower.begin();
	    list <d_Ref<NRnode> >::iterator dne = pnrn->lower.end();
	    while(dni != dne && dni->ptr() != nri->ptr()){
	      ++dni;
	    }
	    if(dni == dne){
	      pnrn->lower.push_back(nri->ptr());
	    }
	  }
	}
	++nri;
      }
    }
    
    if(!r.an.empty()) {
      hash_map<string, list<Annotation*>, Hash<string> >::iterator iter = r.an.begin();
      hash_map<string, list<Annotation*>, Hash<string> >::iterator end = r.an.end();
      list<Annotation*>::iterator ai;
      list<Annotation*>::iterator ae;
      while(iter != end){
	ai = (iter->second).begin();
	ae = (iter->second).end();
	
	while(ai != ae){
	  //	      r.newAnnotations.push_back(*ai);
	  string s((*ai)->persistentID());
	  _ptr[s].push_back(*ai);
	  _ptrAnnotation[s].push_back(*ai);
	  list<d_Ref<NameRecord> >::iterator niter = (*ai)->to.begin();
	  list<d_Ref<NameRecord> >::iterator nrend = (*ai)->to.end();
	  while(niter != nrend){
	    //_hnr[niter->persistentID()]->annotations.push_back(*ai);
	    //		string niterPID(NameRecord::persistentID(niter->persistentID().c_str()));
	    string niterPID;
	    if(NULL == niter->ptr()){
	      niterPID = niter->persistentID();
	    }
	    else{
	      niterPID = niter->ptr()->persistentID();
	    }
	    /*
	      if(_hnr[niterPID] == NULL){
	      _hnr[niterPID] = new NameRecord(niterPID.c_str());
	      }
	      */
	    if(_hnr[niterPID] == NULL){
	      NRnode *nrn = new NRnode(niterPID);
	      //		  if(r.nrn.find(nrn->persistentID())!= r.nrn.end())
	      //		    cerr << "NR exists " <<  niterPID<<endl;
	    }
	    //		  _hnr[niterPID]->annotations.push_back(*ai);
	    
	    list<d_Ref<NameRecord> >::iterator fiter = (*ai)->from.begin();
	    list<d_Ref<NameRecord> >::iterator fend = (*ai)->from.end();
	    while(fiter != fend){
	      if(niter->ptr() != fiter->ptr()){
		//_hnr[niter->persistentID()]->from.insert(_hnr[fiter->persistentID()]);
		string fiterPID;// (NameRecord::persistentID(fiter->persistentID().c_str()));
		if(NULL != fiter->ptr())
		  fiterPID = fiter->ptr()->persistentID();
		else
		  fiterPID = fiter->persistentID();
		if(NULL != niter->ptr()){
		  if(_hnr[niterPID] != NULL){
		    if(_hnr[fiterPID] != NULL)
		      _hnr[niterPID]->from.push_back(_hnr[fiterPID]);
		  }
		  else{
		    //			cerr << "not found " << niterPID << endl;
		  }
		}
	      }
	      ++fiter;
	    }
	    ++niter;
	  }
	  niter = (*ai)->from.begin();
	  nrend = (*ai)->from.end();
	  if(ai != ae)
	    while(niter != nrend){
	      list<d_Ref<NameRecord> >::iterator titer = (*ai)->to.begin();
	      list<d_Ref<NameRecord> >::iterator tend = (*ai)->to.end();
	      
	      string niterPID("");
	      if(NULL != niter->ptr()) 
		niterPID = niter->ptr()->persistentID();
	      else
		niterPID = niter->persistentID();
	      
	      while(titer != tend){
		if(niter->ptr() != titer->ptr()){
		  //_hnr[niter->persistentID()]->to.insert(_hnr[titer->persistentID()]);
		  string titerPID;//(NameRecord::persistentID(titer->persistentID().c_str()));
		  if(NULL != titer->ptr()) 
		    titerPID = titer->ptr()->persistentID();
		  else
		    titerPID = titer->persistentID();
		  
		  if(_hnr[niterPID] != NULL){
		    if(_hnr[titerPID] != NULL)
		      _hnr[niterPID]->to.push_back(_hnr[titerPID]);
		  }
		}
		++titer;
	      }
	      ++niter;
	    }
	  ++ai;
	}
	++iter;
      }
    }
  }
  
  /*
   * Citation objects are stored in _ptr and _ptrCitation 
   */
  if(!r.ci.empty()) {
    hash_map<string, list<Citation*>, Hash<string> >::iterator iter = r.ci.begin();
    hash_map<string, list<Citation*>, Hash<string> >::iterator end = r.ci.end();
    list<Citation*>::iterator i;
    list<Citation*>::iterator e;
    while(iter != end){
      i = (iter->second).begin(); e = (iter->second).end();
      while(i != e){
	string s((*i)->persistentID());
	_ptrCitation[s].push_back(*i);
	_ptr[s].push_back(*i);
	++i;
      }
      ++iter;
    }
  }
  
  if(!r.nr.empty())
    {
      hash_map<string, list<NameRecord*>, Hash<string> >::iterator iter = r.nr.begin();
      hash_map<string, list<NameRecord*>, Hash<string> >::iterator end = r.nr.end();
      list<NameRecord*>::iterator i;
      list<NameRecord*>::iterator e;
      while(iter != r.nr.end()/*end*/){
	i = (iter->second).begin();
	e = (iter->second).end();
	while(i != /*e*/(iter->second).end()){
	  string s((*i)->persistentID());
	  _ptr[s].push_back(*i);
	  _ptrNameRecord[s].push_back(*i);
#if 0
	  _nodes.push_back(*i);
	  NRnode *pnrn = &*(--(_nodes.end()));
	  pnrn->resolve(r);
	  _hnr[(*i)->persistentID()] = pnrn;
	  //	  _hnr[(*i)->persistentID()] = &*(--(_nodes.end()));
#else
	  NRnode *pnrn = new NRnode(*i);
	  NRnode *res = pnrn->resolve(r);
	  if(pnrn != res){
	    //	    delete pnrn;
	  }
	  else{ //?
	    _hnr[(*i)->persistentID()] = pnrn;
	    ncnt++;
	  }
	  //	  else
	  //	    r.newNRnodes.push_back(pnrn);
#endif
	  count++;
	  ++i;
	}
	++iter;
      }

      if(!r.unrn.empty())
	{
	  hash_map<string, list<d_Ref<NRnode>*>, Hash<string> >::iterator diter = r.unrn.begin();
	  hash_map<string, list<NRnode*>, Hash<string> >::iterator nend = r.nrn.end();
	  hash_map<string, list<NRnode*>, Hash<string> >::iterator ptrnend = _ptrNRnode.end();
	  hash_map<string, list<NRnode*>, Hash<string> >::iterator nfnd;
	  while(diter != r.unrn.end()){
	    list<d_Ref<NRnode>* >::iterator iter = diter->second.begin();
	    while(iter != diter->second.end()){
	      if(NULL == (*iter)->ptr()){
		string s((*iter)->persistentID());
		if(s.length()){
		  nfnd = r.nrn.find(s);
		  if(nend == nfnd){
		    nfnd = _ptrNRnode.find(s);
		    if(ptrnend == nfnd)
		      nfnd = nend;
		  }
		  
		  if(nfnd != nend){
		    list<NRnode*>::iterator niter = nfnd->second.begin();
		    list<NRnode*>::iterator _nend = nfnd->second.end();
		    while(niter != _nend){
		      if((*niter)->persistentID() == s){
			(*iter)->ptr(*niter);
			break;
		      }
		      ++niter;
		    }
		  }
		  else{  
		    NRnode *p = new NRnode(s);
		    NRnode *q = p->resolve(r);
		    if(p != q)
		      delete p;
		    else {
		      str = s;
		      nrndeduced++;
		      newDeduce++;
		      if(_verbose)
			cerr << "%deduce " << 4 - looplimit << ":"<< newDeduce << " " << p->persistentID() << "\n";
		    }
		    //		    (*iter)->ptr(q);
		  }
		}
	      }
	      ++iter;
	    }
	    ++diter;
	  }
	}
      
      if(!r.nrn.empty()){
	hash_map<string, list<NRnode*>, Hash<string> >::iterator iter = r.nrn.begin();
	hash_map<string, list<NRnode*>, Hash<string> >::iterator end  = r.nrn.end();
	list<NRnode*>::iterator i;
	list<NRnode*>::iterator e;
	while(iter != /*end*/r.nrn.end()){
	  i = (iter->second).begin();
	  e = (iter->second).end();
	  while(i != e){
	    //r.newNRnodes.push_back(*i);
	    string s("");
	    if(NULL != (*i)->namerecord.ptr())
	      s = (*i)->namerecord->persistentID();
	    else
	      s = (*i)->namerecord.persistentID();
	    _hnr[s] = (*i);
	    string ns((*i)->persistentID());
	    _ptr[ns].push_back(*i);
	    _ptrNRnode[ns].push_back(*i);
	    _ptrNRnodeNR[s].push_back(*i);

	    string nm((*i)->namerecord->generic());
	    if(nm.length()){
	      _generic[nm].push_back(*i);
	      nm += " ";
	    }
	    _specific[(*i)->namerecord->specific()].push_back(*i);
	    nm += (*i)->namerecord->specific();
	    _name[nm].push_back(*i);


#if 1
	    //handling equiv....at where?
	    NameRecord *pnr = (*i)->namerecord.ptr();
	    if(NULL != pnr){
	      list<d_Ref<Annotation> >::iterator ai = pnr->annotation.begin();
	      list<d_Ref<Annotation> >::iterator ae = pnr->annotation.end();
	      bool equiv = false;

	      while(ai != ae && equiv) {
		if(NULL != ai -> ptr())
		  equiv = !strncmp((*ai)->annotation(), "equiv", 5);
		++ai;
	      }

	      if(equiv){
		list<d_Ref<NameRecord> >::iterator niter = (*ai)->from.begin();
		list<d_Ref<NameRecord> >::iterator nrend = (*ai)->from.end();
		while(niter != nrend){
		  NameRecord *_pnr = niter->ptr();
		  if(NULL != _pnr && _pnr != pnr){
		    string nm(_pnr->generic());
		    _generic[nm].push_back(*i);
		    _specific[_pnr->specific()].push_back(*i);
		    nm += " ";
		    nm += _pnr->specific();
		    _name[nm].push_back(*i);
		  }
		  niter++;
		}
#if 0
		niter = (*ai)->to.begin();
		nrend = (*ai)->to.end();
		while(niter != nrend){
		  NameRecord *_pnr = niter->ptr();
		  if(NULL != _pnr && _pnr != pnr){
		    string nm(_pnr->generic());
		    _generic[nm].push_back(*i);
		    _specific[_pnr->->specific()].push_back(*i);
		    nm += " ";
		    nm += _pnr->specific();
		    _name[nm].push_back(*i);
		    niter++;
		  }
		}
#endif
	      }
	    }
#endif
	    ++i;
	  }
	  ++iter;
	}
	
#if 1
	if(!r.unrn.empty())
	  {
	    hash_map<string, list<d_Ref<NRnode>*>, Hash<string> >::iterator diter = r.unrn.begin();
	    hash_map<string, list<NRnode*>, Hash<string> >::iterator nend = r.nrn.end();
	    hash_map<string, list<NRnode*>, Hash<string> >::iterator ptrnend = _ptrNRnode.end();
	    hash_map<string, list<NRnode*>, Hash<string> >::iterator nfnd;
	    while(diter != r.unrn.end()){
	      list<d_Ref<NRnode>* >::iterator iter = diter->second.begin();
	      while(iter != diter->second.end()){
		if(NULL == (*iter)->ptr()){
		  string s((*iter)->persistentID());
		  if(s.length()){
		    nfnd = r.nrn.find(s);
		    if(nend == nfnd){
		      nfnd = _ptrNRnode.find(s);
		      if(ptrnend == nfnd)
			nfnd = nend;
		    }
		    
		    if(nfnd != nend){
		      list<NRnode*>::iterator niter = nfnd->second.begin();
		      list<NRnode*>::iterator _nend = nfnd->second.end();
		      while(niter != _nend){
			if((*niter)->persistentID() == s){
			  (*iter)->ptr(*niter);
			  break;
			}
			++niter;
		      }
		    }
		    else{  
		      NRnode *p = new NRnode(s);
		      NRnode *q = p->resolve(r);
		      if(p != q)
			delete p;
		      else {
			str = s;
			nrndeduced++;
			newDeduce++;
			if(_verbose)
			  cerr << "%deduce " << 4 - looplimit << ":"<< newDeduce << " " << p->persistentID() << "\n";
		      }
		      //		      (*iter)->ptr(q);
		    }
		  }
		}
		++iter;
	      }
	      ++diter;
	    }
	  }
#endif
	//hierarchy reconstruct
	//hopefully it is already solved by constructor of NRnode....
	list<d_Ref<NRnode> >::iterator nri = _nodes.begin();
	list<d_Ref<NRnode> >::iterator nre = _nodes.end();
	while(nri != nre){
	  NameRecord *pnr = nri->ptr()->namerecord->higher().ptr();
	  if(NULL != pnr){
	    NRnode *pnrn = _hnr[pnr->persistentID()];
	    if(NULL != pnrn){
	      if(NULL == nri->ptr()->higher.ptr())
		(*nri)->higher.ptr(pnrn);
	      list <d_Ref<NRnode> >::iterator dni = pnrn->lower.begin();
	      list <d_Ref<NRnode> >::iterator dne = pnrn->lower.end();
	      while(dni != dne && dni->ptr() != nri->ptr()){
		++dni;
	      }
	      if(dni == dne){
		pnrn->lower.push_back(nri->ptr());
	      }
	    }
	  }
	  ++nri;
	}
      }

      if(!r.an.empty())
	{
	  hash_map<string, list<Annotation*>, Hash<string> >::iterator iter = r.an.begin();
	  hash_map<string, list<Annotation*>, Hash<string> >::iterator end = r.an.end();
	  list<Annotation*>::iterator ai;
	  list<Annotation*>::iterator ae;
	  while(iter != end){
	    ai = (iter->second).begin();
	    ae = (iter->second).end();
	    
	    while(ai != ae){
	      //	      r.newAnnotations.push_back(*ai);
	      string s((*ai)->persistentID());
	      _ptr[s].push_back(*ai);
	      _ptrAnnotation[s].push_back(*ai);
	      list<d_Ref<NameRecord> >::iterator niter = (*ai)->to.begin();
	      list<d_Ref<NameRecord> >::iterator nrend = (*ai)->to.end();
	      while(niter != nrend){
		//_hnr[niter->persistentID()]->annotations.push_back(*ai);
		//		string niterPID(NameRecord::persistentID(niter->persistentID().c_str()));
		string niterPID;
		if(NULL == niter->ptr()){
		  niterPID = niter->persistentID();
		}
		else{
		  niterPID = niter->ptr()->persistentID();
		}
		/*
		if(_hnr[niterPID] == NULL){
		  _hnr[niterPID] = new NameRecord(niterPID.c_str());
		}
		*/
		if(_hnr[niterPID] == NULL){
		  NRnode *nrn = new NRnode(niterPID);
		  //		  if(r.nrn.find(nrn->persistentID())!= r.nrn.end())
		  //		    cerr << "NR exists " <<  niterPID<<endl;
		}
		//		  _hnr[niterPID]->annotations.push_back(*ai);
		
		list<d_Ref<NameRecord> >::iterator fiter = (*ai)->from.begin();
		list<d_Ref<NameRecord> >::iterator fend = (*ai)->from.end();
		while(fiter != fend){
		  if(niter->ptr() != fiter->ptr()){
		    //_hnr[niter->persistentID()]->from.insert(_hnr[fiter->persistentID()]);
		    string fiterPID;// (NameRecord::persistentID(fiter->persistentID().c_str()));
		    if(NULL != fiter->ptr())
		      fiterPID = fiter->ptr()->persistentID();
		    else
		      fiterPID = fiter->persistentID();
		    if(NULL != niter->ptr()){
		      if(_hnr[niterPID] != NULL){
			if(_hnr[fiterPID] != NULL)
			  _hnr[niterPID]->from.push_back(_hnr[fiterPID]);
		      }
		      else{
			//			cerr << "not found " << niterPID << endl;
		      }
		    }
		  }
		  ++fiter;
		}
		++niter;
	      }
	      niter = (*ai)->from.begin();
	      nrend = (*ai)->from.end();
	      if(ai != ae)
		while(niter != nrend){
		  list<d_Ref<NameRecord> >::iterator titer = (*ai)->to.begin();
		  list<d_Ref<NameRecord> >::iterator tend = (*ai)->to.end();

		  string niterPID("");
		  if(NULL != niter->ptr()) 
		    niterPID = niter->ptr()->persistentID();
		  else
		    niterPID = niter->persistentID();
		  while(titer != tend){
		    if(niter->ptr() != titer->ptr()){
		      //_hnr[niter->persistentID()]->to.insert(_hnr[titer->persistentID()]);
		      string titerPID;//(NameRecord::persistentID(titer->persistentID().c_str()));
		      if(NULL != titer->ptr()) 
			titerPID = titer->ptr()->persistentID();
		      else
			titerPID = titer->persistentID();
		      if(_hnr[niterPID] != NULL){
			if(_hnr[titerPID] != NULL)
			  _hnr[niterPID]->to.push_back(_hnr[titerPID]);
		      }
		    }
		    ++titer;
		  }
		  ++niter;
		}
	      ++ai;
	    }
	    ++iter;
	  }
	}
    }
  
  
#if 0
  hash_map<string, NRnode*, Hash<string> >::const_iterator iter = _hnr.begin();
  hash_map<string, NRnode*, Hash<string> >::const_iterator end = _hnr.end();
  while(iter != end){
    //    _ptrNRnode[(iter->second)->persistentID()].push_back(iter->second);
    //    _ptr[(iter->second)->persistentID()].push_back(iter->second);
    string nm((iter->second)->namerecord->generic());
    if(nm.length()){
      _generic[nm].push_back(iter->second);
      nm += " ";
    }
    _specific[(iter->second)->namerecord->specific()].push_back(iter->second);
    nm += (iter->second)->namerecord->specific();
    _name[nm].push_back(iter->second);
    
    ++iter;
  }
#else
#endif
  
  //sort and unify
  unifyHashMappedList(_name/*, NRnode::gt()*/);
  unifyHashMappedList(_generic);
  unifyHashMappedList(_specific);
  unifyHashMappedList(_ptrNRnode);
  unifyHashMappedList(_ptrNRnodeNR);
  unifyHashMappedList(_ptrNameRecord);
  
  return true;  // any good idea?
}

bool Nomencurator::_resolve(
		   hash_map<string, list<NameRecord*>, Hash<string> > nr,
		   hash_map<string, list<Annotation*>, Hash<string> > an,
		   hash_map<string, list<Citation*>, Hash<string> > ci,
		   hash_map<string, list<Publication*>, Hash<string> > pub,
		   hash_map<string, list<Author*>, Hash<string> > au,
		   hash_map<string, list<NRnode*>, Hash<string> > nrn,
		   hash_map<string, list<d_Ref<NameRecord>*>, Hash<string> > unr,
		   hash_map<string, list<d_Ref<Annotation>*>, Hash<string> > uan,
		   hash_map<string, list<d_Ref<Citation>*>, Hash<string> > uci,
		   hash_map<string, list<d_Ref<Publication>*>, Hash<string> > upub,
		   hash_map<string, list<d_Ref<Author>*>, Hash<string> > uau,
		   hash_map<string, list<d_Ref<NRnode>*>, Hash<string> > unrn
		   )
{
  Resolver r(nr, an, ci, pub, au, nrn);
  bool ret = _resolve(r);
   nr = r.nr;   an = r.an;    ci = r.ci;   pub = r.pub;   au = r.au;    nrn = r.nrn;
  unr = r.unr; uan = r.uan;  uci = r.uci; upub = r.upub; uau = r.uau;  unrn = r.unrn;

  return ret;
}


Nomencurator& operator += (Nomencurator& n, NameRecord* nr)
{
  if(NULL != nr){
    Nomencurator tmp;
    tmp._nr.push_back(nr);
    n.merge(tmp);
  }
  return n;
}

Nomencurator& operator += (Nomencurator& n, Author* au)
{
  if(NULL != au){
    Nomencurator tmp;
    tmp._auth.push_back(au);
    n.merge(tmp);
  }
  return n;
}

Nomencurator& operator += (Nomencurator& n, Citation* cit)
{
  if(NULL != cit){
    Nomencurator tmp;
    tmp._cite.push_back(cit);
    n.merge(tmp);
  }
  return n;
}

Nomencurator& operator += (Nomencurator& n, Publication* pub)
{
  if(NULL != pub){
    Nomencurator tmp;
    tmp._pub.push_back(pub);
    n.merge(tmp);
  }
  return n;
}

Nomencurator& operator += (Nomencurator& n, Annotation* an)
{
  if(NULL != an){
    Nomencurator tmp;
    tmp._anno.push_back(an);
    n.merge(tmp);
  }
  return n;
}

Nomencurator& operator += (Nomencurator& n, NRnode* nrn)
{
  if(NULL != nrn){
    Nomencurator tmp;
    tmp._nodes.push_back(nrn);
    n.merge(tmp);
  }
  return n;
}
