/*
 *
 * Publication.cxx: an implementation of persistent Publication class
 * for a Nomenclature Heuristic Model.
 * It may be called also N3, Nomenclature Netowrok Navigator
 * or more simply, MkII.
 *
 * Copyright (c) 1999 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: Publication.cxx,v 1.14 1999/12/05 18:24:50 nozomi Exp $
 *	$Log: Publication.cxx,v $
 *	Revision 1.14  1999/12/05 18:24:50  nozomi
 *	improve merge code
 *	
 *	Revision 1.13  1999/09/23 06:36:46  nozomi
 *	resolve method using Resolver class
 *	
 *	Revision 1.12  1999/08/23 06:38:55  nozomi
 *	fix '&' handling in author parse
 *	
 *	Revision 1.11  1999/08/09 17:24:13  nozomi
 *	spell correcion
 *	
 *	Revision 1.10  1999/08/06 07:42:02  nozomi
 *	allocation bug fix
 *	
 *	Revision 1.9  1999/08/05 08:16:12  nozomi
 *	ISSN/ISBN support (as ISXN)
 *	
 *	Revision 1.8  1999/03/24 11:01:37  nozomi
 *	bug fix relating Autor persistentID handling
 *	
 *	Revision 1.7  1999/03/22 09:19:46  nozomi
 *	merge() enhanced
 *	
 *	Revision 1.6  1999/03/21 13:49:28  nozomi
 *	short format support for journal name
 *	
 *	Revision 1.5  1999/03/20 16:11:09  nozomi
 *	Data merge code, still restrictive, works only when persistentIDs are identical.
 *	
 *	Revision 1.4  1999/03/15 16:42:41  nozomi
 *	gcc 2.7 support macro
 *	
 *	Revision 1.3  1999/03/15 13:15:16  nozomi
 *	volume/number inconsistency removed
 *	input data sensitivity (hopefully) reduced.
 *	
 *	Revision 1.2  1999/03/15 11:26:12  nozomi
 *	persistentID related modifications
 *	
 *	Revision 1.1  1999/03/14 18:21:26  nozomi
 *	minor modification in resolving function
 *	
 *	Revision 1.0.0  1999/03/14 02:18:25  nozomi
 *	Initial version, though aka MkII
 *	
 *
 */

#include "Publication.h"
#include "NameRecord.h"
#include "Annotation.h"
#include "d_common.h"
#if (__GNUC_MINOR__ > 7)
#include <string>
#else
#include <string.h>
#endif
#include <stdio.h>

static const char* endchar(const char* a);
static const char* nameEnd(const char* f);
static string surname(const d_Ref<Author> &da);

const char* endchar(const char* a)
{
  if(0 == *a) return a;
  const char* e = a;
  int l = strlen(a);
  int i;
  for(i = 0; i < l && isspace(*e); e++, i++){}
  for(; i < l && !isspace(*e); e++, i++){}
  if('.' != *e) --e;
  return e;
}

const char* nameEnd(const char* f)
{
  int l = strlen(f);
  int i = 0;
  const char *e = endchar(f);
  if(',' == *e) e++;
  f = e;
  while(',' != *e && /*'.' != *e &&*/ *e != '&' && (strncasecmp(f, "and", 3) || !isspace(*(f+3)))){
    f = e;
    for(;i < l && !isspace(*f);f++, i++){}
    for(;i < l && isspace(*f);f++, i++){}
    e = endchar(f);
    for(;i < l && isspace(*f);f++, i++){}
  }
  if(*e == ',' || i == l) return e;
  if(*e == '&') return --e;
  --e;
  for(;i > 0 && (isspace(*e) || ',' == *e);e--, i--){}
  return e;
}

// create Author from given authrs string
// it assumes _year has already set

void Publication::_analyzeAuthors(const char* a)
{
  //re-initialize
  if(!authorlist.empty())
    authorlist.clear();

  int l = strlen(a);
  if(0 == l) return; //no author

  const char *f;
  int i;

  //skip heading ws
  //  for(i = 0, f = a;i < l && isspace(*f);f++, i++){}
  for(i = 0, f = a;i < l && !isalpha(*f);f++, i++){}
  if(i == l) return; //only ws

  const char* head = f;

  int words = 0;
  int ands = 0;
  int ampersands = 0;
  int etal = 0;
  int aftand = 0;
  int commas = 0;
  int dots = 0;
  bool isetal;
  i = f - a;
  while(i < l){
    for(;i < l && isspace(*f);f++, i++){}
    if(i != l){
      isetal = false;
      words++;
      if(ands || ampersands)
	aftand++;
      if('&' == *f)
	ampersands++;
      if(isspace(*(f+3)) && !strncasecmp(f, "and", 3))
	ands++;
      if(!strncasecmp(f, "et", 2)){
	if((!strncasecmp(f, "etal", 4)  && (isspace(*(f+4)) || '.' == *(f+4) || 0 == *(f+4))) ||
	   (!strncasecmp(f, "et al", 5) && (isspace(*(f+5)) || '.' == *(f+5) || 0 == *(f+5))) ||
	   (!strncasecmp(f, "et.al", 5) && (isspace(*(f+5)) || '.' == *(f+5) || 0 == *(f+5))))
	  {etal++;isetal = true;}
	else{
	  const char *e  = f;
	  int j = i;
	  for(;j < l && !isspace(*e);e++, j++){}
	  for(;j < l && isspace(*e);e++, j++){}
	  if(!strncasecmp(e, "al", 2) && (isspace(*(f+2)) || '.' == *(f+2)))
	    {etal++; isetal = true;}
	}
      }
      for(;i < l && !isspace(*f);f++, i++){
	if(!isetal)
	  switch(*f){
	  case ',': commas++;break;
	  case '.': dots++;break;
	  }
      }
    }
  }

  //single author
  if(ampersands == 0 && ands == 0 && etal == 0)
    {
      Author au(head, _year);
      d_Ref<Author> *da = new d_Ref<Author>;
      da->persistentID(au.persistentID());
      authorlist.push_back(*da); return;
    }

  // one effective author
  if(1 == etal)
    {
      f = a + l - 1;
      
      //search etal
      i = 0;
      while(strncmp(f, "et", 2)){
	f--;
	for(i = 0;i < l && isspace(*f);f--, i++){}
	for(;i < l && !isspace(*f);f--, i++){}
	f++;
      }
      head = strndup(head, f-head);
      Author au(head, _year);
      d_Ref<Author> *da = new d_Ref<Author>;
      da->persistentID(au.persistentID());
      authorlist.push_back(*da);
      delete [] head;
      return;
    }

  f = head;
  if(aftand > 1){
    while('&' != *f && !(!strncasecmp(f, "and", 3) && isspace(*(f+3)))){
      f = nameEnd(f);
      head = strndup(head, f-head - 1/*(isspace(*f) || ',' == *f)?1:0*/);
      Author au(head, _year);
      d_Ref<Author> *da = new d_Ref<Author>;
      da->persistentID(au.persistentID());
      authorlist.push_back(*da);
      //authorlist.push_back(new Author(head, _year)); 
      delete [] head;
      for(i = f - a;i < l && (',' == *f || '.' == *f) ;f++, i++){}
      for(;i < l && isspace(*f);f++, i++){}
      for(;i < l && (',' == *f || '.' == *f) ;f++, i++){}
      for(;i < l && isspace(*f);f++, i++){}
      for(i = f - a;i > 0 && !isspace(*f);f--, i--){}
      for(i = f - a;i < l && isspace(*f);f++, i++){}
      head = f;
    }
    for(i = f - a;i < l && !isspace(*f);f++, i++){}
    for(i = f - a;i < l && isspace(*f);f++, i++){}
    Author au(f, _year);
    d_Ref<Author> *da = new d_Ref<Author>;
    da->persistentID(au.persistentID());
    authorlist.push_back(*da);
    return;
  }

  while('&' != *f && !(!strncasecmp(f, "and", 3) && isspace(*(f+3)))){
    while(!isspace(*f) && ',' != *f){f++;}
    head = strndup(head, f-head);
    Author au(head, _year);
    d_Ref<Author> *da = new d_Ref<Author>;
    da->persistentID(au.persistentID());
    authorlist.push_back(*da);
    delete [] head;
    if(',' == *f)f++;
    while(isspace(*f)){f++;}
    head = f;
  }
  for(i = f - a;i < l && !isspace(*f);f++, i++){}
  for(i = f - a;i < l && isspace(*f);f++, i++){}

  Author au(f, _year);
  d_Ref<Author> *da = new d_Ref<Author>;
  da->persistentID(au.persistentID());
  authorlist.push_back(*da);

}

Publication::Publication(void)
  : d_Object(_classname), 
    authorlist(), citations(), annotations(),
    _authors(strdup()), _affiliations(strdup()),
    _title(strdup()),
    _ISXN(strdup()),
    _year(Date::invalidYear), 
    _volume(strdup()), _number(strdup()), _from(0), _to(0),
    _received(), _revised(),_accepted()  {}


Publication::~Publication(void)
{ delete[] _authors; delete[] _affiliations; delete[] _title;  delete[] _ISXN; delete[] _volume; delete[] _number;}

Publication::Publication(const string &s)
  : d_Object(_classname),
    authorlist(), citations(), annotations(),
    _authors(strdup()), _affiliations(strdup()),
    _title(strdup()), _ISXN(strdup()),
    _year(Date::invalidYear), 
    _volume(strdup()), _number(strdup()),
    _from(0),  _to(0),
    _received(), _revised(),_accepted()
{_Publication(s);}

void Publication::_Publication(const string &s)
{
  int i = 0;
  int l = s.length();
  bool pubptr = false;
  if(!strncmp(s.c_str(), "Publication::", 13)){
    pubptr = true;
    for(i = 0; i < l && ':' != s[i]; i++){}
    i += 2;
  }
  string word[7];
  int j;
  for(j = 0; j < 7; j++){ word[j] == "";} 

  int head;
  for(j = 0; i < l && j < 7; j++){
    for(head = i; i < l && '_' != s[i]; i++){}
    if(i!= head)
      word[j] = string(s, head, i - head);
    else
      word[j] = string("");
    i++;
  }

  //it should be set BEFORE _analyzeAuthors(_authors);
  if(word[1].length()) _year = atoi(word[1].c_str());

  if(0 != (l = word[0].length())){ 
    if(pubptr)
      {
	int j;
	int sp = 0;
	for(j = 0; j < l ; j ++){
	  if(isspace(word[0][j]) &&
	     strncasecmp(string(word[0], j+1).c_str(), "sen", 3) &&
	     strncasecmp(string(word[0], j+1).c_str(), "jr", 2) )  sp++;}
	switch(sp){
	case 0:
	  break;
	case 1:
	  for(j = 0; j < l && !isspace(word[0][j]); j++){}
	  word[0].insert(j, " and");
	  break;
	default:
	  j = 0;
	  while(sp > 1){
	    for(; j < l && !isspace(word[0][j]); j++){}
	    word[0].insert(j, ", ");
	    j++; sp--;
	  }
	  for(; j < l && !isspace(word[0][j]); j++){}
	  word[0].insert(j, " and");
	  break;
	}
      }
    delete [] _authors; _authors = strdup(word[0]);
    _analyzeAuthors(_authors);
  }

  if(word[2].length()){ delete [] _title;  _title  = strdup(word[2]); }
  if(word[3].length()){ delete [] _volume; _volume = strdup(word[3]); }
  if(word[4].length()){ delete [] _number; _number = strdup(word[4]); }
  if(word[5].length()){ _from = atoi(word[5].c_str()); }
  if(word[6].length()){ _to = atoi(word[6].c_str()); }
}

Publication::Publication(const char *s)
  : d_Object(_classname),
    authorlist(), citations(), annotations(),
    _authors(strdup()), _affiliations(strdup()),
    _title(strdup()),
    _ISXN(strdup()),
    _year(Date::invalidYear), 
    _volume(strdup()), _number(strdup()), _from(0),  _to(0),
    _received(), _revised(),_accepted()
{
  int l;
  if(NULL == s || 0 == (l = strlen(s)))
    { _title = strdup(); return; }
  
  const char *f;
  int i;
  bool found;
  const char *tmp;
  if(!strncmp(s, "Publication::", 13)){_Publication(string(s));return;}

  for(i = 0, f = s, found = false;
      i < l && '|' != *f /*&& (isspace(*f) || isalnum(*f))*/; f++, i++){}

  if(i == l)
    {
      //skip heading ws
      for(i = 0, f = s; i < l && isspace(*f); f++, i++){}
      //skip tailing ws
      for(i = 0, tmp = s + l -1; i < tmp - f && isspace(*tmp); tmp--, i++){}
      _title = strndup(f, tmp + 1 - f);return;
    }
  
  const char separator = *f;

  //skip heading ws
  for(i = 0, f = s; i < l && isspace(*f); f++, i++){}
  //skip tailing ws
  for(i = 0, tmp = f - 1; i < tmp - f && isspace(*tmp); tmp--, i++){}
  int al = tmp + 1 - f;
  _title = strndup(f, al);
  for(i = 0, f = s; i < l && separator != *f++; i++){} if(i ==l) return;
  for(; i < l && isspace(*f);f++, i++){} if(i ==l) return;
  for(tmp = f; i < l && separator != *tmp;tmp++, i++){} if(i ==l) return;
  _volume = strndup(f, tmp-f);
  for(; i < l && isdigit(*f);f++, i++){} if(i ==l) return;
  for(; i < l && isspace(*f); f++, i++){} if(i ==l) return;
  if('(' == *f)
    {
      ++f;
      tmp = f;
      while(*tmp != ')') tmp++;
      _number = strndup(f, tmp - f);
      for(; i < l && ')' != (*f++); i++){} if(i ==l) return;
      for(; i < l && !isdigit(*f);f++, i++){} if(i ==l) return;
    }
  _from = _to = atoi(f);
  for(; i < l && isdigit(*f);f++, i++){} if(i ==l) return;
  if('-' == *f)
  for(; i < l && !isdigit(*f);f++, i++){} if(i ==l) return;
  _to = atoi(f);
  return;
} 
  
Publication::Publication(const NamedField *nf, const char* fn)
  : d_Object(_classname),
    authorlist(), citations(), annotations(),
    _year(Date::invalidYear),
    _from(0),  _to(0),
    _received(), _revised(),_accepted()
{
  const char* empty = "";
  if(strcmp(nf->entryName(), fn)) {*this = Publication("");return;}
  const char* s;
  if(strlen(s = nf->strcmp(fn))){ *this = Publication(s);return;}
  if(NULL == (s = nf->strcasecmp("affiliations"))) s = empty;
  _affiliations = new_strdup::strdup(s);
  if(NULL == (s = nf->strcasecmp("title"))) s = empty;
  _title = new_strdup::strdup(s);
  if(NULL == (s = nf->strcasecmp("ISXN"))) s = empty;
  _ISXN = new_strdup::strdup(s);
  if(NULL == (s = nf->strncasecmp("vol", 3))) s = empty;
  _volume = new_strdup::strdup(s);
  if(NULL != (s = nf->strcasecmp("year"))) _year = atoi(s);
  if(NULL == (s = nf->strncasecmp("num", 3))) s = empty;
  _number = new_strdup::strdup(s);

  if(NULL != (s = nf->strncasecmp("rec", 3))) _received=Date(s);
  if(NULL != (s = nf->strncasecmp("rev", 3))) _revised=Date(s);
  if(NULL != (s = nf->strncasecmp("acc", 3))) _accepted=Date(s);

  if(NULL == (s = nf->strncmp("author", 6))) s = empty;
  _authors = new_strdup::strdup(s);
  if(NULL == nf->strcmp("Author"))
    _analyzeAuthors(_authors);
  else{
    list<NamedField>::const_iterator tmp;
    for(tmp = nf->subEntries.begin(); tmp != nf->subEntries.end(); ++tmp)
      {
	if(!strcmp(tmp->entryName(), "Author")){
	  s = tmp->contents();
	  if(strlen(s)){
	    d_Ref<Author> *da = new d_Ref<Author>;
	    da->persistentID(s);
	    //	    authorlist.push_front(*da);
	    authorlist.push_back(*da);
	  }
	  else{
	    //instance is given
	    //	    authorlist.push_front(d_Ref<Author>(new Author(*tmp)));
	    authorlist.push_back(d_Ref<Author>(new Author(*tmp)));
	  }
	}
      }
  }
  if(NULL != (s = nf->strncasecmp("pages", 4)))
    {
      _to = _from = atoi(s);
      const char *f;
      int i, l = strlen(s);
      for (i = 0, f = s;i < l && isdigit(*f);f++, i++){}
      if(i < l){
	for (;i < l && !isdigit(*f);f++, i++){}
	_to = atoi(f);
      }
    }
  else if(NULL != (s = nf->strncasecmp("page", 4))) _from = _to = atoi(s);
  else{
    if(NULL != (s = nf->strcasecmp("from"))) _from = atoi(s);
    if(NULL != (s = nf->strcasecmp("to"))) _to = atoi(s);
  }

  if(NULL != nf->strcmp("Citation")){
    list<NamedField>::const_iterator tmp;
    for(tmp = nf->subEntries.begin(); tmp != nf->subEntries.end(); ++tmp)
      {
	if(!strcmp(tmp->entryName(), "Citation")){
	  s = tmp->contents();
	  if(strlen(s)){
	    d_Ref<Citation> *dn = new d_Ref<Citation>;
	    dn->persistentID(s);
	    citations.push_back(*dn);
	  }
	  else{
	    citations.push_back(d_Ref<Citation>(new Citation(*tmp)));
	  }
	}
      }
  }
}

Publication::Publication(const char* a, int yr, const char* t, const char *isxn, const char *v, int bp, int ep, const char *n, const char* af)
  : d_Object(_classname),
    authorlist(), citations(), annotations(),
    _authors(new_strdup::strdup(a)), 
    _affiliations(new_strdup::strdup(af)),
    _title(new_strdup::strdup(t)),
    _ISXN(new_strdup::strdup(isxn)),
    _year(yr),  _volume(new_strdup::strdup(v)), _number(new_strdup::strdup(n)),
    _from(bp),  _to(ep),
    _received(), _revised(),_accepted() {_analyzeAuthors(a);}

Publication::Publication(const char* a, int yr, const char* t, const char *isxn, int v, int bp, int ep, int n, char* af)
  : d_Object(_classname),
    authorlist(), citations(), annotations(),
    _authors(new_strdup::strdup(a)), 
    _affiliations(new_strdup::strdup(af)),
    _title(new_strdup::strdup(t)),
    _ISXN(new_strdup::strdup(isxn)),
    _year(yr),  _volume(strdup()), _number(strdup()), _from(bp),  _to(ep),
    _received(), _revised(),_accepted() { if(v) volume(v); if(n) number(v); _analyzeAuthors(a);}

void Publication::_Publication(const Publication* p)
{
  _authors = new_strdup::strdup(p->_authors);
  _affiliations = new_strdup::strdup(p->_affiliations);
  _title = new_strdup::strdup(p->_title);
  _ISXN = new_strdup::strdup(p->_ISXN);

  _received = p->_received;
  _revised = p->_revised;
  _accepted = p->_accepted;

  _year = p->_year;
  _volume = p->_volume;
  _number = p->_number;

  _from = p->_from;
  _to = p->_to;

  authorlist = p->authorlist;
  //  nameRecords = p->nameRecords;
  citations = p->citations;
  annotations = p->annotations;
}

static string surname(const d_Ref<Author> &da)
{
  if(NULL != da.ptr())
    return da.ptr()->surname();

#if 1
  Author a(da.persistentID());
  return a.surname();
#else
  string s(da.persistentID());
  int l = s.length();
  if(0 == l) return s;
  int i, head;
  for(i = 0; i < l && ':' != s[i] ; i++){}
  if(i != l) head = i+2;
  else head = 0;
  for(i = head; i < l && '_' != s[i] && ',' != s[i] && !isspace(s[i]); i++){}
  return string(s, head, i - head);
#endif
}

string Publication::authorWithYear(void) const
{
  string s("");
  if(!authorlist.empty()){
    list<d_Ref<Author> >::const_iterator iter = authorlist.begin();
    s += surname(*iter);
    if(++iter != authorlist.end())
      for(;iter != authorlist.end(); ++iter){
	s += " "; s+= surname(*iter);
	//if(iter->ptr() != NULL){
	//	  s += " "; s+= iter->ptr()->surname();
	//	}
      }
  }
  else{
    s += _authors;
  }
  s += "_";
  if(_year != Date::invalidYear){char y[5]; sprintf(y, "%d", _year); s+= y;}
  return s;
}

string Publication::emptyPID(void)
{
  string s(_classname);
  s += "::______";
  return s;
}

string Publication::persistentID(void) const 
{
  string s("Publication::");
  s += authorWithYear();
  s += "_";
  s += _title;  s += "_";
  s+= _volume;  s += "_";
  s+= _number;  s += "_";
  if(_from){char f[10]; sprintf(f, "%d", _from); s+= f;}
  s += "_";
  if(_to){char t[10]; sprintf(t, "%d", _to); s+= t;}
  return s;
}

ostream& operator<<(ostream & s, const Publication& p)
{
  s << "Publication = {\n";
  s << " persistentID = {" <<  p.persistentID() << "}\n";
  if(strlen(p._authors)) s << " authors = {" << p._authors << "}\n";
  if(strlen(p._affiliations)) s << " affiliations = {" << p._affiliations << "}\n";
  if(p._year) s << " year = {" << p._year << "}\n";
  if(strlen(p._ISXN)) s << " ISXN = {" << p._ISXN << "}\n";
  if(strlen(p._title)) s << " title = {" << p._title << "}\n";
  if(strlen(p._volume)) s << " volume = {" << p._volume << "}\n";
  if(strlen(p._number)) s << " number = {" << p._number << "}\n";
  if(p._from) s << " from = {" << p._from << "}\n";
  if(p._to) s << " to = {" << p._to << "}\n";
  if(p._received.year() != Date::invalidYear) s << " received = {" << p._received << "}\n";
  if(p._revised.year() != Date::invalidYear) s << " revised = {" << p._revised << "}\n";
  if(p._accepted.year() != Date::invalidYear) s << " accepted = {" << p._accepted << "}\n";
  if(!p.authorlist.empty())
    {
      list<d_Ref<Author> >::const_iterator tmp;
      for(tmp = p.authorlist.begin(); tmp != p.authorlist.end(); ++tmp){
	s << " Author = {" << tmp->persistentID() << "}\n";
      }
    }

  if(!p.citations.empty())
    {
      list<d_Ref<Citation> >::const_iterator tmp;
      for(tmp = p.citations.begin(); tmp != p.citations.end(); ++tmp){
	s << " Citation = {" << tmp->persistentID() << "}\n";
      }
    }

  if(!p.annotations.empty())
    {
      list<d_Ref<Annotation> >::const_iterator tmp;
      for(tmp = p.annotations.begin(); tmp != p.annotations.end(); ++tmp){
	s << " Annotation = {" << tmp->persistentID() << "}\n";
      }
    }

  s << "}\n";
  return s;
}

void Publication::year(int y)
{
  _year = y;
  if(!authorlist.empty()){
    list<d_Ref<Author> >::iterator tmp;
    for(tmp = authorlist.begin(); tmp != authorlist.end(); ++tmp)
      if(tmp->ptr())
	(*tmp)->year(y);
      else {
	string s(tmp->persistentID());
	int siter = 0;
	int send = s.length()+1;
	while(siter != send && '_' != s[siter]){++siter;}
	string id(s, 0, siter+1);
	char year[6];
	sprintf(year, "%d_", y - Author::earliestPublicationAt);
	id+= year;
	sprintf(year, "%d", y);
	id+= year;
	tmp->persistentID(id);
      }
  }
}

int Publication::from(int p)
{int tmp = _from; _from = p; return tmp;}

int Publication::to(int p)
{int tmp = _to; _to = p; return tmp;}

void Publication::authors(const char* a)
{
  delete [] _authors;  _authors = new_strdup::strdup(a); _analyzeAuthors(a);
  return;
}


Date Publication::recieved(Date d)
{
  if(strlen(_authors)){ Date tmp = _received; _received = d; return tmp; }
  else
    return d;
}

Date Publication::revised(Date d)
{
  if(strlen(_authors) || strlen(_title))
    { Date tmp = _revised; _revised = d; return tmp;}
  else return d;
}

Date Publication::accepted(Date d)
{
  if(strlen(_authors)){ Date tmp = _accepted; _accepted = d; return tmp; }
  else return d;
}

void Publication::volume(int v)
{ 
  char vbuf[256];    sprintf(vbuf, "%d", v);
  delete [] _volume; _volume = new_strdup::strdup(vbuf);
}


void Publication::number(int n)
{ 
  char nbuf[256];    sprintf(nbuf, "%d", n);
  delete [] _number; _number = new_strdup::strdup(nbuf);
}

Publication* Publication::resolve(Resolver &r)
{
  string s(persistentID());
  hash_map<string, list<Publication*>, Hash<string> >::iterator piter = r.pub.find(s);
  if(piter != r.pub.end()){
    list<Publication*>::iterator iter = piter->second.begin();
    list<Publication*>::iterator end = piter->second.end();
    while(iter != end){
      if(this == *iter)
	return this;
      else if((*iter)->merge(this))
      	return *iter;
      ++iter;
    }
  }
  
  r.pub[s].push_back(this);
  r.newPublications.push_back(this);

  hash_map<string, list<d_Ref<Publication>*>, Hash<string> >::iterator diter = r.upub.find(s);

  if(diter != r.upub.end()){
    register list<d_Ref<Publication>*>::iterator iter = diter->second.begin();
    register list<d_Ref<Publication>*>::iterator end = diter->second.end();
    register list<d_Ref<Publication>*>::iterator current;
    while(iter != end){
      current = iter;
      ++iter;
      if(s == (*current)->persistentID() && NULL == (*current)->ptr()){
	(*current)->ptr(this);
	diter->second.erase(current);
      }
    }
  }

  if(!citations.empty()){
    list<d_Ref<Citation> >::iterator iter = citations.begin();
    list<d_Ref<Citation> >::iterator end  = citations.end();
    while(iter != end){
      Citation *p = iter->ptr();
      if(NULL != p){
	Citation *q = p->resolve(r);
	if(p != q)
	  iter->ptr(q);
      }
      else
	iter->resolve(r.ci, r.uci);
      ++iter;
    }
  }

  if(!authorlist.empty()){
    list<d_Ref<Author> >::iterator iter = authorlist.begin();
    list<d_Ref<Author> >::iterator end = authorlist.end();
    while(iter != end){
      Author* p= iter->ptr();
      if(NULL != p){
	Author* q = p->resolve(r);
	if(p != q)
	   iter->ptr(q);
      }
      
      else{
#if 1
	string s(iter->persistentID());
	int i;
	int l = s.length();
	for(i = 8; i < l && '_' != s[i]; i++){}
	Author auth(string(s, 8 , i - 8).c_str(), _year); //8 == strlen("Author::");
	string fn(auth.fullname());
	hash_map<string, list<Author*>, Hash<string> >::iterator aiter
	  = r.au.find(s);
	if(aiter == r.au.end()){
	  auth.publications.push_back(this);
	  Author *p = new Author(auth);
	  r.au[s].push_back(p);
	  r.newAuthors.push_back(p);
	}
	else{
#if 1
	  iter->resolve(r.au, r.uau);
#else
	  list<Author*>::iterator iter = aiter->second.begin();
	  list<Author*>::iterator end = aiter->second.end();
	  while(iter != end && s != (*iter)->persistentID()){
	    cout << "% " << s << " " << (*iter)->persistentID()<<"\n";
	    ++iter;
	  }
	  if(iter != end){
	    (*iter)->year(_year);
	    (*iter)->publications.push_back(this);
	  }
	  else{
	    auth.publications.push_back(this);
	    Author *p = new Author(auth);
	    r.au[s].push_back(p);
	    r.newAuthors.push_back(p);
	  }
#endif
	}
#else
	iter->resolve(r.au, r.uau);
#endif
      }
      ++iter;
    }
  }

  if(!annotations.empty()){
    list<d_Ref<Annotation> >::iterator iter = annotations.begin();
    list<d_Ref<Annotation> >::iterator end = annotations.end();
    while(iter != end){
      Annotation *p = iter->ptr();
      if(NULL != p){
	Annotation *q = p->resolve(r);
	if(q != p)
	  iter->ptr(q);
      }
      else
	iter->resolve(r.an, r.uan);
      ++iter;
    }
  }

  return this/*NULL*/;
}

Publication* Publication::resolve(hash_map<string, list<NameRecord*>, Hash<string> > &nr,
				  hash_map<string, list<Annotation*>, Hash<string> > &an,
				  hash_map<string, list<Citation*>, Hash<string> > &ci,
				  hash_map<string, list<Publication*>, Hash<string> > &pub,
				  hash_map<string, list<Author*>, Hash<string> > &au,
				  hash_map<string, list<d_Ref<NameRecord>*>, Hash<string> > &unr,
				  hash_map<string, list<d_Ref<Annotation>*>, Hash<string> > &uan,
				  hash_map<string, list<d_Ref<Citation>*>, Hash<string> > &uci,
				  hash_map<string, list<d_Ref<Publication>*>, Hash<string> > &upub,
				  hash_map<string, list<d_Ref<Author>*>, Hash<string> > &uau)
{
  string s(persistentID());

  hash_map<string, list<Publication*>, Hash<string> >::iterator piter = pub.find(s);
  if(piter != pub.end()){
    list<Publication*>::iterator iter = piter->second.begin();
    list<Publication*>::iterator end = piter->second.end();
    while(iter != end){
      if(this == *iter)
	return NULL;
      else if((*iter)->merge(this))
      	return *iter;
      ++iter;
    }
  }

  pub[s].push_back(this);

  hash_map<string, list<d_Ref<Publication>*>, Hash<string> >::iterator diter = upub.find(s);

  if(diter != upub.end()){
    list<d_Ref<Publication>*>::iterator iter = diter->second.begin();
    list<d_Ref<Publication>*>::iterator end = diter->second.end();
    while(iter != end){
      if(s == (*iter)->persistentID() && NULL == (*iter)->ptr())
	(*iter)->ptr(this);
      ++iter;
    }
  }

  if(!citations.empty()){
    list<d_Ref<Citation> >::iterator iter = citations.begin();
    list<d_Ref<Citation> >::iterator end  = citations.end();
    while(iter != end){
      if(NULL != iter->ptr()){
	Citation *p = (*iter)->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau);
	if(NULL != p){
	  //	  delete iter->ptr();
	  //	  iter->pter(p);
	}
      }
      else
	iter->resolve(ci, uci);
      ++iter;
    }
  }

  if(!authorlist.empty()){
    list<d_Ref<Author> >::iterator iter = authorlist.begin();
    list<d_Ref<Author> >::iterator end = authorlist.end();
    while(iter != end){
      if(NULL != iter->ptr()){
	Author* p= (*iter)->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau);
	if(NULL != p){
	  //	  delete iter->ptr();
	  //	  iter->ptr(p);
	}
      }
      
      else{
#if 1
	string s(iter->persistentID());
	int i;
	int l = s.length();
	for(i = 8; i < l && '_' != s[i]; i++){}
	Author auth(string(s, 8 , i - 8).c_str(), _year); //8 == strlen("Author::");
	string fn(auth.fullname());
	hash_map<string, list<Author*>, Hash<string> >::iterator aiter
	  = au.find(s);
	if(aiter == au.end()){
	  auth.publications.push_back(this);
	  au[s].push_back(new Author(auth));
	}
	else{
#if 1
	  iter->resolve(au, uau);
#else
	  list<Author*>::iterator iter = aiter->second.begin();
	  list<Author*>::iterator end = aiter->second.end();
	  while(iter != end && s != (*iter)->persistentID()){
	    cout << "% " << s << " " << (*iter)->persistentID()<<"\n";
	    ++iter;
	  }
	  if(iter != end){
	    (*iter)->year(_year);
	    (*iter)->publications.push_back(this);
	  }
	  else{
	    auth.publications.push_back(this);
	    au[s].push_back(new Author(auth));
	  }
#endif
	}
#else
	iter->resolve(au, uau);
#endif
      }
      ++iter;
    }
  }

  if(!annotations.empty()){
    list<d_Ref<Annotation> >::iterator iter = annotations.begin();
    list<d_Ref<Annotation> >::iterator end = annotations.end();
    while(iter != end){
      if(NULL != iter->ptr()){
	Annotation *p = (*iter)->resolve(nr, an, ci, pub, au, unr, uan, uci, upub, uau);
	if(NULL != p){
	  //	  delete iter->ptr();
	  //	  iter->ptr(p);
	}
      }
      else
	iter->resolve(an, uan);
      ++iter;
    }
  }

  return NULL;
}

static bool shrunken(const char* s1, const char* s2);

bool Publication::merge(Publication *p)
{
  static const bool refuse = false;

  bool ret = refuse;

  /*
   * identical object can't be merged
   */
  if(this == p){
    return refuse;
  }

  _persistentIDmatch:
  if(persistentID() == p->persistentID()){
    Publication empty;
    if(persistentID().length() <= empty.persistentID().length()){
      return refuse; //empty record
    }

    //    cerr <<"merge " << persistentID() << " and " << p->persistentID()<<endl;

    //must merge...
    if(strlen(_authors) < strlen(p->_authors)){ //too optimistic
      delete [] _authors; 
      _authors = p->_authors;
      p->_authors =new_strdup::strdup(_authors);
    }
    if(strlen(_affiliations) < strlen(p->_affiliations)){ //too optimistic
      delete [] _affiliations;
      _affiliations = p->_affiliations;
      p->_affiliations =new_strdup::strdup(_affiliations);
    }

#if 1
    list <Author*> al;
    list <Author*> pal;
    register list<d_Ref<Author> >::iterator iter = authorlist.begin();
    register list<d_Ref<Author> >::iterator end  = authorlist.end();
    while(iter != end){
      if(NULL != iter->ptr())
	al.push_back(iter->ptr());
      iter++;
    }
      
    iter = p->authorlist.begin();
    end  = p->authorlist.end();
    while(iter != end){
      if(NULL != iter->ptr())
	pal.push_back(iter->ptr());
      iter++;
    }

    al.merge(pal);
    al.unique();
#if 0
    authorlist.clear();
    register list<Author*>::iterator i = al.begin();
    register list<Author*>::iterator e = al.end();
    while(i != e){
      authorlist.push_back(*i);
      i++;
    }
#endif
#else
    authorlist.merge(p->authorlist);
    authorlist.unique();
#endif    

    citations.sort();
    p->citations.sort();
    citations.merge(p->citations);
    citations.unique();

    return !refuse;
  }
#if 1
  if(!strcmp(_title, p->_title)){
    if(_year != p->_year ||
       _volume != p->_volume ||
       //_number != p->_number ||
       _from != p->_from ||
       _to != p->_to)
      return refuse;

    ret = !refuse;
    if(_volume == 0 && p->_volume != 0)
      _volume = p->_volume;
    if(strlen(_authors) < strlen(p->_authors)){
      delete [] _authors;
      _authors = new_strdup::strdup(p->_authors);
      //work on author list
    }
    if(strlen(_affiliations) < strlen(p->_affiliations)){
      delete [] _affiliations;
      _affiliations = new_strdup::strdup(p->_affiliations);
    }
    return ret;
  }

  if(shrunken(_title, p->_title)){
    if(strlen(_title) < strlen(p->_title)){
      delete [] _title;
      _title = new_strdup::strdup(p->_title);
    }
    else{
      delete [] p->_title;
      p->_title = new_strdup::strdup(_title);
    }
    if(persistentID() == p->persistentID())
      goto _persistentIDmatch;
  }
#endif
  return refuse;

}


bool shrunken(const char* s1, const char* s2)
{
  bool ret = true;
  list<string> l1, l2;
  const char *head, *c;
  head = c = s1;
  while(*c){
    while(*c && !isspace(*c)){c++;}
    if(isupper(*head)){
      l1.push_back(string(head, c-head));
    }
    while(*c && isspace(*c)){c++;}
    head = c;
  }
  head = c = s2;
  while(*c){
    while(*c && !isspace(*c)){c++;}
    if(isupper(*head)){
      l2.push_back(string(head, c-head));
    }
    while(*c && isspace(*c)){c++;}
    head = c;
  }
  int wc = 0;
  list<string>::iterator iter = l1.begin();
  list<string>::iterator end = l1.end();
  while(iter != end){++iter;++wc;}
  iter = l2.begin();
  end = l2.end();
  while(iter != end){++iter;--wc;}
  if(wc) return false;
  iter = l2.begin();
  list<string>::iterator iter1 = l1.begin();
  while(iter != end){
    ret |= !strscmp(iter->c_str(), iter1->c_str());
    ++iter; ++iter1;
  }
  return ret;
}

bool operator <(const Publication &p1, const Publication &p2)
{
  if(p1._year != p2._year)
    return p1._year < p2._year;
  int strc = strcasecmp(p1._title, p1._title);
  if(strc != 0)
    return strc < 0;

  strc = strcasecmp(p1._volume, p2._volume);
  if(strc != 0)
    return strc < 0;

  if(p1._from != p2._from)
    return p1._from < p2._from;

  if(p1._to != p2._to)
    return p1._to < p2._to;

  return strcasecmp(p1.persistentID().c_str(), p2.persistentID().c_str()) < 0;
}

bool operator ==(const Publication &p1, const Publication &p2)
{
  if(&p1 == &p2 || p1.persistentID() == p2.persistentID())
    return true;
  return false;
}

