/*
 *
 * Publication.h: a definition of persistent Publication class
 * for a Nomenclature Heuristic Model.
 * It may be called also N3, Nomenclature Netowrok Navigator
 * or more simply, MkII.
 *
 * Copyright (c) 1999 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: Publication.h,v 1.6 1999/12/05 18:21:25 nozomi Exp $
 *	$Log: Publication.h,v $
 *	Revision 1.6  1999/12/05 18:21:25  nozomi
 *	improve redundant include guard
 *	
 *	Revision 1.5  1999/09/23 06:37:14  nozomi
 *	resolve method using Resolver class
 *	
 *	Revision 1.4  1999/08/18 13:32:04  nozomi
 *	change include files
 *	
 *	Revision 1.3  1999/08/05 08:15:47  nozomi
 *	ISSN/ISBN support (as ISXN)
 *	
 *	Revision 1.2  1999/03/20 16:11:40  nozomi
 *	bool Publication::merge(Publication *) add.
 *	
 *	Revision 1.1  1999/03/15 16:43:00  nozomi
 *	gcc 2.7 support macro
 *	
 *	Revision 1.0.0  1999/03/14 02:18:32  nozomi
 *	Initial version, though aka MkII
 *	
 *
 */

//
// class Publication is the lowest level abstraction of publication entitiy,
// i.e. what you would see in library, for example.  Adding ordinary bibliographic 
// informations (authors, journal name, year, voule, number, pages), it has place 
// to hold Date of received, revised and accepted, list of pointers to Author object, 
// list of pointers to Citation object, and list of pointers to Annotation object.
// The list of pointers to Annotation object may be unnecessary.
// Citation class and Annotation class are second and third level abstraction respectively.
// Author class has mixed abstraction level currently; it contains information abstracted
// from Publication and hence second level abstraction.  However, Author class also contaions
// information directly abstracted other source and hence it would be considered 1.5th level
// abstraction.
// Author instances are generated by parsing given string of authors name, if Author class
// instance or pointor to Author object is not given.
//

#if (!defined(_INCLUDED_PUBLICATION_H))
#define _INCLUDED_PUBLICATION_H

#include "d_common.h"
#include "Date.h"
#include "d_Object.h"
#include "d_Ref.h"

#if (__GNUC_MINOR__ > 7)
#if (!defined(_INCLUDED_STRING))
#include <string>
#defin _INCLUDED_STRING
#endif

#if (!defined(_INCLUDED_LIST))
#include <list>
#defin _INCLUDED_LIST
#endif

#else

#if (!defined(_INCLUDED_STRING))
#include <string.h>
#defin _INCLUDED_STRING
#endif

#if (!defined(_INCLUDED_LIST))
#include <list.h>
#defin _INCLUDED_LIST
#endif

#endif

class Citation;
class NameRecord;
class Annotation;
class Resolver;
class Author;

class Publication:public d_Object{
friend class Citation;
public:

  //constructors
  Publication(void);
  Publication(const char *s);
  Publication(const string &s);
  Publication(const char* a, int yr, const char* t, const char * isxn, const char * v, int bp = 0, int ep =0, const char *n = "", const char* af = "");
  Publication(const char* a, int yr, const char* t = "", const char * isxn = "", int v = 0, int bp = 0, int ep =0, int n = 0, char* af = "");
  Publication(const Publication&);
  Publication& operator=(const Publication&);
  Publication(const Publication*);
  Publication(const NamedField *nf, const char* fn = "Publication");

  //destructor
  ~Publication(void);

  //accessors
  string authors(void) const;   //list of authors
  Author* author(int)const;
  Date recieved(void) const;    //received Date 
  Date revised(void) const;     //reception Date in reviced form
  Date accepted(void) const;    //accepted Date
  int  year(void) const;        //publication year
  string volume(void) const;    //volume  
  string title(void) const;     //book title or journal name
  string journal(void) const;   //journal name or book title
  string ISXN(void) const;      //ISSN/ISBN
  string number(void) const;    //number
  string pages(void) const;     //pages range of publication
  int from(void) const;
  int to(void) const;

  //manipulators
  void authors(const char*);
  Date recieved(Date);
  Date revised(Date);
  Date accepted(Date);
  void year(int);
  void volume(int);
  void volume(const char*);
  void title(const char*);
  void journal(const char*);
  void ISXN(const char*);
  void number(int);
  void number(const char*);
  //  string  pages(string);
  int from(int);
  int to(int);

  Publication* resolve(Resolver &);
  Publication* resolve(hash_map<string, list<NameRecord*>, Hash<string> > &nr,
		       hash_map<string, list<Annotation*>, Hash<string> > &an,
		       hash_map<string, list<Citation*>, Hash<string> > &ci,
		       hash_map<string, list<Publication*>, Hash<string> > &pub,
		       hash_map<string, list<Author*>, Hash<string> > &au,
		       hash_map<string, list<d_Ref<NameRecord>*>, Hash<string> > &unr,
		       hash_map<string, list<d_Ref<Annotation>*>, Hash<string> > &uan,
		       hash_map<string, list<d_Ref<Citation>*>, Hash<string> > &uci,
		       hash_map<string, list<d_Ref<Publication>*>, Hash<string> > &upub,
		       hash_map<string, list<d_Ref<Author>*>, Hash<string> > &uau);

  bool merge(Publication *p);

  string persistentID(void) const;
  static string emptyPID(void);
  string authorWithYear(void) const;

  list<d_Ref<Author> > authorlist;
  list<d_Ref<Citation> > citations;

  //Are Annotation objects really members of Publication?
  list<d_Ref<Annotation> > annotations;

  // Each name record entity is `described' in single publicaton entity,
  // and hence it is natural that each instance of NameRecord class
  // belongs to single object of Publication class uniquely.
  // It means that Publication instance holds a list object of NameRecord
  // instances which are described in the publication entity, no as list of
  // pointer to NameRecord object.  However, main purpose of the nomenclature
  // database is navigation in name space; neither collection of publications 
  // nor list of taxonomists with CV.  For this reason, Publication class has
  // list of pointer to NameRecord, but not object of NameRecord.  Publication
  // and Author are exist only for identification in name space, but not the 
  // object of the database.  We need these records to identify NameRecord 
  // instances, but ounece identified, we do not need read them on the memory.

  //I/O
  friend istream& operator>>(istream & s, Publication& p);
  friend ostream& operator<<(ostream & s, const Publication& p);
  ostream& printTo(ostream & os) const;

  // comparators
  friend bool operator==(const Publication&, const Publication&);
  friend bool operator!=(const Publication&, const Publication&);
  friend bool operator <(const Publication&, const Publication&);

private:
  static const char * const _classname = "Publication";
  char *_authors;
  char *_affiliations;

  char *_title; // title of book or journal name etc.
  char *_ISXN; // title of book or journal name etc.
  int _year; // year of publication
  char * _volume; // voloule
  char * _number; // number
  int _from;  //beginning page
  int _to;    //ending page
  Date _received, _revised, _accepted; // date of receive, reviced and accepted

  void initAuth(string, int);
  void _Publication(const Publication *);
  void _Publication(const char *);
  void _Publication(const string&);
  void _analyzeAuthors(const char*);
};

inline ostream& Publication::printTo(ostream & os) const { return os << *(Publication*)this; }

#if (!defined(_INCLUDED_CITATION_H))
#include "Citation.h"
#endif

#include "NameRecord.h"
#include "Annotation.h"

#if (!defined(_INCLUDED_AUTHOR_H))
#include "Author.h"
#endif

#if (!defined(_INCLUDED_RESOLVER_H))
#include "Resolver.h"
#endif

//constructors
inline Publication::Publication(const Publication& p)
  :d_Object(_classname)/*, 
   _authors(strdup()), _affiliations(strdup()),_title(strdup()),
   _volume(strdup()), _number(strdup())*/
 {_Publication(&p);}

inline Publication::Publication(const Publication* p)
  :d_Object(_classname) {_Publication(p);}

inline Publication & Publication::operator=(const Publication& p)
{_Publication(&p);return *this;}

//I/O
inline istream& operator>>(istream & s, Publication &p)
{ NamedField nf; s >> nf; p = Publication(&nf); return s;}

inline bool operator!=(const Publication &p1, const Publication &p2)
{return !(p1 == p2);}
//access
inline string Publication::authors(void) const
{ return string(_authors);}

inline Date Publication::recieved(void) const
{ return _received; }

inline Date Publication::revised(void) const
{ return _revised; }

inline Date Publication::accepted(void) const
{ return _accepted; }

inline int Publication::year(void) const { return _year;}

inline string Publication::volume(void) const
{ return string(_volume); }

inline string Publication::title(void) const
{ return string(_title); }

inline string Publication::journal(void) const
{ return string(_title); }

inline string Publication::ISXN(void) const
{ return string(_ISXN); }

inline string Publication::number(void) const
{ return string(_number); }

inline int Publication::from(void) const
{ return _from; }

inline int Publication::to(void) const
{ return _to; }

inline void Publication::title(const char *t)
{ delete [] _title; _title = new_strdup::strdup(t);}

inline void Publication::journal(const char* j)
{ title(j); }

inline void Publication::ISXN(const char *i)
{ delete [] _ISXN; _ISXN = new_strdup::strdup(i);}

inline void Publication::number(const char* v)
{delete [] _number; _number = new_strdup::strdup(v); }

inline void Publication::volume(const char* v)
{delete [] _volume; _volume = new_strdup::strdup(v); }


//#include "NameRecord.h"

#endif //_INCLUDED_PUBLICATION_H


