/*
 *
 * socketstream.cxx: an implementation of socketsteam for C++
 * 
 *
 * Copyright (c) 1999 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: socketstream.cxx,v 1.1 1999/03/17 13:58:57 nozomi Exp $
 *	$Log: socketstream.cxx,v $
 *	Revision 1.1  1999/03/17 13:58:57  nozomi
 *	Initial version
 *	
 *	
 *
 */
#include "socketstream.h"
#include <unistd.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <arpa/inet.h>  //inet_aton()

#if (!defined(UNIX_PATH_MAX))
#define UNIX_PATH_MAX 108
#endif

Socket::Socket(void) 
  : _socket(-1), _connect(-1), _listen(-1),
    _address(NULL), _length(NULL) {}

Socket::~Socket(void) { if(_socket > -1) ::close (_socket); }

Socket::Socket(int domain, int type, int protocol)
  : _socket(::socket(domain, type, protocol)), _connect(-1), _listen(-1),
    _address(NULL), _length(NULL) {}

Socket::Socket(int domain, int type, int protocol, sockaddr* address, socklen_t *length)
  : _socket(::socket(domain, type, protocol)), _connect(-1), _listen(-1),
    _address(address), _length(length) {}

int Socket::accept(void)
{
  if(_listen)
    return -1;
  return _connect = ::accept(_socket, _address, _length);
}

unixsocket::unixsocket(void)
  : Socket(PF_UNIX, SOCK_STREAM, 0, (sockaddr*)&_address, &_addrLength),
    _addrLength(0)
{_address.sun_family =AF_UNIX;  *_address.sun_path = 0; }

int unixsocket::_unixsocket(const char* s)
{
  _address.sun_family = AF_UNIX;
  *_address.sun_path = 0; 
  if(strlen(s) > UNIX_PATH_MAX)
    return 0;
  strcpy(_address.sun_path, s);
  return _addrLength = sizeof(_address.sun_family) + sizeof(_address.sun_path);
}

unixsocket::unixsocket(const char* s)
  : Socket(PF_UNIX, SOCK_STREAM, 0, (sockaddr*)&_address, &_addrLength), _addrLength(0)
{ _unixsocket(s); connect();}

unixsocket::unixsocket(const char* s, int queue)
  : Socket(PF_UNIX, SOCK_STREAM, 0, (sockaddr*)&_address, &_addrLength), _addrLength(0)
{ 
  unlink(s);
  _unixsocket(s);
  if(bind()) return;
  listen(queue);
}

ip4socket::ip4socket(void)
  : Socket(PF_INET, SOCK_STREAM, 0, (sockaddr*)&_address, &_addrLength),
    _addrLength(0)
{_address.sin_family =AF_INET;  _address.sin_port = 0;}

void ip4socket::_ip4socket(int port)
{
  _address.sin_family = AF_INET;
  _address.sin_port = htons(port); 
  _addrLength = sizeof(_address);
}

void ip4socket::_ip4socket(hostent *host, int port)
{
  if(!host) return;
  memcpy(&(_address.sin_addr), host->h_addr_list[0], sizeof(_address.sin_addr));
  _ip4socket(port);
  connect();
}

ip4socket::ip4socket(hostent *host, int port)
  : Socket(PF_INET, SOCK_STREAM, 0, (sockaddr*)&_address, &_addrLength), _addrLength(0)
{ _ip4socket(host, port);}


ip4socket::ip4socket(const char *host, int port)
  : Socket(PF_INET, SOCK_STREAM, 0, (sockaddr*)&_address, &_addrLength), _addrLength(0)
{ 
  hostent *_host;
  in_addr inaddr;
  if(inet_aton(host, &inaddr))
    _host = gethostbyaddr((char *)&inaddr, sizeof(inaddr), AF_INET);
  else
    _host = gethostbyname(host);
  if(!_host) return;
  _ip4socket(_host, port);
}

ip4socket::ip4socket(int port, int queue)
  : Socket(PF_INET, SOCK_STREAM, 0, (sockaddr*)&_address, &_addrLength), _addrLength(0)
{ 
  _ip4socket(port);
  memset(&(_address.sin_addr), 0, sizeof(_address.sin_addr));
  int opt = 1;
  setsockopt(socket(), SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt));
  if(bind()) return;
  listen(queue);
}


socketstream::socketstream(void) : fstream(), _socket(NULL){}

socketstream::socketstream(Socket* socket)
  : fstream(), _socket(socket){}

int socketstream::connect(void)
{
  if(0 == _socket->connect()) attach();
  return _socket->_connect;
}

#if 0 //sigh.....
socketstream::operator fstream(void) const
{
  if(_socket->listen) return fstream(*this);
  else return fstream(accept());
}
#endif

unixstream::unixstream(void)
  : socketstream((Socket*)&_socket), _socket(){}

unixstream::unixstream(const char* s)
  : socketstream((Socket*)&_socket), _socket(s){attach();}

unixstream::unixstream(const char* s, int queue)
  : socketstream((Socket*)&_socket), _socket(s, queue){}


ip4stream::ip4stream(void)
  : socketstream((Socket*)&_socket), _socket(){}

ip4stream::ip4stream(hostent *host, int port)
  : socketstream((Socket*)&_socket), _socket(host, port){attach();}

ip4stream::ip4stream(const char *host, int port)
  : socketstream((Socket*)&_socket), _socket(host, port){attach();}

ip4stream::ip4stream(int port, int queue)
  : socketstream((Socket*)&_socket), _socket(port, queue){}

#if 0
ax25stream::ax25stream(void)
  : socketstream(PF_AX25, SOCK_STREAM, 0)
{
}

ipxstream::ipxstream(void)
  : socketstream(PF_IPX, SOCK_STREAM, 0)
{
}


atalkstream::atalkstream(void)
  : socketstream(PF_APPLETALK, SOCK_STREAM, 0)
{
}

netromstream::netromstream(void)
  : socketstream(PF_NETROM, SOCK_STREAM, 0)
{
}
#endif
