/*
 *
 * socketstream.h: definition of socketsteam for C++
 * 
 *
 * Copyright (c) 1999 Nozomi `James' Ytow
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Where this Software is combined with software released under the terms of 
 * the GNU Public License ("GPL") and the terms of the GPL would require the 
 * combined work to also be released under the terms of the GPL, the terms
 * and conditions of this License will apply in addition to those of the
 * GPL with the exception of any terms or conditions of this License that
 * conflict with, or are expressly prohibited by, the GPL.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: socketstream.h,v 1.3 1999/03/23 14:21:24 nozomi Exp $
 *	$Log: socketstream.h,v $
 *	Revision 1.3  1999/03/23 14:21:24  nozomi
 *	socklen_t seems uncommon
 *	
 *	Revision 1.2  1999/03/17 13:59:34  nozomi
 *	Initial working version
 *	
 *	Revision 1.1  1999/03/15 16:43:01  nozomi
 *	gcc 2.7 support macro
 *	
 *	Revision 1.0  1999/03/14 02:25:15  nozomi
 *	Initial version, though aka MkII
 *	
 *
 */

#if (!defined( __socketstream_h))
#define  __socketstream_h

#if (__GNUC_MINOR__ > 7)
#include <fstream>
#else
#include <fstream.h>
#endif

#include <sys/socket.h>
#include <sys/un.h>

#if (!defined(socklen_t))
#define socklen_t unsigned int 
#endif

class Socket{
  friend class socketstream;
public:
  //constructors and the destractor
  Socket(void);
  Socket(int domain, int type, int protocol);
  Socket(int domain, int type, int protocol, sockaddr* address, socklen_t *length);
  ~Socket(void);

  //accesser
  int socket(void) const;
  int listen(void) const;
  sockaddr *address(void);
  socklen_t *length(void);
  void address(sockaddr *address);
  void length(socklen_t *length);

  //for server
  int listen(int queue);
  int accept(void);
  int unlisten(void);

  //low level functions
  int bind(void);
  int connect(void);
  int disconnect(void);

private:
  int _socket;
  int _connect;
  int _listen;
  sockaddr *_address;
  socklen_t *_length;
};

class unixsocket:public Socket {
  friend class unixstream;
public:
  //constructors and the destractor
  unixsocket(void);
  unixsocket(const char* s);
  unixsocket(const char* s, int queue);

private:
  sockaddr_un _address;
  socklen_t _addrLength;
  int _unixsocket(const char* s);
};

#include <sys/socket.h>   //sockaddr_in shutdonw
#include <netinet/in.h> //sockaddr_in, hto and nto families
#include <netdb.h>  //hsotent, gethostbyname, gethostbyaddr

class ip4socket:public Socket {
  friend class ip4stream;
public:
  //constructors and the destractor
  ip4socket(void);
  ip4socket(int port, int queue);
  ip4socket(const char* host, int port);
  ip4socket(hostent *host, int port);

private:
  sockaddr_in _address;
  socklen_t _addrLength;
  void _ip4socket(int port);
  void _ip4socket(const char *host, int port);
  void _ip4socket(hostent *host, int port);
};


class socketstream: public fstream {
public:
  //constructors and the destractor
  socketstream(void);
  socketstream(Socket* socket);

  //accesser
  int socket(void) const;

  int accept(void);
  int bind(void);
  int connect(void);
  int listen(int queue);
  void attach(void);
private:
  Socket *_socket;
};

class unixstream: public socketstream {
public:
  unixstream(void);
  unixstream(const char* s);
  unixstream(const char* s, int queue);
private:
  unixsocket _socket;
};

class ip4stream: public socketstream {
public:
  ip4stream(void);
  ip4stream(const char* host, int port);
  ip4stream(hostent *host, int port);
  ip4stream(int port, int queue);
private:
  ip4socket _socket;
};

#if 0
class ax25stream: public socketstream {
public:
  ax25stream(void);
  ~ax25stream(void);
};

class ipxstream: public socketstream {
public:
  ipxstream(void);
  ~ipxstream(void);
};

class atalkstream: public socketstream {
public:
  atalkstream(void);
  ~atalkstream(void);
};

class netromstream: public socketstream {
public:
  netromstream(void);
  ~netromstream(void);
};
#endif

#include <unistd.h>
inline int Socket::socket(void) const {return _socket;}
inline int Socket::listen(void) const {return _listen;}
inline sockaddr * Socket::address(void) {return _address;}
inline void Socket::address(sockaddr *address) { _address = address;}
inline socklen_t * Socket::length(void) {return _length;}
inline void Socket::length(socklen_t *length){ _length = length;}
inline int Socket::bind(void) { return ::bind(_socket, _address, *_length);}
inline int Socket::listen(int queue) { return _listen = ::listen(_socket, queue);}
inline int Socket::connect(void) {return _connect = ::connect(_socket, _address, *_length);}
inline int Socket::unlisten(void) {return ::close(_listen);}
inline int Socket::disconnect(void) {shutdown(_connect, 2);return ::close(_connect);}

inline int socketstream::socket(void) const {return _socket->socket();}
inline int socketstream::accept(void) {return _socket->accept();}
inline int socketstream::bind(void) {return _socket->bind();}
inline int socketstream::listen(int queue) { return _socket->listen(queue);}
inline void socketstream::attach(void) { fstream::attach(_socket->_socket);}

#endif
